% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plotPair.R
\encoding{UTF-8}
\name{plotPair}
\alias{plotPair}
\title{Plot of the (estimated) simultaneous influence of two variables}
\usage{
plotPair(
  pair,
  yvarname,
  statusvarname = NULL,
  data,
  levelsorder1 = NULL,
  levelsorder2 = NULL,
  pvalue = TRUE,
  returnseparate = FALSE,
  intobj = NULL
)
}
\arguments{
\item{pair}{Character string vector of length two, where the first character string gives the name of the first member of the respective pair to plot and the second character string gives the name of the second member.}

\item{yvarname}{Name of outcome variable.}

\item{statusvarname}{Name of status variable, only applicable to survival data.}

\item{data}{Data frame containing the variables.}

\item{levelsorder1}{Optional. Order the categories of the first variable should have in the plot (if it is categorical). Character string vector, where the
i-th entry contains the name of the category that should take the i-th place in the ordering of the categories of the first variable.}

\item{levelsorder2}{Optional. Order the categories of the second variable should have in the plot (if it is categorical). Character string vector specified in an analogous
way as \code{levelsorder1}.}

\item{pvalue}{Set to \code{TRUE} (default) to add to the plot a p-value from a test for interaction effect obtained using a classical
parametric regression approach. For categorical outcomes logistic regression is used, for metric outcomes linear
regression and for survival outcomes Cox regression. See the 'Details' section of \code{\link{plotEffects}} for further details.}

\item{returnseparate}{Set to \code{TRUE} to return invisibly the two generated ggplot plots separately in the form of a list. The
latter option is useful, because it allows to manipulate the resulting plots (label size etc.) and makes it possible to consider
only one of the two plots. The default is \code{FALSE}, which results in the two plots being returned together in the form of a 
\code{ggarrange} object.}

\item{intobj}{Optional. Object of class \code{interactionfor}. If this is provided, the ordering of the categories
obtained when constructing the interaction forest will be used for categorical variables. See Hornung & Boulesteix (2021) for details.}
}
\value{
A ggplot2 plot.
}
\description{
This function allows to visualise the (estimated) bivariable influence of a single specific pair of variables on the outcome. The estimation
and plotting is performed in the same way as in \code{\link{plotEffects}}. However, \code{plotPair} does not require an \code{interactionfor} object
and can thus be used also without a constructed interaction forest.
}
\details{
See the 'Details' section of \code{\link{plotEffects}}.
}
\examples{
\dontrun{

## Load package:

library("diversityForest")



## Visualise the estimated bivariable influence of 'toothed' and 'feathers' on
## the probability of type="mammal":

data(zoo)
plotPair(pair = c("toothed", "feathers"), yvarname="type", data = zoo)



## Visualise the estimated bivariable influence of 'creat' and 'hgb' on
## survival (more precisely, on the log hazards ratio compared to the
## median effect):

library("survival")
mgus2compl <- mgus2[complete.cases(mgus2),]
plotPair(pair=c("creat", "hgb"), yvarname="futime", statusvarname = "death", data=mgus2compl)

# Problem: The outliers in the left plot make it difficult to see what is going
# on in the region with creat values smaller than about two even though the
# majority of values lie there.

# --> Solution: We re-run the above line setting returnseparate = TRUE, because
# this allows to get the two ggplot plots separately, which can then be manipulated
# to change the x-axis range in order to remove the outliers:

ps <- plotPair(pair=c("creat", "hgb"), yvarname="futime", statusvarname = "death", 
               data=mgus2compl, returnseparate = TRUE)

# Change the x-axis range:
library("ggplot2")
ps[[1]] + xlim(c(0.5,2))
# Save the plot:
# ggsave(file="mypathtofolder/FigureXY1.pdf", width=7, height=6)

# We can, for example, also change the label sizes of the second plot:
# With original label sizes:
ps[[2]]
# With larger label sizes:
ps[[2]] +  theme(axis.title=element_text(size=15))
# Save the plot:
# library("ggplot2")
# ggsave(file="mypathtofolder/FigureXY2.pdf", width=7, height=6)

}

}
\references{
\itemize{
  \item Hornung, R. & Boulesteix, A.-L. (2021). Interaction Forests: Identifying and exploiting interpretable quantitative and qualitative interaction effects. Technical Report No. 237, Department of Statistics, University of Munich. \url{https://epub.ub.uni-muenchen.de/75432/index.html}.
  \item Hornung R. (2020) Diversity Forests: Using split sampling to allow for complex split procedures in random forest. Technical Report No. 234, Department of Statistics, University of Munich. \url{https://epub.ub.uni-muenchen.de/73377/index.html}.
  }
}
\seealso{
\code{\link{plotEffects}}, \code{\link{plot.interactionfor}}
}
\author{
Roman Hornung
}
