#' Chainrule
#'
#' Calculates the partial derivatives of the function \eqn{h(x_1,x_2,...,x_{K})=f(g(x_1,x_2,...,x_{K}))} up to order four. Here \eqn{K} is the number of inputs for function \eqn{g(\cdot)}.
#' The function \eqn{f(\cdot)} can only have a single input.
#'
#'
#' @param f1 vector of first derivatives of \eqn{f(\cdot)} evaluated at \eqn{g(\cdot)}.
#' @param f2 vector of second derivatives of \eqn{f(\cdot)} evaluated at \eqn{g(\cdot)}.
#' @param f3 vector of third derivatives of \eqn{f(\cdot)} evaluated at \eqn{g(\cdot)}.
#' @param f4 vector of fourth derivatives of \eqn{f(\cdot)} evaluated at \eqn{g(\cdot)}.
#' @param g1 matrix of first derivatives of \eqn{g(\cdot)}.
#' @param g2 matrix of second derivatives of \eqn{g(\cdot)}.
#' @param g3 matrix of third derivatives of \eqn{g(\cdot)}.
#' @param g4 matrix of fourth derivatives of \eqn{g(\cdot)}.
#' @param deriv derivative of order \code{deriv}. Available are 1,2,3,4.
#' @param xg optional, index arrays for upper triangular for g, generated by \code{\link[mgcv:trind.generator]{trind.generator}}.
#'
#' @return A list with partial derivatives. The index of the list corresponds to a matrix with all partial derivatives of that order.
#'
#' @details Mostly internal function, which is helpful in calculating the partial derivatives of the loglikelihood.
#'
#'
#' @examples
#' x<-1 #For K=1, x_1 value is set to 1.
#'
#' g<-1/x #g(x_1) = 1/x
#' g1<-matrix(-1/x^2,ncol=1)
#' g2<-matrix(2/x^3,ncol=1)
#' g3<-matrix(-6/x^4,ncol=1)
#' g4<-matrix(24/x^5,ncol=1)
#'
#' f<-exp(g) #f(g(x)) = exp(g(x))
#' f1<-f2<-f3<-f4<-exp(g)
#'
#' chainrule(f1, f2, f3, f4, g1, g2, g3, g4, deriv=4)
#'
#' @export
#'
#chainrule
chainrule<-function(f1=NULL, f2=NULL, f3=NULL, f4=NULL, g1=NULL, g2=NULL, g3=NULL, g4=NULL, deriv=2, xg=NULL){
  #Check for correct inputs
  if(any(is.null(f1),is.null(f2),is.null(g1),is.null(g2))){
    stop(paste("Insufficient inputs for deriv=2", "\n", ""))
  }

  # if(any(is.null(f3),is.null(f4),is.null(g3),is.null(g4))&deriv>2){
  #   stop(paste("Insufficient inputs for deriv>2", "\n", ""))
  # }

  #Get number of parameters
  nparg<-ncol(g1)

  #Number of observations
  n<-nrow(g1)

  #Check if index arrays for upper triangular is available, else create it
  if(is.null(xg)){
    xg<-mgcv::trind.generator(nparg)
  }

  xg$i1<-1:nparg
  xg$i1r<-1:nparg

  #Initialize derivative matrix h
  h1<-matrix(NA, nrow=n, ncol=length(xg$i1r))
  h2<-matrix(NA, nrow=n, ncol=length(xg$i2r))
  h3<-matrix(NA, nrow=n, ncol=length(xg$i3r))
  h4<-matrix(NA, nrow=n, ncol=length(xg$i4r))

  for(i in 1:nparg){
    #First derivative
    h1[,i]<-f1*g1[,i]

    for(j in i:nparg){
      #Second derivative
      h2[,xg$i2[i,j]]<-f2*g1[,xg$i1[j]]*g1[,xg$i1[i]]+f1*g2[,xg$i2[i,j]]

      if(deriv>2){
        for(k in j:nparg){
          #Third derivative
          h3[,xg$i3[i,j,k]]<-f3*g1[,xg$i1[k]]*g1[,xg$i1[j]]*g1[,xg$i1[i]]+
                             f2*g2[,xg$i2[j,k]]*g1[,xg$i1[i]]+
                             f2*g1[,xg$i1[j]]*g2[,xg$i2[i,k]]+
                             f2*g1[,xg$i1[k]]*g2[,xg$i2[i,j]]+
                             f1*g3[,xg$i3[i,j,k]]

          for(l in k:nparg){
            #Fourth derivative
            h4[,xg$i4[i,j,k,l]]<-f4*g1[,xg$i1[l]]*g1[,xg$i1[k]]*g1[,xg$i1[j]]*g1[,xg$i1[i]]+
                                 f3*g2[,xg$i2[k,l]]*g1[,xg$i1[j]]*g1[,xg$i1[i]]+
                                 f3*g1[,xg$i1[k]]*g2[,xg$i2[j,l]]*g1[,xg$i1[i]]+
                                 f3*g1[,xg$i1[k]]*g1[,xg$i1[j]]*g2[,xg$i2[i,l]]+
                                 #next line
                                 f3*g1[,xg$i1[l]]*g2[,xg$i2[j,k]]*g1[,xg$i1[i]]+
                                 f2*g3[,xg$i3[j,k,l]]*g1[,xg$i1[i]]+
                                 f2*g2[,xg$i2[j,k]]*g2[,xg$i2[i,l]]+
                                 #next line
                                 f3*g1[,xg$i1[l]]*g1[,xg$i1[j]]*g2[,xg$i2[i,k]]+
                                 f2*g2[,xg$i2[j,l]]*g2[,xg$i2[i,k]]+
                                 f2*g1[,xg$i1[j]]*g3[,xg$i3[i,k,l]]+
                                 #next line
                                 f3*g1[,xg$i1[l]]*g1[,xg$i1[k]]*g2[,xg$i2[i,j]]+
                                 f2*g2[,xg$i2[k,l]]*g2[,xg$i2[i,j]]+
                                 f2*g1[,xg$i1[k]]*g3[,xg$i3[i,j,l]]+
                                 #next line
                                 f2*g1[,xg$i1[l]]*g3[,xg$i3[i,j,k]]+
                                 f1*g4[,xg$i4[i,j,k,l]]

          }
        }
      }
    }
  }

  #Write output as a list
  out<-list(h1=h1, h2=h2, h3=h3, h4=h4)

  #Return output
  return(out)
}

