#' @title
#' Implementation of the NSGA-II EMOA algorithm by Deb.
#'
#' @description
#' The NSGA-II merges the current population and the generated offspring and
#' reduces it by means of the following procedure: It first applies the non
#' dominated sorting algorithm to obtain the nondominated fronts. Starting with
#' the first front, it fills the new population until the i-th front does not fit.
#' It then applies the secondary crowding distance criterion to select the missing
#' individuals from the i-th front.
#'
#' @note
#' This is a pure R implementation of the NSGA-II algorithm. It hides the regular
#' \pkg{ecr2} interface and offers a more R like interface while still being quite
#' adaptable.
#'
#' @references
#' Deb, K., Pratap, A., and Agarwal, S. A Fast and Elitist Multiobjective Genetic
#' Algorithm: NSGA-II. IEEE Transactions on Evolutionary Computation, 6 (8) (2002),
#' 182-197.
#'
#' @keywords optimize
#'
#' @template arg_fitness_fun
#' @template arg_n_objectives
#' @template arg_n_dim
#' @template arg_minimize
#' @template arg_lower
#' @template arg_upper
#' @param mu [\code{integer(1)}]\cr
#'   Number of individuals in the population.
#'   Default is 100.
#' @param lambda [\code{integer(1)}]\cr
#'   Offspring size, i.e., number of individuals generated by variation operators
#'   in each iteration.
#'   Default is 100.
#' @template arg_mutator
#' @template arg_recombinator
#' @template arg_terminators
#' @param ... [any]\cr
#'   Further arguments passed down to fitness function.
#' @return [\code{ecr_multi_objective_result}]
#' @export
nsga2 = function(
  fitness.fun,
  n.objectives = NULL,
  n.dim = NULL,
  minimize = NULL,
  lower = NULL,
  upper = NULL,
  mu = 100L,
  lambda = mu,
  mutator = setup(mutPolynomial, eta = 25, p = 0.2, lower = lower, upper = upper),
  recombinator = setup(recSBX, eta = 15, p = 0.7, lower = lower, upper = upper),
  terminators = list(stopOnIters(100L)),
  ...) {

  res = ecr(fitness.fun = fitness.fun, n.objectives = n.objectives,
    n.dim = n.dim, minimize = minimize, lower = lower, upper = upper,
    mu = mu, lambda = lambda, representation = "float", survival.strategy = "plus",
    parent.selector = selSimple,
    mutator = mutator,
    recombinator = recombinator,
    survival.selector = selNondom,
    terminators = terminators)
  return(res)
}
