
#' Summarize
#' 
#' Summarizes an object
#'
#' The return value is object of class 'editsummary' for which a print method is implemented.
#'
#' @method summary editmatrix
#' @param object an R object
#' @param useBlocks \code{logical} Summarize each block?
#' @param ... Arguments to pass to or from other methods 
#'
#' @rdname summary
#' @export
#' @example ../examples/editmatrix.R
summary.editmatrix <- function(object, useBlocks=TRUE, ...){
    if (useBlocks){ 
        B <- blocks(object)
    } else {
        B <- list(object)
    }
    A <- array(c(
        sapply(B,nrow),
        sapply(B,function(b) sum(getOps(b) == '==')),
        sapply(B,function(b) sum(getOps(b) != '==')),
        sapply(B,function(b) length(getVars(b)) )
        ), 
        dim=c(length(B),4),
        dimnames=list(
            block = 1:length(B),
            count = c('edits','equalities','inequalities','variables')
        )
    )
    structure(A,
        class=c('editsummary'),
        type ='editmatrix',
        normalized = isNormalized(object))
}


#' Summary of edit set
#'
#'
#' @method summary editarray
#' 
#' @rdname summary
#' @export
#' @example ../examples/editarray.R
summary.editarray <- function(object, useBlocks=TRUE, ...){
    if ( useBlocks ){
        B <- blocks(object)
    } else {
        B <- list(object)
    }
    A <- array(c(
        sapply(B,nrow),
        sapply(B,function(b) length(getVars(b)))
        ),
        dim=c(length(B),2),
        dimnames=list(
            block=1:length(B),
            count=c('edits','variables')
        )
    )
    structure(A,
        class='editsummary',
        type ='editarray')
}



#'
#' @method print editsummary
#' @export
#' @keywords internal
print.editsummary <- function(x,...){
    nrm <- ''
    if ( attr(x,'type')=='editmatrix' ){
        nrm <- 'normalized'
        if (!attr(x,'normalized')) nrm = paste('non-',nrm,sep='')
    }
    cat('Summary of',nrm,attr(x,'type'),'\n')
    print(x[,,drop=FALSE])

}

#' Summarize error location
#' 
#' Generates an object of class 'locationsummary' for which a print method is implemented.
#'
#' @method summary errorLocation
#' @rdname summary
#' @export
#' @example ../examples/localizeErrors.R
summary.errorLocation <- function(object,...){
    prb <- c(0,0.5,1)
     
    err.per.var <- c(quantile(colSums(object$adapt),probs=prb,names=FALSE))
    err.per.rec <- c(quantile(rowSums(object$adapt),probs=prb,names=FALSE))

    A <- sapply(object$status[,1:5], function(x) quantile(x,probs=prb,names=FALSE))
    dimnames(A) <- list(c('min','median','max'), status=names(object$status)[1:5])
    A <- cbind(err.per.var,err.per.rec,A)
    tt <- sum(object$adapt)
    total <- c(tt,tt,colSums(object$status[,1:5]))
    te <- sum(object$status$maxDurationExceeded)
    structure(rbind(A,total),
        nexceeded = c(te,te/nrow(object$adapt)),
        user = object$user,
        timestamp = object$timestamp,
        call = as.character(as.expression(object$call)),
        class=c('locationsummary')
    )
}

#'
#' @method print locationsummary
#' @export
#' @keywords internal
print.locationsummary <- function(x,...){
    cat("Summary of 'errorLocation' object, generated by",attr(x,'user'),'at',attr(x,'timestamp'),'\n')
    cat("by calling",attr(x,'call'),'\n\n')
    cat('Results:\n')
    print(x[,,drop=FALSE])
    cat(attr(x,'nexceeded')[1],
        ' records exceeded maximum search time (',
        round(attr(x,'nexceeded')[2]*100,2), '%)\n',sep='')

}





