% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/eicm-fit.r
\name{eicm.fit}
\alias{eicm.fit}
\title{Estimate a EICM model}
\usage{
eicm.fit(occurrences, env = NULL, traits = NULL, intercept = TRUE,
  n.latent = 0, forbidden = NULL, mask.sp = NULL,
  exclude.prevalence = 0, options = NULL, initial.values = NULL,
  regularization = c(ifelse(n.latent > 0, 0.5, 0), 0.5),
  regularization.type = "ridge", fast = FALSE,
  optim.method = ifelse(fast, "ucminf", "L-BFGS-B"), optim.control = if
  (fast) list(trace = 1, maxeval = 10000, gradstep = c(0.001, 0.001), grtol
  = 0.1) else list(trace = 1, maxit = 10000, ndeps = 1e-04))
}
\arguments{
\item{occurrences}{a binary (0/1) sample x species matrix, possibly including NAs.}

\item{env}{an optional sample x environmental variable matrix, for the known environmental predictors.}

\item{traits}{an optional species x trait matrix. Currently, it is only used for excluding
species interactions \emph{a priori}.}

\item{intercept}{logical specifying whether to add a column for the species-level intercepts.}

\item{n.latent}{the number of latent variables to estimate.}

\item{forbidden}{a formula defining which species interactions are not to be estimated. See details.
This constraint is cumulative with other constraints (\code{mask.sp} and \code{exclude.prevalence}).}

\item{mask.sp}{a scalar or a binary square species x species matrix defining which species interactions to exclude
(0) or include (1) \emph{a priori}. If a scalar (0 or 1), 0 excludes all interactions, 1 allows all interactions.
If a matrix, species in the columns affect species in the rows, so, setting \code{mask.sp[3, 8] <- 0}
means that species #8 is assumed \emph{a priori} to not affect species #3.
This constraint is cumulative with other constraints (\code{forbidden} and \code{exclude.prevalence}).}

\item{exclude.prevalence}{exclude species interactions which are caused by species
with prevalence equal or lower than this value. This constraint is cumulative with
other constraints (\code{forbidden} and \code{mask.sp})}

\item{options}{a \code{eicm.options} object defining options for fitting. Usually not needed, use
\code{forbidden}, \code{mask.sp} and \code{exclude.prevalence} instead.}

\item{initial.values}{the starting values for all parameters. Used only for speeding up
fitting when there are previous estimates available.}

\item{regularization}{a two-element numeric vector defining the regularization lambdas used for
environmental coefficients and for species interactions respectively. See details.}

\item{regularization.type}{one of "lasso" or "ridge", defining the type of penalty to apply.}

\item{fast}{a logical defining whether to do a fast - but less accurate - estimation, or a normal estimation.}

\item{optim.method}{the optimization function to use. Should be set to the default.}

\item{optim.control}{the optimization parameters to use. Should be set to the defaults.}
}
\value{
A fitted \code{eicm} object.
}
\description{
Estimates the parameter values of a EICM model from the provided observation data.
This is the low-level estimation function. Users should use \code{\link{eicm}} instead, particularly
if estimating latent variables and species interactions.
}
\details{
By default, all species interactions are estimated. Uers can control which species interactions
are to be estimated with the arguments \code{forbidden}, \code{mask.sp} and \code{exclude.prevalence},
which place cumulative restrictions on which interactions to estimate. See \code{vignette("eicm")}
for commented examples.
}
\note{
If estimating latent variables and species interactions, use \code{\link{eicm}} instead.
}
\examples{
# Simulate some random occurrence data
nenv <- 2
nsp <- 10
nsamples <- 200

env <- matrix(rnorm(nenv * nsamples), ncol=nenv, nrow=nsamples)
env.coefs <- matrix(runif((nenv + 1) * nsp, -4, 4), nrow=nsp)
sp.coefs <- matrix(0, nrow=nsp, ncol=nsp)
sp.coefs[3, 5] <- 3
sp.coefs[4, 8] <- 2

# Define a true model
truemodel <- as.eicm(env=env, env.coefs=env.coefs, sp.coefs=sp.coefs)

# realize the model
simulated.data <- predict(truemodel, nrepetitions=1)

\donttest{
# fit the model without species interactions
fittedNoInt <- eicm.fit(simulated.data, env, mask.sp=0)

# fit the model with all species interactions
fittedInt <- eicm.fit(simulated.data, env, mask.sp=1)

# compute confidence intervals for all parameters
fittedInt <- confint(fittedInt, ncores=2)

# plot estimated parameters and confidence intervals
plot(fittedInt, type="confint", truemodel=truemodel)
}
}
\seealso{
\code{\link{eicm}}, \code{\link{confint.eicm}}, \code{\link{plot.eicm}}
}
