#' SEforR estimates the standard error for a set of parameters
#' @title Estimate the standard error for a set of parameters
#' @author Marc Girondot
#' @return A vector of standard errors estimate
#' @param NestsResult A NestsResult file generated by searchR
#' @param parameters A set of parameters if result is not provided.
#' @param fixed.parameters Another set of parameters if result is not provided.
#' @param temperatures Timeseries of temperatures formatted using formatNests(). Will replace the one in result.
#' @param derivate Function used to fit embryo growth: dydt.Gompertz, dydt.exponential or dydt.linear
#' @param test Mean and SD of size of hatchlings
#' @param weight Weights of the different nests to estimate likelihood
#' @param M0 Measure of hatchling size proxi at laying date
#' @param series The name or number of the series to be used
#' @param hessian If TRUE, will return a list with both the Hessian matrix and SE estimates
#' @description It is an alternative to MCMC using the Fisher Information Matrix.\cr
#' If too many temperatures without information are included, the estimation will fail.
#' @examples
#' \dontrun{
#' library(embryogrowth)
#' data(resultNest_4p)
#' newp <- GenerateAnchor(nests=resultNest_4p, temperatures=seq(from=20,
#'  to=35, length.out=10))
#' SE <- SEforR(NestsResult=resultNest_4p, parameters=newp)
#' SE <- structure(c(NA, NA, 1.18147243592998, 0.610526666309349, 0.496674180881528, 
#' 0.283356315073372, 0.162524360284674, 0.265653521222035, 1.33618216197102, 
#' 14.7268588035904), .Names = c("293.15", "294.816666666667", "296.483333333333", 
#' "298.15", "299.816666666667", "301.483333333333", "303.15", "304.816666666667", 
#' "306.483333333333", "308.15"))
#' plotR(result=resultNest_4p, parameters=newp, ylim=c(0,0.4), SE=SE)
#' }
#' @export

SEforR <- function(NestsResult=NULL, parameters=NULL, fixed.parameters=NULL, 
                               temperatures=NULL, derivate=NULL, 
                               test=NULL, M0=NULL, series="all",
                               weight=NULL, hessian=FALSE) {
  
  # NestsResult=NULL; parameters=NULL;  fixed.parameters=NULL; temperatures=NULL; derivate=NULL; test=NULL; M0=NULL; series="all"; weight=NULL
  # newp <- GenerateAnchor(nests=resultNest_4p, temperatures=seq(from=20, to=35, length.out=10))
  # NestsResult=resultNest_4p; parameters=newp
  
  if (class(NestsResult)=="NestsResult") {
    # temperatures est un objet Nests
    if (is.null(temperatures)) temperatures <- NestsResult$data
    if (is.null(derivate)) derivate <- NestsResult$derivate
    if (is.null(weight)) weight <- NestsResult$weight
    if (is.null(test)) test <- NestsResult$test
    if (is.null(M0)) M0 <- NestsResult$M0
    if (is.null(fixed.parameters)) fixed.parameters <- NestsResult$fixed.parameters
    if (is.null(parameters)) parameters <- NestsResult$par
  }

  if (!requireNamespace("numDeriv", quietly = TRUE)) {
    stop("numDeriv package is absent; Please install it first")
  }
  
  mathessian <- try(numDeriv::hessian(func=info.nests, 
                            x=parameters, 
                            method="Richardson", 
                            temperatures=temperatures, 
                            derivate=derivate, 
                            weight=weight,
                            test=test, 
                            M0=M0, 
                            fixed.parameters=fixed.parameters)
                    , silent=TRUE)
  if (inherits(mathessian, "try-error")) {
    res <- rep(NA, length(parameters))
    mathessian <- NULL
  } else {
    inversemathessian <- try(solve(mathessian), silent=TRUE)
    if (inherits(inversemathessian, "try-error")) {
      res <- rep(NA, length(parameters))
    } else {
      res <- diag(inversemathessian)
      res <- sapply(res, function(x) ifelse(x<0, NA, sqrt(x)))
    }
  }
  
  names(res) <- names(parameters)
  
  if (any(is.na(res))) {
    if (all(is.na(res))) {
      print("SE of parameters cannot be estimated.")
      print("Probably the model is badly fitted.")
    } else {
      print("Probably flat likelihood is observed around some parameters.")
      print("Try using GRTRN_MHmcmc() function to get the SE of these parameters.")
    }
  }
  
  if (hessian) {
    return(invisible(list(hessian=mathessian, SE=res)))
    } else {
  return(res)
      }
  
}
