% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict.engressionBagged.R
\name{predict.engressionBagged}
\alias{predict.engressionBagged}
\title{Prediction Function for Bagged Engression Models}
\usage{
\method{predict}{engressionBagged}(
  object,
  Xtest = NULL,
  type = c("mean", "sample", "quantile")[1],
  trim = 0.05,
  quantiles = 0.1 * (1:9),
  nsample = 200,
  drop = TRUE,
  ...
)
}
\arguments{
\item{object}{A trained bagged engression model returned from the engressionBagged function.}

\item{Xtest}{A matrix or data frame representing the predictors in the test set. If NULL, out-of-bag samples from the training
set are used for prediction (default: NULL).}

\item{type}{The type of prediction to make. "mean" for point estimates, "sample" for samples from the estimated distribution,
or "quantile" for quantiles of the estimated distribution (default: "mean").}

\item{trim}{The proportion of extreme values to trim when calculating the mean (default: 0.05).}

\item{quantiles}{The quantiles to estimate if type is "quantile" (default: 0.1*(1:9)).}

\item{nsample}{The number of samples to draw if type is "sample" (default: 200).}

\item{drop}{A boolean indicating whether to drop dimensions of length 1 from the output (default: TRUE).}

\item{...}{additional arguments (currently ignored)}
}
\value{
A matrix or array of predictions.
#'
}
\description{
This function computes predictions from a trained bagged Engression model. It allows for the generation of point estimates,
quantiles, or samples from the estimated distribution.
}
\examples{
\donttest{
  n = 1000
  p = 5
  X = matrix(rnorm(n*p),ncol=p)
  Y = (X[,1]+rnorm(n)*0.1)^2 + (X[,2]+rnorm(n)*0.1) + rnorm(n)*0.1
  Xtest = matrix(rnorm(n*p),ncol=p)
  Ytest = (Xtest[,1]+rnorm(n)*0.1)^2 + (Xtest[,2]+rnorm(n)*0.1) + rnorm(n)*0.1

  ## fit bagged engression object
  engb = engressionBagged(X,Y,K=3)
  print(engb)

  ## prediction on test data
  Yhat = predict(engb,Xtest,type="mean")
  cat("\n correlation between predicted and realized values:  ", signif(cor(Yhat, Ytest),3))
  plot(Yhat, Ytest,xlab="estimated conditional mean", ylab="observation")

  ## out-of-bag prediction
  Yhat_oob = predict(engb,type="mean")
  cat("\n correlation between predicted and realized values on oob data:  ")
  print(signif(cor(Yhat_oob, Y),3))
  plot(Yhat_oob, Y,xlab="estimated conditional mean", ylab="observation")

  ## quantile prediction
  Yhatquant = predict(engb,Xtest,type="quantiles")
  ord = order(Yhat)
  matplot(Yhat[ord], Yhatquant[ord,], type="l", col=2,lty=1,xlab="prediction", ylab="observation")
  points(Yhat[ord],Ytest[ord],pch=20,cex=0.5)

  ## sampling from estimated model
  Ysample = predict(engb,Xtest,type="sample",nsample=1)
     
}

}
