% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/model_bedload.R
\name{model_bedload}
\alias{model_bedload}
\title{Model the seismic spectrum due to bedload transport in rivers}
\usage{
model_bedload(
  gsd,
  d_s,
  s_s,
  r_s,
  q_s,
  h_w,
  w_w,
  a_w,
  f = c(1, 100),
  r_0,
  f_0,
  q_0,
  e_0,
  v_0,
  x_0,
  n_0,
  n_c,
  res = 100,
  eseis = FALSE,
  ...
)
}
\arguments{
\item{gsd}{\code{data frame} grain-size distribution function. Must be 
provided as data frame with two variables: grain-size class (first column)
and vol- or wgt.-percentage per class (second column). See examples for 
details.}

\item{d_s}{\code{Numeric} value, mean sediment grain diameter (m). 
Alternative to \code{gsd}.}

\item{s_s}{\code{Numeric} value, standard deviation of sediment grain 
diameter (m). Alternative to \code{gsd}.}

\item{r_s}{\code{Numeric} value, specific sediment density (kg / m^3)}

\item{q_s}{\code{Numeric} value, unit sediment flux (m^2 / s)}

\item{h_w}{\code{Numeric} value, fluid flow depth (m)}

\item{w_w}{\code{Numeric} value, fluid flow width (m)}

\item{a_w}{\code{Numeric} value, fluid flow inclination angle (radians)}

\item{f}{\code{Numeric} vector, frequency range to be modelled. 
If of length two the argument is interpreted as representing the lower and 
upper limit and the final length of the frequency vector is set by the 
argument \code{res}. If \code{f} contains more than two values it is 
interpreted as the actual frequency vector and the value of \code{res} is 
ignored.}

\item{r_0}{\code{Numeric} value, distance of seismic station to source}

\item{f_0}{\code{Numeric} value, reference frequency (Hz)}

\item{q_0}{\code{Numeric} value, ground quality factor at \code{f_0}.
"Reasonable value may be \code{20}" (Tsai et al. 2012).}

\item{e_0}{\code{Numeric} value, exponent characterizing quality factor 
increase with frequency (dimensionless). "Reasonable value may be 
\code{0}" (Tsai et al. 2012).}

\item{v_0}{\code{Numeric} value, phase speed of the Rayleigh wave at 
\code{f_0} (m/s). Assuming a shear wave velocity of about 2200 m/s, 
Tsai et al. (2012) yield a value of 1295 m/s for this parameter.}

\item{x_0}{\code{Numeric} value, exponent of the power law variation of 
Rayleigh wave velocities with frequency}

\item{n_0}{\code{Numeric} vector of length two, Greens function 
displacement amplitude coefficients. Cf. N_ij in eq. 36 in Gimbert et 
al. (2014)}

\item{n_c}{\code{Numeric} value, option to include single particle hops 
coherent in time, causing spectrum modulation due to secondary effects. 
Omitted is no value is specified, here. Usual values may be between 2 and 
4.}

\item{res}{\code{Numeric} value, output resolution, i.e. length of the 
spectrum vector. Default is 1000.}

\item{eseis}{\code{Character} value, option to return an eseis object 
instead of a data frame. Default is \code{FALSE}.}

\item{\dots}{Further arguments passed to the function.}
}
\value{
\code{eseis} object containing the modelled spectrum.
}
\description{
The function calculates a seismic spectrum as predicted by the model 
of Tsai et al. (2012) for river bedload transport. The code was written to 
R by Sophie Lagarde and integrated to the R package 'eseis' by Michael 
Dietze.
}
\details{
The model uses a set of predefined constants. These can also be changed
by the user, using the \code{...} argument:
\itemize{
  \item \code{g = 9.81}, gravitational acceleration (m/s^2)
  \item \code{r_w = 1000}, fluid specific density (kg/m^3)
  \item \code{k_s = 3 * d_50}, roughness length (m)
  \item \code{log_lim = c(0.0001, 100), limits of grain-size distribution 
  function template (m)}
  \item \code{log_length = 10000, length of grain-size distribution 
  function template}
  \item \code{nu = 10^(-6)}, specific density of the fluid (kg/m^3)
  \item \code{power_d = 3}, grain-size power exponent
  \item \code{gamma = 0.9}, gamma parameter, after Parker (1990)
  \item \code{s_c = 0.8}, drag coefficient parameter
  \item \code{s_p = 3.5}, drag coefficient parameter
  \item \code{c_1 = 2 / 3}, inter-impact time scaling, after 
  Sklar & Dietrich (2004)
}

When no user defined grain-size distribution function is provided,the 
function calculates the raised cosine distribution function as defined 
in Tsai et al. (2012) using the default range and resolution as specified 
by \code{log_lim} and \code{log_length} (see additional arguments list 
above). These default values are appropriate for mean sediment sizes 
between 0.001 and 10 m and log standard deivations between 0.05 and 1. 
When more extreme distributions are to be used, it is necessary to either 
adjust the arguments \code{log_lim} and \code{log_length} or use a 
user defined distribution function.
}
\examples{

## calculate spectrum (i.e., fig. 1b in Tsai et al., 2012)
p_bedload <- model_bedload(d_s = 0.7,
                           s_s = 0.1,
                           r_s = 2650,
                           q_s = 0.001,
                           h_w = 4,
                           w_w = 50,
                           a_w = 0.005,
                           f = c(0.1, 20),
                           r_0 = 600,
                           f_0 = 1,
                           q_0 = 20,
                           e_0 = 0,
                           v_0 = 1295,
                           x_0 = 0.374,
                           n_0 = 1,
                           res = 100,
                           eseis = TRUE)

## plot spectrum
plot_spectrum(data = p_bedload, 
              ylim = c(-170, -110))
              
## define empiric grain-size distribution
gsd_empiric <- data.frame(d = c(0.70, 0.82, 0.94, 1.06, 1.18, 1.30),
                          p = c(0.02, 0.25, 0.45, 0.23, 0.04, 0.00))
                  
## calculate spectrum
p_bedload <- model_bedload(gsd = gsd_empiric,
                           r_s = 2650,
                           q_s = 0.001,
                           h_w = 4,
                           w_w = 50,
                           a_w = 0.005,
                           f = c(0.1, 20),
                           r_0 = 600,
                           f_0 = 1,
                           q_0 = 20,
                           e_0 = 0,
                           v_0 = 1295,
                           x_0 = 0.374,
                           n_0 = 1,
                           res = 100,
                           eseis = TRUE)
                  
## plot spectrum
plot_spectrum(data = p_bedload, 
              ylim = c(-170, -110))
              
## define mean and sigma for parametric distribution function
d_50 <- 1
sigma <- 0.1

## define raised cosine distribution function following Tsai et al. (2012)
d_1 <- 10^seq(log10(d_50 - 5 * sigma), 
              log10(d_50 + 5 * sigma), 
              length.out = 20)

sigma_star <- sigma / sqrt(1 / 3 - 2 / pi^2)

p_1 <- (1 / (2 * sigma_star) * 
          (1 + cos(pi * (log(d_1) - log(d_50)) / sigma_star))) / d_1
p_1[log(d_1) - log(d_50) > sigma_star] <- 0
p_1[log(d_1) - log(d_50) < -sigma_star] <- 0
p_1 <- p_1 / sum(p_1)

gsd_raised_cos <- data.frame(d = d_1,
                             p = p_1)
             
}
\references{
Tsai, V. C., B. Minchew, M. P. Lamb, and J.-P. Ampuero (2012), A 
physical model for seismic noise generation from sediment transport in 
rivers, Geophys. Res. Lett., 39, L02404, doi:10.1029/2011GL050255.
}
\author{
Sophie Lagarde, Michael Dietze
}
\keyword{eseis}
