set_default <- function(obj, repl) {
  arg_names <- names(repl)
  for (i in seq_along(arg_names)) {
    if (is.null(obj[[arg_names[[i]]]])) {
      obj[[arg_names[[i]]]] <- repl[[i]]
    }
  }
  obj
}

#'Plot Method for Class \code{"esemifar_fc"}
#'
#'Create basic R plots for forecasting objects of class \code{"esemifar_fc"}.
#'
#'@param x an object of class \code{"esemifar_fc"}, for example generated by a
#'call to \code{\link{predict.esemifar}}.
#'@param y currently without use; for compatibility only.
#'@param t a numeric vector with series of time points for observations and
#'forecasts.
#'@param ... further arguments of \code{\link[stats]{plot.ts}} to adjust
#'for example the axis limits via \code{xlim} and \code{ylim}.
#'
#'@details
#'This is a plot method to visualize the forecasting results for an ESEMIFAR
#'model. Common plot arguments can be implemented to change the
#'appearance.
#'
#'@return
#'This method returns \code{NULL}.
#'
#'@author
#'\itemize{
#'\item Dominik Schulz (Research Assistant) (Department of Economics, Paderborn
#'University), \cr
#'Author and Package Creator
#'}
#'
#'@export
#'
#'@importFrom utils head
#'
#'@examples
#'lgdp <- log(esemifar::gdpG7$gdp)
#'est <- tsmoothlm(lgdp, pmax = 1, qmax = 1)
#'# Under normality
#'fc <- predict(est, n.ahead = 10, method = "norm", expo = TRUE)
#'plot(fc)
#'

plot.esemifar_fc <- function(x, y = NULL, t = NULL, ...) {

  dots <- list(...)
  obs <- x$obs
  if (is.null(t)) {
    t <- 1:length(c(obs, x$mean))
  }
  y_low_def <- min(obs, x$lower)
  y_up_def <- max(obs, x$upper)
  defaults <- list(
    xlab = "Time",
    main = paste0('The observations together with point and interval forecasts'),
    ylab = "Value",
    xlim = c(t[[1]], utils::tail(t, 1)),
    ylim = c(y_low_def, y_up_def)
  )
  dots <- set_default(dots, defaults)
  if (!is.null(dots[["col"]])) {
    col <- dots$col
    col1 <- col[[1]]
    col2 <- col[[2]]
    dots[["col"]] <- NULL
  } else {
    col1 <- "black"
    col2 <- "blue"
  }
  dots[["type"]] = "n"
  dots[["x"]] <- 0
  dots[["y"]] <- 0
  do.call(plot, args = dots)
  m <- length(x$lower[1, ])
  t_fc <- tail(t, length(x$mean))

  j <- 0

  if (m > 1) {
    for (i in m:2) {
      graphics::polygon(c(t_fc, rev(t_fc)), c(c(x$lower[, i]), rev(c(x$lower[, (i - 1)]))),
              border = NA, col = ggplot2::alpha(col2, 0.1 + j * 0.15))
      graphics::polygon(c(t_fc, rev(t_fc)), c(c(x$upper[, i]), rev(c(x$upper[, i - 1]))),
              border = NA, col = ggplot2::alpha(col2, 0.1 + j * 0.15))
      j <- j + 1
    }
  }

  graphics::polygon(c(t_fc, rev(t_fc)), c(c(x$lower[, 1]), rev(c(x$upper[, 1]))),
    border = NA, col = ggplot2::alpha(col2, 0.1 + j * 0.15))

  graphics::lines(head(t, length(obs)), c(x$obs), col = col1)
  graphics::lines(tail(t, length(x$mean)), c(x$mean), col = col2)

}
