% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gev_bayes.R
\name{gev_bayes}
\alias{gev_bayes}
\title{GEV Bayesian Inference with Adjustment for Missing Data}
\usage{
gev_bayes(
  data,
  block_length,
  block,
  adjust = TRUE,
  discard = 0,
  init = "quartiles",
  prior = revdbayes::set_prior(prior = "flat", model = "gev"),
  n = 1000,
  ...
)
}
\arguments{
\item{data}{Either
\itemize{
\item a numeric vector containing a time series of raw data,
\item an object returned from \code{\link{block_maxima}}, a list with components
\code{maxima}, \code{notNA} and \code{n},
\item a data frame or named list containing the same information, that is, the
variables \code{maxima}, \code{notNA} and \code{n}, as an object returned from
\code{\link{block_maxima}}, such as the data frame \code{\link{BrestSurgeMaxima}}.
}}

\item{block_length}{A numeric scalar. Used calculate the maxima of disjoint
blocks of \code{block_length} contiguous values in the vector \code{data}.
If \code{length(data)} is not an integer multiple of \code{block_length} then
the values at the end of \code{data} that do not constitute a complete block
of length \code{block_length} are discarded, without warning.}

\item{block}{A numeric vector with the same length as \code{data}. The value of
\code{block[i]} indicates the block into which \code{data[i]} falls. For example,
\code{block} could provide the year in which observation \code{i} was observed.}

\item{adjust}{A logical scalar or a numeric scalar in \verb{[0, 100]}.
\itemize{
\item If \code{adjust = TRUE} then the adjustment, described in \strong{Details}, for the
numbers of non-missing values underlying each block maximum is performed.
\item If \code{adjust = FALSE} then no adjustment is made, that is, the block maxima
are treated as if the underlying raw data have no missing values.
}}

\item{discard}{A numeric scalar. Any block maximum for which greater than
\code{discard} percent of the underlying raw values were missing is discarded.
Whether or not an adjustment for missingness is made for the block maxima
that remain is determined by \code{adjust}.}

\item{init}{Either a character scalar, one of \code{"quartiles"} or \code{"moments"},
or a numeric vector of length 3 giving initial estimates of the GEV
location, scale and shape parameters: \eqn{\mu}, \eqn{\sigma} and
\eqn{\xi}. If \code{init = "quartiles"} then initial estimates of \eqn{\mu} and
\eqn{\sigma} are based on sample quartiles of block maxima, ignoring the
underlying numbers of non-missing raw data, and a value of 0 for
\eqn{\xi}. If \code{init = "moments"} then instead we use the sample mean and
variance of these maxima and an initial value of 0.1 for \eqn{\xi}.}

\item{prior}{Specifies a prior distribution for the GEV parameters. This is
most easily set using \code{\link[revdbayes:set_prior]{revdbayes::set_prior}}. The default is a prior
\eqn{\pi(\mu, \sigma, \xi) \propto \sigma^{-1}} for \eqn{\sigma > 0}. See
\code{\link[revdbayes:set_prior]{revdbayes::set_prior}} for details.}

\item{n}{A non-negative integer. The number of values to simulate from the
posterior distribution for \eqn{(\mu, \sigma, \xi)}.}

\item{...}{Further arguments to be passed to \code{\link[rust:ru]{rust::ru}}.}
}
\value{
An object returned from \code{\link[rust:ru]{rust::ru}}. The following components are
added to this list
\itemize{
\item \code{model}: = \code{"gev"}.
\item \verb{data,prior}: the inputs \code{data} and \code{prior}.
\item \code{call}: the call to \code{gev_bayes}.
\item \code{maxima}: the vector of block maxima used to fit the model.
\item \code{notNA}: the number of non-missing raw values on which the maxima in
\code{maxima} are based.
\item \code{n}: the maximal block length, that is, the largest number of values that
could have been observed in each of these blocks.
\item \code{adjust}: a logical scalar indicating whether or not the adjustment in
the \strong{Details} section of \code{\link{gev_mle}} was performed. This is \code{TRUE}
only if the input argument \code{adjust} was \code{TRUE}.
\item \verb{adjust,discard} : the values of these input arguments.
}

The class of the returned object is
\code{c("evpost", "ru", "bayes", "evmissing")}.
Objects of class \code{"evpost"} have \code{\link[revdbayes:print.evpost]{print}},
\code{\link[revdbayes:summary.evpost]{summary}} and \code{\link[revdbayes:plot.evpost]{plot}}
S3 methods.
}
\description{
Performs Bayesian inference using a GEV distribution using block maxima,
with the option to make an adjustment for the numbers of non-missing raw
values in each block.
}
\details{
The likelihood described in \code{\link{gev_mle}} is combined with the prior
density provided by \code{prior} to produce, up to proportionality, a
posterior density for \eqn{(\mu, \sigma, \xi)}.

A function to evaluate the log-posterior is passed to \code{\link[rust:ru]{rust::ru}} to
simulate a random sample from this posterior distribution using the
generalised ratio-of-uniforms method, using relocation of the mode of the
density to the origin to increase efficiency. The value of \code{init} is used
as an initial estimate in a search for the posterior mode. Arguments to
\code{\link[rust:ru]{rust::ru}} can be passed via \code{...}. The default setting is
\code{trans = "none"}, that is, no transformation of the margins, and
\code{rotate = TRUE}, rotation of the parameter axes to improve isotropy
with a view to increasing efficiency.
}
\examples{
## Simulate data with missing values

set.seed(24032025)
blocks <- 50
block_length <- 365

# Simulate raw data from an exponential distribution
sdata <- sim_data(blocks = blocks, block_length = block_length)

block_length <- sdata$block_length
# Sample from the posterior based on block maxima from full data
post1 <- gev_bayes(sdata$data_full, block_length = block_length)
summary(post1)

# Sample with adjustment for the number of non-missing values per block
post2 <- gev_bayes(sdata$data_miss, block_length = block_length)
summary(post2)

# Do not make the adjustment
post3 <- gev_bayes(sdata$data_miss, block_length = block_length,
                   adjust = FALSE)
summary(post3)

# Remove all block maxima with greater than 25\% missing values and
# do not make the adjustment
post4 <- gev_bayes(sdata$data_miss, block_length = block_length,
                   adjust = FALSE, discard = 25)
summary(post4)

## Brest sea surge data

post <- gev_bayes(BrestSurgeMaxima)
summary(post)
plot(post)
}
