#' Extract fitted values from models
#' 
#' Extracts the fitted values from each of the models in a mable. A tsibble will
#' be returned containing these fitted values. Fitted values will be 
#' automatically back-transformed if a transformation was specified.
#' 
#' @param object A mable or time series model.
#' @param ... Other arguments passed to the model method for `fitted()`
#' 
#' @importFrom stats fitted
#' @export
fitted.mdl_df <- function(object, ...){
  mbl_vars <- mable_vars(object)
  kv <- key_vars(object)
  object <- mutate(as_tibble(object), 
              dplyr::across(all_of(mbl_vars), function(x) lapply(x, fitted, ...)))
  object <- pivot_longer(object, mbl_vars, names_to = ".model", values_to = ".fitted")
  unnest_tsbl(object, ".fitted", parent_key = c(kv, ".model"))
}

#' @rdname fitted.mdl_df
#' @export
fitted.mdl_ts <- function(object, ...){
  bt <- map(object$transformation, invert_transformation)
  
  fits <- as.matrix(fitted(object$fit, ...))
  fits <- map2(bt, split(fits, col(fits)), function(bt, fit) bt(fit))
  
  nm <- if(length(fits) == 1) ".fitted" else map_chr(object$response, expr_name)
  
  out <- object$data[index_var(object$data)]
  out[nm] <- fits
  out
}