% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_monthly_stats.R
\name{calc_monthly_stats}
\alias{calc_monthly_stats}
\title{Calculate monthly summary statistics}
\usage{
calc_monthly_stats(
  data,
  dates = Date,
  values = Value,
  groups = STATION_NUMBER,
  station_number,
  percentiles = c(10, 90),
  roll_days = 1,
  roll_align = "right",
  water_year_start = 1,
  start_year,
  end_year,
  exclude_years,
  months = 1:12,
  transpose = FALSE,
  spread = FALSE,
  ignore_missing = FALSE
)
}
\arguments{
\item{data}{A data frame of daily data that contains columns of dates, flow values, and (optional) groups (e.g. station numbers).
Leave blank if using \code{station_number} argument.}

\item{dates}{Column in \code{data} that contains dates formatted YYYY-MM-DD. Only required if dates column name is not '
Date' (default). Leave blank if using \code{station_number} argument.}

\item{values}{Column in \code{data} that contains numeric flow values, in units of cubic metres per second.
Only required if values column name is not 'Value' (default). Leave blank if using \code{station_number} argument.}

\item{groups}{Column in \code{data} that contains unique identifiers for different data sets, if applicable. Only required if
groups column name is not 'STATION_NUMBER'. Function will automatically group by a column named 'STATION_NUMBER' if present.
Remove the 'STATION_NUMBER' column beforehand to remove this grouping. Leave blank if using \code{station_number} argument.}

\item{station_number}{A character string vector of seven digit Water Survey of Canada station numbers (e.g. \code{"08NM116"}) of
which to extract daily streamflow data from a HYDAT database. Requires \code{tidyhydat} package and a HYDAT database.
Leave blank if using \code{data} argument.}

\item{percentiles}{Numeric vector of percentiles to calculate. Set to NA if none required. Default \code{c(10,90)}.}

\item{roll_days}{Numeric value of the number of days to apply a rolling mean. Default \code{1}.}

\item{roll_align}{Character string identifying the direction of the rolling mean from the specified date, either by the first 
('left'), last ('right), or middle ('center') day of the rolling n-day group of observations. Default \code{'right'}.}

\item{water_year_start}{Numeric value indicating the month of the start of the water year for analysis. Default \code{1}.}

\item{start_year}{Numeric value of the first year to consider for analysis. Leave blank to use the first year of the source data.}

\item{end_year}{Numeric value of the last year to consider for analysis. Leave blank to use the last year of the source data.}

\item{exclude_years}{Numeric vector of years to exclude from analysis. Leave blank to include all years.}

\item{months}{Numeric vector of months to include in analysis (e.g.. \code{6:8} for Jun-Aug). Leave blank to summarize 
all months (default \code{1:12}).}

\item{transpose}{Logical value indicating if each month statistic should be individual rows. Default \code{FALSE}.}

\item{spread}{Logical value indicating if each month statistic should be the column name. Default \code{FALSE}.}

\item{ignore_missing}{Logical value indicating whether dates with missing values should be included in the calculation. If
\code{TRUE} then a statistic will be calculated regardless of missing dates. If \code{FALSE} then only statistics from time periods 
with no missing dates will be returned. Default \code{FALSE}.}
}
\value{
A tibble data frame with the following columns:
  \item{Year}{calendar or water year selected}
  \item{Month}{month of the year}
  \item{Mean}{mean of all daily flows for a given month and year}
  \item{Median}{median of all daily flows for a given month and year}
  \item{Maximum}{maximum of all daily flows for a given month and year}
  \item{Minimum}{minimum of all daily flows for a given month and year}
  \item{P'n'}{each n-th percentile selected for a given month and year}
  Default percentile columns:
  \item{P10}{10th percentile of all daily flows for a given month and year}
  \item{P90}{90th percentile of all daily flows for a given month and year}
  Transposing data creates a column of 'Statistics' for each month, labeled as 'Month-Statistic' (ex "Jan-Mean"),
  and subsequent columns for each year selected.
  Spreading data creates columns of Year and subsequent columns of Month-Statistics  (ex 'Jan-Mean').
}
\description{
Calculates monthly mean, median, maximum, minimum, and percentiles for each month of all years of daily flow values 
   from a streamflow dataset. Calculates the statistics from all daily discharge values from all years, unless specified.
}
\examples{
# Run if HYDAT database has been downloaded (using tidyhydat::download_hydat())
if (file.exists(tidyhydat::hy_downloaded_db())) {

# Calculate statistics using a data frame and data argument with defaults
flow_data <- tidyhydat::hy_daily_flows(station_number = "08NM116")
calc_monthly_stats(data = flow_data,
                   start_year = 1980)

# Calculate statistics using station_number argument with defaults
calc_monthly_stats(station_number = "08NM116",
                   start_year = 1980)

# Calculate statistics regardless if there is missing data for a given year
calc_monthly_stats(station_number = "08NM116",
                   ignore_missing = TRUE)
                  
# Calculate statistics for water years starting in October
calc_monthly_stats(station_number = "08NM116",
                   start_year = 1980,
                   water_year_start = 10)
                  
# Calculate statistics with custom years
calc_monthly_stats(station_number = "08NM116",
                   start_year = 1981,
                   end_year = 2010,
                   exclude_years = c(1991,1993:1995))
                  
# Calculate statistics for 7-day flows, with 25 and 75th percentiles
calc_monthly_stats(station_number = "08NM116",
                   roll_days = 7,
                   percentiles = c(25,75),
                   ignore_missing = TRUE)
                   
}
}
