#' Read an Acoustic Receiver File
#'
#' This function takes a raw acoustic detection file generated by a Lotek,
#' Teknologic, or ATS JSATS receiver and determines which reader function to use
#' to process it into a dataframe which can be used by the filtering functions
#' in this package.
#'
#' @param path the path to the folder containing the desired file
#' @param file the path of the desired file
#' @param timezone the Olsen Named time zone, default is "America/Los_Angeles"
#' @returns A dataframe converting the raw detection data into rows of detections
#' @export
#' @examples
#' # Read in any ATS, LOTEK, TEKNO file
#'
#' # list of files
#' path = system.file("extdata", package = "filteRjsats")
#' files <- list.files(path)
#'
#' # Read the files
#' raw_data <- list()
#' for(i in 1:length(files)){
#' raw_data[[i]] <- read_jsats(path = path, file = files[i],
#' timezone = "America/Los_Angeles")
#' }
#' # Warnings are expected due to the formatting of ATS files
read_jsats <- function(path, file, timezone="America/Los_Angeles"){
  # file_name <- stringr::str_split(file, pattern = '\\.')[[1]][1]
  file_type <- ifelse(stringr::str_detect(file,'L'),
                      "Lotek",
                      ifelse(stringr::str_detect(file,'.SUM'),
                             "Teknologic",
                             "ATS"))
  jsats_file <- data.frame()
  if(file_type == "Lotek") jsats_file <- read_lotek(path, file, timezone)
  if(file_type == "Teknologic") jsats_file <- read_tekno(path, file,
                                                         timezone)
  if(file_type == "ATS") jsats_file <- read_ats(path, file, timezone)
  if(!(file_type %in% c("Lotek","ATS","Teknologic")))
    errorCondition("Receiver make should be one of Lotek, Tekno, or ATS")
  jsats_file
}
