% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/idio.R
\name{fnets.var}
\alias{fnets.var}
\title{\code{l1}-regularised Yule-Walker estimation for VAR processes}
\usage{
fnets.var(
  x,
  center = TRUE,
  method = c("lasso", "ds"),
  lambda = NULL,
  var.order = 1,
  tuning.args = list(tuning = c("cv", "bic"), n.folds = 1, penalty = NULL, path.length
    = 10, do.plot = FALSE),
  do.threshold = FALSE,
  n.iter = NULL,
  tol = 0,
  n.cores = min(parallel::detectCores() - 1, 3)
)
}
\arguments{
\item{x}{input time series matrix, with each row representing a variable}

\item{center}{whether to de-mean the input \code{x} row-wise}

\item{method}{a string specifying the method to be adopted for VAR process estimation; possible values are:
\itemize{
   \item{\code{"lasso"}}{ Lasso-type \code{l1}-regularised \code{M}-estimation}
   \item{\code{"ds"}}{ Dantzig Selector-type constrained \code{l1}-minimisation}
}}

\item{lambda}{regularisation parameter; if \code{lambda = NULL}, \code{tuning} is employed to select the parameter}

\item{var.order}{order of the VAR process; if a vector of integers is supplied, the order is chosen via \code{tuning}}

\item{tuning.args}{a list specifying arguments for \code{tuning}
for selecting the regularisation parameter (and VAR order). It contains:
\itemize{
   \item{\code{tuning}} a string specifying the selection procedure for \code{var.order} and \code{lambda}; possible values are:
   \itemize{
      \item{\code{"cv"}}{ cross validation}
      \item{\code{"bic"}}{ information criterion}
   }
   \item{\code{n.folds}}{ if \code{tuning = "cv"}, positive integer number of folds}
   \item{\code{penalty}}{ if \code{tuning = "bic"}, penalty multiplier between 0 and 1; if \code{penalty = NULL}, it is set to \code{1/(1+exp(dim(x)[1])/dim(x)[2]))}} by default
   \item{\code{path.length}}{ positive integer number of regularisation parameter values to consider; a sequence is generated automatically based in this value}
   \item{\code{do.plot}}{ whether to plot the output of the cross validation step}
}}

\item{do.threshold}{whether to perform adaptive thresholding of VAR parameter estimator with \link[fnets]{threshold}}

\item{n.iter}{maximum number of descent steps, by default depends on \code{var.order}; applicable when \code{method = "lasso"}}

\item{tol}{numerical tolerance for increases in the loss function; applicable when \code{method = "lasso"}}

\item{n.cores}{number of cores to use for parallel computing, see \link[parallel]{makePSOCKcluster}; applicable when \code{method = "ds"}}
}
\value{
a list which contains the following fields:
\item{beta}{ estimate of VAR parameter matrix; each column contains parameter estimates for the regression model for a given variable}
\item{Gamma}{ estimate of the innovation covariance matrix}
\item{lambda}{ regularisation parameter}
\item{var.order}{ VAR order}
\item{mean.x}{ if \code{center = TRUE}, returns a vector containing row-wise sample means of \code{x}; if \code{center = FALSE}, returns a vector of zeros}
}
\description{
Estimates the VAR parameter matrices via \code{l1}-regularised Yule-Walker estimation
and innovation covariance matrix via constrained \code{l1}-minimisation.
}
\details{
Further information can be found in Barigozzi, Cho and Owens (2022).
}
\examples{
library(fnets)

set.seed(123)
n <- 500
p <- 50
idio <- sim.var(n, p)
x <- idio$data

fv <- fnets.var(x,
  center = TRUE, method = "lasso", var.order = 1,
  tuning.args = list(tuning = "cv", n.folds = 1, path.length = 10, do.plot = TRUE),
  n.cores = 2
)
norm(fv$beta - t(idio$A), "F") / norm(t(idio$A), "F")
}
\references{
Barigozzi, M., Cho, H. & Owens, D. (2022) FNETS: Factor-adjusted network estimation and forecasting for high-dimensional time series. arXiv preprint arXiv:2201.06110.

Owens, D., Cho, H. & Barigozzi, M. (2022) fnets: An R Package for Network Estimation and Forecasting via Factor-Adjusted VAR Modelling. arXiv preprint arXiv:2301.11675.
}
