% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dom_height.R
\name{dom_height}
\alias{dom_height}
\title{Calculate the dominant height of forest inventory data plots}
\usage{
dom_height(
  df,
  th,
  dbh,
  plot,
  obs,
  dom,
  .groups,
  merge_data = FALSE,
  dh_name = "DH"
)
}
\arguments{
\item{df}{A data frame.}

\item{th}{Quoted name of the total height variable.}

\item{dbh}{Quoted name of the diameter at breast height variable. Used to filter out trees with no diameter measurement.}

\item{plot}{Quoted name of the plot variable. used to differentiate the data's plots. If this argument is missing, the defined groups in the data frame will be used, If there are no groups in the data, the function will fail.}

\item{obs}{Quoted name of the observations variable. This will be used to tell which trees are dominant, i.e. it's the variable that tells the type of tree; if it is normal, dominant, suppressed, etc. If this argument is not supplied, the function will calculate the average value of 2 trees with bigger height values in each plot, and use that as the dominant value.}

\item{dom}{Character value for the dominant tree code used in the observations variable variable supplied  in the \code{obs} argument. This is used alongside the \code{obs} argument to differentiate dominant trees from the others.}

\item{.groups}{Optional argument. Quoted name(s) of grouping variables that can be added to differentiate subdivisions of the data. Default: \code{NA}.}

\item{merge_data}{If \code{TRUE}, will merge the original data frame with the dominant height table. Default: \code{FALSE}.}

\item{dh_name}{Character value for the name of the dominant height variable created. Default: \code{"DH"}}
}
\value{
A data frame with the the dominant height values by plot.
}
\description{
This function is used to get a data frame with Dominant height values for each plot
in an forest inventory data.
}
\examples{
library(forestmangr)
data("exfm9")
exfm9

# Let's say we need to get the dominant height (DH) values for a forest inventory data.
# If we don't have a variable that tells which trees are dominant, it's ok. We can
# still estimate DH using this function. It will average the top 2 trees of each plot:
dom_height(df=exfm9,th="TH",dbh="DBH",plot="PLOT")

# Of course, if we do have a variable that differentiate the dominant trees, it's
# best we use it. For that we use the obs argument, and the dom argument.
# In this data, the OBS variable is used to tell the type of tree.
# Let's check the levels in our OBS variable, to see which one is associated 
# with dominant trees.

levels(as.factor(exfm9$OBS))

# So, the "D" level must be the one that tells which trees are dominant. Let's use it: 
dom_height(df=exfm9,th="TH",dbh="DBH",plot="PLOT",obs="OBS",dom="D")

# If there are subdivisions of the data, like different strata, they can be included in the
# .groups argument: 
dom_height(df=exfm9,th="TH",dbh="DBH",plot="PLOT",obs="OBS",dom="D",.groups="STRATA")

# It's possible to automatically bind the dominant heights table to the original data, 
# using the merge_data argument:

dom_height(df=exfm9,th="TH",dbh="DBH",plot="PLOT",obs="OBS",
dom="D",.groups="STRATA", merge_data=TRUE)

}
\author{
Sollano Rabelo Braga \email{sollanorb@gmail.com}
}
