% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/LRpower.R
\name{LRpower}
\alias{LRpower}
\title{Power simulation for kinship LR}
\usage{
LRpower(
  numeratorPed,
  denominatorPed,
  truePed = numeratorPed,
  ids,
  markers = NULL,
  source = "true",
  nsim = 1,
  threshold = NULL,
  disableMutations = NA,
  alleles = NULL,
  afreq = NULL,
  Xchrom = FALSE,
  knownGenotypes = NULL,
  plot = FALSE,
  plotMarkers = NULL,
  seed = NULL,
  verbose = TRUE
)
}
\arguments{
\item{numeratorPed, denominatorPed}{\code{ped} objects (or lists of such),
describing the two relationships under comparison.}

\item{truePed}{A \code{ped} object (or a list of such), describing the true
relationship. By default equal to \code{numeratorPed}.}

\item{ids}{Individuals available for genotyping.}

\item{markers}{A vector indicating the names or indices of markers attached
to the source pedigree. If NULL (default), then all markers attached to the
source pedigree are used. If \code{alleles} or \code{afreq} is non-NULL, then this
parameter is ignored.}

\item{source}{Either "true" (default), "numerator" or "denominator",
indicating which pedigree is used as source for marker data.}

\item{nsim}{A positive integer: the number of simulations.}

\item{threshold}{A numeric vector with one or more positive numbers used as
LR thresholds.}

\item{disableMutations}{Not implemented yet.}

\item{alleles, afreq, Xchrom}{If these are given, they are used (together with
\code{knownGenotypes}) to create a marker object on the fly.}

\item{knownGenotypes}{A list of triplets \verb{(a, b, c)}, indicating that
individual \code{a} has genotype \code{b/c}. Ignored unless \code{alleles} or \code{afreq} is
non-NULL.}

\item{plot}{Either a logical or the character "plotOnly". If the latter, a
plot is drawn, but no further computations are done.}

\item{plotMarkers}{A vector of marker names or indices whose genotypes are to
be included in the plot.}

\item{seed}{An integer seed for the random number generator (optional).}

\item{verbose}{A logical.}
}
\value{
A \code{LRpowerResult} object, which is essentially a list with the
following entries:
\itemize{
\item \code{LRperSim}: A numeric vector of length \code{nsim} containing the total LR for
each simulation.
\item \code{meanLRperMarker}: The mean LR per marker, over all simulations.
\item \code{meanLR}: The mean total LR over all simulations.
\item \code{meanLogLR}: The mean total \code{log10(LR)} over all simulations.
\item \code{IP}: A named numeric of the same length as \code{threshold}. For each element
of \code{threshold}, the fraction of simulations resulting in a LR exceeding the
given number.
\item \code{time}: The total computation time.
\item \code{params}: A list containing the input parameters \code{missing}, \code{markers},
\code{nsim}, \code{threshold} and \code{disableMutations}
}
}
\description{
This function uses simulations to estimate the likelihood ratio (LR)
distribution in a given kinship testing scenario. In the most general
setting, three pedigrees are involved: the two pedigrees being compared, and
the true relationship (which may differ from the other two). A subset of
individuals are available for genotyping. Some individuals may already be
genotyped; all simulations are then conditional on these.
}
\examples{

# Paternity LR of siblings
ids = c("A", "B")
truth = nuclearPed(children = ids)
claim = nuclearPed(fa = "A", mo = "NN", children = "B")
unrel = singletons(ids)

# Simulation parameters
nsim = 10   # increase!
thresh = 1

# Simulation 1:
als = 1:5
afr = runif(5)
afr = afr/sum(afr)

pow1 = LRpower(claim, unrel, truth, ids = ids, nsim = nsim,
               threshold = thresh, alleles = als, afreq = afr,
               seed = 123)
pow1

# Simulation 2: Same, but using an attached marker
truth = addMarker(truth, alleles = als, afreq = afr)

pow2 = LRpower(claim, unrel, truth, ids = ids, nsim = nsim,
               threshold = thresh, markers = 1, seed = 123)

stopifnot(identical(pow1$LRperSim, pow2$LRperSim))

\donttest{
# True pedigree has inbred founders
truth2 = setFounderInbreeding(truth, value = 0.5)

pow3 = LRpower(claim, unrel, truth2, ids = ids, nsim = nsim,
               threshold = thresh, markers = 1, seed = 123) # plot = TRUE
pow3
}

}
