% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spa_functions.R
\name{spa_eval}
\alias{spa_eval}
\title{Capturing the membership degree of a point}
\usage{
spa_eval(pgo, point)
}
\arguments{
\item{pgo}{A \code{pgeometry} object of any type.}

\item{point}{An \code{sfg} object of the type \code{POINT}.}
}
\value{
A numeric value between 0 and 1 that indicates the membership degree of a point (i.e., \code{sfg} object) in a spatial plateau object (i.e., \code{pgeometry} object).
}
\description{
This function evaluates the membership degree of a given point in a spatial plateau object of any type.
It returns a value in [0, 1] that indicates to which extent the point belongs to the \code{pgeometry} object.
}
\details{
The goal of this function is to return the membership degree of a simple point object (i.e., \code{sfg} object) in a given spatial plateau object (i.e., \code{pgeometry} object).
This evaluation depends on the following basic cases:
\itemize{
\item if the simple point object belongs to the interior or boundary of \emph{one} component of the spatial plateau object, it returns the membership degree of that component.
\item if the simple point object intersects more components (e.g., boundaries of region components, or different line components), it returns the maximum membership degree of all intersected components.
\item if the simple point object is disjoint to the support of the spatial plateau object, it returns 0.
}
}
\examples{

library(tibble)
library(sf)
library(FuzzyR)

# some basic examples 

pts1 <- rbind(c(1, 2), c(3, 2))
pts2 <- rbind(c(1, 1), c(2, 3), c(2, 1))
pts3 <- rbind(c(2, 2), c(3, 3))

cp1 <- component_from_sfg(st_multipoint(pts1), 0.3)
cp2 <- component_from_sfg(st_multipoint(pts2), 0.6)
cp3 <- component_from_sfg(st_multipoint(pts3), 1.0)

pp <- create_pgeometry(list(cp1, cp2, cp3), "PLATEAUPOINT")

spa_eval(pp, st_point(c(1, 2)))
spa_eval(pp, st_point(c(1, 3)))

# other examples with plateau regions

set.seed(345)

# some random points to create plateau region objects by using the function spa_creator
tbl = tibble(x = runif(10, min= 0, max = 20), 
             y = runif(10, min = 0, max = 30), 
             z = runif(10, min = 0, max = 100))

#getting the convex hull on the points to clipping the construction of plateau region objects
pts <- st_as_sf(tbl, coords = c(1, 2))
ch <- st_convex_hull(do.call(c, st_geometry(pts)))

pregions <- spa_creator(tbl, fuzz_policy = "fcp", k = 2, base_poly = ch)

# capturing the membership degree of a specific point in each object
spa_eval(pregions$pgeometry[[1]], st_point(c(5, 15)))
spa_eval(pregions$pgeometry[[2]], st_point(c(5, 15)))

}
\references{
\href{https://ieeexplore.ieee.org/document/8491565}{Carniel, A. C.; Schneider, M. Spatial Plateau Algebra: An Executable Type System for Fuzzy Spatial Data Types. In Proceedings of the 2018 IEEE International Conference on Fuzzy Systems (FUZZ-IEEE 2018), pp. 1-8, 2018.}
}
