% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plotMethods.R
\name{autoplot.funData}
\alias{autoplot.funData}
\alias{autolayer.funData}
\title{Visualize functional data objects using ggplot}
\usage{
autoplot.funData(
  object,
  obs = seq_len(nObs(object)),
  geom = "line",
  plotNA = FALSE,
  ...
)

autolayer.funData(
  object,
  obs = seq_len(nObs(object)),
  geom = "line",
  plotNA = FALSE,
  ...
)
}
\arguments{
\item{object}{A \code{funData} object on a one- or two-dimensional domain.}

\item{obs}{A vector of numerics giving the observations to plot. Defaults to 
all observations in \code{object}. For two-dimensional functions (images) 
\code{obs} must have length 1.}

\item{geom}{A character string describing the geometric object to use.
Defaults to \code{"line"}. See \pkg{ggplot2} for details.}

\item{plotNA}{Logical. If \code{TRUE}, missing values are interpolated using 
the \code{\link{approxNA}} function (only for one-dimensional functions). 
Defaults to \code{FALSE}. See Details.}

\item{...}{Further parameters passed to \code{\link[ggplot2]{geom_line}} (for
one dimensional domains, e.g. \code{alpha, color, fill, linetype, size}) or
to \code{\link[ggplot2]{geom_raster}} (for two-dimensional domains, e.g.
\code{hjust, vjust, interpolate}).}
}
\value{
A \code{\link[ggplot2]{ggplot}} object that can be customized  using 
  all functionalities of the \pkg{ggplot2} package.
}
\description{
This function allows to plot \code{funData} objects based on the 
\pkg{ggplot2} package. The function provides a wrapper that rearranges the 
data in a \code{funData} object on a one- or two-dimensional domain and 
provides a basic \code{\link[ggplot2]{ggplot}} object, which can be 
customized using all functionalities of the \pkg{ggplot2} package.
}
\details{
If some observations contain missing values (coded via \code{NA}), the 
functions can be interpolated using the option \code{plotNA = TRUE}. This 
option relies on the \code{\link[zoo]{na.approx}} function in package 
\code{\link[zoo]{zoo}} and is currently implemented for one-dimensional 
functions only in the function \code{\link{approxNA}}.
}
\examples{
# Install / load package ggplot2 before running the examples
library("ggplot2")

# One-dimensional
argvals <- seq(0,2*pi,0.01)
object <- funData(argvals,
                   outer(seq(0.75, 1.25, length.out = 11), sin(argvals)))

g <- autoplot(object) # returns ggplot object
g # plot the object

# add the mean function in red
g + autolayer(meanFunction(object),  col = 2)

# Two-dimensional
X <- array(0, dim = c(2, length(argvals), length(argvals)))
X[1,,] <- outer(argvals, argvals, function(x,y){sin((x-pi)^2 + (y-pi)^2)})
X[2,,] <- outer(argvals, argvals, function(x,y){sin(2*x*pi) * cos(2*y*pi)})
object2D <- funData(list(argvals, argvals), X)

\donttest{
autoplot(object2D, obs = 1)
autoplot(object2D, obs = 2)}
\dontrun{autoplot(object2D)} # must specify obs!

### More examples ###
\donttest{
par(mfrow = c(1,1))

# using plotNA (needs packages zoo and gridExtra)
\dontshow{requireNamespace("zoo", quietly = TRUE)}
\dontshow{requireNamespace("gridExtra", quietly = TRUE)}
objectMissing <- funData(1:5, rbind(c(1, NA, 5, 4, 3), c(10, 9, NA, NA, 6)))
g1 <- autoplot(objectMissing) # the default
g2 <- autoplot(objectMissing, plotNA = TRUE) # requires zoo

gridExtra::grid.arrange(g1 + ggtitle("plotNA = FALSE (default)"),
                        g2 + ggtitle("plotNA = TRUE")) # requires gridExtra

# Customizing plots (see ggplot2 documentation for more details)
# parameters passed to geom_line are passed via the ... argument
gFancy <- autoplot(object, color = "red", linetype = 2) 
gFancy

# new layers can be added directly to the ggplot object
gFancy + theme_bw() # add new layers to the ggplot object
gFancy + ggtitle("Fancy Plot with Title and Axis Legends") + 
         xlab("The x-Axis") + ylab("The y-Axis")

autoplot(object2D, obs = 1) + ggtitle("Customized 2D plot") + theme_minimal() +
          scale_fill_gradient(high = "green", low = "blue", name = "Legend here")
}
}
\seealso{
\code{\linkS4class{funData}}, \code{\link[ggplot2]{ggplot}}, 
  \code{\link{plot.funData}}
}
