% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/facX.R
\name{faX}
\alias{faX}
\title{Factor Extraction (faX) Routines}
\usage{
faX(R, numFactors = NULL, facMethod = "fals", faControl = NULL,
  digits = NULL)
}
\arguments{
\item{R}{(Matrix) A correlation matrix used for factor extraction.}

\item{numFactors}{(Numeric) The number of factors to extract for subsequent rotation.}

\item{facMethod}{(Character) The method used for factor extraction. The supported options are "fals" for unweighted least squares, "faml" for maximum likelihood, and "fapa" for iterated principal axis factoring. The default method is "fals".
\itemize{
  \item \strong{"fals"}: Factors are extracted using the unweighted least squares estimation procedure using the \code{\link[fungible]{fals}} function.
  \item \strong{"faml"}: Factors are extracted using the maximum likelihood estimation procedure using the \code{\link[stats]{factanal}} function.
  \item \strong{"fapa"}: Factors are extracted using the iterated principal axis factoring estimation procedure using the \code{\link{fapa}} function.
}}

\item{faControl}{(List) A list of optional parameters passed to the factor extraction routines.
\itemize{
  \item \strong{treatHeywood}: (Logical) In fals, if treatHeywood is true, a penalized least squares function is used to bound the communality estimates below 1.0. The default is TRUE.
  \item \strong{nStart}: (Numeric) In faml, determine the number of starting values to try. The default is 10 start values.
  \item \strong{maxCommunality}: (Numeric) In faml, set the maximum communality value for the estimated solution. The default maximum is .995.
  \item \strong{epsilon}: (Numeric) In fapa, the numeric threshold designating when the algorithm has converged. The default value is 1e-4.
  \item \strong{communality}: (Character) In fapa, the routine requires an initial communality estimate. Select how communalities are initially estimated. The default is squared multiple correlation ("SMC").
  \itemize{
    \item \strong{"SMC"}: Initial communalities are estimated by taking the squared multiple correlations of each indicator after regressing the indicator on the remaining variables. The following equation is employed to find the squared multiple correlation: \eqn{1 - 1 / diag(R^-1)}.
    \item \strong{"maxRsqr"}: Initial communalities equal the largest squared correlation in each column of the correlation matrix.
    \item \strong{"unity"}: Initial communalities equal 1.0 for all variables.
  }
  \item \strong{maxITR}: (Numeric) In fapa, the maximum number of iterations to reach convergence. The default is 15,000
}}

\item{digits}{(Numeric) The number of digits to round all output to.}
}
\value{
This function returns a list of output relating to the extracted factor loadings.
\itemize{
  \item \strong{loadings}: (Matrix) An unrotated factor structure matrix.
  \item \strong{h2}: (Vector) A vector containing the item communality estimates.
  \item \strong{uniqueness}: (Vector) A vector of the item uniqueness estimates (1 - h2).
  \item \strong{Heywood}: (Logical) Whether a Heywood case (i.e., a communality value > 1.0) was detected.
}
}
\description{
This function can be used to extract an unrotated factor structure matrix using the following algorithms: (a) unweighted least squares ("fals"); (b) maximum likelihood ("faml"); and (c) iterated principal axis factoring ("fapa").
}
\details{
\itemize{
  \item \strong{Initial communality estimate}: According to Widaman and Herringer (1985), the initial communality estimate does not have much bearing on the resulting solution \emph{when the a stringent convergence criterion is used}. In their analyses, a convergence criterion of .001 (i.e., slightly less stringent than the default of 1e-4) is sufficiently stringent to produce virtually identical communality estimates irrespective of the initial estimate used. It should be noted that all four methods for estimating the initial communality in Widaman and Herringer (1985) are the exact same used in this function. Based on their findings, it is not recommended to use a convergence criterion lower than 1e-3.
}
}
\examples{
## Generate an example factor structure matrix
lambda <- matrix(c(.62, .00, .00,
                   .54, .00, .00,
                   .41, .00, .00,
                   .00, .31, .00,
                   .00, .58, .00,
                   .00, .62, .00,
                   .00, .00, .38,
                   .00, .00, .43,
                   .00, .00, .37),
                 nrow = 9, ncol = 3, byrow = TRUE)

## Find the model implied correlation matrix
R <- lambda \%*\% t(lambda)
diag(R) <- 1

## Extract (principal axis) factors using the factExtract function
Out1 <- faX(R          = R,
            numFactors = 3,
            facMethod  = "fapa",
            faControl  = list(communality = "maxRsqr",
                              epsilon     = 1e-4))

## Extract (least squares) factors using the factExtract function
Out2 <- faX(R          = R,
            numFactors = 3,
            facMethod  = "fals",
            faControl  = list(treatHeywood = TRUE))

}
\references{
Widaman, K. F., & Herringer, L. G. (1985). Iterative least squares estimates of communality: Initial estimate need not affect stabilized value. \emph{Psychometrika, 50}(4), 469-477.
}
\author{
\itemize{
  \item Casey Giordano (Giord023@umn.edu)
  \item Niels G. Waller (nwaller@umn.edu)
}
}
