% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geostan_fit-methods.R
\name{predict.geostan_fit}
\alias{predict.geostan_fit}
\title{Predict method for \code{geostan_fit} models}
\source{
Goulard, Michael, Thibault Laurent, and Christine Thomas-Agnan (2017). About predictions in spatial autoregressive models: optimal and almost optimal strategies. \emph{Spatial Economic Analysis} 12 (2-3): 304-325.

LeSage, James (2014). What Regional Scientists Need to Know about Spatial Econometrics. \emph{The Review of Regional Science} 44: 13-32 (2014 Southern Regional Science Association Fellows Address).

LeSage, James, & Robert kelley Pace (2009). \emph{Introduction to Spatial Econometrics}. Chapman and Hall/CRC.
}
\usage{
\method{predict}{geostan_fit}(
  object,
  newdata,
  alpha = as.matrix(object, pars = "intercept"),
  center = object$x_center,
  summary = TRUE,
  type = c("link", "response"),
  add_slx = FALSE,
  approx = FALSE,
  K = 15,
  ...
)
}
\arguments{
\item{object}{A fitted model object of class \code{geostan_fit}.}

\item{newdata}{A data frame in which to look for variables with which to predict. Note that if the model formula includes an offset term, \code{newdata} must contain the offset column (see examples below). If covariates in the model were centered using the \code{centerx} argument, the \code{predict.geostan_fit} method will automatically center the predictors in \code{newdata} using the values stored in \code{object$x_center}. If \code{newdata} is missing, the fitted values of the model will be returned.}

\item{alpha}{An N-by-1 matrix of MCMC samples for the intercept; this is provided by default. If used, note that the intercept needs to be provided on the scale of the linear predictor. This argument might be used if there is a need to incorporate the spatial trend term (as a spatially-varying intercept).}

\item{center}{Optional vector of numeric values or a logical scalar to pass to \code{\link[base]{scale}}. Defaults to using \code{object$x_center}. If the model was fit using \code{centerx = TRUE}, then covariates were centered and their mean values are stored in \code{object$x_center} and the \code{predict} method will use them automatically to center \code{newdata}; if the model was fit with \code{centerx = FALSE}, then \code{object$x_center = FALSE} and \code{newdata} will not be centered.}

\item{summary}{If \code{FALSE}, a matrix containing samples from the posterior distribution at each observation is returned. The default, \code{TRUE}, will summarize results by providing an estimate (mean) and credible interval (formed by taking quantiles of the MCMC samples).}

\item{type}{By default, results from \code{predict} are on the scale of the linear predictor (\verb{type = "link")}). The alternative (\code{type = "response"}) is on the scale of the response variable. For example, the default return values for a Poisson model on the log scale, and using \code{type = "response"} will return the original scale of the outcome variable (by exponentiating the log values).}

\item{add_slx}{Logical. If \code{add_slx = TRUE}, any spatially-lagged covariates that were specified through the 'slx' argument (of the model fitting function, e.g., \code{stan_glm}) will be added to the linear predictor. The spatial lag terms will be calculated internally using \code{object$C}, the spatial weights matrix used to fit the model. Hence, \code{newdata} must have \code{N = object$N} rows. Predictions from spatial lag models (SAR models of type 'SLM' and 'SDLM') always include the SLX terms (i.e., any value passed to \code{add_slx} will be overwritten with \code{TRUE}).}

\item{approx}{For SAR models of type 'SLM' or 'SDLM' only; use an approximation for matrix inversion? See details below.}

\item{K}{Number of matrix powers to use with \code{approx}.}

\item{...}{Not used}
}
\value{
If \code{summary = FALSE}, a matrix of samples is returned. If \code{summary = TRUE} (the default), a data frame is returned.
}
\description{
Obtain predicted values from a fitted model by providing new covariate values.
}
\details{
The primary purpose of the predict method is to explore marginal effects of covariates. The uncertainty present in these predictions refers to uncertainty in the expected value of the model. The expectation does not include the error term of the model (nb: one expects actual observations to form a cloud of points around the expected value). By contrast, \link[geostan]{posterior_predict} returns the complete (posterior) predictive distribution of the model (the expectation plus noise).

The model formula will be taken from \code{object$formula}, and then a model matrix will be created by passing \code{newdata} to the \link[stats]{model.frame} function (as in: \code{model.frame(object$formula, newdata)}. Parameters are taken from \code{as.matrix(object, pars = c("intercept", "beta"))}.

The examples illustrate how to use the function in most cases.

Special considerations apply to models with spatially-lagged covariates and a spatially-lagged dependent variable (i.e., the 'SLM' and 'SDLM' models fit by \link[geostan]{stan_sar}).
\subsection{Spatial lag of X}{

Spatially-lagged covariates which were included via the \code{slx} argument will, by default, not be included in the predicted values. (The user can have greater control by manually adding the spatially-lagged covariate to the main model formula.) The \code{slx} term will be be included in predictions if \code{add_slx = TRUE} or if the fitted model is a SAR model of type 'SLM' or 'SDLM'. In either of those cases, \code{newdata} must have the same number of rows as were used to fit the original data.
}

\subsection{Spatial lag of Y}{

The typical 'marginal effect' interpretation of the regression coefficients does not hold for the SAR models of type 'SLM' or 'SDLM'. For details on these 'spillover effects', see LeSage and Pace (2009), LeSage (2014), and \link[geostan]{impacts}.

Predictions for the spatial lag model (SAR models of type 'SLM') are equal to:
\deqn{
 (I - \rho W)^{-1} X \beta
}
where \eqn{X \beta} contains the intercept and covariates. Predictions for the spatial Durbin lag model (SAR models of type 'SDLM') are equal to:
\deqn{
 (I - \rho W)^{-1} (X \beta + WX \gamma)
}
where \eqn{WX \gamma} are spatially lagged covariates multiplied by their coefficients. For this reason, the \code{predict.geostan_fit} method requires that \code{newdata} have as many rows as the original data (so that \code{nrow(newdata) == nrow(object$C)}); the spatial weights matrix will be taken from \code{object$C}.

The inverse of the matrix \eqn{(I - \rho W)} can be time consuming to compute (especially when iterating over MCMC samples). You can use \code{approx = TRUE} to approximate the inverse using a series of matrix powers. The argument \eqn{K} controls how many powers to use for the approximation. As a rule, higher values of \eqn{\rho} require larger \eqn{K} to obtain accuracy. Notice that \eqn{\rho^K} should be close to zero for the approximation to hold. For example, for \eqn{\rho = .5} a value of \eqn{K=8} may suffice (\eqn{0.5^8 = 0.004}), but larger values of \eqn{\rho} require higher values of \eqn{K}.
}

\subsection{Generalized linear models}{

In generalized linear models (such as Poisson and Binomial models) marginal effects plots on the response scale may be sensitive to the level of other covariates in the model and to geographic location (given a spatially-varying mean value). If the model includes a spatial autocorrelation component (for example, you used a spatial CAR, SAR, or ESF model, or used the \code{re} argument for random effects), by default these terms will be fixed at zero for the purposes of calculating marginal effects. If you want to change this, you can introduce a varying intercept manually using the \code{alpha} argument.
}
}
\examples{
data(georgia)
georgia$income <- georgia$income / 1e3

fit <- stan_glm(deaths.male ~ offset(log(pop.at.risk.male)) + log(income),
               data = georgia,
               re = ~ GEOID,
               centerx = TRUE,
               family = poisson(),
               chains = 2, iter = 600) # for speed only

# note: pop.at.risk.male=1 leads to offset of log(pop.at.risk.male)=0
# so that the predicted values are rates
newdata <- data.frame(
             income = seq(min(georgia$income),
                          max(georgia$income),
                           length.out = 200),
             pop.at.risk.male = 1)

preds <- predict(fit, newdata, type = "response")
head(preds)
plot(preds$income,
     preds$mean * 10e3,
     type = "l",
     ylab = "Deaths per 10,000",
     xlab = "Income ($1,000s)")

# here the predictions are rates per 10,000
newdata$pop.at.risk.male <- 10e3
preds <- predict(fit, newdata, type = "response")
head(preds)

# plot range
y_lim <- c(min(preds$`2.5\%`), max(preds$`97.5\%`))

# plot line
plot(preds$income,
    preds$mean,
    type = "l",    
    ylab = "Deaths per 10,000",
    xlab = "Income ($1,000s)",
    ylim = y_lim,
    axes = FALSE)

# add shaded cred. interval
x <- c(preds$income, rev(preds$income))
y <- c(preds$`2.5\%`, rev(preds$`97.5\%`))
polygon(x = x, y = y,
       col = rgb(0.1, 0.2, 0.3, 0.3),
       border = NA)

# add axes
yat = seq(0, 300, by = 20)
axis(2, at = yat)

xat = seq(0, 200, by = 10)
axis(1, at = xat)

# show county incomes
rug(georgia$income)
}
