% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geom-peak.R
\name{geom_peak}
\alias{geom_peak}
\title{Visualise peaks in a power spectrum or periodogram}
\usage{
geom_peak(mapping = NULL, data = NULL, stat = "identity",
  position = "identity", ..., na.rm = TRUE, show.legend = NA,
  inherit.aes = TRUE, peak_rank = 1, conversion = hours)
}
\arguments{
\item{mapping}{Set of aesthetic mappings created by \code{\link[=aes]{aes()}} or
\code{\link[=aes_]{aes_()}}. If specified and \code{inherit.aes = TRUE} (the
default), it is combined with the default mapping at the top level of the
plot. You must supply \code{mapping} if there is no plot mapping.}

\item{data}{The data to be displayed in this layer. There are three
options:

If \code{NULL}, the default, the data is inherited from the plot
data as specified in the call to \code{\link[=ggplot]{ggplot()}}.

A \code{data.frame}, or other object, will override the plot
data. All objects will be fortified to produce a data frame. See
\code{\link[=fortify]{fortify()}} for which variables will be created.

A \code{function} will be called with a single argument,
the plot data. The return value must be a \code{data.frame.}, and
will be used as the layer data.}

\item{stat}{The statistical transformation to use on the data for this
layer, as a string.}

\item{position}{Position adjustment, either as a string, or the result of
a call to a position adjustment function.}

\item{...}{Other arguments passed on to \code{\link[=layer]{layer()}}. These are
often aesthetics, used to set an aesthetic to a fixed value, like
\code{color = "red"} or \code{size = 3}. They may also be parameters
to the paired geom/stat.}

\item{na.rm}{If \code{FALSE}, the default, missing values are removed with
a warning. If \code{TRUE}, missing values are silently removed.}

\item{show.legend}{logical. Should this layer be included in the legends?
\code{NA}, the default, includes if any aesthetics are mapped.
\code{FALSE} never includes, and \code{TRUE} always includes.
It can also be a named logical vector to finely select the aesthetics to
display.}

\item{inherit.aes}{If \code{FALSE}, overrides the default aesthetics,
rather than combining with them. This is most useful for helper functions
that define both data and aesthetics and shouldn't inherit behaviour from
the default plot specification, e.g. \code{\link[=borders]{borders()}}.}

\item{peak_rank}{numerical vector specifying the rank(s) of peak(s) to draw}

\item{conversion}{function to convert values of \code{x} to a specific unit.
The default, \code{hours}, will write x (time) in decimal hours.}
}
\description{
This function draws points on the x-y coordinates of selected peaks and write their (y) value on the bottom of the plot.
}
\details{
In the input data, peaks are encoded as an additional column/aesthetic with values
corresponding to peak ranks (and \code{0} when the point is not a peak).
In other word, the mapping must provide \code{x}, \code{y} and \code{peak}.
Only peaks matching \code{peak_rank} will be drawn (see example).
}
\examples{
# We make a data frame by hand with five rows
# There are two peaks: in position 4 and 2

df <- data.frame(x = hours(1:5),
                 y = c(1, 2, 0, 4, 1),
                 peak = c(0, 2, 0, 1, 0))
#  We draw the plot as a line
pl <-  ggplot(df, aes(x, y, peak = peak)) +
                  geom_line() +
                  scale_x_hours()
pl
# Now we could add the peak values as an extra layer:
# The first peak
pl + geom_peak()
# The first ans second peak
pl + geom_peak(peak_rank = 1:2)
# The second only
pl + geom_peak(peak_rank = 2)

# Just like with other geoms,
# we can change colour, size, alpha, shape, ... :
pl + geom_peak(colour = "red", size = 10, alpha = .5, shape = 20)

## In the context of circadian analysis,
# Using the zeitgebr package:
\donttest{
require(zeitgebr)
# We make toy data
metadata <- data.table(id = sprintf("toy_experiment|\%02d", 1:40),
                       region_id = 1:40,
                       condition = c("A", "B"),
                       sex = c("M", "M", "F", "F"))
dt <- toy_activity_data(metadata, seed = 107)
# We shift period of the group "A" by 0.01
dt[, t := ifelse(xmv(condition) == "A", t, t * 1.01)]
# We  compute a periodogram for each individual
per_dt <- periodogram(moving, dt, FUN = chi_sq_periodogram)
per_dt <- find_peaks(per_dt)
out <- ggperio(per_dt, aes(y = power - signif_threshold, colour = condition, peak = peak)) +
                    stat_pop_etho() +
                    facet_wrap( ~ id, labeller = id_labeller)
out
out + geom_peak(colour="black")
}
}
\references{
\itemize{
\item The relevant \href{https://rethomics.github.io/ggetho.html#periodograms}{rethomic tutorial section}
}
}
\seealso{
\itemize{
\item \link{ggperio} to create a periodogram
\item \link[zeitgebr:find_peaks]{zeitgebr::find_peaks} to automatically add a \code{peak} column on periodogram data
}

Other layers: \code{\link{stat_bar_tile_etho}},
  \code{\link{stat_ld_annotations}},
  \code{\link{stat_pop_etho}}
}
\concept{layers}
