#' Scale bar
#' @description Adds a scale bar to maps created with ggplot or ggmap.
#' @param data the same \code{\link{data.frame}} passed to \code{\link{ggplot}} to plot the map.
#' @param location string indicating the symbol's location in the plot. Possible options: "topright" (default), "bottomright", "bottomleft" and "topleft".
#' @param dist distance in km to represent with each segment of the scale bar.
#' @param height number between 0 and 1 to indicate the height of the scale bar, as a proportion of the y axis.
#' @param st.dist number between 0 and 1 to indicate the distance between the scale bar and the scale bar text, as a proportion of the y axis.
#' @param st.bottom logical. If TRUE (default) the scale bar text is displayed at the bottom of the scale bar, if FALSE, it is displayed at the top.
#' @param st.size number to indicate the scale bar text size. It is passed to the size argument of \code{\link{annotate}} function.
#' @param dd2km logical. If TRUE \code{dist} it is assumed that map coordinates are in decimal degrees, if FALSE, it assumed they are in meters.
#' @param model choice of ellipsoid model ("WGS84", "GRS80", "Airy", "International", "Clarke", "GRS67") Used when dd2km is TRUE.
#' @param anchor named \code{\link{vector}} with coordinates to control the symbol position. For \code{location = "topright"}, \code{anchor} defines the coordinates of the symbol's topright corner and so forth. The x coordinate must be named as x and the y coordinate as y.
#' @param x.min if \code{data} is not defined, number with the minimum x coordinate. Useful for ggmap.
#' @param x.max if \code{data} is not defined, number with the maximum x coordinate. Useful for ggmap.
#' @param y.min if \code{data} is not defined, number with the minimum y coordinate. Useful for ggmap.
#' @param y.max if \code{data} is not defined, number with the maximum y coordinate. Useful for ggmap.
#' @param facet.var if faceting, variable name used for faceting. This is useful for placing the scalebar only in one facet and must be used together with \code{facet.lev}.
#' @param facet.lev one of the level names in \code{facet.var}. The scale bar will be drawn only in the \code{facet.lev} facet.
#' @export
#' @examples
#' library(rgdal); library(broom)
#' dsn <- system.file('extdata', package = 'ggsn')
 
#' ## Map in geographic coordinates.
#' map <- readOGR(dsn, 'sp')
#' map@@data$id <- 1:nrow(map@@data)
#' map.df <- merge(tidy(map), map, by = 'id')
#' 
#' ggplot(data = map.df, aes(long, lat, group = group, fill = nots)) +
#'     geom_polygon() +
#'     coord_equal() +
#'     geom_path() +
#'     scale_fill_brewer(name = 'Animal abuse\nnotifications', palette = 8) +
#'     scalebar(map.df, dist = 5, dd2km = TRUE, model = 'WGS84')
#'
scalebar <- function(data = NULL, location = "bottomright", dist, height = 0.02, st.dist = 0.02, st.bottom = TRUE, st.size = 5, dd2km = NULL, model, x.min, x.max, y.min, y.max, anchor = NULL, facet.var = NULL, facet.lev = NULL){
    if (is.null(data)) {
        if (is.null(x.min) | is.null(x.max) |
            is.null(y.min) | is.null(y.max) ) {
            stop('If data is not defined, x.min, x.max, y.min and y.max must be.')
        }
        data <- data.frame(long = c(x.min, x.max), lat = c(y.min, y.max))
    }
    if (location == 'bottomleft') {
        if (is.null(anchor)) {
            x <- min(data$long)
            y <- min(data$lat)
        } else {
            x <- as.numeric(anchor['x'])
            y <- as.numeric(anchor['y'])
        }
        direction <- 1
        
    }
    if (location == 'bottomright') {
        if (is.null(anchor)) {
            x <- max(data$long)
            y <- min(data$lat)
        } else {
            x <- as.numeric(anchor['x'])
            y <- as.numeric(anchor['y'])
        }
        direction <- -1
        
    }
    if (location == 'topleft') {
        if (is.null(anchor)) {
            x <- min(data$long)
            y <- max(data$lat)
        } else {
            x <- as.numeric(anchor['x'])
            y <- as.numeric(anchor['y'])
        }
        direction <- 1
        
    }
    if (location == 'topright') {
        if (is.null(anchor)) {
            x <- max(data$long)
            y <- max(data$lat)
        } else {
            x <- as.numeric(anchor['x'])
            y <- as.numeric(anchor['y'])
        }
        direction <- -1
        
    }
    if (!st.bottom) {
        st.dist <-
            y + (max(data$lat) - min(data$lat)) * (height + st.dist)
    } else {
        st.dist <- y - (max(data$lat) - min(data$lat)) * st.dist
    }
    height <- y + (max(data$lat) - min(data$lat)) * height
    
    if (!is.null(dd2km)) {
        break1 <- gcDestination(lon = x, lat = y, bearing = 90 * direction,
                                dist = dist, dist.units = 'km',
                                model = model)[1, 1]
        break2 <- gcDestination(lon = x, lat = y, bearing = 90 * direction,
                                dist = dist*2, dist.units = 'km',
                                model = model)[1, 1]
    } else {
        if (location == 'bottomleft' | location == 'topleft') {
            break1 <- x + dist * 1e3
            break2 <- x + dist * 2e3
        } else {
            break1 <- x - dist * 1e3
            break2 <- x - dist * 2e3
        }
        
    }
    box1 <- data.frame(x = c(x, x, rep(break1, 2), x),
                       y = c(y, height, height,y, y), group = 1)
    box2 <- data.frame(x = c(rep(break1, 2), rep(break2, 2), break1),
                       y=c(y, rep(height, 2), y, y), group = 1)
    if (!is.null(facet.var) & !is.null(facet.lev)) {
        box1[ , facet.var] <- facet.lev
        box2[ , facet.var] <- facet.lev
    }
    legend <- data.frame(text = c(0, dist, dist * 2))
    
    gg.box1 <- geom_polygon(data = box1, aes(x, y), fill = 'white',
                            color = 'black')
    gg.box2 <- geom_polygon(data = box2, aes(x, y), fill = 'black',
                            color = 'black')
    x.st.pos <- c(box1[c(1, 3), 1], box2[3, 1])
    if (location == 'bottomright' | location == 'topright') {
        x.st.pos <- rev(x.st.pos)
    }
    legend2 <- cbind(data[1:3, ], x = x.st.pos, y = st.dist,
                          label = paste0(legend[, "text"], "km"))
    if (!is.null(facet.var) & !is.null(facet.lev)) {
        legend2[ , facet.var] <- facet.lev
        legend2[ , facet.var] <- facet.lev
    }
    if (!is.null(facet.var) & !is.null(facet.lev)) {
        gg.legend <- geom_text(data = legend2, aes(x, y, label = label),
                               size = st.size)
    } else {
        gg.legend <- annotate('text', label = paste0(legend[,'text'], 'km'),
                              x = x.st.pos, y = st.dist, size = st.size)    
    }
    return(list(gg.box1, gg.box2, gg.legend))
}