% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/giniCI.R
\name{giniCI}
\alias{giniCI}
\title{Computing Gini-based Composite Indicators}
\usage{
giniCI(inds, method = c("equal", "gini", "reci"),
       agg = c("ari", "geo"), hv = TRUE, ci.pol = c("pos", "neg"),
       time = NULL, ref.time = NULL, only.ci = FALSE)
}
\arguments{
\item{inds}{a matrix or data frame of indicators to be aggregated.}

\item{method}{weighting method to be used. See \code{'Details'}.}

\item{agg}{aggregation function to be used, with \code{"ari"} denoting the
weighted arithmetic mean and \code{"geo"} denoting the weighted geometric mean.}

\item{hv}{a logical value indicating whether horizontal variability adjustment
should be applied.}

\item{ci.pol}{a character value indicating the polarity of composite index.
Use \code{"pos"} if increasing values of the composite index correspond to
positive variations of the phenomenon (e.g., socio-economic developments).
Otherwise, use \code{"neg"} if increasing values of the composite index
correspond to to negative variations of the phenomenon (e.g., vulnerability
and poverty).}

\item{time}{a vector of temporal factors for indicators. The length of \code{time}
must equal the number of rows in \code{inds}. If \code{NULL}, the input indicators
are treated as cross-sectional data.}

\item{ref.time}{a value denoting the reference time for weighting. If
provided, weights will be derived using only observations at the reference
time.}

\item{only.ci}{a logical value indicating whether only the composite index
should be returned.}
}
\value{
A list containing the following components:
\item{ci}{the composite index.}
\item{w}{the weights assigned.}
\item{pen}{the horizontal variability penalties (if \code{hv = TRUE}).}
If \code{only.ci = TRUE}, the function will return only the composite index.
}
\description{
Compute a composite index with weighting schemes based on the Gini
coefficient of constituent indicators, with options for aggregation methods
and  horizontal variability adjustment.
}
\details{
The defaut method is \code{"equal"} that produces equal weights \eqn{1/n} where
\eqn{n} is the number of indicators. For methods \code{"gini"} (Gini-based
weighting) and \code{"reci"} (reciprocal Gini-based weighting), weights are
defined based on the Gini coefficient of indicators. Let \eqn{G_i} be the
Gini coefficient of the \eqn{i}-th indicator, the weights by methods
\code{"gini"} and \code{"reci"} are respectively computed as
\eqn{w^\mathrm{gini}_i = \frac{G_i}{\sum^n_{i = 1} G_i}} and
\eqn{w^\mathrm{reci}_i = \frac{1 / G_i}{\sum^n_{i = 1} 1 / G_i}}.

Temporal factors can be applied to methods \code{"gini"} and \code{"reci"}. If either
\code{time} or \code{ref.time} is \code{NULL}, the weighting process is run on all
observations. If both \code{time} and \code{ref.time} are not \code{NULL}, only observations
at the reference time are used for weight computation.

When aggregating the indicators, the aggregate score for the \eqn{j}-th unit
is computed by applying the chosen aggregation function with the obtained
weights to values in the \eqn{j}-th row. If \code{hv = TRUE}, horizontal
variability adjustment is executed by introducing a penalty for units with
unbalanced values among dimensions. The penalty for the \eqn{j}-th unit is
defined as the the index of dispersion (variance-to-mean ratio) of values in
the \eqn{j}-th row. If \code{ci.pol = "pos"}, the penalties is subtracted from the
aggregate scores to form the composite index. If \code{ci.pol = "neg"} the
penalties is added to the aggregate scores to form the composite index.
}
\note{
Methods \code{"gini"} and \code{"reci"} require non-negative indicators for the
calculation of Gini coefficients. In addition, option \code{hv = TRUE} cannot be
used if any row contains negative values. Therefore, it may be necessary to
use \code{\link{normalize}} to scale the indicators to non-negative ranges before
computing the composite index.

A Gini coefficient of zero occurs when the indicators are constant or do not
change over the reference time. If a zero Gini coefficient is obtained for
for one or more indicators, method \code{"gini"} returns the corresponding weights
as zero while method \code{"reci"} cannot be applied.
}
\examples{
data(bli)

# Indicator polarity
bli.pol = c("neg", "pos", "pos", "pos", "pos", "neg",
            "pos", "pos", "pos", "neg", "pos")

# Goalpost normalization without using time factors
bli.norm <- normalize(inds = bli[, 3:13], method = "goalpost",
                      ind.pol = bli.pol)

# Goalpost normalization using time factors and a reference time
bli.norm.2014 <- normalize(inds = bli[, 3:13], method = "goalpost",
                           ind.pol = bli.pol, time = bli$YEAR,
                           ref.time = 2014)

# Adjusted Mazziotta-Pareto index
bli.ampi <- giniCI(bli.norm, ci.pol = "pos")
bli.ampi$ci

# Gini-based weighted arithmetic mean with reference time
bli.gini <- giniCI(bli.norm.2014, method = "gini", ci.pol = "pos",
                   time = bli$YEAR, ref.time = 2014)
bli.gini$ci
bli.gini$w

# Reciprocal Gini-based weighted geometric mean with reference time
bli.reci <- giniCI(bli.norm.2014, method = "reci", agg = "geo",
                   ci.pol = "pos", time = bli$YEAR, ref.time = 2014)
bli.reci$ci
bli.reci$w

}
\references{
Mazziotta, M. and Pareto, A. (2016). On a Generalized Non-compensatory
Composite Index for Measuring Socio-economic Phenomena. \emph{Social
Indicators Research}, \bold{127}, 983--1003.

Ciommi, M., Gigliarano, C., Emili, A., Taralli, S., and Chelli, F. M. (2017).
A new class of composite indicators for measuring well-being at the local
level: An application to the Equitable and Sustainable Well-being (BES) of
the Italian Provinces. \emph{Ecological Indicators}, \bold{76}, 281--296.
}
\seealso{
\code{\link{normalize}}, \code{\link{rankComp}}.
}
\author{
Viet Duong Nguyen, Chiara Gigliarano, Mariateresa Ciommi
}
