/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5HFmodule.h" 

#include "H5private.h"   
#include "H5B2private.h" 
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5FLprivate.h" 
#include "H5HFpkg.h"     
#include "H5MFprivate.h" 

typedef struct H5HF_huge_bt2_ctx_t {
    uint8_t sizeof_size; 
    uint8_t sizeof_addr; 
} H5HF_huge_bt2_ctx_t;

static void  *H5HF__huge_bt2_crt_context(void *udata);
static herr_t H5HF__huge_bt2_dst_context(void *ctx);

static herr_t H5HF__huge_bt2_indir_store(void *native, const void *udata);
static herr_t H5HF__huge_bt2_indir_compare(const void *rec1, const void *rec2, int *result);
static herr_t H5HF__huge_bt2_indir_encode(uint8_t *raw, const void *native, void *ctx);
static herr_t H5HF__huge_bt2_indir_decode(const uint8_t *raw, void *native, void *ctx);
static herr_t H5HF__huge_bt2_indir_debug(FILE *stream, int indent, int fwidth, const void *record,
                                         const void *_udata);

static herr_t H5HF__huge_bt2_filt_indir_store(void *native, const void *udata);
static herr_t H5HF__huge_bt2_filt_indir_compare(const void *rec1, const void *rec2, int *result);
static herr_t H5HF__huge_bt2_filt_indir_encode(uint8_t *raw, const void *native, void *ctx);
static herr_t H5HF__huge_bt2_filt_indir_decode(const uint8_t *raw, void *native, void *ctx);
static herr_t H5HF__huge_bt2_filt_indir_debug(FILE *stream, int indent, int fwidth, const void *record,
                                              const void *_udata);

static herr_t H5HF__huge_bt2_dir_store(void *native, const void *udata);
static herr_t H5HF__huge_bt2_dir_compare(const void *rec1, const void *rec2, int *result);
static herr_t H5HF__huge_bt2_dir_encode(uint8_t *raw, const void *native, void *ctx);
static herr_t H5HF__huge_bt2_dir_decode(const uint8_t *raw, void *native, void *ctx);
static herr_t H5HF__huge_bt2_dir_debug(FILE *stream, int indent, int fwidth, const void *record,
                                       const void *_udata);

static herr_t H5HF__huge_bt2_filt_dir_store(void *native, const void *udata);
static herr_t H5HF__huge_bt2_filt_dir_compare(const void *rec1, const void *rec2, int *result);
static herr_t H5HF__huge_bt2_filt_dir_encode(uint8_t *raw, const void *native, void *ctx);
static herr_t H5HF__huge_bt2_filt_dir_decode(const uint8_t *raw, void *native, void *ctx);
static herr_t H5HF__huge_bt2_filt_dir_debug(FILE *stream, int indent, int fwidth, const void *record,
                                            const void *_udata);

const H5B2_class_t H5HF_HUGE_BT2_INDIR[1] = {{
    
    H5B2_FHEAP_HUGE_INDIR_ID,          
    "H5B2_FHEAP_HUGE_INDIR_ID",        
    sizeof(H5HF_huge_bt2_indir_rec_t), 
    H5HF__huge_bt2_crt_context,        
    H5HF__huge_bt2_dst_context,        
    H5HF__huge_bt2_indir_store,        
    H5HF__huge_bt2_indir_compare,      
    H5HF__huge_bt2_indir_encode,       
    H5HF__huge_bt2_indir_decode,       
    H5HF__huge_bt2_indir_debug         
}};

const H5B2_class_t H5HF_HUGE_BT2_FILT_INDIR[1] = {{
    
    H5B2_FHEAP_HUGE_FILT_INDIR_ID,          
    "H5B2_FHEAP_HUGE_FILT_INDIR_ID",        
    sizeof(H5HF_huge_bt2_filt_indir_rec_t), 
    H5HF__huge_bt2_crt_context,             
    H5HF__huge_bt2_dst_context,             
    H5HF__huge_bt2_filt_indir_store,        
    H5HF__huge_bt2_filt_indir_compare,      
    H5HF__huge_bt2_filt_indir_encode,       
    H5HF__huge_bt2_filt_indir_decode,       
    H5HF__huge_bt2_filt_indir_debug         
}};

const H5B2_class_t H5HF_HUGE_BT2_DIR[1] = {{
    
    H5B2_FHEAP_HUGE_DIR_ID,          
    "H5B2_FHEAP_HUGE_DIR_ID",        
    sizeof(H5HF_huge_bt2_dir_rec_t), 
    H5HF__huge_bt2_crt_context,      
    H5HF__huge_bt2_dst_context,      
    H5HF__huge_bt2_dir_store,        
    H5HF__huge_bt2_dir_compare,      
    H5HF__huge_bt2_dir_encode,       
    H5HF__huge_bt2_dir_decode,       
    H5HF__huge_bt2_dir_debug         
}};

const H5B2_class_t H5HF_HUGE_BT2_FILT_DIR[1] = {{
    
    H5B2_FHEAP_HUGE_FILT_DIR_ID,          
    "H5B2_FHEAP_HUGE_FILT_DIR_ID",        
    sizeof(H5HF_huge_bt2_filt_dir_rec_t), 
    H5HF__huge_bt2_crt_context,           
    H5HF__huge_bt2_dst_context,           
    H5HF__huge_bt2_filt_dir_store,        
    H5HF__huge_bt2_filt_dir_compare,      
    H5HF__huge_bt2_filt_dir_encode,       
    H5HF__huge_bt2_filt_dir_decode,       
    H5HF__huge_bt2_filt_dir_debug         
}};

H5FL_DEFINE_STATIC(H5HF_huge_bt2_ctx_t);

static void *
H5HF__huge_bt2_crt_context(void *_f)
{
    H5F_t               *f = (H5F_t *)_f;  
    H5HF_huge_bt2_ctx_t *ctx;              
    void                *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(f);

    
    if (NULL == (ctx = H5FL_MALLOC(H5HF_huge_bt2_ctx_t)))
        HGOTO_ERROR(H5E_HEAP, H5E_CANTALLOC, NULL, "can't allocate callback context");

    
    ctx->sizeof_addr = H5F_SIZEOF_ADDR(f);
    ctx->sizeof_size = H5F_SIZEOF_SIZE(f);

    
    ret_value = ctx;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__huge_bt2_dst_context(void *_ctx)
{
    H5HF_huge_bt2_ctx_t *ctx = (H5HF_huge_bt2_ctx_t *)_ctx; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(ctx);

    
    ctx = H5FL_FREE(H5HF_huge_bt2_ctx_t, ctx);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF__huge_bt2_indir_found(const void *nrecord, void *op_data)
{
    FUNC_ENTER_PACKAGE_NOERR

    *(H5HF_huge_bt2_indir_rec_t *)op_data = *(const H5HF_huge_bt2_indir_rec_t *)nrecord;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF__huge_bt2_indir_remove(const void *nrecord, void *_udata)
{
    H5HF_huge_remove_ud_t *udata     = (H5HF_huge_remove_ud_t *)_udata; 
    herr_t                 ret_value = SUCCEED;                         

    FUNC_ENTER_PACKAGE

    
    if (H5MF_xfree(udata->hdr->f, H5FD_MEM_FHEAP_HUGE_OBJ, ((const H5HF_huge_bt2_indir_rec_t *)nrecord)->addr,
                   ((const H5HF_huge_bt2_indir_rec_t *)nrecord)->len) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "unable to free space for huge object on disk");

    
    udata->obj_len = ((const H5HF_huge_bt2_indir_rec_t *)nrecord)->len;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__huge_bt2_indir_store(void *nrecord, const void *udata)
{
    FUNC_ENTER_PACKAGE_NOERR

    *(H5HF_huge_bt2_indir_rec_t *)nrecord = *(const H5HF_huge_bt2_indir_rec_t *)udata;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HF__huge_bt2_indir_compare(const void *_rec1, const void *_rec2, int *result)
{
    FUNC_ENTER_PACKAGE_NOERR

    *result = (int)(((const H5HF_huge_bt2_indir_rec_t *)_rec1)->id -
                    ((const H5HF_huge_bt2_indir_rec_t *)_rec2)->id);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HF__huge_bt2_indir_encode(uint8_t *raw, const void *_nrecord, void *_ctx)
{
    H5HF_huge_bt2_ctx_t             *ctx     = (H5HF_huge_bt2_ctx_t *)_ctx; 
    const H5HF_huge_bt2_indir_rec_t *nrecord = (const H5HF_huge_bt2_indir_rec_t *)_nrecord;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(ctx);

    
    H5F_addr_encode_len(ctx->sizeof_addr, &raw, nrecord->addr);
    H5_ENCODE_LENGTH_LEN(raw, nrecord->len, ctx->sizeof_size);
    H5_ENCODE_LENGTH_LEN(raw, nrecord->id, ctx->sizeof_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HF__huge_bt2_indir_decode(const uint8_t *raw, void *_nrecord, void *_ctx)
{
    H5HF_huge_bt2_ctx_t       *ctx     = (H5HF_huge_bt2_ctx_t *)_ctx; 
    H5HF_huge_bt2_indir_rec_t *nrecord = (H5HF_huge_bt2_indir_rec_t *)_nrecord;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(ctx);

    
    H5F_addr_decode_len(ctx->sizeof_addr, &raw, &nrecord->addr);
    H5_DECODE_LENGTH_LEN(raw, nrecord->len, ctx->sizeof_size);
    H5_DECODE_LENGTH_LEN(raw, nrecord->id, ctx->sizeof_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HF__huge_bt2_indir_debug(FILE *stream, int indent, int fwidth, const void *_nrecord,
                           const void H5_ATTR_UNUSED *_udata)
{
    const H5HF_huge_bt2_indir_rec_t *nrecord = (const H5HF_huge_bt2_indir_rec_t *)_nrecord;

    FUNC_ENTER_PACKAGE_NOERR

    Rfprintf(stream, "%*s%-*s {%" PRIuHADDR ", %" PRIuHSIZE ", %" PRIuHSIZE "}\n", indent, "", fwidth,
            "Record:", nrecord->addr, nrecord->len, nrecord->id);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF__huge_bt2_filt_indir_found(const void *nrecord, void *op_data)
{
    FUNC_ENTER_PACKAGE_NOERR

    *(H5HF_huge_bt2_filt_indir_rec_t *)op_data = *(const H5HF_huge_bt2_filt_indir_rec_t *)nrecord;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF__huge_bt2_filt_indir_remove(const void *nrecord, void *_udata)
{
    H5HF_huge_remove_ud_t *udata     = (H5HF_huge_remove_ud_t *)_udata; 
    herr_t                 ret_value = SUCCEED;                         

    FUNC_ENTER_PACKAGE

    
    if (H5MF_xfree(udata->hdr->f, H5FD_MEM_FHEAP_HUGE_OBJ,
                   ((const H5HF_huge_bt2_filt_indir_rec_t *)nrecord)->addr,
                   ((const H5HF_huge_bt2_filt_indir_rec_t *)nrecord)->len) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "unable to free space for huge object on disk");

    
    udata->obj_len = ((const H5HF_huge_bt2_filt_indir_rec_t *)nrecord)->obj_size;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__huge_bt2_filt_indir_store(void *nrecord, const void *udata)
{
    FUNC_ENTER_PACKAGE_NOERR

    *(H5HF_huge_bt2_filt_indir_rec_t *)nrecord = *(const H5HF_huge_bt2_filt_indir_rec_t *)udata;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HF__huge_bt2_filt_indir_compare(const void *_rec1, const void *_rec2, int *result)
{
    FUNC_ENTER_PACKAGE_NOERR

    *result = (int)(((const H5HF_huge_bt2_filt_indir_rec_t *)_rec1)->id -
                    ((const H5HF_huge_bt2_filt_indir_rec_t *)_rec2)->id);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HF__huge_bt2_filt_indir_encode(uint8_t *raw, const void *_nrecord, void *_ctx)
{
    H5HF_huge_bt2_ctx_t                  *ctx = (H5HF_huge_bt2_ctx_t *)_ctx; 
    const H5HF_huge_bt2_filt_indir_rec_t *nrecord = (const H5HF_huge_bt2_filt_indir_rec_t *)_nrecord;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(ctx);

    
    H5F_addr_encode_len(ctx->sizeof_addr, &raw, nrecord->addr);
    H5_ENCODE_LENGTH_LEN(raw, nrecord->len, ctx->sizeof_size);
    UINT32ENCODE(raw, nrecord->filter_mask);
    H5_ENCODE_LENGTH_LEN(raw, nrecord->obj_size, ctx->sizeof_size);
    H5_ENCODE_LENGTH_LEN(raw, nrecord->id, ctx->sizeof_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HF__huge_bt2_filt_indir_decode(const uint8_t *raw, void *_nrecord, void *_ctx)
{
    H5HF_huge_bt2_ctx_t            *ctx     = (H5HF_huge_bt2_ctx_t *)_ctx; 
    H5HF_huge_bt2_filt_indir_rec_t *nrecord = (H5HF_huge_bt2_filt_indir_rec_t *)_nrecord;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(ctx);

    
    H5F_addr_decode_len(ctx->sizeof_addr, &raw, &nrecord->addr);
    H5_DECODE_LENGTH_LEN(raw, nrecord->len, ctx->sizeof_size);
    UINT32DECODE(raw, nrecord->filter_mask);
    H5_DECODE_LENGTH_LEN(raw, nrecord->obj_size, ctx->sizeof_size);
    H5_DECODE_LENGTH_LEN(raw, nrecord->id, ctx->sizeof_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HF__huge_bt2_filt_indir_debug(FILE *stream, int indent, int fwidth, const void *_nrecord,
                                const void H5_ATTR_UNUSED *_udata)
{
    const H5HF_huge_bt2_filt_indir_rec_t *nrecord = (const H5HF_huge_bt2_filt_indir_rec_t *)_nrecord;

    FUNC_ENTER_PACKAGE_NOERR

    Rfprintf(stream, "%*s%-*s {%" PRIuHADDR ", %" PRIuHSIZE ", %x, %" PRIuHSIZE ", %" PRIuHSIZE "}\n", indent,
            "", fwidth, "Record:", nrecord->addr, nrecord->len, nrecord->filter_mask, nrecord->obj_size,
            nrecord->id);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF__huge_bt2_dir_remove(const void *nrecord, void *_udata)
{
    H5HF_huge_remove_ud_t *udata     = (H5HF_huge_remove_ud_t *)_udata; 
    herr_t                 ret_value = SUCCEED;                         

    FUNC_ENTER_PACKAGE

    
    if (H5MF_xfree(udata->hdr->f, H5FD_MEM_FHEAP_HUGE_OBJ, ((const H5HF_huge_bt2_indir_rec_t *)nrecord)->addr,
                   ((const H5HF_huge_bt2_indir_rec_t *)nrecord)->len) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "unable to free space for huge object on disk");

    
    udata->obj_len = ((const H5HF_huge_bt2_indir_rec_t *)nrecord)->len;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__huge_bt2_dir_store(void *nrecord, const void *udata)
{
    FUNC_ENTER_PACKAGE_NOERR

    *(H5HF_huge_bt2_dir_rec_t *)nrecord = *(const H5HF_huge_bt2_dir_rec_t *)udata;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HF__huge_bt2_dir_compare(const void *_rec1, const void *_rec2, int *result)
{
    const H5HF_huge_bt2_dir_rec_t *rec1 = (const H5HF_huge_bt2_dir_rec_t *)_rec1;
    const H5HF_huge_bt2_dir_rec_t *rec2 = (const H5HF_huge_bt2_dir_rec_t *)_rec2;

    FUNC_ENTER_PACKAGE_NOERR

    if (rec1->addr < rec2->addr)
        *result = -1;
    else if (rec1->addr > rec2->addr)
        *result = 1;
    else if (rec1->len < rec2->len)
        *result = -1;
    else if (rec1->len > rec2->len)
        *result = 1;
    else
        *result = 0;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HF__huge_bt2_dir_encode(uint8_t *raw, const void *_nrecord, void *_ctx)
{
    H5HF_huge_bt2_ctx_t           *ctx     = (H5HF_huge_bt2_ctx_t *)_ctx; 
    const H5HF_huge_bt2_dir_rec_t *nrecord = (const H5HF_huge_bt2_dir_rec_t *)_nrecord;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(ctx);

    
    H5F_addr_encode_len(ctx->sizeof_addr, &raw, nrecord->addr);
    H5_ENCODE_LENGTH_LEN(raw, nrecord->len, ctx->sizeof_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HF__huge_bt2_dir_decode(const uint8_t *raw, void *_nrecord, void *_ctx)
{
    H5HF_huge_bt2_ctx_t     *ctx     = (H5HF_huge_bt2_ctx_t *)_ctx; 
    H5HF_huge_bt2_dir_rec_t *nrecord = (H5HF_huge_bt2_dir_rec_t *)_nrecord;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(ctx);

    
    H5F_addr_decode_len(ctx->sizeof_addr, &raw, &nrecord->addr);
    H5_DECODE_LENGTH_LEN(raw, nrecord->len, ctx->sizeof_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HF__huge_bt2_dir_debug(FILE *stream, int indent, int fwidth, const void *_nrecord,
                         const void H5_ATTR_UNUSED *_udata)
{
    const H5HF_huge_bt2_dir_rec_t *nrecord = (const H5HF_huge_bt2_dir_rec_t *)_nrecord;

    FUNC_ENTER_PACKAGE_NOERR

    Rfprintf(stream, "%*s%-*s {%" PRIuHADDR ", %" PRIuHSIZE "}\n", indent, "", fwidth,
            "Record:", nrecord->addr, nrecord->len);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF__huge_bt2_filt_dir_found(const void *nrecord, void *op_data)
{
    FUNC_ENTER_PACKAGE_NOERR

    *(H5HF_huge_bt2_filt_dir_rec_t *)op_data = *(const H5HF_huge_bt2_filt_dir_rec_t *)nrecord;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF__huge_bt2_filt_dir_remove(const void *nrecord, void *_udata)
{
    H5HF_huge_remove_ud_t *udata     = (H5HF_huge_remove_ud_t *)_udata; 
    herr_t                 ret_value = SUCCEED;                         

    FUNC_ENTER_PACKAGE

    
    if (H5MF_xfree(udata->hdr->f, H5FD_MEM_FHEAP_HUGE_OBJ,
                   ((const H5HF_huge_bt2_filt_dir_rec_t *)nrecord)->addr,
                   ((const H5HF_huge_bt2_filt_dir_rec_t *)nrecord)->len) < 0)
        HGOTO_ERROR(H5E_HEAP, H5E_CANTFREE, FAIL, "unable to free space for huge object on disk");

    
    udata->obj_len = ((const H5HF_huge_bt2_filt_dir_rec_t *)nrecord)->obj_size;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5HF__huge_bt2_filt_dir_store(void *nrecord, const void *udata)
{
    FUNC_ENTER_PACKAGE_NOERR

    *(H5HF_huge_bt2_filt_dir_rec_t *)nrecord = *(const H5HF_huge_bt2_filt_dir_rec_t *)udata;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HF__huge_bt2_filt_dir_compare(const void *_rec1, const void *_rec2, int *result)
{
    const H5HF_huge_bt2_filt_dir_rec_t *rec1 = (const H5HF_huge_bt2_filt_dir_rec_t *)_rec1;
    const H5HF_huge_bt2_filt_dir_rec_t *rec2 = (const H5HF_huge_bt2_filt_dir_rec_t *)_rec2;

    FUNC_ENTER_PACKAGE_NOERR

    if (rec1->addr < rec2->addr)
        *result = -1;
    else if (rec1->addr > rec2->addr)
        *result = 1;
    else if (rec1->len < rec2->len)
        *result = -1;
    else if (rec1->len > rec2->len)
        *result = 1;
    else
        *result = 0;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HF__huge_bt2_filt_dir_encode(uint8_t *raw, const void *_nrecord, void *_ctx)
{
    H5HF_huge_bt2_ctx_t                *ctx = (H5HF_huge_bt2_ctx_t *)_ctx; 
    const H5HF_huge_bt2_filt_dir_rec_t *nrecord = (const H5HF_huge_bt2_filt_dir_rec_t *)_nrecord;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(ctx);

    
    H5F_addr_encode_len(ctx->sizeof_addr, &raw, nrecord->addr);
    H5_ENCODE_LENGTH_LEN(raw, nrecord->len, ctx->sizeof_size);
    UINT32ENCODE(raw, nrecord->filter_mask);
    H5_ENCODE_LENGTH_LEN(raw, nrecord->obj_size, ctx->sizeof_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HF__huge_bt2_filt_dir_decode(const uint8_t *raw, void *_nrecord, void *_ctx)
{
    H5HF_huge_bt2_ctx_t          *ctx     = (H5HF_huge_bt2_ctx_t *)_ctx; 
    H5HF_huge_bt2_filt_dir_rec_t *nrecord = (H5HF_huge_bt2_filt_dir_rec_t *)_nrecord;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(ctx);

    
    H5F_addr_decode_len(ctx->sizeof_addr, &raw, &nrecord->addr);
    H5_DECODE_LENGTH_LEN(raw, nrecord->len, ctx->sizeof_size);
    UINT32DECODE(raw, nrecord->filter_mask);
    H5_DECODE_LENGTH_LEN(raw, nrecord->obj_size, ctx->sizeof_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5HF__huge_bt2_filt_dir_debug(FILE *stream, int indent, int fwidth, const void *_nrecord,
                              const void H5_ATTR_UNUSED *_udata)
{
    const H5HF_huge_bt2_filt_dir_rec_t *nrecord = (const H5HF_huge_bt2_filt_dir_rec_t *)_nrecord;

    FUNC_ENTER_PACKAGE_NOERR

    Rfprintf(stream, "%*s%-*s {%" PRIuHADDR ", %" PRIuHSIZE ", %x, %" PRIuHSIZE "}\n", indent, "", fwidth,
            "Record:", nrecord->addr, nrecord->len, nrecord->filter_mask, nrecord->obj_size);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
