/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5PLmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5MMprivate.h" 
#include "H5PLpkg.h"     

#define H5PL_INITIAL_CACHE_CAPACITY 16

#define H5PL_CACHE_CAPACITY_ADD 16

typedef struct H5PL_plugin_t {
    H5PL_type_t type;   
    H5PL_key_t  key;    
    H5PL_HANDLE handle; 
} H5PL_plugin_t;

static herr_t H5PL__expand_cache(void);

static H5PL_plugin_t *H5PL_cache_g = NULL;

static unsigned int H5PL_num_plugins_g = 0;

static unsigned int H5PL_cache_capacity_g = 0;

herr_t
H5PL__create_plugin_cache(void)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    H5PL_num_plugins_g = 0;

    H5PL_cache_capacity_g = H5PL_INITIAL_CACHE_CAPACITY;

    if (NULL ==
        (H5PL_cache_g = (H5PL_plugin_t *)H5MM_calloc((size_t)H5PL_cache_capacity_g * sizeof(H5PL_plugin_t))))
        HGOTO_ERROR(H5E_PLUGIN, H5E_CANTALLOC, FAIL, "can't allocate memory for plugin cache");

done:
    
    if (FAIL == ret_value) {
        if (H5PL_cache_g)
            H5PL_cache_g = (H5PL_plugin_t *)H5MM_xfree(H5PL_cache_g);
        H5PL_cache_capacity_g = 0;
    }

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5PL__close_plugin_cache(bool *already_closed )
{
    unsigned int u; 
    herr_t       ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE_NOERR

    
    if (H5PL_cache_g) {

        
        for (u = 0; u < H5PL_num_plugins_g; u++)
            H5PL__close((H5PL_cache_g[u]).handle);

        
        H5PL_cache_g          = (H5PL_plugin_t *)H5MM_xfree(H5PL_cache_g);
        H5PL_num_plugins_g    = 0;
        H5PL_cache_capacity_g = 0;

        
        *already_closed = false;
    }
    else
        *already_closed = true;

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5PL__expand_cache(void)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    H5PL_cache_capacity_g += H5PL_CACHE_CAPACITY_ADD;

    
    if (NULL == (H5PL_cache_g = (H5PL_plugin_t *)H5MM_realloc(H5PL_cache_g, (size_t)H5PL_cache_capacity_g *
                                                                                sizeof(H5PL_plugin_t))))
        HGOTO_ERROR(H5E_PLUGIN, H5E_CANTALLOC, FAIL, "allocating additional memory for plugin cache failed");

    
    memset(H5PL_cache_g + H5PL_num_plugins_g, 0, (size_t)H5PL_CACHE_CAPACITY_ADD * sizeof(H5PL_plugin_t));

done:
    
    if (FAIL == ret_value)
        H5PL_cache_capacity_g -= H5PL_CACHE_CAPACITY_ADD;

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5PL__add_plugin(H5PL_type_t type, const H5PL_key_t *key, H5PL_HANDLE handle)
{
    herr_t ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    if (H5PL_num_plugins_g >= H5PL_cache_capacity_g)
        if (H5PL__expand_cache() < 0)
            HGOTO_ERROR(H5E_PLUGIN, H5E_CANTALLOC, FAIL, "can't expand plugin cache");

    
    H5PL_cache_g[H5PL_num_plugins_g].type   = type;
    H5PL_cache_g[H5PL_num_plugins_g].key    = *key;
    H5PL_cache_g[H5PL_num_plugins_g].handle = handle;

    H5PL_num_plugins_g++;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5PL__find_plugin_in_cache(const H5PL_search_params_t *search_params, bool *found, const void **plugin_info)
{
    unsigned int u; 
    herr_t       ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    
    assert(search_params);
    assert(found);
    assert(plugin_info);

    
    *found       = false;
    *plugin_info = NULL;

    
    for (u = 0; u < H5PL_num_plugins_g; u++) {
        bool matched = false; 

        
        if (search_params->type != H5PL_cache_g[u].type)
            continue;

        
        switch (search_params->type) {
            case H5PL_TYPE_FILTER:
                
                if (search_params->key->id == H5PL_cache_g[u].key.id)
                    matched = true;

                break;

            case H5PL_TYPE_VOL:
                if (search_params->key->vol.kind == H5VL_GET_CONNECTOR_BY_NAME) {
                    
                    if (H5PL_cache_g[u].key.vol.kind != H5VL_GET_CONNECTOR_BY_NAME)
                        continue;

                    
                    if (!strcmp(search_params->key->vol.u.name, H5PL_cache_g[u].key.vol.u.name))
                        matched = true;
                }
                else {
                    assert(search_params->key->vol.kind == H5VL_GET_CONNECTOR_BY_VALUE);

                    
                    if (H5PL_cache_g[u].key.vol.kind != H5VL_GET_CONNECTOR_BY_VALUE)
                        continue;

                    
                    if (search_params->key->vol.u.value == H5PL_cache_g[u].key.vol.u.value)
                        matched = true;
                }

                break;

            case H5PL_TYPE_VFD:
                if (search_params->key->vfd.kind == H5FD_GET_DRIVER_BY_NAME) {
                    
                    if (H5PL_cache_g[u].key.vfd.kind != H5FD_GET_DRIVER_BY_NAME)
                        continue;

                    
                    if (!strcmp(search_params->key->vfd.u.name, H5PL_cache_g[u].key.vfd.u.name))
                        matched = true;
                }
                else {
                    assert(search_params->key->vfd.kind == H5FD_GET_DRIVER_BY_VALUE);

                    
                    if (H5PL_cache_g[u].key.vfd.kind != H5FD_GET_DRIVER_BY_VALUE)
                        continue;

                    
                    if (search_params->key->vfd.u.value == H5PL_cache_g[u].key.vfd.u.value)
                        matched = true;
                }

                break;

            case H5PL_TYPE_ERROR:
            case H5PL_TYPE_NONE:
            default:
                HGOTO_ERROR(H5E_PLUGIN, H5E_CANTGET, FAIL, "Invalid plugin type specified");
        }

        
        if (matched) {
            H5PL_get_plugin_info_t get_plugin_info_function;
            const void            *info;

            
            H5_WARN_OBJ_FXN_POINTER_CONVERSION_OFF
            if (NULL == (get_plugin_info_function = (H5PL_get_plugin_info_t)H5PL_GET_LIB_FUNC(
                             H5PL_cache_g[u].handle, "H5PLget_plugin_info")))
                HGOTO_ERROR(H5E_PLUGIN, H5E_CANTGET, FAIL, "can't get function for H5PLget_plugin_info");
            H5_WARN_OBJ_FXN_POINTER_CONVERSION_ON

            
            if (NULL == (info = (*get_plugin_info_function)()))
                HGOTO_ERROR(H5E_PLUGIN, H5E_CANTGET, FAIL, "can't get plugin info");

            
            *found       = true;
            *plugin_info = info;

            
            break;
        }
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
