# from Keith Jewell - https://stat.ethz.ch/pipermail/r-help/2009-December/222371.html
inhull <- function(testpts, calpts, hull=geometry::convhulln(calpts), tol=mean(mean(abs(calpts)))*sqrt(.Machine$double.eps)) {
#++++++++++++++++++++
# R implementation of the Matlab code by John D'Errico 04 Mar 2006 (Updated 30 Oct 2006)
# downloaded from http://www.mathworks.com/matlabcentral/fileexchange/10226-inhull
# with some modifications and simplifications
#
# Efficient test for points inside a convex hull in n dimensions
#
#% arguments: (input)
#%  testpts - nxp array to test, n data points, in p dimensions
#%       If you have many points to test, it is most efficient to
#%       call this function once with the entire set.
#%
#%  calpts - mxp array of vertices of the convex hull, as used by
#%       convhulln.
#%
#%  hull - (OPTIONAL) tessellation (or triangulation) generated by convhulln
#%       If hull is left empty or not supplied, then it will be
#%       generated.
#%
#%  tol - (OPTIONAL) tolerance on the tests for inclusion in the
#%       convex hull. You can think of tol as the distance a point
#%       may possibly lie outside the hull, and still be perceived
#%       as on the surface of the hull. Because of numerical slop
#%       nothing can ever be done exactly here. I might guess a
#%       semi-intelligent value of tol to be
#%
#%         tol = 1.e-13*mean(abs(calpts(:)))
#%
#%       In higher dimensions, the numerical issues of floating
#%       point arithmetic will probably suggest a larger value
#%       of tol.
#%
# In this R implementation default tol=mean(mean(abs(calpts)))*sqrt(.Machine$double.eps)
#       DEFAULT: tol = 1e-6
#
# VALUE: Matlab returns a vector of TRUE (inside/on) or FALSE (outside)
#       This R implementation returns an integer vector of length n
#       1 = inside hull
#      -1 = inside hull
#       0 = on hull (to precision indicated by tol)
#--------------------------------------------------------
# ensure arguments are matrices (not data frames) and get sizes
   calpts <- as.matrix(calpts)
   testpts <- as.matrix(testpts)
   p <- dim(calpts)[2]   # columns in calpts
   cx <- dim(testpts)[1]  # rows in testpts
   nt <- dim(hull)[1]    # number of simplexes in hull
# find normal vectors to each simplex
   nrmls <- matrix(NA, nt, p)         # predefine each nrml as NA, degenerate
   degenflag <- matrix(TRUE, nt, 1)
   for (i in  1:nt) {
    nullsp <- t(MASS::Null(t(calpts[hull[i,-1],] - matrix(calpts[hull[i,1],],p-1,p, byrow=TRUE))))
    if (dim(nullsp)[1] == 1) { nrmls[i,] <- nullsp
       degenflag[i] <- FALSE}}
# Warn of degenerate faces, and remove corresponding normals
   if(length(degenflag[degenflag]) > 0) 
warning(length(degenflag[degenflag])," degenerate faces in convex hull")
   nrmls <- nrmls[!degenflag,]
   nt <- dim(nrmls)[1]
# find center point in hull, and any (1st) point in the plane of each simplex
   center = apply(calpts, 2, mean)
   a <- calpts[hull[!degenflag,1],]
# scale normal vectors to unit length and ensure pointing inwards
   nrmls <- nrmls/matrix(apply(nrmls, 1, function(x) sqrt(sum(x^2))), nt, p)
   dp <- sign(apply((matrix(center, nt, p, byrow=TRUE)-a) * nrmls, 1, sum))
   nrmls <- nrmls*matrix(dp, nt, p)
# if  min across all faces of dot((x - a),nrml) is
#      +ve then x is inside hull
#      0   then x is on hull
#      -ve then x is outside hull
# Instead of dot((x - a),nrml)  use dot(x,nrml) - dot(a, nrml)
   aN <- diag(a %*% t(nrmls))
   val <- apply(testpts %*% t(nrmls) - matrix(aN, cx, nt, byrow=TRUE), 1,min)
# code  values inside 'tol' to zero, return sign as integer
   val[abs(val) < tol] <- 0
   as.integer(sign(val))
}

padrange <- function(data, delta=0.05)
{
	r <- range(data)
	rmin = r[1] - delta * diff(r)
	rmax = r[2] + delta * diff(r)
	
	return(c(rmin, rmax))
}


rangeboxrandom <- function(dataset, npoints)
{
  minv <- apply(dataset,2,min)
  maxv <- apply(dataset,2,max)
  
  result <- matrix(NA, nrow=npoints, ncol=length(minv), dimnames=list(NULL, dimnames(dataset)[[2]]))
  
  for (i in 1:length(minv)) # across each dimension
  {
    result[,i] <- runif(npoints, minv[i], maxv[i])
  }
  
  volume = prod(maxv - minv)
  density = npoints / volume
  bw = (maxv - minv)/2
  
  return(list(rp=result, volume=volume, density=density, bw=bw))
}


nonconvexfeatures <- function(hv, numconvexhullpoints=NULL, memory_reduction_factor=1, check_convexhull=TRUE, check_memory=TRUE, minnumpoints = 5, npointsdesired=1000, inflationfactors=seq(1,2,by=0.2), plotdiag=FALSE)
{
	if (plotdiag)
	{
		if(ncol(hv@Data) != 2)
		{
			warning('Diagnostic graphs only available for n=2 dimensions.')
		}
	}
	
	# set diagnostic plot information
	plotdiag <- plotdiag & (ncol(hv@Data) == 2) # FOR MAKING DIAGNOSTIC PLOTS
	prx <- padrange(hv@Data[,1])
	pry <- padrange(hv@Data[,2])
	if (plotdiag)
	{
		pdf('diag.pdf',width=12,height=8)
		par(mar=c(0,0,0,0))
		par(mfrow=c(2,3))
	}
	
	# initialize a list of results
	finalresult <- vector(mode="list",length=length(inflationfactors))
	names(finalresult) <- paste("Inflation factor ", round(inflationfactors,digits=3))
		
	# make sure we are using a hypervolume with data (i.e. not the output of set operations)
	if(all(is.nan(hv@Data)))
	{
		stop('Hypervolume must be associated with datapoints')
	}	

	if (plotdiag)
	{
		plot(hv@Data[,1], hv@Data[,2],xlim=prx, ylim=pry, axes=F,pch=16,col='black')
		legend('topleft','A',bty='n',cex=1.5,adj=c(2,0))
		box()
	}
	
	if (plotdiag)
	{
		rpx <- hv@RandomUniformPointsThresholded[,1]
		rpy <- hv@RandomUniformPointsThresholded[,2]
		
		ivals <- sample(length(rpx), 5000)
		rpx <- rpx[ivals]
		rpy <- rpy[ivals]
		plot(rpx, rpy, xlim=prx, ylim=pry, axes=F,cex=0.5,pch=16,col='blue')
		points(hv@Data[,1], hv@Data[,2],xlim=prx, ylim=pry, pch=16,col='black')
		legend('topleft','B',bty='n',cex=1.5,adj=c(2,0))
		box()
	}
	
	
	# resample the data to make the convex hull run faster (exponentially slow as dimensionality increases)
	data_reduced <- hv@Data
	if (!is.null(numconvexhullpoints)) # resample data, being more likely to take points far away from center
	{
		if (numconvexhullpoints > nrow(data_reduced))
		{
			numconvexhullpoints = nrow(data_reduced)
			warning("Cannot request more points for convex hull than are in the data. Choosing maximum available.")
		}
		data_reduced <- data_reduced[sample(nrow(data_reduced), numconvexhullpoints, prob=rowSums(scale(data_reduced, center=TRUE, scale=FALSE)^2)),]
	}			
	# FIND THE CONVEX HULL of the data points (does not work on intersection volumes)	
	convexhull <- geometry::convhulln(data_reduced,options="FA")
	hull_matrix <- convexhull$hull #extract coordinates of vertices
	hull_volume <- convexhull$vol # extract volume
	
	cat(sprintf("Convex hull calculated with %.0f simplices.\nCalculation of inequality constraints will require allocation of %.0f double-precision numbers (approximately %.1f GB memory)\n", nrow(hull_matrix), nrow(hull_matrix)^2, nrow(hull_matrix)^2 * 8 / 1024 / 1024 /1024))
	
	if (check_convexhull)
	{
		stop('Set check_convexhull=F to continue.\n')
	}	
  
	# REJECTION SAMPLING FROM BOUNDING BOX TO FILL IN CONVEX POLYTOPE APPROACH  
	ntrials <- 1000 # number of candidate points to propose at a time
	done = FALSE
	inpoints <- NULL
	niter <- 0
	
	# generate a bounding box, and test if each point is in the convex hull or not
	while(!done)
	{
	  # try a set of test points
	  testpoints_rangebox <- rangeboxrandom(data_reduced, ntrials)
	  chullin <- (inhull(testpts= testpoints_rangebox$rp, calpts=data_reduced, hull=hull_matrix)==1)
	  # figure out which are 'in'
	  inpoints_temp <- testpoints_rangebox$rp[chullin,]
	  
	  niter <- niter + 1
	  inpoints <- rbind(inpoints, inpoints_temp)
	  
	  done = nrow(inpoints) >= npointsdesired
	  
	  cat(sprintf('Rejection sampling: iteration %.0f - %.0f / %.0f points accepted\n', niter, nrow(inpoints), npointsdesired))
	}
	finaldensity <- niter * testpoints_rangebox$density
	dimnames(inpoints) <- list(NULL,dimnames(hv@Data)[[2]])
	
	# MAKE a new hypervolume with the convex hull shape and volume
	hv_chull <- new("Hypervolume",
	                RandomUniformPointsThresholded=inpoints, 
	                Dimensionality=ncol(inpoints), 
	                Volume=hull_volume,
	                PointDensity=niter * testpoints_rangebox$density, 
	                Bandwidth= testpoints_rangebox$bw, 
	                RepsPerPoint=ntrials*niter, 
	                QuantileThresholdDesired=0, 
	                QuantileThresholdObtained=0
	)
	
	if (plotdiag)
	{
		plot(hv_chull@RandomUniformPointsThresholded[,1], hv_chull@RandomUniformPointsThresholded[,2],xlim=prx, ylim=pry, axes=F,cex=0.5,pch=16,col='green')
		points(hv@Data[,1], hv@Data[,2],xlim=prx, ylim=pry, pch=16,col='black')
		legend('topleft','C',bty='n',cex=1.5,adj=c(2,0))
		box()
	}	
		
	
	# FIND THE DIFFERENCE between the convex hull shape and the real hypervolume
	cat("Beginning set operations...")
	hvs_overlap <- hypervolume_set(hv, hv_chull, check_memory=check_memory, reduction_factor = memory_reduction_factor)
	cat("Finished set operations.\n")
	
	# find the distance between all points in the difference
	randompoints <- hvs_overlap@HVList$Unique_2@RandomUniformPointsThresholded
	if (is.null(randompoints) || nrow(randompoints) == 0)
	{
		
		message('No random points found in intersection between hypervolume and its convex hull. This may indicate one or more of the following scenarios: 1) no holes, 2) insufficient resolution because npointsdesired is too low (you should specify a larger value), 3) approximation of convex hull is too conservative (you should increase numconvexhullpoints to provide better wrap around data).\n\nFunction will return NULL.')
		
		return(NULL)
	}
	
	if (plotdiag)
	{
		plot(randompoints[,1], randompoints[,2],xlim=prx, ylim=pry, axes=F,cex=0.5,pch=16,col='gray')
		points(hv@Data[,1], hv@Data[,2],xlim=prx, ylim=pry, pch=16,col='black')
		legend('topleft','D',bty='n',cex=1.5,adj=c(2,0))
		box()
	}	
	
	cat(sprintf("Retaining %.0f random points in intersection between hypervolume and convex hull.\nMaking %.0f pairwise calculations...", nrow(randompoints), nrow(randompoints)^2))
	
	distances <- dist(randompoints, method="euclidean")
	
	# infer which points are 'connected' to each other (if they are within the critical distance)	
	criticaldistance <- hvs_overlap@HVList$Unique_2@PointDensity ^(-1/hvs_overlap@HVList$Unique_2@Dimensionality)
	
	distances_thresholded <- (as.matrix(distances) < criticaldistance)
	distances_thresholded[lower.tri(distances_thresholded,diag=T)] <- FALSE
	cat("done.\n")
	
	# identify the connected components
	cat("Building graph for random point connections...")
	network <- igraph::graph.adjacency(distances_thresholded,mode="undirected")
	cat("done.\n")
	
	ebw <- igraph::edge.betweenness(network, directed=F)
	cat("Trimming high connectivity edges...")
	network <- network - igraph::E(network)[ebw > quantile(ebw,0.99)]
	cat("done.\n")

	if (plotdiag)
	{
		
		plot(randompoints[,1], randompoints[,2],xlim=prx, ylim=pry, axes=F,cex=0.5,pch=16,col='cyan')
		points(hv@Data[,1], hv@Data[,2],xlim=prx, ylim=pry, pch=16,col='black')

		indices <- which(distances_thresholded > 0, arr.ind=T)
		for (i in 1:nrow(indices))
		{
			vfrom <- randompoints[indices[i,1],]
			vto <- randompoints[indices[i,2],]
			segments(vfrom[1],vfrom[2],vto[1],vto[2],col='cyan',lwd=0.5)
		}
		legend('topleft','E',bty='n',cex=1.5,adj=c(2,0))
		box()
	}

	cat("Calculating connected components...")
	cc <- igraph::clusters(network)
	cat("done.\n")
	
	# remove small clusters
	smallclusters <- which(cc$csize < minnumpoints)
	cc$membership[cc$membership %in% smallclusters] <- NA
	cat(sprintf('Discarding %.0f clusters of size less than %.0f points, retaining %.0f clusters.\n', length(smallclusters), minnumpoints, length(which(cc$csize >= minnumpoints))))

	if (!all(is.na(cc$membership)))
	{	
		randompoints_trimmed <- randompoints[which(!is.na(cc$membership)),]
		cutdist <- as.matrix(distances)[which(!is.na(cc$membership)),which(!is.na(cc$membership))]
	
		for (j in 1:length(inflationfactors))
		{
			# re-merge the clusters		
			cat(sprintf("Beginning merge for inflation factor %.2f...\n", inflationfactors[j]))
			
			cutdist_thresholded <- (cutdist < criticaldistance * inflationfactors[j])
			
	
			network_cut <- igraph::graph.adjacency(cutdist_thresholded,mode="undirected")
			cc_cut <- igraph::clusters(network_cut)
			
			# make hypervolumelist from the holes
			nclusters <- max(cc_cut$membership,na.rm=T)
			if (is.finite(nclusters))
			{
				finalnonconvexfeatures <- vector(mode="list",length=nclusters)
				
				cat(sprintf("Beginning dent/hole designation..."))
				
				featureflags <- inhull(testpts= randompoints_trimmed, calpts=data_reduced, hull=hull_matrix, tol= criticaldistance * inflationfactors[j])
				
				for (i in 1:nclusters)
				{
					# determine if the feature is a hole or a dent
					featurepoints_this <- subset(randompoints_trimmed, cc_cut$membership == i)
					featureflags_this <- subset(featureflags, cc_cut$membership == i)
					
					if (any(featureflags_this ==0)) # if any points are on the boundary
					{
						featuretype <- "Dent"
					}
					else
					{
						featuretype <- "Hole"
					}
					
					thishv <- hvs_overlap@HVList$Unique_2 # copy some information
					thishv@RandomUniformPointsThresholded <- featurepoints_this
					thishv@Volume <- thishv@Volume * nrow(thishv@RandomUniformPointsThresholded) / nrow(randompoints)
					thishv@Name <- sprintf("%s",featuretype) # still need to segment this into categories
					
					finalnonconvexfeatures[[i]] <- thishv
				}
				names(finalnonconvexfeatures) <- lapply(finalnonconvexfeatures, function(x) { x@Name })	
				
				cat(sprintf("done.\n"))
			}
			else
			{
				finalnonconvexfeatures[[i]] <- hv
				cat(sprintf('No non-convex-features found for inflation factor %.2f.\n', inflationfactors[j]))
			}
			
	
			finalresult[[j]] <- new("HypervolumeList",HVList=c(hv, finalnonconvexfeatures))

			cat('merge done.\n')
		}
		
		if (plotdiag==TRUE) # choose to plot results for the largest inflation factor
		{
			plot(0, 0, type='n', xlim=prx, ylim=pry, axes=F)
			
			hvt <- finalresult[[length(inflationfactors)]]@HVList
			
			for (i in 2:nclusters)
			{
				points(hvt[[i]]@RandomUniformPointsThresholded[,1], hvt[[i]]@RandomUniformPointsThresholded[,2],col=rainbow(nclusters*3)[i],cex=0.5,pch=16)
			}
			
			points(hv@Data[,1], hv@Data[,2],xlim=prx, ylim=pry, pch=16,col='black')
			
			legend('topleft','F',bty='n',cex=1.5,adj=c(2,0))
			box()
			
			dev.off()
		}
	}
	else
	{
		finalresult <- NULL
	}

	# return the final hypervolumelist
	cat('Returning original hypervolume and all non-convex features.\n')

	return(finalresult)
}