% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/comorbid.R, R/z-deprecated.R
\name{comorbid}
\alias{comorbid}
\alias{icd10_comorbid}
\alias{icd9_comorbid}
\alias{icd9_comorbid_ahrq}
\alias{icd10_comorbid_ahrq}
\alias{icd9_comorbid_elix}
\alias{icd10_comorbid_elix}
\alias{icd9_comorbid_quan_elix}
\alias{icd10_comorbid_quan_elix}
\alias{icd9_comorbid_quan_deyo}
\alias{icd10_comorbid_quan_deyo}
\alias{comorbid_ccs}
\alias{icd9_comorbid_ccs}
\alias{icd10_comorbid_ccs}
\alias{comorbid_ahrq}
\alias{comorbid_elix}
\alias{comorbid_quan_elix}
\alias{comorbid_quan_deyo}
\alias{comorbid_charlson}
\alias{icd_comorbid}
\alias{icd_comorbid_ahrq}
\alias{icd_comorbid_elix}
\alias{icd_comorbid_hcc}
\alias{icd_comorbid_quan_deyo}
\alias{icd_comorbid_quan_elix}
\title{Find comorbidities from ICD-9 codes.}
\usage{
comorbid(x, map, ...)

icd10_comorbid(x, map, visit_name = NULL, icd_name = NULL,
  short_code = NULL, short_map = guess_short(map), return_df = FALSE,
  icd10_comorbid_fun = icd10_comorbid_reduce, ...)

icd9_comorbid(x, map, visit_name = NULL, icd_name = NULL,
  short_code = guess_short(x, icd_name = icd_name),
  short_map = guess_short(map), return_df = FALSE, preclean = TRUE,
  visitId = NULL, icd9Field = NULL, ...)

icd9_comorbid_ahrq(x, ..., abbrev_names = TRUE, hierarchy = TRUE)

icd10_comorbid_ahrq(x, ..., abbrev_names = TRUE, hierarchy = TRUE)

icd9_comorbid_elix(x, ..., abbrev_names = TRUE, hierarchy = TRUE)

icd10_comorbid_elix(x, ..., abbrev_names = TRUE, hierarchy = TRUE)

icd9_comorbid_quan_elix(x, ..., abbrev_names = TRUE, hierarchy = TRUE)

icd10_comorbid_quan_elix(x, ..., abbrev_names = TRUE, hierarchy = TRUE)

icd9_comorbid_quan_deyo(x, ..., abbrev_names = TRUE, hierarchy = TRUE)

icd10_comorbid_quan_deyo(x, ..., abbrev_names = TRUE, hierarchy = TRUE)

comorbid_ccs(x, icd_name = get_icd_name(x), ...)

icd9_comorbid_ccs(x, ..., single = TRUE, lvl = NULL,
  map = icd::icd9_map_single_ccs, short_map = TRUE)

icd10_comorbid_ccs(x, ..., single = TRUE, lvl = NULL)

comorbid_ahrq(x, ...)

comorbid_elix(x, ...)

comorbid_quan_elix(x, ...)

comorbid_quan_deyo(x, ...)

comorbid_charlson(...)

icd_comorbid(...)

icd_comorbid_ahrq(...)

icd_comorbid_elix(...)

icd_comorbid_hcc(...)

icd_comorbid_quan_deyo(...)

icd_comorbid_quan_elix(...)
}
\arguments{
\item{x}{\code{data.frame} containing a column of patient-visit identifiers
and a column of ICD codes. The \code{data.frame} should be in \sQuote{long}
format, like the example \code{vermont_dx} data. If it is in \sQuote{wide}
format, it must be converted to \sQuote{long} using
\code{\link{wide_to_long}} before calling any comorbidity functions.}

\item{map}{list of the comorbidities with each list item containing a vector
of decimal ICD-9 codes. This is in the form of a list, with the names of
the items corresponding to the comorbidities (e.g. \sQuote{HTN}, or
\sQuote{diabetes}) and the contents of each list item being a character
vector of short-form (no decimal place, zero left-padded) ICD codes. There
is no default: the user should use the family of functions, e.g.
\code{comorbid_ahrq}, since these also name the fields correctly, apply any
hierarchical rules (see \code{hierarchy} below)}

\item{...}{arguments passed on to other functions}

\item{visit_name}{The name of the column in the data frame which contains the
patient or visit identifier. Typically this is the visit identifier, since
patients come leave and enter hospital with different ICD-9 codes. It is a
character vector of length one. If left empty, or \code{NULL}, then an
attempt is made to guess which field has the ID for the patient encounter
(not a patient ID, although this can of course be specified directly). The
guesses proceed until a single match is made. Data frames may be wide with
many matching fields, so to avoid false positives, anything but a single
match is rejected. If there are no successful guesses, and \code{visit_id}
was not specified, then the first column of the data frame is used.}

\item{icd_name}{The name of the column in the \code{data.frame} which
contains the ICD codes. This is a character vector of length one. If it is
\code{NULL}, \code{icd9} will attempt to guess the column name, looking for
progressively less likely possibilities until it matches a single column.
Failing this, it will take the first column in the data frame. Specifying
the column using this argument avoids the guesswork.}

\item{short_code}{single logical value which determines whether the ICD-9
code provided is in short (\code{TRUE}) or decimal (\code{FALSE}) form.
Where reasonable, this is guessed from the input data.}

\item{short_map}{Same as short, but applied to \code{map} instead of the data
frame of ICD codes, \code{x}. All the codes in a mapping should be of the
same type, i.e. short or decimal.}

\item{return_df}{single logical value, if \code{TRUE}, return the result as a data frame with the first column being
the \code{visit_id}, and the second being the count. If \code{visit_id} was a factor or named differently in the
input, this is preserved.}

\item{icd10_comorbid_fun}{function Internal parameter, default will be fast
and accurate. A function which calculates comorbidities for ICD-10 codes,
in which the comorbidity map only specifies parent codes, not every
possible child.}

\item{preclean}{single logical value, which, if \code{TRUE} causes ICD-9
'short' code input to be padded to correct three (or four for E code)
length before applying the comorbidity map. For very large data sets, e.g.
ten million rows, this is much slower than the comorbidity calculation. If
you know that the source ICD-9 codes are already well formed (or have
already run \code{icd9_add_leading_zeroes}), then \code{preclean} can be
set to \code{FALSE} to save time.}

\item{visitId}{Deprecated. Use \code{visit_name} instead.}

\item{icd9Field}{Deprecated. Use \code{icd_name} instead.}

\item{abbrev_names}{single logical value that defaults to \code{TRUE}, in
which case the shorter human-readable names stored in e.g.
\code{ahrqComorbidNamesAbbrev} are applied to the data frame column names.}

\item{hierarchy}{single logical value that defaults to \code{TRUE}, in which
case the hierarchy defined for the mapping is applied. E.g. in Elixhauser,
you can't have uncomplicated and complicated diabetes both flagged.}

\item{single}{a logical value, if \code{TRUE} then use single level CCS,
otherwise use multi level}

\item{lvl}{If multiple level CCS, then level must be selected as a number
between one and four.}
}
\description{
This is the main function which extracts comorbidities from a set of ICD-9
codes. Some comorbidity schemes have rules, for example, what to do when both
'hypertension' and 'hypertension with complications' are present. These rules
are applied by default; if the exact fields from the original mappings are
needed, use \code{hierarchy = FALSE}. For comorbidity counting, Charlson or
VanWalraven scores the default should be used to apply the rules. For more
about computing Hierarchical Condition Codes (HCC), see
\code{\link{comorbid_hcc}} For more about comorbidities following the
Clinical Classification Software (CCS) rules from AHRQ, see
\code{\link{comorbid_ccs}}.
}
\details{
The order of visits may change depending on the original sequence,
  and the underlying algorithm used. Usually this would be the order of the
  first occurence of each visit/patient identifier.

  The threading of the C++ can be controlled using e.g.
  \code{option(icd.threads = 4)}. If it is not set, the number of cores in
  the machine is used. 'OpenMP' environment variables also work.

\code{data.frame}s of patient data may have columns within them
  which are of class \code{icd9}, \code{icd10} etc., but do not themselves
  have a class: therefore, the S3 mechanism for dispatch is not suitable. I
  may add a wrapper function which looks inside a \code{data.frame} of
  comorbidities, and dispatches to the appropriate function, but right now
  the user must call the \code{icd9_} or \code{icd10_} prefixed function
  directly.
}
\section{Functions}{
\itemize{
\item \code{icd10_comorbid}: ICD-10 comorbidities

\item \code{icd9_comorbid}: Get comorbidities from \code{data.frame} of ICD-9
codes

\item \code{icd9_comorbid_ahrq}: AHRQ comorbidities for ICD-9 codes

\item \code{icd10_comorbid_ahrq}: AHRQ comorbidities for ICD-10 codes

\item \code{icd9_comorbid_elix}: Elixhauser comorbidities for ICD-9 codes

\item \code{icd10_comorbid_elix}: Elixhauser comorbidities for ICD-10 codes

\item \code{icd9_comorbid_quan_elix}: Quan's Elixhauser comorbidities for ICD-9 codes

\item \code{icd10_comorbid_quan_elix}: Quan's Elixhauser comorbidities for ICD-10 codes

\item \code{icd9_comorbid_quan_deyo}: Quan's Deyo (Charlson) comorbidities for ICD-9 codes

\item \code{icd10_comorbid_quan_deyo}: Quan's Deyo (Charlson) comorbidities for ICD-10 codes

\item \code{comorbid_ccs}: Use AHRQ CCS for comorbidity classification

\item \code{icd9_comorbid_ccs}: Compute AHRQ Clinical Classifications Software (CCS)
scores from ICD-9 codes

\item \code{icd10_comorbid_ccs}: Compute AHRQ Clinical Classifications Software (CCS)
scores from ICD-10 codes

\item \code{comorbid_ahrq}: AHRQ comorbidities, infers whether to use ICD-9 or
ICD-10 codes

\item \code{comorbid_elix}: Elixhauser comorbidities, infers whether to use ICD-9 or
ICD-10 codes

\item \code{comorbid_quan_elix}: Quan's Elixhauser comorbidities, infers whether to use
ICD-9 or ICD-10 codes

\item \code{comorbid_quan_deyo}: Quan's Deyo (Charlson) comorbidities, infers whether to
use ICD-9 or ICD-10 codes

\item \code{comorbid_charlson}: Calculate comorbidities using Charlson categories
according to Quan/Deyo ICD categories. Synonymous with
\code{link{comorbid_quan_deyo}} in this release.
}}

\section{Deprecated function names}{
 Future versions of \pkg{icd} will drop
  the \code{icd_} prefix. For example, \code{charlson} should be used in
  favor of \code{icd_charlson}. To distinguish \pkg{icd} function calls,
  consider using the namespace prefix \code{icd::} instead, e.g.,
  \code{icd::charlson}. Functions which specifically operate on either ICD-9
  or ICD-10 codes or their subtypes will retain the prefix. E.g.
  \code{\link{icd9_comorbid_ahrq}}. \pkg{icd} specific classes also retain
  the prefix, e.g., \code{\link{icd_wide_data}}.
}

\examples{
vermont_dx[1:5, 1:10]
vd <- wide_to_long(vermont_dx)
# get first few rows and columns of Charlson comorbidities using Quan's mapping
comorbid_quan_deyo(vd)[1:5, 1:14]

# get summary AHRQ (based on Elixhauser) comorbidities for ICD-10 Uranium data:
summary(comorbid_ahrq(uranium_pathology))

pts <- icd_long_data(visit_name = c("2", "1", "2", "3", "3"),
                 icd9 = c("39891", "40110", "09322", "41514", "39891"))
comorbid(pts, icd9_map_ahrq, short_code = TRUE) # visit_name is now sorted
pts <- icd_long_data(
           visit_name = c("1", "2", "3", "4", "4"),
           icd_name = c("20084", "1742", "30410", "41514", "95893"),
           date = as.Date(c("2011-01-01", "2011-01-02", "2011-01-03",
             "2011-01-04", "2011-01-04")))

pt_hccs <- comorbid_hcc(pts, date_name = "date")
head(pt_hccs)

pts10 <- icd_long_data(
  visit_name = c("a", "b", "c", "d", "e"),
  icd_name =c("I058", NA, "T82817A", "", "I69369"),
  date = as.Date(
    c("2011-01-01", "2011-01-02", "2011-01-03", "2011-01-03", "2011-01-03")))

icd10_comorbid(pts10, map = icd10_map_ahrq)
# or if library(icd) hasn't been called first:
icd::icd10_comorbid(pts10, map = icd::icd10_map_ahrq)
# or most simply:
icd::icd10_comorbid_ahrq(pts10)

# specify a simple custom comorbidity map:
my_map <- list("malady" = c("100", "2000"),
               "ailment" = c("003", "040"))
two_pts <- data.frame(visit_id = c("v01", "v01", "v02", "v02"),
  icd9 = as.icd9(c("040", "000", "100", "000")),
  stringsAsFactors = FALSE)
comorbid(two_pts, map = my_map)
}
\seealso{
\link{icd9_map_single_ccs}

Consider using \code{\link{comorbid_ahrq}} instead of
  \code{\link{comorbid_elix}} for more recently updated mappings based on the
  Elixhauser scheme.
}
