% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/importance_perm.R
\name{importance_perm}
\alias{importance_perm}
\title{Compute permutation-based predictor importance}
\usage{
importance_perm(
  wflow,
  data,
  metrics = NULL,
  type = "original",
  size = 500,
  times = 10,
  eval_time = NULL,
  event_level = "first"
)
}
\arguments{
\item{wflow}{A fitted \code{\link[workflows:workflow]{workflows::workflow()}}.}

\item{data}{A data frame of the data passed to \code{\link[workflows:fit-workflow]{workflows::fit.workflow()}},
including the outcome and case weights (if any).}

\item{metrics}{A \code{\link[yardstick:metric_set]{yardstick::metric_set()}} or \code{NULL}.}

\item{type}{A character string for which \emph{level} of predictors to compute.
A value of \code{"original"} (default) will return values in the same
representation of \code{data}. Using \code{"derived"} will compute them for any derived
features/predictors, such as dummy indicator columns, etc.}

\item{size}{How many data points to predict for each permutation iteration.}

\item{times}{How many iterations to repeat the calculations.}

\item{eval_time}{For censored regression models, a vector of time points at
which the survival probability is estimated. This is only needed if a dynamic
metric is used, such as the Brier score or the area under the ROC curve.}

\item{event_level}{A single string. Either \code{"first"} or \code{"second"} to specify
which level of \code{truth} to consider as the "event". This argument is only
applicable when \code{estimator = "binary"}.}
}
\value{
A tibble with extra classes \code{"importance_perm"} and either
"\verb{original_importance_perm"} or "\verb{derived_importance_perm"}. The columns are:
\itemize{
\item \code{.metric} the name of the performance metric:
\item \code{predictor}: the predictor
\item \code{n}: the number of usable results (should be the same as \code{times})
\item \code{mean}: the average of the differences in performance. For each metric,
larger values indicate worse performance (i.e., higher importance).
\item \code{std_err}: the standard error of the differences.
\item \code{importance}: the mean divided by the standard error.
\item For censored regression models, an additional \code{.eval_time} column may also
be included (depending on the metric requested).
}
}
\description{
\code{\link[=importance_perm]{importance_perm()}} computes model-agnostic variable importance scores by
permuting individual predictors (one at a time) and measuring how worse
model performance becomes.
}
\details{
The function can compute importance at two different levels.
\itemize{
\item The "original" predictors are the unaltered columns in the source data set.
For example, for a categorical predictor used with linear regression, the
original predictor is the factor column.
\item "Derived" predictors are the final versions given to the model. For the
categorical predictor example, the derived versions are the binary
indicator variables produced from the factor version.
}

This can make a difference when pre-processing/feature engineering is used.
This can help us understand \emph{how} a predictor can be important

Importance scores are computed for each predictor (at the specified level)
and each performance metric. If no metric is specified, defaults are used:
\itemize{
\item Classification: \code{\link[yardstick:brier_class]{yardstick::brier_class()}}, \code{\link[yardstick:roc_auc]{yardstick::roc_auc()}}, and
\code{\link[yardstick:accuracy]{yardstick::accuracy()}}.
\item Regression:  \code{\link[yardstick:rmse]{yardstick::rmse()}} and \code{\link[yardstick:rsq]{yardstick::rsq()}}.
\item Censored regression: \code{\link[yardstick:brier_survival]{yardstick::brier_survival()}}
}

For censored data, importance is computed for each evaluation time (when a
dynamic metric is specified).

By default, no parallelism is used to process models in \pkg{tune}; you have
to opt-in.
\subsection{Using future to parallel process}{

You should install the package and choose your flavor of parallelism using
the \link[future:plan]{plan} function. This allows you to specify the number of
worker processes and the specific technology to use.

For example, you can use:

\if{html}{\out{<div class="sourceCode r">}}\preformatted{   library(future)
   plan(multisession, workers = 4)
}\if{html}{\out{</div>}}

and work will be conducted simultaneously (unless there is an exception; see
the section below).

See \code{\link[future:plan]{future::plan()}} for possible options other than \code{multisession}.
}

\subsection{Using mirai  to parallel process}{

To configure parallel processing with \pkg{mirai}, use the
\code{\link[mirai:daemons]{mirai::daemons()}} function. The first argument, \code{n}, determines the number
of parallel workers. Using \code{daemons(0)} reverts to sequential processing.

The arguments \code{url} and \code{remote} are used to set up and launch parallel
processes over the network for distributed computing. See \code{\link[mirai:daemons]{mirai::daemons()}}
documentation for more details.
}
}
\examples{
if (rlang::is_installed(c("modeldata", "recipes", "workflows", "parsnip"))) {
  library(modeldata)
  library(recipes)
  library(workflows)
  library(dplyr)
  library(parsnip)

  set.seed(12)
  dat_tr <-
    sim_logistic(250, ~ .1 + 2 * A - 3 * B + 1 * A *B, corr = .7) |>
    dplyr::bind_cols(sim_noise(250, num_vars = 10))

  rec <-
    recipe(class ~ ., data = dat_tr) |>
    step_interact(~ A:B) |>
    step_normalize(all_numeric_predictors()) |>
    step_pca(contains("noise"), num_comp = 5)

  lr_wflow <- workflow(rec, logistic_reg())
  lr_fit <- fit(lr_wflow, dat_tr)

  set.seed(39)
  orig_res <- importance_perm(lr_fit, data = dat_tr, type = "original",
                              size = 100, times = 3)
  orig_res

  set.seed(39)
  deriv_res <- importance_perm(lr_fit, data = dat_tr, type = "derived",
                               size = 100, times = 3)
  deriv_res
}
}
