% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/estimate_peak.R
\name{estimate_peak}
\alias{estimate_peak}
\title{Estimate the peak date of an incidence curve using bootstrap}
\usage{
estimate_peak(x, n = 100, alpha = 0.05)
}
\arguments{
\item{x}{An \code{incidence} object.}

\item{n}{The number of bootstrap datasets to be generated; defaults to 100.}

\item{alpha}{The type 1 error chosen for the confidence interval; defaults to
0.05.}
}
\value{
A list containing the following items:
\itemize{
\item \code{observed}: the peak incidence of the original dataset
\item \code{estimated}: the mean peak time of the bootstrap datasets
\item \code{ci}: the confidence interval based on bootstrap datasets
\item \code{peaks}: the peak times of the bootstrap datasets
}
}
\description{
This function can be used to estimate the peak of an epidemic curve stored as
\code{incidence}, using bootstrap. See \link[incidence:bootstrap]{incidence::bootstrap} for more information
on the resampling.
}
\details{
Input dates are resampled with replacement to form bootstrapped
datasets; the peak is reported for each, resulting in a distribution of
peak times. When there are ties for peak incidence, only the first date is
reported.

Note that the bootstrapping approach used for estimating the peak time makes
the following assumptions:
\itemize{
\item the total number of event is known (no uncertainty on total incidence)
\item dates with no events (zero incidence) will never be in bootstrapped datasets
\item the reporting is assumed to be constant over time, i.e. every case is
equally likely to be reported
}
}
\examples{

if (require(outbreaks) && require(ggplot2)) { withAutoprint({
  i <- incidence(fluH7N9_china_2013$date_of_onset)
  i
  plot(i)

  ## one simple bootstrap
  x <- bootstrap(i)
  x
  plot(x)

  ## find 95\% CI for peak time using bootstrap
  peak_data <- estimate_peak(i)
  peak_data
  summary(peak_data$peaks)

  ## show confidence interval
  plot(i) + geom_vline(xintercept = peak_data$ci, col = "red", lty = 2)

  ## show the distribution of bootstrapped peaks
  df <- data.frame(peak = peak_data$peaks)
  plot(i) + geom_density(data = df,
                         aes(x = peak, y = 10 * ..scaled..),
                         alpha = .2, fill = "red", color = "red")

})}

}
\seealso{
\link[incidence:bootstrap]{incidence::bootstrap} for the bootstrapping underlying this
approach and \link[incidence:find_peak]{incidence::find_peak} to find the peak in a single
\code{incidence} object.
}
\author{
Thibaut Jombart \email{thibautjombart@gmail.com}, with inputs on
caveats from Michael Höhle.
}
