#' Standardize initialization
#'
#' @description
#' This function is an implementation of the standardize initialization
#' strategy.
#'
#' @details
#' For more details see the help vignette:
#' \code{vignette("workflow", package = "ino")}
#'
#' @param x
#' An object of class \code{ino}.
#' @param arg
#' A character, the name(s) of the argument(s) to be standardized.
#' The argument must be of class \code{matrix} or \code{data.frame}.
#' Per default, \code{arg = "data"}.
#' @param by_col
#' A boolean, set to \code{TRUE} (the default) to standardize column-wise, set
#' to \code{FALSE} to standardize by rows.
#' @param center
#' A boolean, passed to \code{\link[base]{scale}}.
#' @param scale
#' A boolean, passed to \code{\link[base]{scale}}.
#' @param col_ign
#' A numeric vector of column indices (or row indices if \code{by_col = FALSE})
#' that are ignored when standardizing.
#' @param initialization
#' An object of class \code{ino_call} which determines the initialization for
#' the standardized \code{ino} object. The \code{ino_call} is generated by any
#' of the strategy functions (any function with the name
#' \code{*_initialization}), when the \code{x} is unspecified.
#' Per default, \code{initialization = random_initialization()}, i.e. random
#' initialization.
#' @param ncores
#' This function is parallelized, set the number of cores here.
#' @param verbose
#' Set to \code{TRUE} (\code{FALSE}) to print (hide) progress.
#'
#' @return
#' The updated \code{ino} object.
#'
#' @export
#'
#' @keywords
#' strategy

standardize_initialization <- function(x, arg = "data", by_col = TRUE, center = TRUE, scale = TRUE, col_ign = NULL,
                                       initialization = random_initialization(), ncores = getOption("ino_ncores"),
                                       verbose = getOption("ino_progress")) {

  ### capture function call if 'x' is not specified
  if (missing(x)) {
    return(ino_call(match.call(expand.dots = TRUE)))
  }

  ### check inputs
  ino_check_inputs(
    "x" = x, "arg" = arg, "by_col" = by_col, "center" = center, "scale" = scale,
    "ncores" = ncores, "col_ign" = col_ign, "initialization" = initialization,
    "verbose" = verbose
  )

  ### initial  message
  ino_status(msg = "standardize initialization", verbose = verbose)

  ### standardize 'arg' argument(s) in 'x'
  x_tmp <- clear_ino(x)
  for (arg_name in arg) {
    x_standardized <- standardize_arg(
      x = clear_ino(x), arg = arg_name, by_col = by_col, center = center,
      scale = scale, col_ign = col_ign
    )
  }

  ### optimize on standardized 'x'
  x_standardized <- do.call(
    what = rlang::call_name(initialization),
    args = c(list("x" = x_standardized), rlang::call_args(initialization))
  )
  name <- paste("standardize", x_standardized$runs$table$.strategy, sep = ">")
  x_standardized$runs$table$.strategy <- name

  ### merge 'x' and 'x_standardized'
  x <- merge_ino(x, x_standardized)

  ### return ino object
  return(x)
}

#' Subset initialization
#'
#' @description
#' This function is an implementation of the subset initialization strategy.
#'
#' @details
#' For more details see the help vignette:
#' \code{vignette("workflow", package = "ino")}
#'
#' @param x
#' An object of class \code{ino}.
#' @param arg
#' A character, the name of the argument to be subsetted.
#' The argument must be of class \code{matrix} or \code{data.frame}.
#' Per default, \code{arg = "data"}.
#' @param how
#' A character, specifying how to select the subset.
#' Can be one of \code{"random"} (default), \code{"first"}, and \code{"kmeans"}.
#' @param prop
#' A numeric between 0 and 1, specifying the proportion of the subset.
#' @param by_row
#' A boolean, set to \code{TRUE} (the default) to subset by row, set to
#' \code{FALSE} to subset by column.
#' @param col_ign
#' A numeric vector of column indices (or row indices if \code{by_row = FALSE})
#' that are ignored when clustering.
#' Only relevant if \code{how = "kmeans"}.
#' @param kmeans_arg
#' A list of additional arguments for \code{\link[stats]{kmeans}}.
#' Per default, \code{kmeans_arg = list(centers = 2)}, which sets the number of
#' clusters to 2.
#' Only relevant if \code{how = "kmeans"}.
#' @param initialization
#' An object of class \code{ino_call} which determines the initialization for
#' the standardized \code{ino} object. The \code{ino_call} is generated by any
#' of the strategy functions, when the \code{x} is unspecified.
#' Per default, \code{initialization = random_initialization()}, i.e. random
#' initialization.
#' @param ncores
#' This function is parallelized, set the number of cores here.
#' @param verbose
#' Set to \code{TRUE} (\code{FALSE}) to print (hide) progress.
#'
#' @return
#' The updated \code{ino} object.
#'
#' @export
#'
#' @keywords
#' strategy
#'
#' @importFrom stats kmeans

subset_initialization <- function(x, arg = "data", how = "random", prop = 0.5, by_row = TRUE, col_ign = NULL,
                                  kmeans_arg = list("centers" = 2), initialization = random_initialization(),
                                  ncores = getOption("ino_ncores"), verbose = getOption("ino_progress")) {

  ### capture function call if 'x' is not specified
  if (missing(x)) {
    return(ino_call(match.call(expand.dots = TRUE)))
  }

  ### check inputs
  ino_check_inputs(
    "x" = x, "arg" = arg, "how" = how, "prop" = prop,
    "by_row" = by_row, "ncores" = ncores, "col_ign" = col_ign,
    "kmeans_arg" = kmeans_arg, "initialization" = initialization,
    "verbose" = verbose
  )

  ### subset 'arg' argument in 'x', optimize on subset, and extract estimates
  x_subset <- subset_arg(
    x = clear_ino(x), arg = arg, how = how, prop = prop, by_row = by_row,
    col_ign = col_ign, kmeans_arg = kmeans_arg
  )
  x_subset <- do.call(
    what = rlang::call_name(initialization),
    args = c(list("x" = x_subset), rlang::call_args(initialization))
  )
  initial_time <- x_subset$runs$table$.time
  strategy_name <- x_subset$runs$table$.strategy[1]
  init <- lapply(x_subset$runs$pars, "[[", ".estimate")

  ### create parameter grid
  grid <- grid_ino(x)

  ### initial  message
  ino_status(msg = "subset initialization", verbose = verbose)

  ### loop over parameter sets 'p' and optimizer 'o'
  loop_grid <- expand.grid(p = 1:length(grid), o = 1:length(x$opt))
  cluster <- parallel::makeCluster(ncores)
  doSNOW::registerDoSNOW(cluster)
  pb <- ino_pb(title = "  grid set ", total = nrow(loop_grid))
  opts <- list(progress = function(n) ino_pp(pb = pb, verbose = verbose))
  i <- 0
  loop_res <- foreach::foreach(
    i = 1:nrow(loop_grid), .inorder = FALSE, .packages = "ino",
    .options.snow = opts
  ) %dopar% {

    ### extract current loop indices
    p <- loop_grid[i, "p"]
    o <- loop_grid[i, "o"]

    ### extract current parameter set
    pars <- grid[[p]]

    ### extract current optimizer
    opt <- x$opt[[o]]

    ### extract initial value
    pars[[x$f$target_arg]] <- init[[i]]

    ### base arguments of the optimizer
    base_args <- list(x$f$f, pars[[x$f$target_arg]])
    names(base_args) <- opt$base_arg_names[1:2]
    f_args <- pars
    f_args[[x$f$target_arg]] <- NULL

    ### optimize
    result <- try_silent(
      do.call_timed(
        what = opt$f,
        args = c(base_args, f_args, opt$args),
        headstart = initial_time[i]
      )
    )
    list("pars" = pars, "result" = result, "opt_name" = names(x$opt)[o])
  }

  ### terminate cluster
  parallel::stopCluster(cluster)

  ### save optimization results
  strategy_name <- paste("subset", strategy_name, sep = ">")
  for (res in seq_along(loop_res)) {
    if (inherits(loop_res[[res]]$result, "fail")) {
      warning("Optimization failed with message", loop_res[[res]]$result,
        immediate. = TRUE
      )
    } else {
      x <- do.call(
        what = result_ino,
        args = append(
          list("x" = x, "strategy" = strategy_name),
          loop_res[[res]]
        )
      )
    }
  }

  ### return ino object
  return(x)
}

#' Fixed initialization
#'
#' @description
#' This function is an implementation of the fixed initialization strategy.
#'
#' @details
#' For more details see the help vignette:
#' \code{vignette("workflow", package = "ino")}
#'
#' @param x
#' An object of class \code{ino}.
#' @param at
#' A vector containing the (fixed) initial values.
#' @param ncores
#' This function is parallelized, set the number of cores here.
#' @param verbose
#' Set to \code{TRUE} (\code{FALSE}) to print (hide) progress.
#'
#' @return
#' The updated \code{ino} object.
#'
#' @export
#'
#' @keywords
#' strategy

fixed_initialization <- function(x, at, ncores = getOption("ino_ncores"),
                                 verbose = getOption("ino_progress")) {

  ### capture function call if 'x' is not specified
  if (missing(x)) {
    return(ino_call(match.call(expand.dots = TRUE)))
  }

  ### initial message
  ino_status(msg = "fixed initialization", verbose = verbose)

  ### check inputs
  ino_check_inputs("x" = x, "at" = at, "ncores" = ncores, "verbose" = verbose)

  ### create parameter grid
  grid <- grid_ino(x)

  ### loop over parameter sets 'p' and optimizer 'o'
  loop_grid <- expand.grid(p = 1:length(grid), o = 1:length(x$opt))
  cluster <- parallel::makeCluster(ncores)
  doSNOW::registerDoSNOW(cluster)
  pb <- ino_pb(title = "  grid set ", total = nrow(loop_grid))
  opts <- list(progress = function(n) ino_pp(pb, verbose = verbose))
  i <- 0
  loop_res <- foreach::foreach(
    i = 1:nrow(loop_grid), .packages = "ino",
    .options.snow = opts
  ) %dopar% {

    ### extract current loop indices
    p <- loop_grid[i, "p"]
    o <- loop_grid[i, "o"]

    ### extract current parameter set
    pars <- grid[[p]]

    ### save initial value in parameter set
    pars[[x$f$target_arg]] <- at

    ### extract current optimizer
    opt <- x$opt[[o]]

    ### base arguments of the optimizer
    base_args <- list(x$f$f, pars[[x$f$target_arg]])
    names(base_args) <- opt$base_arg_names[1:2]
    f_args <- pars
    f_args[[x$f$target_arg]] <- NULL

    ### optimize
    result <- try_silent(
      do.call_timed(what = opt$f, args = c(base_args, f_args, opt$args))
    )
    list("pars" = pars, "result" = result, "opt_name" = names(x$opt)[o])
  }

  ### terminate cluster
  parallel::stopCluster(cluster)

  ### save optimization results
  for (res in seq_along(loop_res)) {
    if (inherits(loop_res[[res]]$result, "fail")) {
      warning("Optimization failed with message", loop_res[[res]]$result,
        immediate. = TRUE
      )
    } else {
      x <- do.call(
        what = result_ino,
        args = append(
          list("x" = x, "strategy" = "fixed"),
          loop_res[[res]]
        )
      )
    }
  }

  ### return (invisibly) updated ino object
  return(invisible(x))
}

#' Random initialization
#'
#' @description
#' This function is an implementation of the random initialization strategy.
#'
#' @details
#' For more details see the help vignette:
#' \code{vignette("workflow", package = "ino")}
#'
#' @param x
#' An object of class \code{ino}.
#' @param sampler
#' The sampler for random initial values. Can be any function that
#' \itemize{
#'   \item has as first argument an integer, say \code{npar},
#'   \item and returns a numeric vector of length \code{npar}.
#' }
#' Per default, \code{sampler = stats::rnorm}, i.e. independent draws from a
#' standard normal distribution as initial value.
#' @param ...
#' Additional argument to \code{sampler} (optional).
#' @param ncores
#' This function is parallelized, set the number of cores here.
#' @param verbose
#' Set to \code{TRUE} (\code{FALSE}) to print (hide) progress.
#'
#' @return
#' The updated \code{ino} object.
#'
#' @export
#'
#' @keywords
#' strategy
#'
#' @importFrom utils capture.output
#' @importFrom foreach %dopar%

random_initialization <- function(x, sampler = stats::rnorm, ..., ncores = getOption("ino_ncores"),
                                  verbose = getOption("ino_progress")) {

  ### capture function call if 'x' is not specified
  if (missing(x)) {
    return(ino_call(match.call(expand.dots = TRUE)))
  }

  ### initial message
  ino_status(msg = "random initialization", verbose = verbose)

  ### check inputs
  ino_check_inputs(
    "x" = x, "sampler" = sampler, "ncores" = ncores,
    "verbose" = verbose
  )

  ### check sampler
  npar <- NULL
  sampler_first_arg <- alist(npar)
  names(sampler_first_arg) <- names(formals(sampler))[1]
  sampler_add_args <- list(...)
  sampler_args <- c(sampler_first_arg, sampler_add_args)
  sampler_init <- function(npar) do.call(what = sampler, args = sampler_args)
  try_sampler <- try_silent(sampler_init(x$f$npar))
  if (inherits(try_sampler, "fail")) {
    stop("'sampler' failed with error message:\n", try_sampler, call. = FALSE)
  }
  if (!is.numeric(try_sampler) || length(try_sampler) != x$f$npar) {
    stop("'sampler' must return a numeric vector of length 'x$f$npar'.\n",
      "Instead, 'sampler' returned:\n",
      paste(capture.output(str(try_sampler)), collapse = "\n"),
      call. = FALSE
    )
  }

  ### create parameter grid
  grid <- grid_ino(x)

  ### loop over parameter sets 'p' and optimizer 'o'
  loop_grid <- expand.grid(p = 1:length(grid), o = 1:length(x$opt))
  cluster <- parallel::makeCluster(ncores)
  doSNOW::registerDoSNOW(cluster)
  pb <- ino_pb(title = "  grid set ", total = nrow(loop_grid))
  opts <- list(progress = function(n) ino_pp(pb = pb, verbose = verbose))
  i <- 0
  loop_res <- foreach::foreach(
    i = 1:nrow(loop_grid), .packages = "ino",
    .options.snow = opts
  ) %dopar% {

    ### extract current loop indices
    p <- loop_grid[i, "p"]
    o <- loop_grid[i, "o"]

    ### extract current parameter set
    pars <- grid[[p]]

    ### draw random initial value
    init <- sampler_init(x$f$npar)

    ### save initial value in parameter set
    pars[[x$f$target_arg]] <- init

    ### extract current optimizer
    opt <- x$opt[[o]]

    ### base arguments of the optimizer
    base_args <- list(x$f$f, pars[[x$f$target_arg]])
    names(base_args) <- opt$base_arg_names[1:2]
    f_args <- pars
    f_args[[x$f$target_arg]] <- NULL

    ### optimize
    result <- try_silent(
      do.call_timed(what = opt$f, args = c(base_args, f_args, opt$args))
    )
    list("pars" = pars, "result" = result, "opt_name" = names(x$opt)[o])
  }

  ### terminate cluster
  parallel::stopCluster(cluster)

  ### save optimization results
  for (res in seq_along(loop_res)) {
    if (inherits(loop_res[[res]]$result, "fail")) {
      warning("Optimization failed with message", loop_res[[res]]$result,
        immediate. = TRUE
      )
    } else {
      x <- do.call(
        what = result_ino,
        args = append(list("x" = x, "strategy" = "random"), loop_res[[res]])
      )
    }
  }

  ### return (invisibly) updated ino object
  return(invisible(x))
}
