% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/api_define_extract.R
\name{define_extract-micro}
\alias{define_extract-micro}
\alias{define_extract_usa}
\alias{define_extract_cps}
\alias{define_extract_ipumsi}
\title{Define an extract request for an IPUMS microdata collection}
\usage{
define_extract_usa(
  description,
  samples,
  variables,
  data_format = "fixed_width",
  data_structure = "rectangular",
  rectangular_on = NULL,
  case_select_who = "individuals",
  data_quality_flags = NULL
)

define_extract_cps(
  description,
  samples,
  variables,
  data_format = "fixed_width",
  data_structure = "rectangular",
  rectangular_on = NULL,
  case_select_who = "individuals",
  data_quality_flags = NULL
)

define_extract_ipumsi(
  description,
  samples,
  variables,
  data_format = "fixed_width",
  data_structure = "rectangular",
  rectangular_on = NULL,
  case_select_who = "individuals",
  data_quality_flags = NULL
)
}
\arguments{
\item{description}{Description of the extract.}

\item{samples}{Vector of samples to include in the extract
request. Use \code{\link[=get_sample_info]{get_sample_info()}} to identify
sample IDs for a given collection.}

\item{variables}{Vector of variable names or a list of detailed
variable specifications to include in the extract
request. Use \code{\link[=var_spec]{var_spec()}} to create a \code{var_spec} object containing a
detailed variable specification. See examples.}

\item{data_format}{Format for the output extract data file. Either
\code{"fixed_width"} or \code{"csv"}.

Note that while \code{"stata"}, \code{"spss"}, or \code{"sas9"} are also accepted, these
file formats are not supported by ipumsr data-reading functions.

Defaults to \code{"fixed_width"}.}

\item{data_structure}{Data structure for the output extract data.
\itemize{
\item \code{"rectangular"} provides person records with all requested household
information attached to respective household members.
\item \code{"hierarchical"} provides household records followed by person records.
}

Defaults to \code{"rectangular"}.}

\item{rectangular_on}{If \code{data_structure} is \code{"rectangular"},
records on which to rectangularize. Currently only \code{"P"} (person records)
is supported.

Defaults to \code{"P"} if \code{data_structure} is \code{"rectangular"} and \code{NULL}
otherwise.}

\item{case_select_who}{Indication of how to interpret any case selections
included for variables in the extract definition.
\itemize{
\item \code{"individuals"} includes records for all individuals who match the
specified case selections.
\item \code{"households"} includes records for all members of each household that
contains an individual who matches the specified case selections.
}

Defaults to \code{"individuals"}. Use \code{\link[=var_spec]{var_spec()}} to add case selections for
specific variables.}

\item{data_quality_flags}{Set to \code{TRUE} to include data quality
flags for all applicable variables in the extract definition. This will
override the \code{data_quality_flags} specification for individual variables
in the definition.

Use \code{\link[=var_spec]{var_spec()}} to add data quality flags for specific variables.}
}
\value{
An object of class \code{\link[=ipums_extract-class]{micro_extract}} containing
the extract definition.
}
\description{
Define the parameters of an IPUMS microdata extract request to be submitted
via the IPUMS API.

Currently supported microdata collections include:
\itemize{
\item \strong{IPUMS USA}: \code{define_extract_usa()}
\item \strong{IPUMS CPS}: \code{define_extract_cps()}
\item \strong{IPUMS International}: \code{define_extract_ipumsi()}
}

Learn more about the IPUMS API in \code{vignette("ipums-api")} and
microdata extract definitions in \code{vignette("ipums-api-micro")}.
}
\examples{
usa_extract <- define_extract_usa(
  description = "2013-2014 ACS Data",
  samples = c("us2013a", "us2014a"),
  variables = c("SEX", "AGE", "YEAR")
)

usa_extract

# Use `var_spec()` to created detailed variable specifications:
usa_extract <- define_extract_usa(
  description = "Example USA extract definition",
  samples = c("us2013a", "us2014a"),
  variables = var_spec(
    "SEX",
    case_selections = "2",
    attached_characteristics = c("mother", "father")
  )
)

# For multiple variables, provide a list of `var_spec` objects and/or
# variable names.
cps_extract <- define_extract_cps(
  description = "Example CPS extract definition",
  samples = c("cps2020_02s", "cps2020_03s"),
  variables = list(
    var_spec("AGE", data_quality_flags = TRUE),
    var_spec("SEX", case_selections = "2"),
    "RACE"
  )
)

cps_extract

# To recycle specifications to many variables, it may be useful to
# create variables prior to defining the extract:
var_names <- c("AGE", "SEX")

my_vars <- purrr::map(
  var_names,
  ~ var_spec(.x, attached_characteristics = "mother")
)

ipumsi_extract <- define_extract_ipumsi(
  description = "Extract definition with predefined variables",
  samples = c("br2010a", "cl2017a"),
  variables = my_vars
)

# Extract specifications can be indexed by name
names(ipumsi_extract$samples)

names(ipumsi_extract$variables)

ipumsi_extract$variables$AGE

\dontrun{
# Use the extract definition to submit an extract request to the API
submit_extract(usa_extract)
}
}
\seealso{
\code{\link[=submit_extract]{submit_extract()}} to submit an extract request for processing.

\code{\link[=save_extract_as_json]{save_extract_as_json()}} and \code{\link[=define_extract_from_json]{define_extract_from_json()}} to share an
extract definition.
}
