% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/inspect.R
\name{plot_smooth}
\alias{plot_smooth}
\title{Visualization of smooths.}
\usage{
plot_smooth(x, view = NULL, cond = list(), plot_all = NULL,
  rm.ranef = NULL, n.grid = 30, rug = TRUE, add = FALSE, se = 1.96,
  shade = TRUE, eegAxis = FALSE, col = NULL, lwd = NULL, lty = NULL,
  print.summary = getOption("itsadug_print"), main = NULL, xlab = NULL,
  ylab = NULL, xlim = NULL, ylim = NULL, h0 = 0, v0 = NULL,
  transform = NULL, transform.view = NULL, legend_plot_all = NULL,
  hide.label = FALSE, ...)
}
\arguments{
\item{x}{A gam object, produced by \code{\link[mgcv]{gam}} or 
\code{\link[mgcv]{bam}}.}

\item{view}{Text string containing the name of the smooth
to be displayed. Note that 
variables coerced to factors in the model formula won't work as view 
variables.}

\item{cond}{A named list of the values to use for the other predictor terms 
(not in view). Used for choosing between smooths that share the same view 
predictors.}

\item{plot_all}{A vector with a name / names of model predictors, 
for which all levels should be plotted.}

\item{rm.ranef}{Logical: whether or not to remove random effects. 
Default is TRUE.}

\item{n.grid}{The number of grid nodes in each direction used for 
calculating the plotted surface.}

\item{rug}{Logical: when TRUE (default) then the covariate to which the 
plot applies is displayed as a rug plot at the foot of each plot of a 1-d 
smooth. Setting to FALSE will speed up plotting for large datasets.}

\item{add}{Logical: whether or not to add the lines to an existing plot, or 
start a new plot (default).}

\item{se}{If less than or equal to zero then only the predicted surface is 
plotted, but if greater than zero, then the predicted values plus 
confidence intervals are plotted. The value of se will be multiplied with 
the standard error (i.e., 1.96 results in 95\%CI and 2.58).}

\item{shade}{Logical: Set to TRUE to produce shaded regions as confidence 
bands for smooths (not avaliable for parametric terms, which are plotted 
using termplot).}

\item{eegAxis}{Logical: whether or not to reverse the y-axis, plotting the 
negative amplitudes upwards as traditionally is done in EEG research.
If eeg.axes is TRUE, labels for x- and y-axis are provided, when not 
provided by the user. Default value is FALSE.}

\item{col}{The colors for the lines and the error bars of the plot.}

\item{lwd}{The line width for the lines of the plot.}

\item{lty}{The line type for the lines of the plot.}

\item{print.summary}{Logical: whether or not to print summary.
Default set to the print info messages option 
(see \code{\link{infoMessages}}).}

\item{main}{Changing the main title for the plot, see also title.}

\item{xlab}{Changing the label for the x axis, 
defaults to a description of x.}

\item{ylab}{Changing the label for the y axis, 
defaults to a description of y.}

\item{xlim}{the x limits of the plot.}

\item{ylim}{the y limits of the plot.}

\item{h0}{A vector indicating where to add solid horizontal lines for 
reference. By default no values provided.}

\item{v0}{A vector indicating where to add dotted vertical lines for 
reference. By default no values provided.}

\item{transform}{Function for transforming the fitted values. 
Default is NULL.}

\item{transform.view}{Function for transforming 
the values on the x-axis. Defaults to NULL (no transformation).}

\item{legend_plot_all}{Legend location. This could be a keyword from 
the list "bottomright", "bottom", "bottomleft", "left", "topleft", "top", 
"topright", "right" and "center", or a list with \code{x} and \code{y} 
coordinate (e.g., \code{list(x=0,y=0)}).}

\item{hide.label}{Logical: whether or not to hide the label 
(i.e., "fitted values"). Default is FALSE.}

\item{...}{other options to pass on to lines and plot, 
see \code{\link[graphics]{par}}}
}
\description{
Plots a smooth from a \code{\link[mgcv]{gam}} or 
\code{\link[mgcv]{bam}} model based on predictions.
In contrast with the default \code{\link[mgcv]{plot.gam}}, this function 
plots the summed effects and optionally removes the random effects.
}
\section{Notes}{

This function plots the summed effects, including intercept and other 
predictors. For plotting partial effects, see the function 
\code{\link[mgcv]{plot.gam}}, or see the examples with 
\code{\link{get_modelterm}} for more flexibility (e.g., plotting using the 
\code{lattice} package or \code{ggplots}).
}
\examples{
data(simdat)

\dontrun{
# Model with random effect and interactions:
m1 <- bam(Y ~ te(Time, Trial)+s(Time, Subject, bs='fs', m=1),
    data=simdat, discrete=TRUE)

# Default plot produces only surface of Time x Trial:
plot(m1, select=1)
# Only the Time component:
plot_smooth(m1, view="Time")
# Note the summary that is printed.

# without random effects:
plot_smooth(m1, view="Time", rm.ranef=TRUE)

# Plot summed effects:
dev.new(width=8, height=4) # use x11(,8,4) on Linux
par(mfrow=c(1,2))
fvisgam(m1, view=c("Time", "Trial"), 
    plot.type='contour', color='topo', main='interaction',
    rm.ranef=TRUE)
arrows(x0=0, x1=2200, y0=-5, y1=-5, col='red', 
    code=2, length=.1, lwd=2, xpd=TRUE)
plot_smooth(m1, view='Time', cond=list(Trial=-5),
    main='Trial=-5', rm.ranef=TRUE)


# Model with random effect and interactions:
m2 <- bam(Y ~ Group + s(Time, by=Group)
    +s(Time, Subject, bs='fs', m=1),
    data=simdat, discrete=TRUE)

# Plot all levels of a predictor:
plot_smooth(m2, view='Time', plot_all="Group",
    rm.ranef=TRUE)
# It also possible to combine predictors in plot_all.
# Note: this is not a meaningfull plot, because Subjects
# fall in only one group. Just for illustration purposes!
plot_smooth(m2, view='Time', plot_all=c("Group", "Subject"))
# Clearly visible difference in confidence interval, because  
# a01 does not occur in Group "Children":
# (Note that this plot generates warning)
plot_smooth(m2, view='Time', plot_all=c("Group", "Subject"), cond=list(Subject="a01"))

# Using transform
# Plot log-transformed dependent predictor on original scale:
plot_smooth(m1, view="Time", rm.ranef=TRUE, transform=exp)

# Notes on transform.view: 
# This will generate an error, because x-values <= 0 will result in NaN:
plot_smooth(m1, view="Time", rm.ranef=TRUE, transform.view=log)
# adjusting the x-axis helps:
plot_smooth(m1, view="Time", rm.ranef=TRUE, transform.view=log,
   xlim=c(1,2000))
}

# and for a quick overview of plotfunctions:
vignette("overview", package="itsadug")

}
\author{
Jacolien van Rij and Martijn Wieling.
}
\seealso{
\code{\link[mgcv]{plot.gam}}, \code{\link{plot_diff}}

Other Functions for model inspection: \code{\link{fvisgam}},
  \code{\link{gamtabs}}, \code{\link{inspect_random}},
  \code{\link{plot_data}}, \code{\link{plot_parametric}},
  \code{\link{plot_topo}}, \code{\link{pvisgam}}
}

