#' Generic summary function for bmeta object in jarbes
#' @param object The object generated by the bmeta function.
#'
#' @param digits The number of significant digits printed. The default value is 3.
#' @param ... \dots
#'
#' @export
summary.bmeta = function(object, digits = 3, ...) {

  bugs.output = object$BUGSoutput
  bugs.summary = bugs.output$summary

  summary.m = list()

  # Model specifications ....
  #
  model.spec = list()
  model.spec$link = "Normal approximation"

  # Hyper-priors parameters............................................
  model.spec$mean.mu     = object$prior$mean.mu
  model.spec$sd.mu       = object$prior$sd.mu
  model.spec$scale.sigma.between = object$prior$scale.sigma.between
  model.spec$df.scale.between    = object$prior$df.scale.between
  #

  summary.m$model.specification = model.spec

  # Posterior of the model parameters
  #
  # The list of parameters will include more complex models, e.g. estimation of
  # the parameters nu from the Beta ...
  #
  summary.m$summary.par = bugs.summary[c("mu", "mu.new", "tau"),]

  row.names(summary.m$summary.par) = c("Mean (Pooled mean)",
                                       "Predictive effect",
                                       "Tau (between studies sd)")

  # DIC
  summary.m$DIC = bugs.output$DIC
  summary.m$pD = bugs.output$pD

  # MCMC setup ...
  mcmc.setup = list()
  mcmc.setup$n.chains = bugs.output$n.chains
  mcmc.setup$n.iter = bugs.output$n.iter
  mcmc.setup$n.burnin = bugs.output$n.burnin
  summary.m$mcmc.setup = mcmc.setup

  class(summary.m) = "summary.bmeta"

  print(summary.m, digits, ...)

}

#' @export

print.summary.bmeta = function(x, digits = 3, ...) {
  cat('Model specifications:\n')
  model.spec = x$model.specification

  cat(paste('  Link function: ', model.spec$link, sep = ''))
  cat('\n')
  cat('\n')
  cat('  Hyper-priors parameters: \n')
  cat(paste('  Prior for mu: Normal', '[', model.spec$mean.mu,', ' ,model.spec$sd.mu^2,']', sep = ''))
  cat('\n')
  cat(paste('  Prior for 1/tau^2: Scale.Gamma', '[', model.spec$scale.sigma.between,', ' ,
            model.spec$df.scale.between,']', sep = ''))
   cat('\n')
  cat('Posterior distributions: \n')
  print(round(x$summary.par, digits))

  cat('\n-------------------\n')


  mcmc = x$mcmc.setup
  cat(paste('MCMC setup (fit using jags): ', mcmc$n.chains, ' chains, each with ', mcmc$n.iter,' iterations', sep = ' '))
  cat('\n')
  cat(paste('(first ', mcmc$n.burnin, ' discarded )', sep = ''))
  cat('\n')
  cat(paste('DIC: ', round(x$DIC, digits), sep = ''))
  cat('\n')
  cat(paste('pD: ',   round(x$pD, digits), sep = ''))
  cat('\n')
}

