\name{plot.dade}
\alias{plot.dade}

\title{Density estimates and partition plot for discriminant analysis
for 2- and 3-dimensional data}
\description{
  Density estimate and partition classes plot
  for 2- and 3-dimensional for kernel, linear and
  quadratic discriminant analysis}
}

\synopsis{
plot.dade(x, y, y.group, prior.prob=NULL, display="part",
    cont=NULL, ncont=NULL, ...)
}

\usage{
## bivariate
\method{plot}{dade}(x, y, y.group, prior.prob=NULL, display="part",
    cont=c(25,50,75), ncont=NULL, xlim, ylim, xlabs="x",
    ylabs="y", drawlabels=TRUE, cex=1, pch, lty, col,
    lcol, ptcol, ...)

## trivariate
\method{plot}{dade}(x, y, y.group, prior.prob=NULL, display="rgl",
    cont=c(25,50), colors, alphavec, origin=c(0,0,0), endpts,
    xlabs="x", ylabs="y", zlabs="z", drawpoints=TRUE, size=3,
    ptcol, ...)

}

\arguments{
  \item{x}{an object of class \code{dade} i.e. output from
    \code{\link{kda.kde}} or \code{\link{pda.pde}}}
  \item{display}{"part" = include plot of partition classes, "" = don't plot
    partition i.e. density estimates only)}
  \item{y}{matrix of test data points}
  \item{y.group}{vector of group labels for test data points}
  \item{prior.prob}{vector of prior probabilities}  
  \item{cont}{vector of percentages (of maximum height) for contour
    level curves}
  \item{ncont}{number of contour level curves}
  \item{cex,pch,lty,xlim,ylim,xlabs,ylabs,zlabs}{usual graphics parameters}
  \item{drawlabels}{if TRUE then draw contour labels}
  \item{col}{vector of colours for partition classes}
  \item{ptcol}{vector of colours for data points of each group}
  \item{lcol}{vector of colours for contour lines of density estimates}
  \item{colors}{vector of colours for contours of density estimates (3-d plot)}
  \item{origin}{origin vector (3-d plot)}
  \item{endpts}{vector of end points for each of the 3 axes (3-d plot)}
  \item{alphavec}{vector of transparency values - one for each contour
    (3-d plot)}
  \item{drawpoints}{if TRUE then data points are plotted (3-d plot)}
  \item{size}{size and of plotting symbol (3-d plot)}
  \item{...}{other graphics parameters}
}
  

\value{
  Plot of 2-d density estimates (and partition) for discriminant analysis is
  sent to graphics window. Plot for 3-d is generated by the
  \code{misc3d} and \code{rgl}  libraries and is sent to RGL window.
}

\details{
  If \code{prior.prob} is set to a particular value then this is used. 
  The default is \code{NULL} which means that the sample proportions are used.

  The object \code{x} contains the training data and its group
  labels.  If \code{y} and \code{y.group} are missing then the training
  data points are plotted.  Otherwise, the test data \code{y} are plotted.
  The plotting symbols are set by \code{pch} (the default is 1 to
  \eqn{\nu}{nu}), one for each group, where \eqn{\nu}{nu} is the
  number of groups, and the colour by \code{ptcol}.

  For 2-d plots:
  
  If \code{display="part"} then a partition induced by the discriminant
  analysis is also plotted. If this is not desired, set
  \code{display=""}. Its colours are controlled by \code{col}
  (the default is \code{heat.colors}). 
  
  Unlike \code{\link{plot.kde}}, the contour plots are automatically
  added to the plot.  Default contours are \code{cont=c(25,50,75)}.
  The line types are set by \code{lty} (the default
  is 1 to \eqn{\nu}{nu}). Also,
  \code{cont} and \code{ncont} control the number of level curves (only
  one of these needs to be set).

  For 3-d plots:
  
  Default contours are \code{cont=c(25,50)}. Colors are
  set one per group - default is \code{heat.colors}. The transparency of
  each contour (within each group) is \code{alphavec}. Default is
  \code{seq(0.1, 0.5, length(cont))}. \code{origin} is the point where
  the three axes meet. \code{endpts} is the vector of the 
  maximum axis values to be plotted.  Default \code{endpts} is the
  maxima for the plotting grid from \code{x} (automatically generated by
  \code{\link{kde}}). 
}
 
\references{  

  Bowman, A.W. & Azzalini, A. (1997) \emph{Applied Smoothing Techniques
    for Data Analysis}. Clarendon Press. Oxford.
  
  Simonoff, J. S., (1996) \emph{Smoothing Methods in Statistics}.
  Springer-Verlag. New York.}
}


\seealso{\code{\link{kda.kde}}, \code{\link{pda.pde}},
  \code{\link{kda}}, \code{\link{pda}}}

\examples{
## bivariate example

library(MASS)
data(iris)
ir <- iris[,c(1,2)]
ir.gr <- iris[,5]
xlab <- "Sepal length (mm)"
ylab <- "Sepal width (mm)"
xlim <- c(4,8)
ylim <- c(2,4.5)

H <- Hkda(ir, ir.gr, bw="plugin", pre="scale")
fhat <- kda.kde(ir, ir.gr, H)
lda.fhat <- pda.pde(ir, ir.gr, type="line")
qda.fhat <- pda.pde(ir, ir.gr, type="quad")

layout(rbind(c(1,2), c(3,4)))
plot(fhat, cont=0, xlab=xlab, ylab=ylab, xlim=xlim, ylim=ylim, 
    pch=c(1,5,10))
plot(fhat, ncont=6, xlab=xlab, ylab=ylab, xlim=xlim, ylim=ylim,
     col=c("transparent", "grey", "#8f8f8f"), drawlabels=FALSE)
plot(lda.fhat, ncont=6, xlim=xlim, ylim=ylim, xlab=xlab, ylab=ylab, 
     disp="")
plot(qda.fhat, ncont=6, xlim=xlim, ylim=ylim,  xlab=xlab, ylab=ylab, 
     lty=c(2,5,3))
layout(1)

## trivariate example
ir <- iris[,1:3]
ir.gr <- iris[,5] 
H <- Hkda(ir, ir.gr, bw="plugin", pre="scale")
fhat <- kda.kde(ir, ir.gr, H)
plot(fhat, cont=c(25,50))
   ## colour indicates species, transparency indicates density heights

qda.fhat <- pda.pde(ir, ir.gr, type="quad")
plot(qda.fhat, cont=c(25,50))
}
\keyword{ hplot}

