#############################################################################
## Estimation of psi_r for normal scales
#############################################################################

psins.1d <- function(r, sigma)
{
  if (r %% 2 ==0)
    psins <- (-1)^(r/2)*factorial(r)/((2*sigma)^(r+1)*factorial(r/2)*pi^(1/2))
  else
    psins <- 0
    
  return(psins)  
}

psins.eta <- function(r, A, Sigma)
{
  if (r %% 2 ==0)
  {
    rr <- r/2 
    d <- ncol(Sigma)
    Sigmainv <- chol2inv(chol(Sigma))
    psins.val <- (4*pi)^(-d/2)*4^(-rr)*det(Sigma)^(-1/2)*nu.noncent(mu=rep(0,d), A=Sigmainv %*% A %*%Sigmainv,Sigma=-2*Sigma, r=r/2) 
  }
  else
    psins.val <- 0
  return(psins.val)
}

psins <- function(r, Sigma, deriv.vec=FALSE, Sdr.mat)
{
  d <- ncol(Sigma)
  
  if (deriv.vec)
  {
    return(drop(dmvnorm.deriv(x=rep(0,d), mu=rep(0,d), deriv.order=r, Sigma=2*Sigma, Sdr.mat=Sdr.mat)))
  }
  else
  {  
    if (d==2) return(psins.2d(r=r, Sigma=Sigma))
    if (d==3) return(psins.3d(r=r, Sigma=Sigma))
    if (d==4) return(psins.4d(r=r, Sigma=Sigma))
    if (d==5) return(psins.5d(r=r, Sigma=Sigma))
    if (d==6) return(psins.6d(r=r, Sigma=Sigma))
  }
}


psins.2d <- function(r, Sigma)
{
  A <- chol2inv(chol(2*Sigma))
  a11 <- A[1,1]
  a22 <- A[2,2]
  a12 <- A[1,2]

  
  if ((r[1] == 0) & (r[2] == 0))   
    psi <- 1
  
  ### second order psi functionals - normal score
  else if ((r[1]==2) & (r[2]==0))
    psi <- -a11
  else if ((r[1]==1) & (r[2]==1))
    psi <- -a12
  else if ((r[1]==0) & (r[2]==2))
    psi <- -a22
  
  ### fourth order psi functionals - normal score
  else if ((r[1]==4) & (r[2]==0))
    psi <-3*a11^2
  else if ((r[1]==3) & (r[2]==1))
    psi <-3*a11*a12
  else if ((r[1]==2) & (r[2]==2))
    psi <-2*a12^2 + a11*a22
  else if ((r[1]==1) & (r[2]==3))
    psi <-3*a12*a22
  else if ((r[1]==0) & (r[2]==4))
    psi <-3*a22^2
  
  ### sixth order psi functionals - normal score 
  else if ((r[1]==6) & (r[2]==0))
    psi <- -15*a11^3
  else if ((r[1]==5) & (r[2]==1))
    psi <- -15*a11^2*a12
  else if ((r[1]==4) & (r[2]==2))
    psi <- -3*a11*(4*a12^2 + a11*a22)
  else if ((r[1]==3) & (r[2]==3))
    psi <- -6*a12^3 - 9*a11*a12*a22
  else if ((r[1]==2) & (r[2]==4))
    psi <- -3*a22*(4*a12^2 + a11*a22)
  else if ((r[1]==1) & (r[2]==5))
    psi <- -15*a12*a22^2
  else if ((r[1]==0) & (r[2]==6))
    psi <- -15*a22^3

  ### eighth  order psi functionals - normal score 
  else if ((r[1]==8) & (r[2]==0))
    psi <- 105*a11^4
  else if ((r[1]==7) & (r[2]==1))
    psi <- 105*a11^3*a12
  else if ((r[1]==6) & (r[2]==2))
    psi <- 15*a11^2*(6*a12^2 + a11*a22)
  else if ((r[1]==5) & (r[2]==3))
    psi <- 15*a11*a12*(4*a12^2 + 3*a11*a22)
  else if ((r[1]==4) & (r[2]==4))
    psi <- 24*a12^4 + 72*a11*a12^2*a22 + 9*a11^2*a22^2
  else if ((r[1]==3) & (r[2]==5))
    psi <- 15*a12*a22*(4*a12^2 + 3*a11*a22)
  else if ((r[1]==2) & (r[2]==6))
    psi <- 15*a22^2*(6*a12^2 + a11*a22)
  else if ((r[1]==1) & (r[2]==7))
    psi <- 105*a12*a22^3
  else if ((r[1]==0) & (r[2]==8))
    psi <- 105*a22^4
  
  return(1/sqrt(2*pi)^2 * sqrt(det(A))*psi) 
}
  
  
psins.3d <- function(r, Sigma)
{
  A <- chol2inv(chol(2*Sigma[1:3, 1:3]))
  a11 <- A[1,1]
  a22 <- A[2,2]
  a33 <- A[3,3]
  a44 <- 0

  a12 <- A[1,2]
  a13 <- A[1,3]
  a14 <- 0
  a23 <- A[2,3]
  a24 <- 0
  a34 <- 0
  
  r <- c(r, 0) 
  if (sum(r)==6)
    psi <- psins.4d.part6(r, a11, a22, a33, a44, a12, a13, a14, a23, a24, a34)
  else if (sum(r)==8)
    psi <- psins.4d.part8(r, a11, a22, a33, a44, a12, a13, a14, a23, a24, a34)

  return(1/sqrt(2*pi)^3 * sqrt(det(A))*psi)
}

psins.4d <- function(r, Sigma)
{
  A <- chol2inv(chol(2*Sigma))
  a11 <- A[1,1]
  a22 <- A[2,2]
  a33 <- A[3,3]
  a44 <- A[4,4]

  a12 <- A[1,2]
  a13 <- A[1,3]
  a14 <- A[1,4]
  a23 <- A[2,3]
  a24 <- A[2,4]
  a34 <- A[3,4]
  
  if (sum(r)==6)
    psi <- psins.4d.part6(r, a11, a22, a33, a44, a12, a13, a14, a23, a24, a34)
  else if (sum(r)==8)
    psi <- psins.4d.part8(r, a11, a22, a33, a44, a12, a13, a14, a23, a24, a34)

  return(1/sqrt(2*pi)^4 * sqrt(det(A))*psi)
}


psins.5d <- function(r, Sigma)
{
  A <- chol2inv(chol(2*Sigma[1:5,1:5]))
  a11 <- A[1,1]
  a22 <- A[2,2]
  a33 <- A[3,3]
  a44 <- A[4,4]
  a55 <- A[3,3]
  a66 <- 0
  
  a12 <- A[1,2]
  a13 <- A[1,3]
  a14 <- A[1,4]
  a15 <- A[1,5]
  a16 <- 0
  a23 <- A[2,3]
  a24 <- A[2,4]
  a25 <- A[2,5]
  a26 <- 0
  a34 <- A[3,4]
  a35 <- A[3,5]
  a36 <- 0
  a45 <- A[4,5]
  a46 <- 0
  a56 <- 0

  r <- c(r,0)  
  if (sum(r)==6)
   psi <- psins.6d.part6(r, a11, a22, a33, a44, a55, a66, a12, a13, a14, a15,
                          a16, a23, a24, a25, a26, a34, a35, a36, a45, a46, a56)
  else if (sum(r)==8)
  {
    psi	<- psins.6d.part81(r, a11, a22, a33, a44, a55, a66, a12, a13, a14, a15,
                          a16, a23, a24, a25, a26, a34, a35, a36, a45, a46, a56)
    if (is.na(psi))
      psi <- psins.6d.part82(r, a11, a22, a33, a44, a55, a66, a12, a13, a14, a15,
                          a16, a23, a24, a25, a26, a34, a35, a36, a45, a46, a56)
    if (is.na(psi))
      psi <- psins.6d.part83(r, a11, a22, a33, a44, a55, a66, a12, a13, a14, a15,
                          a16, a23, a24, a25, a26, a34, a35, a36, a45, a46, a56)
    if (is.na(psi))
      psi <- psins.6d.part84(r, a11, a22, a33, a44, a55, a66, a12, a13, a14, a15,
                          a16, a23, a24, a25, a26, a34, a35, a36, a45, a46, a56)
  }
  return(1/sqrt(2*pi)^6 * sqrt(det(A))*psi)
}

 
psins.6d <- function(r, Sigma)
{
  A <- chol2inv(chol(2*Sigma))
  a11 <- A[1,1]
  a22 <- A[2,2]
  a33 <- A[3,3]
  a44 <- A[4,4]
  a55 <- A[3,3]
  a66 <- A[4,4]
  
  a12 <- A[1,2]
  a13 <- A[1,3]
  a14 <- A[1,4]
  a15 <- A[1,5]
  a16 <- A[1,6]
  a23 <- A[2,3]
  a24 <- A[2,4]
  a25 <- A[2,5]
  a26 <- A[2,6]
  a34 <- A[3,4]
  a35 <- A[3,5]
  a36 <- A[3,6]
  a45 <- A[4,5]
  a46 <- A[4,6]
  a56 <- A[5,6]
  
  if (sum(r)==6)
   psi <- psins.6d.part6(r, a11, a22, a33, a44, a55, a66, a12, a13, a14, a15,
                          a16, a23, a24, a25, a26, a34, a35, a36, a45, a46, a56)
  else if (sum(r)==8)
  {
    psi	<- psins.6d.part81(r, a11, a22, a33, a44, a55, a66, a12, a13, a14, a15,
                          a16, a23, a24, a25, a26, a34, a35, a36, a45, a46, a56)
    if (is.na(psi))
      psi <- psins.6d.part82(r, a11, a22, a33, a44, a55, a66, a12, a13, a14, a15,
                          a16, a23, a24, a25, a26, a34, a35, a36, a45, a46, a56)
    if (is.na(psi))
      psi <- psins.6d.part83(r, a11, a22, a33, a44, a55, a66, a12, a13, a14, a15,
                          a16, a23, a24, a25, a26, a34, a35, a36, a45, a46, a56)
    if (is.na(psi))
      psi <- psins.6d.part84(r, a11, a22, a33, a44, a55, a66, a12, a13, a14, a15,
                          a16, a23, a24, a25, a26, a34, a35, a36, a45, a46, a56)
  }
  return(1/sqrt(2*pi)^6 * sqrt(det(A))*psi)
}

 


psins.4d.part6 <- function(r, a11, a22, a33, a44, a12, a13, a14, a23, a24, a34)
{
  if ((r[1]==6) & (r[2]==0) & (r[3]==0) & (r[4]==0))
    psi <- -15*a11^3
  if ((r[1]==5) & (r[2]==1) & (r[3]==0) & (r[4]==0))
    psi <- -15*a11^2*a12
  if ((r[1]==5) & (r[2]==0) & (r[3]==1) & (r[4]==0))
    psi <- -15*a11^2*a13
  if ((r[1]==5) & (r[2]==0) & (r[3]==0) & (r[4]==1))
    psi <- -15*a11^2*a14
  if ((r[1]==4) & (r[2]==2) & (r[3]==0) & (r[4]==0))
    psi <- -3*a11*(4*a12^2 + a11*a22)
  if ((r[1]==4) & (r[2]==1) & (r[3]==1) & (r[4]==0))
    psi <- -3*a11*(4*a12*a13 + a11*a23)
  if ((r[1]==4) & (r[2]==1) & (r[3]==0) & (r[4]==1))
    psi <- -3*a11*(4*a12*a14 + a11*a24)
  if ((r[1]==4) & (r[2]==0) & (r[3]==2) & (r[4]==0))
    psi <- -3*a11*(4*a13^2 + a11*a33)
  if ((r[1]==4) & (r[2]==0) & (r[3]==1) & (r[4]==1))
    psi <- -3*a11*(4*a13*a14 + a11*a34)
  if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==2))
    psi <- -3*a11*(4*a14^2 + a11*a44)
  if ((r[1]==3) & (r[2]==3) & (r[3]==0) & (r[4]==0))
    psi <- -6*a12^3 - 9*a11*a12*a22
  if ((r[1]==3) & (r[2]==2) & (r[3]==1) & (r[4]==0))
    psi <- -3*(2*a12^2*a13 + a11*a13*a22 + 2*a11*a12*a23)
  if ((r[1]==3) & (r[2]==2) & (r[3]==0) & (r[4]==1))
    psi <- -3*(2*a12^2*a14 + a11*a14*a22 + 2*a11*a12*a24)
  if ((r[1]==3) & (r[2]==1) & (r[3]==2) & (r[4]==0))
    psi <- -3*(2*a12*a13^2 + 2*a11*a13*a23 + a11*a12*a33)
  if ((r[1]==3) & (r[2]==1) & (r[3]==1) & (r[4]==1))
    psi <- -3*(2*a12*a13*a14 + a11*a14*a23 + a11*a13*a24 + a11*a12*a34)
  if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==2))
    psi <- -3*(2*a12*a14^2 + 2*a11*a14*a24 + a11*a12*a44)
  if ((r[1]==3) & (r[2]==0) & (r[3]==3) & (r[4]==0))
    psi <- -6*a13^3 - 9*a11*a13*a33
  if ((r[1]==3) & (r[2]==0) & (r[3]==2) & (r[4]==1))
    psi <- -3*(2*a13^2*a14 + a11*a14*a33 + 2*a11*a13*a34)
  if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==2))
    psi <- -3*(2*a13*a14^2 + 2*a11*a14*a34 + a11*a13*a44)
  if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==3))
    psi <- -6*a14^3 - 9*a11*a14*a44
  if ((r[1]==2) & (r[2]==4) & (r[3]==0) & (r[4]==0))
    psi <- -3*a22*(4*a12^2 + a11*a22)
  if ((r[1]==2) & (r[2]==3) & (r[3]==1) & (r[4]==0))
    psi <- -3*(2*a12*a13*a22 + 2*a12^2*a23 + a11*a22*a23)
  if ((r[1]==2) & (r[2]==3) & (r[3]==0) & (r[4]==1))
    psi <- -3*(2*a12*a14*a22 + 2*a12^2*a24 + a11*a22*a24)
  if ((r[1]==2) & (r[2]==2) & (r[3]==2) & (r[4]==0))
    psi <- -2*a13^2*a22 - 8*a12*a13*a23 - 2*a11*a23^2 - 2*a12^2*a33 - a11*a22*a33
  if ((r[1]==2) & (r[2]==2) & (r[3]==1) & (r[4]==1))
    psi <- -2*a13*a14*a22 - 4*a12*a14*a23 - 4*a12*a13*a24 - 2*a11*a23*a24 - 2*a12^2*a34 - a11*a22*a34
  if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==2))
    psi <- -2*a14^2*a22 - 8*a12*a14*a24 - 2*a11*a24^2 - 2*a12^2*a44 - a11*a22*a44
  if ((r[1]==2) & (r[2]==1) & (r[3]==3) & (r[4]==0))
    psi <- -3*(2*a13^2*a23 + 2*a12*a13*a33 + a11*a23*a33)
  if ((r[1]==2) & (r[2]==1) & (r[3]==2) & (r[4]==1))
    psi <- -4*a13*a14*a23 - 2*a13^2*a24 - 2*a12*a14*a33 - a11*a24*a33 - 4*a12*a13*a34 - 2*a11*a23*a34
  if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==2))
    psi <- -2*a14^2*a23 - 4*a13*a14*a24 - 4*a12*a14*a34 - 2*a11*a24*a34 - 2*a12*a13*a44 - a11*a23*a44
  if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==3))
    psi <- -3*(2*a14^2*a24 + 2*a12*a14*a44 + a11*a24*a44)
  if ((r[1]==2) & (r[2]==0) & (r[3]==4) & (r[4]==0))
    psi <- -3*a33*(4*a13^2 + a11*a33)
  if ((r[1]==2) & (r[2]==0) & (r[3]==3) & (r[4]==1))
    psi <- -3*(2*a13*a14*a33 + 2*a13^2*a34 + a11*a33*a34)
  if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==2))
    psi <- -2*a14^2*a33 - 8*a13*a14*a34 - 2*a11*a34^2 - 2*a13^2*a44 - a11*a33*a44
  if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==3))
    psi <- -3*(2*a14^2*a34 + 2*a13*a14*a44 + a11*a34*a44)
  if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==4))
    psi <- -3*a44*(4*a14^2 + a11*a44)
  if ((r[1]==1) & (r[2]==5) & (r[3]==0) & (r[4]==0))
    psi <- -15*a12*a22^2
  if ((r[1]==1) & (r[2]==4) & (r[3]==1) & (r[4]==0))
    psi <- -3*a22*(a13*a22 + 4*a12*a23)
  if ((r[1]==1) & (r[2]==4) & (r[3]==0) & (r[4]==1))
    psi <- -3*a22*(a14*a22 + 4*a12*a24)
  if ((r[1]==1) & (r[2]==3) & (r[3]==2) & (r[4]==0))
    psi <- -3*(2*a13*a22*a23 + 2*a12*a23^2 + a12*a22*a33)
  if ((r[1]==1) & (r[2]==3) & (r[3]==1) & (r[4]==1))
    psi <- -3*(a14*a22*a23 + a13*a22*a24 + 2*a12*a23*a24 + a12*a22*a34)
  if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==2))
    psi <- -3*(2*a14*a22*a24 + 2*a12*a24^2 + a12*a22*a44)
  if ((r[1]==1) & (r[2]==2) & (r[3]==3) & (r[4]==0))
    psi <- -3*(2*a13*a23^2 + a13*a22*a33 + 2*a12*a23*a33)
  if ((r[1]==1) & (r[2]==2) & (r[3]==2) & (r[4]==1))
    psi <- -2*a14*a23^2 - 4*a13*a23*a24 - a14*a22*a33 - 2*a12*a24*a33 -
      2*a13*a22*a34 -  4*a12*a23*a34
  if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==2))
    psi <- -4*a14*a23*a24 - 2*a13*a24^2 - 2*a14*a22*a34 - 4*a12*a24*a34 - a13*a22*a44 - 2*a12*a23*a44
  if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==3))
    psi <- -3*(2*a14*a24^2 + a14*a22*a44 + 2*a12*a24*a44)
  if ((r[1]==1) & (r[2]==1) & (r[3]==4) & (r[4]==0))
    psi <- -3*a33*(4*a13*a23 + a12*a33)
  if ((r[1]==1) & (r[2]==1) & (r[3]==3) & (r[4]==1))
    psi <- -3*(a14*a23*a33 + a13*a24*a33 + 2*a13*a23*a34 + a12*a33*a34)
  if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==2))
    psi <- -2*a14*a24*a33 - 4*a14*a23*a34 - 4*a13*a24*a34 - 2*a12*a34^2 - 
      2*a13*a23*a44 - a12*a33*a44
  if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==3))
    psi <- -3*(2*a14*a24*a34 + a14*a23*a44 + a13*a24*a44 + a12*a34*a44)
  if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==4))
    psi <- -3*a44*(4*a14*a24 + a12*a44)
  if ((r[1]==1) & (r[2]==0) & (r[3]==5) & (r[4]==0))
    psi <- -15*a13*a33^2
  if ((r[1]==1) & (r[2]==0) & (r[3]==4) & (r[4]==1))
    psi <- -3*a33*(a14*a33 + 4*a13*a34)
  if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==2))
    psi <- -3*(2*a14*a33*a34 + 2*a13*a34^2 + a13*a33*a44)
  if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==3))
    psi <- -3*(2*a14*a34^2 + a14*a33*a44 + 2*a13*a34*a44)
  if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==4))
    psi <- -3*a44*(4*a14*a34 + a13*a44)
  if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==5))
    psi <- -15*a14*a44^2
  if ((r[1]==0) & (r[2]==6) & (r[3]==0) & (r[4]==0))
    psi <- -15*a22^3
  if ((r[1]==0) & (r[2]==5) & (r[3]==1) & (r[4]==0))
    psi <- -15*a22^2*a23
  if ((r[1]==0) & (r[2]==5) & (r[3]==0) & (r[4]==1))
    psi <- -15*a22^2*a24
  if ((r[1]==0) & (r[2]==4) & (r[3]==2) & (r[4]==0))
    psi <- -3*a22*(4*a23^2 + a22*a33)
  if ((r[1]==0) & (r[2]==4) & (r[3]==1) & (r[4]==1))
    psi <- -3*a22*(4*a23*a24 + a22*a34)
  if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==2))
    psi <- -3*a22*(4*a24^2 + a22*a44)
  if ((r[1]==0) & (r[2]==3) & (r[3]==3) & (r[4]==0))
    psi <- -6*a23^3 - 9*a22*a23*a33
  if ((r[1]==0) & (r[2]==3) & (r[3]==2) & (r[4]==1))
    psi <- -3*(2*a23^2*a24 + a22*a24*a33 + 2*a22*a23*a34)
  if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==2))
    psi <- -3*(2*a23*a24^2 + 2*a22*a24*a34 + a22*a23*a44)
  if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==3))
   psi <- -6*a24^3 - 9*a22*a24*a44
  if ((r[1]==0) & (r[2]==2) & (r[3]==4) & (r[4]==0))
    psi <- -3*a33*(4*a23^2 + a22*a33)
  if ((r[1]==0) & (r[2]==2) & (r[3]==3) & (r[4]==1))
    psi <- -3*(2*a23*a24*a33 + 2*a23^2*a34 + a22*a33*a34)
  if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==2))
    psi <- -2*a24^2*a33 - 8*a23*a24*a34 - 2*a22*a34^2 - 2*a23^2*a44 - a22*a33*a44
  if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==3))
    psi <- -3*(2*a24^2*a34 + 2*a23*a24*a44 + a22*a34*a44)
  if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==4))
    psi <- -3*a44*(4*a24^2 + a22*a44)
  if ((r[1]==0) & (r[2]==1) & (r[3]==5) & (r[4]==0))
    psi <- -15*a23*a33^2
  if ((r[1]==0) & (r[2]==1) & (r[3]==4) & (r[4]==1))
    psi <- -3*a33*(a24*a33 + 4*a23*a34)
  if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==2))
    psi <- -3*(2*a24*a33*a34 + 2*a23*a34^2 + a23*a33*a44)
  if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==3))
    psi <- -3*(2*a24*a34^2 + a24*a33*a44 + 2*a23*a34*a44)
  if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==4))
    psi <- -3*a44*(4*a24*a34 + a23*a44)
  if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==5))
    psi <- -15*a24*a44^2
  if ((r[1]==0) & (r[2]==0) & (r[3]==6) & (r[4]==0))
    psi <- -15*a33^3
  if ((r[1]==0) & (r[2]==0) & (r[3]==5) & (r[4]==1))
    psi <- -15*a33^2*a34
  if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==2))
    psi <- -3*a33*(4*a34^2 + a33*a44)
  if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==3))
    psi <- -6*a34^3 - 9*a33*a34*a44
  if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==4))
    psi <- -3*a44*(4*a34^2 + a33*a44)
  if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==5))
    psi <- -15*a34*a44^2
  if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==6))
    psi <- -15*a44^3

   return(psi)
}

psins.4d.part8 <- function(r, a11, a22, a33, a44, a12, a13, a14, a23, a24, a34)
{

  if ((r[1]==8) & (r[2]==0) & (r[3]==0) & (r[4]==0))
    psi <-105*a11^4
  if ((r[1]==7) & (r[2]==1) & (r[3]==0) & (r[4]==0))
    psi <- 105*a11^3*a12
  if ((r[1]==7) & (r[2]==0) & (r[3]==1) & (r[4]==0))
    psi <-105*a11^3*a13
  if ((r[1]==7) & (r[2]==0) & (r[3]==0) & (r[4]==1))
    psi <- 105*a11^3*a14
  if ((r[1]==6) & (r[2]==2) & (r[3]==0) & (r[4]==0))
    psi <- 15*a11^2*(6*a12^2 + a11*a22)
  if ((r[1]==6) & (r[2]==1) & (r[3]==1) & (r[4]==0))
    psi <- 15*a11^2*(6*a12*a13 + a11*a23)
  if ((r[1]==6) & (r[2]==1) & (r[3]==0) & (r[4]==1))
    psi <- 15*a11^2*(6*a12*a14 + a11*a24)
  if ((r[1]==6) & (r[2]==0) & (r[3]==2) & (r[4]==0))
    psi <- 15*a11^2*(6*a13^2 + a11*a33)
  if ((r[1]==6) & (r[2]==0) & (r[3]==1) & (r[4]==1))
    psi <- 15*a11^2*(6*a13*a14 + a11*a34)
  if ((r[1]==6) & (r[2]==0) & (r[3]==0) & (r[4]==2))
    psi <- 15*a11^2*(6*a14^2 + a11*a44)
  if ((r[1]==5) & (r[2]==3) & (r[3]==0) & (r[4]==0))
    psi <- 15*a11*a12*(4*a12^2 + 3*a11*a22)
  if ((r[1]==5) & (r[2]==2) & (r[3]==1) & (r[4]==0))
    psi <-15*a11*(4*a12^2*a13 + a11*a13*a22 + 2*a11*a12*a23)
  if ((r[1]==5) & (r[2]==2) & (r[3]==0) & (r[4]==1))
    psi <-15*a11*(4*a12^2*a14 + a11*a14*a22 + 2*a11*a12*a24)
  if ((r[1]==5) & (r[2]==1) & (r[3]==2) & (r[4]==0))
    psi <-15*a11*(4*a12*a13^2 + 2*a11*a13*a23 + a11*a12*a33)
  if ((r[1]==5) & (r[2]==1) & (r[3]==1) & (r[4]==1))
    psi <-15*a11*(4*a12*a13*a14 + a11*a14*a23 + a11*a13*a24 + a11*a12*a34)
  if ((r[1]==5) & (r[2]==1) & (r[3]==0) & (r[4]==2))
    psi <-15*a11*(4*a12*a14^2 + 2*a11*a14*a24 + a11*a12*a44)
  if ((r[1]==5) & (r[2]==0) & (r[3]==3) & (r[4]==0))
    psi <-15*a11*a13*(4*a13^2 + 3*a11*a33)
  if ((r[1]==5) & (r[2]==0) & (r[3]==2) & (r[4]==1))
    psi <-15*a11*(4*a13^2*a14 + a11*a14*a33 + 2*a11*a13*a34)
  if ((r[1]==5) & (r[2]==0) & (r[3]==1) & (r[4]==2))
    psi <-15*a11*(4*a13*a14^2 + 2*a11*a14*a34 + a11*a13*a44)
  if ((r[1]==5) & (r[2]==0) & (r[3]==0) & (r[4]==3))
    psi <-15*a11*a14*(4*a14^2 + 3*a11*a44)
  if ((r[1]==4) & (r[2]==4) & (r[3]==0) & (r[4]==0))
    psi <-24*a12^4 + 72*a11*a12^2*a22 + 9*a11^2*a22^2
  if ((r[1]==4) & (r[2]==3) & (r[3]==1) & (r[4]==0))
    psi <-24*a12^3*a13 + 36*a11*a12*a13*a22 + 36*a11*a12^2*a23 + 9*a11^2*a22*a23
  if ((r[1]==4) & (r[2]==3) & (r[3]==0) & (r[4]==1))
    psi <-24*a12^3*a14 + 36*a11*a12*a14*a22 + 36*a11*a12^2*a24 + 9*a11^2*a22*a24
  if ((r[1]==4) & (r[2]==2) & (r[3]==2) & (r[4]==0))
    psi <-
      3*(16*a11*a12*a13*a23 + 4*a12^2*(2*a13^2 + a11*a33) + 
         a11*(4*a13^2*a22 + 2*a11*a23^2 + a11*a22*a33))
  if ((r[1]==4) & (r[2]==2) & (r[3]==1) & (r[4]==1))
    psi <- 3*(8*a11*a12*(a14*a23 + a13*a24) + 4*a12^2*(2*a13*a14 + a11*a34) + 
              a11*(4*a13*a14*a22 + 2*a11*a23*a24 + a11*a22*a34))
  if ((r[1]==4) & (r[2]==2) & (r[3]==0) & (r[4]==2))
    psi <-
      3*(16*a11*a12*a14*a24 + 4*a12^2*(2*a14^2 + a11*a44) + 
         a11*(4*a14^2*a22 + 2*a11*a24^2 + a11*a22*a44))
  if ((r[1]==4) & (r[2]==1) & (r[3]==3) & (r[4]==0))
    psi <-
      9*a11*a23*(4*a13^2 + a11*a33) + 12*a12*(2*a13^3 + 3*a11*a13*a33)
  if ((r[1]==4) & (r[2]==1) & (r[3]==2) & (r[4]==1))
    psi <-
      3*(4*a12*(2*a13^2*a14 + a11*a14*a33 + 2*a11*a13*a34) + 
  a11*(8*a13*a14*a23 + 4*a13^2*a24 + a11*a24*a33 + 2*a11*a23*a34))
  if ((r[1]==4) & (r[2]==1) & (r[3]==1) & (r[4]==2))
    psi <-
      3*(4*a12*(2*a13*a14^2 + 2*a11*a14*a34 + a11*a13*a44) + 
         a11*(4*a14^2*a23 + 8*a13*a14*a24 + 2*a11*a24*a34 + a11*a23*a44))
  if ((r[1]==4) & (r[2]==1) & (r[3]==0) & (r[4]==3))
    psi <-
      9*a11*a24*(4*a14^2 + a11*a44) + 12*a12*(2*a14^3 + 3*a11*a14*a44)
  if ((r[1]==4) & (r[2]==0) & (r[3]==4) & (r[4]==0))
    psi <-
      24*a13^4 + 72*a11*a13^2*a33 + 9*a11^2*a33^2
  if ((r[1]==4) & (r[2]==0) & (r[3]==3) & (r[4]==1))
  psi <-
    24*a13^3*a14 + 36*a11*a13*a14*a33 + 36*a11*a13^2*a34 + 9*a11^2*a33*a34
  if ((r[1]==4) & (r[2]==0) & (r[3]==2) & (r[4]==2))
    psi <-
      3*(16*a11*a13*a14*a34 + 4*a13^2*(2*a14^2 + a11*a44) + 
         a11*(4*a14^2*a33 + 2*a11*a34^2 + a11*a33*a44))
  if ((r[1]==4) & (r[2]==0) & (r[3]==1) & (r[4]==3))
    psi <-
      9*a11*a34*(4*a14^2 + a11*a44) + 12*a13*(2*a14^3 + 3*a11*a14*a44)
  if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==4))
    psi <-
      24*a14^4 + 72*a11*a14^2*a44 + 9*a11^2*a44^2
  if ((r[1]==3) & (r[2]==5) & (r[3]==0) & (r[4]==0))
    psi <-
      15*a12*a22*(4*a12^2 + 3*a11*a22)
  if ((r[1]==3) & (r[2]==4) & (r[3]==1) & (r[4]==0))
    psi <-
      36*a12^2*a13*a22 + 9*a11*a13*a22^2 + 24*a12^3*a23 + 36*a11*a12*a22*a23
  if ((r[1]==3) & (r[2]==4) & (r[3]==0) & (r[4]==1))
    psi <-
      36*a12^2*a14*a22 + 9*a11*a14*a22^2 + 24*a12^3*a24 + 36*a11*a12*a22*a24
  if ((r[1]==3) & (r[2]==3) & (r[3]==2) & (r[4]==0))
    psi <-
      3*(12*a12^2*a13*a23 + 6*a11*a13*a22*a23 + 2*a12^3*a33 + 
         3*a12*(2*a13^2*a22 + 2*a11*a23^2 + a11*a22*a33))
  if ((r[1]==3) & (r[2]==3) & (r[3]==1) & (r[4]==1))
    psi <-
      3*(6*a12^2*(a14*a23 + a13*a24) + 3*a11*a22*(a14*a23 + a13*a24) + 
         2*a12^3*a34 + 3*a12*(2*a13*a14*a22 + 2*a11*a23*a24 + a11*a22*a34))
  if ((r[1]==3) & (r[2]==3) & (r[3]==0) & (r[4]==2))
    psi <-
      3*(12*a12^2*a14*a24 + 6*a11*a14*a22*a24 + 2*a12^3*a44 + 
         3*a12*(2*a14^2*a22 + 2*a11*a24^2 + a11*a22*a44))
  if ((r[1]==3) & (r[2]==2) & (r[3]==3) & (r[4]==0))
    psi <-
      3*(2*a13^3*a22 + 12*a12*a13^2*a23 + 6*a11*a12*a23*a33 + 
         3*a13*(2*a11*a23^2 + 2*a12^2*a33 + a11*a22*a33))
  if ((r[1]==3) & (r[2]==2) & (r[3]==2) & (r[4]==1))
    psi <-
      3*(2*a13^2*(a14*a22 + 2*a12*a24) + 2*a12^2*a14*a33 + 
         2*a13*(4*a12*a14*a23 + 2*a11*a23*a24 + 2*a12^2*a34 + a11*a22*a34) + 
         a11*(2*a14*a23^2 + a14*a22*a33 + 2*a12*a24*a33 + 4*a12*a23*a34))
  if ((r[1]==3) & (r[2]==2) & (r[3]==1) & (r[4]==2))
    psi <-
      3*(a13*(2*a14^2*a22 + 8*a12*a14*a24 + 2*a11*a24^2 + 2*a12^2*a44 + 
              a11*a22*a44) + 2*(2*a12^2*a14*a34 + a11*a14*(2*a23*a24 + a22*a34) + 
                              a12*(2*a14^2*a23 + 2*a11*a24*a34 + a11*a23*a44)))
  if ((r[1]==3) & (r[2]==2) & (r[3]==0) & (r[4]==3))
    psi <-
      3*(2*a14^3*a22 + 12*a12*a14^2*a24 + 6*a11*a12*a24*a44 + 
         3*a14*(2*a11*a24^2 + 2*a12^2*a44 + a11*a22*a44))
  if ((r[1]==3) & (r[2]==1) & (r[3]==4) & (r[4]==0))
    psi <-
      24*a13^3*a23 + 36*a12*a13^2*a33 + 36*a11*a13*a23*a33 + 9*a11*a12*a33^2
  if ((r[1]==3) & (r[2]==1) & (r[3]==3) & (r[4]==1))
    psi <-
      3*(2*a13^3*a24 + 6*a13^2*(a14*a23 + a12*a34) + 
         3*a11*a33*(a14*a23 + a12*a34) + 3*a13*(2*a12*a14*a33 + a11*a24*a33 + 
                                                2*a11*a23*a34))
  if ((r[1]==3) & (r[2]==1) & (r[3]==2) & (r[4]==2))
    psi <-
      3*(2*a11*a14*(a24*a33 + 2*a23*a34) + 2*a13^2*(2*a14*a24 + a12*a44) + 
         2*a13*(2*a14^2*a23 + 4*a12*a14*a34 + 2*a11*a24*a34 + a11*a23*a44) + 
         a12*(2*a14^2*a33 + 2*a11*a34^2 + a11*a33*a44))
  if ((r[1]==3) & (r[2]==1) & (r[3]==1) & (r[4]==3))
    psi <-
      3*(2*a14^3*a23 + 6*a14^2*(a13*a24 + a12*a34) + 3*a11*(a13*a24 + a12*a34)*
         a44 + 3*a14*(2*a11*a24*a34 + 2*a12*a13*a44 + a11*a23*a44))
  if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==4))
    psi <-
      24*a14^3*a24 + 36*a12*a14^2*a44 + 36*a11*a14*a24*a44 + 9*a11*a12*a44^2
  if ((r[1]==3) & (r[2]==0) & (r[3]==5) & (r[4]==0))
    psi <-
      15*a13*a33*(4*a13^2 + 3*a11*a33)
  if ((r[1]==3) & (r[2]==0) & (r[3]==4) & (r[4]==1))
    psi <-
      36*a13^2*a14*a33 + 9*a11*a14*a33^2 + 24*a13^3*a34 + 36*a11*a13*a33*a34
  if ((r[1]==3) & (r[2]==0) & (r[3]==3) & (r[4]==2))
    psi <-
      3*(12*a13^2*a14*a34 + 6*a11*a14*a33*a34 + 2*a13^3*a44 + 
         3*a13*(2*a14^2*a33 + 2*a11*a34^2 + a11*a33*a44))
  if ((r[1]==3) & (r[2]==0) & (r[3]==2) & (r[4]==3))
    psi <-
      3*(2*a14^3*a33 + 12*a13*a14^2*a34 + 6*a11*a13*a34*a44 + 
         3*a14*(2*a11*a34^2 + 2*a13^2*a44 + a11*a33*a44))
  if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==4))
    psi <-
      24*a14^3*a34 + 36*a13*a14^2*a44 + 36*a11*a14*a34*a44 + 9*a11*a13*a44^2
  if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==5))
    psi <-
      15*a14*a44*(4*a14^2 + 3*a11*a44)
  if ((r[1]==2) & (r[2]==6) & (r[3]==0) & (r[4]==0))
    psi <-
      15*a22^2*(6*a12^2 + a11*a22)
  if ((r[1]==2) & (r[2]==5) & (r[3]==1) & (r[4]==0))
    psi <-
      15*a22*(2*a12*a13*a22 + 4*a12^2*a23 + a11*a22*a23)
  if ((r[1]==2) & (r[2]==5) & (r[3]==0) & (r[4]==1))
    psi <-
      15*a22*(2*a12*a14*a22 + 4*a12^2*a24 + a11*a22*a24)
  if ((r[1]==2) & (r[2]==4) & (r[3]==2) & (r[4]==0))
    psi <-
      3*(2*a13^2*a22^2 + 16*a12*a13*a22*a23 + 4*a12^2*(2*a23^2 + a22*a33) + 
         a11*a22*(4*a23^2 + a22*a33))
  if ((r[1]==2) & (r[2]==4) & (r[3]==1) & (r[4]==1))
    psi <-
      3*(8*a12*a14*a22*a23 + 2*a13*a22*(a14*a22 + 4*a12*a24) + 
         4*a12^2*(2*a23*a24 + a22*a34) + a11*a22*(4*a23*a24 + a22*a34))
  if ((r[1]==2) & (r[2]==4) & (r[3]==0) & (r[4]==2))
    psi <-
      3*(2*a14^2*a22^2 + 16*a12*a14*a22*a24 + 4*a12^2*(2*a24^2 + a22*a44) + 
         a11*a22*(4*a24^2 + a22*a44))
  if ((r[1]==2) & (r[2]==3) & (r[3]==3) & (r[4]==0))
    psi <-
      3*(6*a13^2*a22*a23 + 6*a12*a13*(2*a23^2 + a22*a33) + 
         a23*(2*a11*a23^2 + 6*a12^2*a33 + 3*a11*a22*a33))
  if ((r[1]==2) & (r[2]==3) & (r[3]==2) & (r[4]==1))
    psi <-
      3*(2*a13^2*a22*a24 + 2*a12*a14*(2*a23^2 + a22*a33) + 
         4*a13*(a14*a22*a23 + 2*a12*a23*a24 + a12*a22*a34) + 
         2*a12^2*(a24*a33 + 2*a23*a34) + a11*(2*a23^2*a24 + a22*a24*a33 + 
                                              2*a22*a23*a34))
if ((r[1]==2) & (r[2]==3) & (r[3]==1) & (r[4]==2))
psi <-
3*(2*a14^2*a22*a23 + 4*a14*(a13*a22*a24 + 2*a12*a23*a24 + a12*a22*a34) + 
  2*a12*a13*(2*a24^2 + a22*a44) + 2*a12^2*(2*a24*a34 + a23*a44) + 
  a11*(2*a23*a24^2 + 2*a22*a24*a34 + a22*a23*a44))
if ((r[1]==2) & (r[2]==3) & (r[3]==0) & (r[4]==3))
psi <-
3*(6*a14^2*a22*a24 + 6*a12*a14*(2*a24^2 + a22*a44) + 
  a24*(2*a11*a24^2 + 6*a12^2*a44 + 3*a11*a22*a44))
if ((r[1]==2) & (r[2]==2) & (r[3]==4) & (r[4]==0))
psi <-
3*(16*a12*a13*a23*a33 + 4*a13^2*(2*a23^2 + a22*a33) + 
  a33*(4*a11*a23^2 + 2*a12^2*a33 + a11*a22*a33))
if ((r[1]==2) & (r[2]==2) & (r[3]==3) & (r[4]==1))
psi <-
3*(4*a12*a14*a23*a33 + 2*a12^2*a33*a34 + 2*a13^2*(2*a23*a24 + a22*a34) + 
  2*a13*(2*a14*a23^2 + a14*a22*a33 + 2*a12*a24*a33 + 4*a12*a23*a34) + 
  a11*(2*a23*a24*a33 + 2*a23^2*a34 + a22*a33*a34))
if ((r[1]==2) & (r[2]==2) & (r[3]==2) & (r[4]==2))
psi <-
2*a11*a24^2*a33 + 2*a14^2*(2*a23^2 + a22*a33) + 8*a11*a23*a24*a34 + 
 4*a12^2*a34^2 + 2*a11*a22*a34^2 + 
 8*a14*(2*a13*a23*a24 + a12*a24*a33 + a13*a22*a34 + 2*a12*a23*a34) + 
 2*a11*a23^2*a44 + 2*a12^2*a33*a44 + a11*a22*a33*a44 + 
 2*a13^2*(2*a24^2 + a22*a44) + 8*a12*a13*(2*a24*a34 + a23*a44)
if ((r[1]==2) & (r[2]==2) & (r[3]==1) & (r[4]==3))
psi <-
3*(2*a14^2*(2*a23*a24 + a22*a34) + 2*a12*(2*a13*a24 + a12*a34)*a44 + 
  2*a14*(2*a13*a24^2 + 4*a12*a24*a34 + a13*a22*a44 + 2*a12*a23*a44) + 
  a11*(2*a24^2*a34 + 2*a23*a24*a44 + a22*a34*a44))
  if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==4))
psi <-
3*(16*a12*a14*a24*a44 + 4*a14^2*(2*a24^2 + a22*a44) + 
  a44*(4*a11*a24^2 + 2*a12^2*a44 + a11*a22*a44))
if ((r[1]==2) & (r[2]==1) & (r[3]==5) & (r[4]==0))
psi <-
15*a33*(4*a13^2*a23 + 2*a12*a13*a33 + a11*a23*a33)
if ((r[1]==2) & (r[2]==1) & (r[3]==4) & (r[4]==1))
psi <-
3*(8*a13*a33*(a14*a23 + a12*a34) + 4*a13^2*(a24*a33 + 2*a23*a34) + 
  a33*(2*a12*a14*a33 + a11*a24*a33 + 4*a11*a23*a34))
if ((r[1]==2) & (r[2]==1) & (r[3]==3) & (r[4]==2))
psi <-
3*(2*a14^2*a23*a33 + 4*a14*(a13*a24*a33 + 2*a13*a23*a34 + a12*a33*a34) + 
  2*a13^2*(2*a24*a34 + a23*a44) + 2*a12*a13*(2*a34^2 + a33*a44) + 
  a11*(2*a24*a33*a34 + 2*a23*a34^2 + a23*a33*a44))
if ((r[1]==2) & (r[2]==1) & (r[3]==2) & (r[4]==3))
psi <-
3*(2*a14^2*(a24*a33 + 2*a23*a34) + 2*a13*(a13*a24 + 2*a12*a34)*a44 + 
  2*a14*(4*a13*a24*a34 + 2*a12*a34^2 + 2*a13*a23*a44 + a12*a33*a44) + 
  a11*(2*a24*a34^2 + a24*a33*a44 + 2*a23*a34*a44))
if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==4))
psi <-
3*(8*a14*(a13*a24 + a12*a34)*a44 + 4*a14^2*(2*a24*a34 + a23*a44) + 
  a44*(4*a11*a24*a34 + 2*a12*a13*a44 + a11*a23*a44))
if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==5))
psi <-
15*a44*(4*a14^2*a24 + 2*a12*a14*a44 + a11*a24*a44)
if ((r[1]==2) & (r[2]==0) & (r[3]==6) & (r[4]==0))
psi <-
15*a33^2*(6*a13^2 + a11*a33)
if ((r[1]==2) & (r[2]==0) & (r[3]==5) & (r[4]==1))
psi <-
15*a33*(2*a13*a14*a33 + 4*a13^2*a34 + a11*a33*a34)
if ((r[1]==2) & (r[2]==0) & (r[3]==4) & (r[4]==2))
psi <-
3*(2*a14^2*a33^2 + 16*a13*a14*a33*a34 + 4*a13^2*(2*a34^2 + a33*a44) + 
  a11*a33*(4*a34^2 + a33*a44))
if ((r[1]==2) & (r[2]==0) & (r[3]==3) & (r[4]==3))
psi <-
3*(6*a14^2*a33*a34 + 6*a13*a14*(2*a34^2 + a33*a44) + 
  a34*(2*a11*a34^2 + 6*a13^2*a44 + 3*a11*a33*a44))
if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==4))
psi <-
3*(16*a13*a14*a34*a44 + 4*a14^2*(2*a34^2 + a33*a44) + 
  a44*(4*a11*a34^2 + 2*a13^2*a44 + a11*a33*a44))
if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==5))
psi <-
15*a44*(4*a14^2*a34 + 2*a13*a14*a44 + a11*a34*a44)
if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==6))
psi <-
15*a44^2*(6*a14^2 + a11*a44)
  if ((r[1]==1) & (r[2]==7) & (r[3]==0) & (r[4]==0))
psi <-
105*a12*a22^3
if ((r[1]==1) & (r[2]==6) & (r[3]==1) & (r[4]==0))
psi <-
15*a22^2*(a13*a22 + 6*a12*a23)
if ((r[1]==1) & (r[2]==6) & (r[3]==0) & (r[4]==1))
psi <-
15*a22^2*(a14*a22 + 6*a12*a24)
if ((r[1]==1) & (r[2]==5) & (r[3]==2) & (r[4]==0))
psi <-
15*a22*(2*a13*a22*a23 + 4*a12*a23^2 + a12*a22*a33)
if ((r[1]==1) & (r[2]==5) & (r[3]==1) & (r[4]==1))
psi <-
15*a22*(a14*a22*a23 + a13*a22*a24 + 4*a12*a23*a24 + a12*a22*a34)
if ((r[1]==1) & (r[2]==5) & (r[3]==0) & (r[4]==2))
psi <-
15*a22*(2*a14*a22*a24 + 4*a12*a24^2 + a12*a22*a44)
if ((r[1]==1) & (r[2]==4) & (r[3]==3) & (r[4]==0))
psi <-
9*a13*a22*(4*a23^2 + a22*a33) + 12*a12*a23*(2*a23^2 + 3*a22*a33)
if ((r[1]==1) & (r[2]==4) & (r[3]==2) & (r[4]==1))
psi <-
3*(a14*a22*(4*a23^2 + a22*a33) + 2*a13*a22*(4*a23*a24 + a22*a34) + 
  4*a12*(2*a23^2*a24 + a22*a24*a33 + 2*a22*a23*a34))
if ((r[1]==1) & (r[2]==4) & (r[3]==1) & (r[4]==2))
psi <-
3*(2*a14*a22*(4*a23*a24 + a22*a34) + a13*a22*(4*a24^2 + a22*a44) + 
  4*a12*(2*a23*a24^2 + 2*a22*a24*a34 + a22*a23*a44))
if ((r[1]==1) & (r[2]==4) & (r[3]==0) & (r[4]==3))
psi <-
9*a14*a22*(4*a24^2 + a22*a44) + 12*a12*a24*(2*a24^2 + 3*a22*a44)
if ((r[1]==1) & (r[2]==3) & (r[3]==4) & (r[4]==0))
psi <-
9*a12*a33*(4*a23^2 + a22*a33) + 12*a13*(2*a23^3 + 3*a22*a23*a33)
if ((r[1]==1) & (r[2]==3) & (r[3]==3) & (r[4]==1))
psi <-
3*(2*a14*a23^3 + 6*a13*a23^2*a24 + 3*a14*a22*a23*a33 + 3*a13*a22*a24*a33 + 
  6*a12*a23*a24*a33 + 6*a13*a22*a23*a34 + 6*a12*a23^2*a34 + 3*a12*a22*a33*a34)
if ((r[1]==1) & (r[2]==3) & (r[3]==2) & (r[4]==2))
psi <-
3*(2*a14*(2*a23^2*a24 + a22*a24*a33 + 2*a22*a23*a34) + 
  2*a13*(2*a23*a24^2 + 2*a22*a24*a34 + a22*a23*a44) + 
  a12*(2*a24^2*a33 + 8*a23*a24*a34 + 2*a22*a34^2 + 2*a23^2*a44 + a22*a33*a44))
if ((r[1]==1) & (r[2]==3) & (r[3]==1) & (r[4]==3))
psi <-
3*(2*a13*a24^3 + 6*a12*a24^2*a34 + 3*a13*a22*a24*a44 + 6*a12*a23*a24*a44 + 
  3*a12*a22*a34*a44 + 3*a14*(2*a23*a24^2 + 2*a22*a24*a34 + a22*a23*a44))
if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==4))
psi <-
9*a12*a44*(4*a24^2 + a22*a44) + 12*a14*(2*a24^3 + 3*a22*a24*a44)
if ((r[1]==1) & (r[2]==2) & (r[3]==5) & (r[4]==0))
psi <-
15*a33*(4*a13*a23^2 + a13*a22*a33 + 2*a12*a23*a33)
  if ((r[1]==1) & (r[2]==2) & (r[3]==4) & (r[4]==1))
psi <-
3*(a14*a33*(4*a23^2 + a22*a33) + 2*a12*a33*(a24*a33 + 4*a23*a34) + 
  4*a13*(2*a23*a24*a33 + 2*a23^2*a34 + a22*a33*a34))
if ((r[1]==1) & (r[2]==2) & (r[3]==3) & (r[4]==2))
psi <-
3*(2*a14*(2*a23*a24*a33 + 2*a23^2*a34 + a22*a33*a34) + 
  a13*(2*a24^2*a33 + 8*a23*a24*a34 + 2*a22*a34^2 + 2*a23^2*a44 + 
    a22*a33*a44) + 2*a12*(2*a24*a33*a34 + 2*a23*a34^2 + a23*a33*a44))
if ((r[1]==1) & (r[2]==2) & (r[3]==2) & (r[4]==3))
psi <-
3*(a14*(2*a24^2*a33 + 8*a23*a24*a34 + 2*a22*a34^2 + 2*a23^2*a44 + 
    a22*a33*a44) + 2*(a13*(2*a24^2*a34 + 2*a23*a24*a44 + a22*a34*a44) + 
    a12*(2*a24*a34^2 + a24*a33*a44 + 2*a23*a34*a44)))
if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==4))
psi <-
3*(a44*(4*a13*a24^2 + 8*a12*a24*a34 + a13*a22*a44 + 2*a12*a23*a44) + 
  4*a14*(2*a24^2*a34 + 2*a23*a24*a44 + a22*a34*a44))
if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==5))
psi <-
15*a44*(4*a14*a24^2 + a14*a22*a44 + 2*a12*a24*a44)
if ((r[1]==1) & (r[2]==1) & (r[3]==6) & (r[4]==0))
psi <-
15*a33^2*(6*a13*a23 + a12*a33)
if ((r[1]==1) & (r[2]==1) & (r[3]==5) & (r[4]==1))
psi <-
15*a33*(a14*a23*a33 + a13*a24*a33 + 4*a13*a23*a34 + a12*a33*a34)
if ((r[1]==1) & (r[2]==1) & (r[3]==4) & (r[4]==2))
psi <-
3*(2*a14*a33*(a24*a33 + 4*a23*a34) + a12*a33*(4*a34^2 + a33*a44) + 
  4*a13*(2*a24*a33*a34 + 2*a23*a34^2 + a23*a33*a44))
if ((r[1]==1) & (r[2]==1) & (r[3]==3) & (r[4]==3))
psi <-
3*(6*a13*a24*a34^2 + 2*a12*a34^3 + 3*a13*a24*a33*a44 + 6*a13*a23*a34*a44 + 
  3*a12*a33*a34*a44 + 3*a14*(2*a24*a33*a34 + 2*a23*a34^2 + a23*a33*a44))
if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==4))
psi <-
3*(a44*(8*a13*a24*a34 + 4*a12*a34^2 + 2*a13*a23*a44 + a12*a33*a44) + 
  4*a14*(2*a24*a34^2 + a24*a33*a44 + 2*a23*a34*a44))
if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==5))
psi <-
15*a44*(4*a14*a24*a34 + a14*a23*a44 + a13*a24*a44 + a12*a34*a44)
if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==6))
psi <-
15*a44^2*(6*a14*a24 + a12*a44)
if ((r[1]==1) & (r[2]==0) & (r[3]==7) & (r[4]==0))
psi <-
105*a13*a33^3
if ((r[1]==1) & (r[2]==0) & (r[3]==6) & (r[4]==1))
psi <-
15*a33^2*(a14*a33 + 6*a13*a34)
if ((r[1]==1) & (r[2]==0) & (r[3]==5) & (r[4]==2))
psi <-
15*a33*(2*a14*a33*a34 + 4*a13*a34^2 + a13*a33*a44)
if ((r[1]==1) & (r[2]==0) & (r[3]==4) & (r[4]==3))
psi <-
9*a14*a33*(4*a34^2 + a33*a44) + 12*a13*a34*(2*a34^2 + 3*a33*a44)
  if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==4))
psi <-
9*a13*a44*(4*a34^2 + a33*a44) + 12*a14*(2*a34^3 + 3*a33*a34*a44)
if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==5))
psi <-
15*a44*(4*a14*a34^2 + a14*a33*a44 + 2*a13*a34*a44)
if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==6))
psi <-
15*a44^2*(6*a14*a34 + a13*a44)
if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==7))
psi <-
105*a14*a44^3
if ((r[1]==0) & (r[2]==8) & (r[3]==0) & (r[4]==0))
psi <-
105*a22^4
if ((r[1]==0) & (r[2]==7) & (r[3]==1) & (r[4]==0))
psi <-
105*a22^3*a23
if ((r[1]==0) & (r[2]==7) & (r[3]==0) & (r[4]==1))
psi <-
105*a22^3*a24
if ((r[1]==0) & (r[2]==6) & (r[3]==2) & (r[4]==0))
psi <-
15*a22^2*(6*a23^2 + a22*a33)
if ((r[1]==0) & (r[2]==6) & (r[3]==1) & (r[4]==1))
psi <-
15*a22^2*(6*a23*a24 + a22*a34)
if ((r[1]==0) & (r[2]==6) & (r[3]==0) & (r[4]==2))
psi <-
15*a22^2*(6*a24^2 + a22*a44)
if ((r[1]==0) & (r[2]==5) & (r[3]==3) & (r[4]==0))
psi <-
15*a22*a23*(4*a23^2 + 3*a22*a33)
if ((r[1]==0) & (r[2]==5) & (r[3]==2) & (r[4]==1))
psi <-
15*a22*(4*a23^2*a24 + a22*a24*a33 + 2*a22*a23*a34)
if ((r[1]==0) & (r[2]==5) & (r[3]==1) & (r[4]==2))
psi <-
15*a22*(4*a23*a24^2 + 2*a22*a24*a34 + a22*a23*a44)
if ((r[1]==0) & (r[2]==5) & (r[3]==0) & (r[4]==3))
psi <-
15*a22*a24*(4*a24^2 + 3*a22*a44)
if ((r[1]==0) & (r[2]==4) & (r[3]==4) & (r[4]==0))
psi <-
24*a23^4 + 72*a22*a23^2*a33 + 9*a22^2*a33^2
if ((r[1]==0) & (r[2]==4) & (r[3]==3) & (r[4]==1))
psi <-
24*a23^3*a24 + 36*a22*a23*a24*a33 + 36*a22*a23^2*a34 + 9*a22^2*a33*a34
if ((r[1]==0) & (r[2]==4) & (r[3]==2) & (r[4]==2))
psi <-
3*(16*a22*a23*a24*a34 + 4*a23^2*(2*a24^2 + a22*a44) + 
  a22*(4*a24^2*a33 + 2*a22*a34^2 + a22*a33*a44))
if ((r[1]==0) & (r[2]==4) & (r[3]==1) & (r[4]==3))
psi <-
9*a22*a34*(4*a24^2 + a22*a44) + 12*a23*(2*a24^3 + 3*a22*a24*a44)
if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==4))
psi <-
24*a24^4 + 72*a22*a24^2*a44 + 9*a22^2*a44^2
if ((r[1]==0) & (r[2]==3) & (r[3]==5) & (r[4]==0))
psi <-
15*a23*a33*(4*a23^2 + 3*a22*a33)
if ((r[1]==0) & (r[2]==3) & (r[3]==4) & (r[4]==1))
psi <-
36*a23^2*a24*a33 + 9*a22*a24*a33^2 + 24*a23^3*a34 + 36*a22*a23*a33*a34
if ((r[1]==0) & (r[2]==3) & (r[3]==3) & (r[4]==2))
psi <-
3*(12*a23^2*a24*a34 + 6*a22*a24*a33*a34 + 2*a23^3*a44 + 
  3*a23*(2*a24^2*a33 + 2*a22*a34^2 + a22*a33*a44))
if ((r[1]==0) & (r[2]==3) & (r[3]==2) & (r[4]==3))
psi <-
3*(2*a24^3*a33 + 12*a23*a24^2*a34 + 6*a22*a23*a34*a44 + 
  3*a24*(2*a22*a34^2 + 2*a23^2*a44 + a22*a33*a44))
if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==4))
psi <-
24*a24^3*a34 + 36*a23*a24^2*a44 + 36*a22*a24*a34*a44 + 9*a22*a23*a44^2
if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==5))
psi <-
15*a24*a44*(4*a24^2 + 3*a22*a44)
if ((r[1]==0) & (r[2]==2) & (r[3]==6) & (r[4]==0))
psi <-
15*a33^2*(6*a23^2 + a22*a33)
if ((r[1]==0) & (r[2]==2) & (r[3]==5) & (r[4]==1))
psi <-
15*a33*(2*a23*a24*a33 + 4*a23^2*a34 + a22*a33*a34)
if ((r[1]==0) & (r[2]==2) & (r[3]==4) & (r[4]==2))
psi <-
3*(2*a24^2*a33^2 + 16*a23*a24*a33*a34 + 4*a23^2*(2*a34^2 + a33*a44) + 
  a22*a33*(4*a34^2 + a33*a44))
if ((r[1]==0) & (r[2]==2) & (r[3]==3) & (r[4]==3))
psi <-
3*(6*a24^2*a33*a34 + 6*a23*a24*(2*a34^2 + a33*a44) + 
  a34*(2*a22*a34^2 + 6*a23^2*a44 + 3*a22*a33*a44))
if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==4))
psi <-
3*(16*a23*a24*a34*a44 + 4*a24^2*(2*a34^2 + a33*a44) + 
  a44*(4*a22*a34^2 + 2*a23^2*a44 + a22*a33*a44))
if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==5))
psi <-
15*a44*(4*a24^2*a34 + 2*a23*a24*a44 + a22*a34*a44)
if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==6))
psi <-
15*a44^2*(6*a24^2 + a22*a44)
if ((r[1]==0) & (r[2]==1) & (r[3]==7) & (r[4]==0))
psi <-
105*a23*a33^3
if ((r[1]==0) & (r[2]==1) & (r[3]==6) & (r[4]==1))
psi <-
15*a33^2*(a24*a33 + 6*a23*a34)
  if ((r[1]==0) & (r[2]==1) & (r[3]==5) & (r[4]==2))
psi <-
15*a33*(2*a24*a33*a34 + 4*a23*a34^2 + a23*a33*a44)
if ((r[1]==0) & (r[2]==1) & (r[3]==4) & (r[4]==3))
psi <-
9*a24*a33*(4*a34^2 + a33*a44) + 12*a23*a34*(2*a34^2 + 3*a33*a44)
if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==4))
psi <-
9*a23*a44*(4*a34^2 + a33*a44) + 12*a24*(2*a34^3 + 3*a33*a34*a44)
if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==5))
psi <-
15*a44*(4*a24*a34^2 + a24*a33*a44 + 2*a23*a34*a44)
if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==6))
psi <-
15*a44^2*(6*a24*a34 + a23*a44)
if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==7))
psi <-
105*a24*a44^3
if ((r[1]==0) & (r[2]==0) & (r[3]==8) & (r[4]==0))
psi <-
105*a33^4
if ((r[1]==0) & (r[2]==0) & (r[3]==7) & (r[4]==1))
psi <-
105*a33^3*a34
if ((r[1]==0) & (r[2]==0) & (r[3]==6) & (r[4]==2))
psi <-
15*a33^2*(6*a34^2 + a33*a44)
if ((r[1]==0) & (r[2]==0) & (r[3]==5) & (r[4]==3))
psi <-
15*a33*a34*(4*a34^2 + 3*a33*a44)
if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==4))
psi <-
24*a34^4 + 72*a33*a34^2*a44 + 9*a33^2*a44^2
if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==5))
psi <-
15*a34*a44*(4*a34^2 + 3*a33*a44)
if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==6))
psi <-
15*a44^2*(6*a34^2 + a33*a44)
if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==7))
psi <- 105*a34*a44^3
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==8))
psi <- 105*a44^4


  return(psi)
}



psins.6d.part6 <- function(r, a11, a22, a33, a44, a55, a66, a12, a13, a14, a15,
                           a16, a23, a24, a25, a26, a34, a35, a36, a45, a46, a56)
{
  if ((r[1]==6) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==0) & (r[6]==0))
    psi <--15*a11^3
if ((r[1]==5) & (r[2]==1) & (r[3]==0) & (r[4]==0) & (r[5]==0) & (r[6]==0))
    psi <- -15*a11^2*a12
if ((r[1]==5) & (r[2]==0) & (r[3]==1) & (r[4]==0) & (r[5]==0) & (r[6]==0))
psi <-
-15*a11^2*a13
if ((r[1]==5) & (r[2]==0) & (r[3]==0) & (r[4]==1) & (r[5]==0) & (r[6]==0))
psi <-
-15*a11^2*a14
if ((r[1]==5) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==1) & (r[6]==0))
psi <-
-15*a11^2*a15
if ((r[1]==5) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==0) & (r[6]==1))
psi <-
-15*a11^2*a16
if ((r[1]==4) & (r[2]==2) & (r[3]==0) & (r[4]==0) & (r[5]==0) & (r[6]==0))
psi <-
-3*a11*(4*a12^2 + a11*a22)
if ((r[1]==4) & (r[2]==1) & (r[3]==1) & (r[4]==0) & (r[5]==0) & (r[6]==0))
psi <-
-3*a11*(4*a12*a13 + a11*a23)
if ((r[1]==4) & (r[2]==1) & (r[3]==0) & (r[4]==1) & (r[5]==0) & (r[6]==0))
psi <-
-3*a11*(4*a12*a14 + a11*a24)
if ((r[1]==4) & (r[2]==1) & (r[3]==0) & (r[4]==0) & (r[5]==1) & (r[6]==0))
psi <-
-3*a11*(4*a12*a15 + a11*a25)
if ((r[1]==4) & (r[2]==1) & (r[3]==0) & (r[4]==0) & (r[5]==0) & (r[6]==1))
psi <-
-3*a11*(4*a12*a16 + a11*a26)
if ((r[1]==4) & (r[2]==0) & (r[3]==2) & (r[4]==0) & (r[5]==0) & (r[6]==0))
psi <-
-3*a11*(4*a13^2 + a11*a33)
if ((r[1]==4) & (r[2]==0) & (r[3]==1) & (r[4]==1) & (r[5]==0) & (r[6]==0))
psi <-
-3*a11*(4*a13*a14 + a11*a34)
if ((r[1]==4) & (r[2]==0) & (r[3]==1) & (r[4]==0) & (r[5]==1) & (r[6]==0))
psi <-
-3*a11*(4*a13*a15 + a11*a35)
if ((r[1]==4) & (r[2]==0) & (r[3]==1) & (r[4]==0) & (r[5]==0) & (r[6]==1))
psi <-
-3*a11*(4*a13*a16 + a11*a36)
if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==2) & (r[5]==0) & (r[6]==0))
psi <-
-3*a11*(4*a14^2 + a11*a44)
if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==1) & (r[5]==1) & (r[6]==0))
psi <-
-3*a11*(4*a14*a15 + a11*a45)
if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==1) & (r[5]==0) & (r[6]==1))
psi <-
-3*a11*(4*a14*a16 + a11*a46)
if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==2) & (r[6]==0))
psi <-
-3*a11*(4*a15^2 + a11*a55)
  if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==1) & (r[6]==1))
psi <-
-3*a11*(4*a15*a16 + a11*a56)
if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==0) & (r[6]==2))
psi <-
-3*a11*(4*a16^2 + a11*a66)
if ((r[1]==3) & (r[2]==3) & (r[3]==0) & (r[4]==0) & (r[5]==0) & (r[6]==0))
psi <-
-6*a12^3 - 9*a11*a12*a22
if ((r[1]==3) & (r[2]==2) & (r[3]==1) & (r[4]==0) & (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a12^2*a13 + a11*a13*a22 + 2*a11*a12*a23)
if ((r[1]==3) & (r[2]==2) & (r[3]==0) & (r[4]==1) & (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a12^2*a14 + a11*a14*a22 + 2*a11*a12*a24)
if ((r[1]==3) & (r[2]==2) & (r[3]==0) & (r[4]==0) & (r[5]==1) & (r[6]==0))
psi <-
-3*(2*a12^2*a15 + a11*a15*a22 + 2*a11*a12*a25)
if ((r[1]==3) & (r[2]==2) & (r[3]==0) & (r[4]==0) & (r[5]==0) & (r[6]==1))
psi <-
-3*(2*a12^2*a16 + a11*a16*a22 + 2*a11*a12*a26)
if ((r[1]==3) & (r[2]==1) & (r[3]==2) & (r[4]==0) & (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a12*a13^2 + 2*a11*a13*a23 + a11*a12*a33)
if ((r[1]==3) & (r[2]==1) & (r[3]==1) & (r[4]==1) & (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a12*a13*a14 + a11*a14*a23 + a11*a13*a24 + a11*a12*a34)
if ((r[1]==3) & (r[2]==1) & (r[3]==1) & (r[4]==0) & (r[5]==1) & (r[6]==0))
psi <-
-3*(2*a12*a13*a15 + a11*a15*a23 + a11*a13*a25 + a11*a12*a35)
if ((r[1]==3) & (r[2]==1) & (r[3]==1) & (r[4]==0) & (r[5]==0) & (r[6]==1))
psi <-
-3*(2*a12*a13*a16 + a11*a16*a23 + a11*a13*a26 + a11*a12*a36)
if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==2) & (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a12*a14^2 + 2*a11*a14*a24 + a11*a12*a44)
if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==1) & (r[5]==1) & (r[6]==0))
psi <-
-3*(2*a12*a14*a15 + a11*a15*a24 + a11*a14*a25 + a11*a12*a45)
if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==1) & (r[5]==0) & (r[6]==1))
psi <-
-3*(2*a12*a14*a16 + a11*a16*a24 + a11*a14*a26 + a11*a12*a46)
if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==0) & (r[5]==2) & (r[6]==0))
psi <-
-3*(2*a12*a15^2 + 2*a11*a15*a25 + a11*a12*a55)
if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==0) & (r[5]==1) & (r[6]==1))
psi <-
-3*(2*a12*a15*a16 + a11*a16*a25 + a11*a15*a26 + a11*a12*a56)
if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==0) & (r[5]==0) & (r[6]==2))
psi <-
-3*(2*a12*a16^2 + 2*a11*a16*a26 + a11*a12*a66)
if ((r[1]==3) & (r[2]==0) & (r[3]==3) & (r[4]==0) & (r[5]==0) & (r[6]==0))
psi <-
-6*a13^3 - 9*a11*a13*a33
if ((r[1]==3) & (r[2]==0) & (r[3]==2) & (r[4]==1) & (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a13^2*a14 + a11*a14*a33 + 2*a11*a13*a34)
  if ((r[1]==3) & (r[2]==0) & (r[3]==2) & (r[4]==0) & (r[5]==1) & (r[6]==0))
psi <-
-3*(2*a13^2*a15 + a11*a15*a33 + 2*a11*a13*a35)
if ((r[1]==3) & (r[2]==0) & (r[3]==2) & (r[4]==0) & (r[5]==0) & (r[6]==1))
psi <-
-3*(2*a13^2*a16 + a11*a16*a33 + 2*a11*a13*a36)
if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==2) & (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a13*a14^2 + 2*a11*a14*a34 + a11*a13*a44)
if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==1) & (r[5]==1) & (r[6]==0))
psi <-
-3*(2*a13*a14*a15 + a11*a15*a34 + a11*a14*a35 + a11*a13*a45)
if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==1) & (r[5]==0) & (r[6]==1))
psi <-
-3*(2*a13*a14*a16 + a11*a16*a34 + a11*a14*a36 + a11*a13*a46)
if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==0) & (r[5]==2) & (r[6]==0))
psi <-
-3*(2*a13*a15^2 + 2*a11*a15*a35 + a11*a13*a55)
if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==0) & (r[5]==1) & (r[6]==1))
psi <-
-3*(2*a13*a15*a16 + a11*a16*a35 + a11*a15*a36 + a11*a13*a56)
if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==0) & (r[5]==0) & (r[6]==2))
psi <-
-3*(2*a13*a16^2 + 2*a11*a16*a36 + a11*a13*a66)
if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==3) & (r[5]==0) & (r[6]==0))
psi <-
-6*a14^3 - 9*a11*a14*a44
if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==2) & (r[5]==1) & (r[6]==0))
psi <-
-3*(2*a14^2*a15 + a11*a15*a44 + 2*a11*a14*a45)
if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==2) & (r[5]==0) & (r[6]==1))
psi <-
-3*(2*a14^2*a16 + a11*a16*a44 + 2*a11*a14*a46)
if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==1) & (r[5]==2) & (r[6]==0))
psi <-
-3*(2*a14*a15^2 + 2*a11*a15*a45 + a11*a14*a55)
if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==1) & (r[5]==1) & (r[6]==1))
psi <-
-3*(2*a14*a15*a16 + a11*a16*a45 + a11*a15*a46 + a11*a14*a56)
if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==1) & (r[5]==0) & (r[6]==2))
psi <-
-3*(2*a14*a16^2 + 2*a11*a16*a46 + a11*a14*a66)
if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==3) & (r[6]==0))
psi <-
-6*a15^3 - 9*a11*a15*a55
if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==2) & (r[6]==1))
psi <-
-3*(2*a15^2*a16 + a11*a16*a55 + 2*a11*a15*a56)
if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==1) & (r[6]==2))
psi <-
-3*(2*a15*a16^2 + 2*a11*a16*a56 + a11*a15*a66)
if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==0) & (r[6]==3))
psi <-
-6*a16^3 - 9*a11*a16*a66
if ((r[1]==2) & (r[2]==4) & (r[3]==0) & (r[4]==0) & (r[5]==0) & (r[6]==0))
psi <-
-3*a22*(4*a12^2 + a11*a22)
if ((r[1]==2) & (r[2]==3) & (r[3]==1) & (r[4]==0) & (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a12*a13*a22 + 2*a12^2*a23 + a11*a22*a23)
if ((r[1]==2) & (r[2]==3) & (r[3]==0) & (r[4]==1) & (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a12*a14*a22 + 2*a12^2*a24 + a11*a22*a24)
if ((r[1]==2) & (r[2]==3) & (r[3]==0) & (r[4]==0) & (r[5]==1) & (r[6]==0))
psi <-
-3*(2*a12*a15*a22 + 2*a12^2*a25 + a11*a22*a25)
if ((r[1]==2) & (r[2]==3) & (r[3]==0) & (r[4]==0) & (r[5]==0) & (r[6]==1))
psi <-
-3*(2*a12*a16*a22 + 2*a12^2*a26 + a11*a22*a26)
if ((r[1]==2) & (r[2]==2) & (r[3]==2) & (r[4]==0) & (r[5]==0) & (r[6]==0))
psi <-
-2*a13^2*a22 - 8*a12*a13*a23 - 2*a11*a23^2 - 2*a12^2*a33 - a11*a22*a33
if ((r[1]==2) & (r[2]==2) & (r[3]==1) & (r[4]==1) & (r[5]==0) & (r[6]==0))
psi <-
-2*a13*a14*a22 - 4*a12*a14*a23 - 4*a12*a13*a24 - 2*a11*a23*a24 - 
 2*a12^2*a34 - a11*a22*a34
if ((r[1]==2) & (r[2]==2) & (r[3]==1) & (r[4]==0) & (r[5]==1) & (r[6]==0))
psi <-
-2*a13*a15*a22 - 4*a12*a15*a23 - 4*a12*a13*a25 - 2*a11*a23*a25 - 
 2*a12^2*a35 - a11*a22*a35
if ((r[1]==2) & (r[2]==2) & (r[3]==1) & (r[4]==0) & (r[5]==0) & (r[6]==1))
psi <-
-2*a13*a16*a22 - 4*a12*a16*a23 - 4*a12*a13*a26 - 2*a11*a23*a26 - 
 2*a12^2*a36 - a11*a22*a36
if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==2) & (r[5]==0) & (r[6]==0))
psi <-
-2*a14^2*a22 - 8*a12*a14*a24 - 2*a11*a24^2 - 2*a12^2*a44 - a11*a22*a44
if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==1) & (r[5]==1) & (r[6]==0))
psi <-
-2*a14*a15*a22 - 4*a12*a15*a24 - 4*a12*a14*a25 - 2*a11*a24*a25 - 
 2*a12^2*a45 - a11*a22*a45
if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==1) & (r[5]==0) & (r[6]==1))
psi <-
-2*a14*a16*a22 - 4*a12*a16*a24 - 4*a12*a14*a26 - 2*a11*a24*a26 - 
 2*a12^2*a46 - a11*a22*a46
if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==0) & (r[5]==2) & (r[6]==0))
psi <-
-2*a15^2*a22 - 8*a12*a15*a25 - 2*a11*a25^2 - 2*a12^2*a55 - a11*a22*a55
if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==0) & (r[5]==1) & (r[6]==1))
psi <-
-2*a15*a16*a22 - 4*a12*a16*a25 - 4*a12*a15*a26 - 2*a11*a25*a26 - 
 2*a12^2*a56 - a11*a22*a56
if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==0) & (r[5]==0) & (r[6]==2))
psi <-
-2*a16^2*a22 - 8*a12*a16*a26 - 2*a11*a26^2 - 2*a12^2*a66 - a11*a22*a66
if ((r[1]==2) & (r[2]==1) & (r[3]==3) & (r[4]==0) & (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a13^2*a23 + 2*a12*a13*a33 + a11*a23*a33)
  if ((r[1]==2) & (r[2]==1) & (r[3]==2) & (r[4]==1) & (r[5]==0) & (r[6]==0))
psi <-
-4*a13*a14*a23 - 2*a13^2*a24 - 2*a12*a14*a33 - a11*a24*a33 - 4*a12*a13*a34 - 
 2*a11*a23*a34
if ((r[1]==2) & (r[2]==1) & (r[3]==2) & (r[4]==0) & (r[5]==1) & (r[6]==0))
psi <-
-4*a13*a15*a23 - 2*a13^2*a25 - 2*a12*a15*a33 - a11*a25*a33 - 4*a12*a13*a35 - 
 2*a11*a23*a35
if ((r[1]==2) & (r[2]==1) & (r[3]==2) & (r[4]==0) & (r[5]==0) & (r[6]==1))
psi <-
-4*a13*a16*a23 - 2*a13^2*a26 - 2*a12*a16*a33 - a11*a26*a33 - 4*a12*a13*a36 - 
 2*a11*a23*a36
if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==2) & (r[5]==0) & (r[6]==0))
psi <-
-2*a14^2*a23 - 4*a13*a14*a24 - 4*a12*a14*a34 - 2*a11*a24*a34 - 
 2*a12*a13*a44 - a11*a23*a44
if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==1) & (r[5]==1) & (r[6]==0))
psi <-
-2*a14*a15*a23 - 2*a13*a15*a24 - 2*a13*a14*a25 - 2*a12*a15*a34 - 
 a11*a25*a34 - 2*a12*a14*a35 - a11*a24*a35 - 2*a12*a13*a45 - a11*a23*a45
if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
-2*a14*a16*a23 - 2*a13*a16*a24 - 2*a13*a14*a26 - 2*a12*a16*a34 - 
 a11*a26*a34 - 2*a12*a14*a36 - a11*a24*a36 - 2*a12*a13*a46 - a11*a23*a46
if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
-2*a15^2*a23 - 4*a13*a15*a25 - 4*a12*a15*a35 - 2*a11*a25*a35 - 
 2*a12*a13*a55 - a11*a23*a55
if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
-2*a15*a16*a23 - 2*a13*a16*a25 - 2*a13*a15*a26 - 2*a12*a16*a35 - 
 a11*a26*a35 - 2*a12*a15*a36 - a11*a25*a36 - 2*a12*a13*a56 - a11*a23*a56
if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
-2*a16^2*a23 - 4*a13*a16*a26 - 4*a12*a16*a36 - 2*a11*a26*a36 - 
 2*a12*a13*a66 - a11*a23*a66
if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a14^2*a24 + 2*a12*a14*a44 + a11*a24*a44)
if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
-4*a14*a15*a24 - 2*a14^2*a25 - 2*a12*a15*a44 - a11*a25*a44 - 4*a12*a14*a45 - 
 2*a11*a24*a45
if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
-4*a14*a16*a24 - 2*a14^2*a26 - 2*a12*a16*a44 - a11*a26*a44 - 4*a12*a14*a46 - 
 2*a11*a24*a46
if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
-2*a15^2*a24 - 4*a14*a15*a25 - 4*a12*a15*a45 - 2*a11*a25*a45 - 
 2*a12*a14*a55 - a11*a24*a55
if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
-2*a15*a16*a24 - 2*a14*a16*a25 - 2*a14*a15*a26 - 2*a12*a16*a45 - 
 a11*a26*a45 - 2*a12*a15*a46 - a11*a25*a46 - 2*a12*a14*a56 - a11*a24*a56
if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
-2*a16^2*a24 - 4*a14*a16*a26 - 4*a12*a16*a46 - 2*a11*a26*a46 - 
 2*a12*a14*a66 - a11*a24*a66
if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
-3*(2*a15^2*a25 + 2*a12*a15*a55 + a11*a25*a55)
if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
-4*a15*a16*a25 - 2*a15^2*a26 - 2*a12*a16*a55 - a11*a26*a55 - 4*a12*a15*a56 - 
 2*a11*a25*a56
if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
-2*a16^2*a25 - 4*a15*a16*a26 - 4*a12*a16*a56 - 2*a11*a26*a56 - 
 2*a12*a15*a66 - a11*a25*a66
if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
-3*(2*a16^2*a26 + 2*a12*a16*a66 + a11*a26*a66)
if ((r[1]==2) & (r[2]==0) & (r[3]==4) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*a33*(4*a13^2 + a11*a33)
if ((r[1]==2) & (r[2]==0) & (r[3]==3) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a13*a14*a33 + 2*a13^2*a34 + a11*a33*a34)
if ((r[1]==2) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*(2*a13*a15*a33 + 2*a13^2*a35 + a11*a33*a35)
if ((r[1]==2) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*(2*a13*a16*a33 + 2*a13^2*a36 + a11*a33*a36)
if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
-2*a14^2*a33 - 8*a13*a14*a34 - 2*a11*a34^2 - 2*a13^2*a44 - a11*a33*a44
if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
-2*a14*a15*a33 - 4*a13*a15*a34 - 4*a13*a14*a35 - 2*a11*a34*a35 - 
 2*a13^2*a45 - a11*a33*a45
  if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
-2*a14*a16*a33 - 4*a13*a16*a34 - 4*a13*a14*a36 - 2*a11*a34*a36 - 
 2*a13^2*a46 - a11*a33*a46
if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
-2*a15^2*a33 - 8*a13*a15*a35 - 2*a11*a35^2 - 2*a13^2*a55 - a11*a33*a55
if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
-2*a15*a16*a33 - 4*a13*a16*a35 - 4*a13*a15*a36 - 2*a11*a35*a36 - 
 2*a13^2*a56 - a11*a33*a56
if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
-2*a16^2*a33 - 8*a13*a16*a36 - 2*a11*a36^2 - 2*a13^2*a66 - a11*a33*a66
if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a14^2*a34 + 2*a13*a14*a44 + a11*a34*a44)
if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
-4*a14*a15*a34 - 2*a14^2*a35 - 2*a13*a15*a44 - a11*a35*a44 - 4*a13*a14*a45 - 
 2*a11*a34*a45
if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
-4*a14*a16*a34 - 2*a14^2*a36 - 2*a13*a16*a44 - a11*a36*a44 - 4*a13*a14*a46 - 
 2*a11*a34*a46
if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
-2*a15^2*a34 - 4*a14*a15*a35 - 4*a13*a15*a45 - 2*a11*a35*a45 - 
 2*a13*a14*a55 - a11*a34*a55
if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
-2*a15*a16*a34 - 2*a14*a16*a35 - 2*a14*a15*a36 - 2*a13*a16*a45 - 
 a11*a36*a45 - 2*a13*a15*a46 - a11*a35*a46 - 2*a13*a14*a56 - a11*a34*a56
if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
-2*a16^2*a34 - 4*a14*a16*a36 - 4*a13*a16*a46 - 2*a11*a36*a46 - 
 2*a13*a14*a66 - a11*a34*a66
if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
-3*(2*a15^2*a35 + 2*a13*a15*a55 + a11*a35*a55)
if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
-4*a15*a16*a35 - 2*a15^2*a36 - 2*a13*a16*a55 - a11*a36*a55 - 4*a13*a15*a56 - 
 2*a11*a35*a56
if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
-2*a16^2*a35 - 4*a15*a16*a36 - 4*a13*a16*a56 - 2*a11*a36*a56 - 
 2*a13*a15*a66 - a11*a35*a66
if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
-3*(2*a16^2*a36 + 2*a13*a16*a66 + a11*a36*a66)
if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==4) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*a44*(4*a14^2 + a11*a44)
if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*(2*a14*a15*a44 + 2*a14^2*a45 + a11*a44*a45)
if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*(2*a14*a16*a44 + 2*a14^2*a46 + a11*a44*a46)
if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==2) & (r[6]==0))
psi <-
-2*a15^2*a44 - 8*a14*a15*a45 - 2*a11*a45^2 - 2*a14^2*a55 - a11*a44*a55
if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==1))
psi <-
-2*a15*a16*a44 - 4*a14*a16*a45 - 4*a14*a15*a46 - 2*a11*a45*a46 - 
 2*a14^2*a56 - a11*a44*a56
if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==2))
psi <-
-2*a16^2*a44 - 8*a14*a16*a46 - 2*a11*a46^2 - 2*a14^2*a66 - a11*a44*a66
if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==3) & (r[6]==0))
psi <-
-3*(2*a15^2*a45 + 2*a14*a15*a55 + a11*a45*a55)
if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==1))
psi <-
-4*a15*a16*a45 - 2*a15^2*a46 - 2*a14*a16*a55 - a11*a46*a55 - 4*a14*a15*a56 - 
 2*a11*a45*a56
if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==2))
psi <-
-2*a16^2*a45 - 4*a15*a16*a46 - 4*a14*a16*a56 - 2*a11*a46*a56 - 
 2*a14*a15*a66 - a11*a45*a66
if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==3))
psi <-
-3*(2*a16^2*a46 + 2*a14*a16*a66 + a11*a46*a66)
  if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==4) & (r[6]==0))
psi <-
-3*a55*(4*a15^2 + a11*a55)
if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==1))
psi <-
-3*(2*a15*a16*a55 + 2*a15^2*a56 + a11*a55*a56)
if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==2))
psi <-
-2*a16^2*a55 - 8*a15*a16*a56 - 2*a11*a56^2 - 2*a15^2*a66 - a11*a55*a66
if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==3))
psi <-
-3*(2*a16^2*a56 + 2*a15*a16*a66 + a11*a56*a66)
if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==4))
psi <-
-3*a66*(4*a16^2 + a11*a66)
if ((r[1]==1) & (r[2]==5) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
-15*a12*a22^2
if ((r[1]==1) & (r[2]==4) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*a22*(a13*a22 + 4*a12*a23)
if ((r[1]==1) & (r[2]==4) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*a22*(a14*a22 + 4*a12*a24)
if ((r[1]==1) & (r[2]==4) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*a22*(a15*a22 + 4*a12*a25)
if ((r[1]==1) & (r[2]==4) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*a22*(a16*a22 + 4*a12*a26)
if ((r[1]==1) & (r[2]==3) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a13*a22*a23 + 2*a12*a23^2 + a12*a22*a33)
if ((r[1]==1) & (r[2]==3) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*(a14*a22*a23 + a13*a22*a24 + 2*a12*a23*a24 + a12*a22*a34)
if ((r[1]==1) & (r[2]==3) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*(a15*a22*a23 + a13*a22*a25 + 2*a12*a23*a25 + a12*a22*a35)
if ((r[1]==1) & (r[2]==3) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*(a16*a22*a23 + a13*a22*a26 + 2*a12*a23*a26 + a12*a22*a36)
if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a14*a22*a24 + 2*a12*a24^2 + a12*a22*a44)
if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*(a15*a22*a24 + a14*a22*a25 + 2*a12*a24*a25 + a12*a22*a45)
if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*(a16*a22*a24 + a14*a22*a26 + 2*a12*a24*a26 + a12*a22*a46)
if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
-3*(2*a15*a22*a25 + 2*a12*a25^2 + a12*a22*a55)
if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
-3*(a16*a22*a25 + a15*a22*a26 + 2*a12*a25*a26 + a12*a22*a56)
if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
-3*(2*a16*a22*a26 + 2*a12*a26^2 + a12*a22*a66)
if ((r[1]==1) & (r[2]==2) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a13*a23^2 + a13*a22*a33 + 2*a12*a23*a33)
if ((r[1]==1) & (r[2]==2) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
-2*a14*a23^2 - 4*a13*a23*a24 - a14*a22*a33 - 2*a12*a24*a33 - 2*a13*a22*a34 - 
 4*a12*a23*a34
if ((r[1]==1) & (r[2]==2) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
-2*a15*a23^2 - 4*a13*a23*a25 - a15*a22*a33 - 2*a12*a25*a33 - 2*a13*a22*a35 - 
 4*a12*a23*a35
if ((r[1]==1) & (r[2]==2) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
-2*a16*a23^2 - 4*a13*a23*a26 - a16*a22*a33 - 2*a12*a26*a33 - 2*a13*a22*a36 - 
 4*a12*a23*a36
if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
-4*a14*a23*a24 - 2*a13*a24^2 - 2*a14*a22*a34 - 4*a12*a24*a34 - a13*a22*a44 - 
 2*a12*a23*a44
if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
-2*a15*a23*a24 - 2*a14*a23*a25 - 2*a13*a24*a25 - a15*a22*a34 - 
 2*a12*a25*a34 - a14*a22*a35 - 2*a12*a24*a35 - a13*a22*a45 - 2*a12*a23*a45
if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
-2*a16*a23*a24 - 2*a14*a23*a26 - 2*a13*a24*a26 - a16*a22*a34 - 
 2*a12*a26*a34 - a14*a22*a36 - 2*a12*a24*a36 - a13*a22*a46 - 2*a12*a23*a46
if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
-4*a15*a23*a25 - 2*a13*a25^2 - 2*a15*a22*a35 - 4*a12*a25*a35 - a13*a22*a55 - 
 2*a12*a23*a55
if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
-2*a16*a23*a25 - 2*a15*a23*a26 - 2*a13*a25*a26 - a16*a22*a35 - 
 2*a12*a26*a35 - a15*a22*a36 - 2*a12*a25*a36 - a13*a22*a56 - 2*a12*a23*a56
if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
-4*a16*a23*a26 - 2*a13*a26^2 - 2*a16*a22*a36 - 4*a12*a26*a36 - a13*a22*a66 - 
 2*a12*a23*a66
if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a14*a24^2 + a14*a22*a44 + 2*a12*a24*a44)
if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
-2*a15*a24^2 - 4*a14*a24*a25 - a15*a22*a44 - 2*a12*a25*a44 - 2*a14*a22*a45 - 
 4*a12*a24*a45
if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
-2*a16*a24^2 - 4*a14*a24*a26 - a16*a22*a44 - 2*a12*a26*a44 - 2*a14*a22*a46 - 
 4*a12*a24*a46
if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
-4*a15*a24*a25 - 2*a14*a25^2 - 2*a15*a22*a45 - 4*a12*a25*a45 - a14*a22*a55 - 
 2*a12*a24*a55
if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
-2*a16*a24*a25 - 2*a15*a24*a26 - 2*a14*a25*a26 - a16*a22*a45 - 
 2*a12*a26*a45 - a15*a22*a46 - 2*a12*a25*a46 - a14*a22*a56 - 2*a12*a24*a56
if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
-4*a16*a24*a26 - 2*a14*a26^2 - 2*a16*a22*a46 - 4*a12*a26*a46 - a14*a22*a66 - 
 2*a12*a24*a66
if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
-3*(2*a15*a25^2 + a15*a22*a55 + 2*a12*a25*a55)
if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
-2*a16*a25^2 - 4*a15*a25*a26 - a16*a22*a55 - 2*a12*a26*a55 - 2*a15*a22*a56 - 
 4*a12*a25*a56
if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
-4*a16*a25*a26 - 2*a15*a26^2 - 2*a16*a22*a56 - 4*a12*a26*a56 - a15*a22*a66 - 
 2*a12*a25*a66
if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
-3*(2*a16*a26^2 + a16*a22*a66 + 2*a12*a26*a66)
if ((r[1]==1) & (r[2]==1) & (r[3]==4) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*a33*(4*a13*a23 + a12*a33)
if ((r[1]==1) & (r[2]==1) & (r[3]==3) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*(a14*a23*a33 + a13*a24*a33 + 2*a13*a23*a34 + a12*a33*a34)
if ((r[1]==1) & (r[2]==1) & (r[3]==3) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*(a15*a23*a33 + a13*a25*a33 + 2*a13*a23*a35 + a12*a33*a35)
if ((r[1]==1) & (r[2]==1) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*(a16*a23*a33 + a13*a26*a33 + 2*a13*a23*a36 + a12*a33*a36)
if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
-2*a14*a24*a33 - 4*a14*a23*a34 - 4*a13*a24*a34 - 2*a12*a34^2 - 
 2*a13*a23*a44 - a12*a33*a44
if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
-(a15*a24*a33) - a14*a25*a33 - 2*a15*a23*a34 - 2*a13*a25*a34 - 
 2*a14*a23*a35 - 2*a13*a24*a35 - 2*a12*a34*a35 - 2*a13*a23*a45 - a12*a33*a45
if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
-(a16*a24*a33) - a14*a26*a33 - 2*a16*a23*a34 - 2*a13*a26*a34 - 
 2*a14*a23*a36 - 2*a13*a24*a36 - 2*a12*a34*a36 - 2*a13*a23*a46 - a12*a33*a46
if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
-2*a15*a25*a33 - 4*a15*a23*a35 - 4*a13*a25*a35 - 2*a12*a35^2 - 
 2*a13*a23*a55 - a12*a33*a55
if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
-(a16*a25*a33) - a15*a26*a33 - 2*a16*a23*a35 - 2*a13*a26*a35 - 
 2*a15*a23*a36 - 2*a13*a25*a36 - 2*a12*a35*a36 - 2*a13*a23*a56 - a12*a33*a56
if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
-2*a16*a26*a33 - 4*a16*a23*a36 - 4*a13*a26*a36 - 2*a12*a36^2 - 
 2*a13*a23*a66 - a12*a33*a66
if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a14*a24*a34 + a14*a23*a44 + a13*a24*a44 + a12*a34*a44)
if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
-2*a15*a24*a34 - 2*a14*a25*a34 - 2*a14*a24*a35 - a15*a23*a44 - a13*a25*a44 - 
 a12*a35*a44 - 2*a14*a23*a45 - 2*a13*a24*a45 - 2*a12*a34*a45
if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
-2*a16*a24*a34 - 2*a14*a26*a34 - 2*a14*a24*a36 - a16*a23*a44 - a13*a26*a44 - 
 a12*a36*a44 - 2*a14*a23*a46 - 2*a13*a24*a46 - 2*a12*a34*a46
if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
-2*a15*a25*a34 - 2*a15*a24*a35 - 2*a14*a25*a35 - 2*a15*a23*a45 - 
 2*a13*a25*a45 - 2*a12*a35*a45 - a14*a23*a55 - a13*a24*a55 - a12*a34*a55
if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
-(a16*a25*a34) - a15*a26*a34 - a16*a24*a35 - a14*a26*a35 - a15*a24*a36 - 
 a14*a25*a36 - a16*a23*a45 - a13*a26*a45 - a12*a36*a45 - a15*a23*a46 - 
 a13*a25*a46 - a12*a35*a46 - a14*a23*a56 - a13*a24*a56 - a12*a34*a56
if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
-2*a16*a26*a34 - 2*a16*a24*a36 - 2*a14*a26*a36 - 2*a16*a23*a46 - 
 2*a13*a26*a46 - 2*a12*a36*a46 - a14*a23*a66 - a13*a24*a66 - a12*a34*a66
if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
-3*(2*a15*a25*a35 + a15*a23*a55 + a13*a25*a55 + a12*a35*a55)
if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
-2*a16*a25*a35 - 2*a15*a26*a35 - 2*a15*a25*a36 - a16*a23*a55 - a13*a26*a55 - 
 a12*a36*a55 - 2*a15*a23*a56 - 2*a13*a25*a56 - 2*a12*a35*a56
if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
-2*a16*a26*a35 - 2*a16*a25*a36 - 2*a15*a26*a36 - 2*a16*a23*a56 - 
 2*a13*a26*a56 - 2*a12*a36*a56 - a15*a23*a66 - a13*a25*a66 - a12*a35*a66
if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
-3*(2*a16*a26*a36 + a16*a23*a66 + a13*a26*a66 + a12*a36*a66)
if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==4) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*a44*(4*a14*a24 + a12*a44)
if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==3) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*(a15*a24*a44 + a14*a25*a44 + 2*a14*a24*a45 + a12*a44*a45)
if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*(a16*a24*a44 + a14*a26*a44 + 2*a14*a24*a46 + a12*a44*a46)
if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==2) & (r[6]==0))
psi <-
-2*a15*a25*a44 - 4*a15*a24*a45 - 4*a14*a25*a45 - 2*a12*a45^2 - 
 2*a14*a24*a55 - a12*a44*a55
if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==1))
psi <-
-(a16*a25*a44) - a15*a26*a44 - 2*a16*a24*a45 - 2*a14*a26*a45 - 
 2*a15*a24*a46 - 2*a14*a25*a46 - 2*a12*a45*a46 - 2*a14*a24*a56 - a12*a44*a56
if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==2))
psi <-
-2*a16*a26*a44 - 4*a16*a24*a46 - 4*a14*a26*a46 - 2*a12*a46^2 - 
 2*a14*a24*a66 - a12*a44*a66
if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==3) & (r[6]==0))
psi <-
-3*(2*a15*a25*a45 + a15*a24*a55 + a14*a25*a55 + a12*a45*a55)
if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==1))
psi <-
-2*a16*a25*a45 - 2*a15*a26*a45 - 2*a15*a25*a46 - a16*a24*a55 - a14*a26*a55 - 
 a12*a46*a55 - 2*a15*a24*a56 - 2*a14*a25*a56 - 2*a12*a45*a56
if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==2))
psi <-
-2*a16*a26*a45 - 2*a16*a25*a46 - 2*a15*a26*a46 - 2*a16*a24*a56 - 
 2*a14*a26*a56 - 2*a12*a46*a56 - a15*a24*a66 - a14*a25*a66 - a12*a45*a66
if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==3))
psi <-
-3*(2*a16*a26*a46 + a16*a24*a66 + a14*a26*a66 + a12*a46*a66)
if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==4) & (r[6]==0))
psi <-
-3*a55*(4*a15*a25 + a12*a55)
if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==1))
psi <-
-3*(a16*a25*a55 + a15*a26*a55 + 2*a15*a25*a56 + a12*a55*a56)
if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==2))
psi <-
-2*a16*a26*a55 - 4*a16*a25*a56 - 4*a15*a26*a56 - 2*a12*a56^2 - 
 2*a15*a25*a66 - a12*a55*a66
if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==3))
psi <-
-3*(2*a16*a26*a56 + a16*a25*a66 + a15*a26*a66 + a12*a56*a66)
  if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==4))
psi <-
-3*a66*(4*a16*a26 + a12*a66)
if ((r[1]==1) & (r[2]==0) & (r[3]==5) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
-15*a13*a33^2
if ((r[1]==1) & (r[2]==0) & (r[3]==4) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*a33*(a14*a33 + 4*a13*a34)
if ((r[1]==1) & (r[2]==0) & (r[3]==4) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*a33*(a15*a33 + 4*a13*a35)
if ((r[1]==1) & (r[2]==0) & (r[3]==4) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*a33*(a16*a33 + 4*a13*a36)
if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a14*a33*a34 + 2*a13*a34^2 + a13*a33*a44)
if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*(a15*a33*a34 + a14*a33*a35 + 2*a13*a34*a35 + a13*a33*a45)
if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*(a16*a33*a34 + a14*a33*a36 + 2*a13*a34*a36 + a13*a33*a46)
if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
-3*(2*a15*a33*a35 + 2*a13*a35^2 + a13*a33*a55)
if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
-3*(a16*a33*a35 + a15*a33*a36 + 2*a13*a35*a36 + a13*a33*a56)
if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
-3*(2*a16*a33*a36 + 2*a13*a36^2 + a13*a33*a66)
if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a14*a34^2 + a14*a33*a44 + 2*a13*a34*a44)
if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
-2*a15*a34^2 - 4*a14*a34*a35 - a15*a33*a44 - 2*a13*a35*a44 - 2*a14*a33*a45 - 
 4*a13*a34*a45
if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
-2*a16*a34^2 - 4*a14*a34*a36 - a16*a33*a44 - 2*a13*a36*a44 - 2*a14*a33*a46 - 
 4*a13*a34*a46
if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
-4*a15*a34*a35 - 2*a14*a35^2 - 2*a15*a33*a45 - 4*a13*a35*a45 - a14*a33*a55 - 
 2*a13*a34*a55
if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
-2*a16*a34*a35 - 2*a15*a34*a36 - 2*a14*a35*a36 - a16*a33*a45 - 
 2*a13*a36*a45 - a15*a33*a46 - 2*a13*a35*a46 - a14*a33*a56 - 2*a13*a34*a56
if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
-4*a16*a34*a36 - 2*a14*a36^2 - 2*a16*a33*a46 - 4*a13*a36*a46 - a14*a33*a66 - 
 2*a13*a34*a66
if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
-3*(2*a15*a35^2 + a15*a33*a55 + 2*a13*a35*a55)
if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
-2*a16*a35^2 - 4*a15*a35*a36 - a16*a33*a55 - 2*a13*a36*a55 - 2*a15*a33*a56 - 
 4*a13*a35*a56
if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
-4*a16*a35*a36 - 2*a15*a36^2 - 2*a16*a33*a56 - 4*a13*a36*a56 - a15*a33*a66 - 
 2*a13*a35*a66
if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
-3*(2*a16*a36^2 + a16*a33*a66 + 2*a13*a36*a66)
if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==4) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*a44*(4*a14*a34 + a13*a44)
if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==3) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*(a15*a34*a44 + a14*a35*a44 + 2*a14*a34*a45 + a13*a44*a45)
if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==3) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*(a16*a34*a44 + a14*a36*a44 + 2*a14*a34*a46 + a13*a44*a46)
if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==2) & (r[6]==0))
psi <-
-2*a15*a35*a44 - 4*a15*a34*a45 - 4*a14*a35*a45 - 2*a13*a45^2 - 
 2*a14*a34*a55 - a13*a44*a55
if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==1) & (r[6]==1))
psi <-
-(a16*a35*a44) - a15*a36*a44 - 2*a16*a34*a45 - 2*a14*a36*a45 - 
 2*a15*a34*a46 - 2*a14*a35*a46 - 2*a13*a45*a46 - 2*a14*a34*a56 - a13*a44*a56
if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==2))
psi <-
-2*a16*a36*a44 - 4*a16*a34*a46 - 4*a14*a36*a46 - 2*a13*a46^2 - 
 2*a14*a34*a66 - a13*a44*a66
if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==3) & (r[6]==0))
psi <-
-3*(2*a15*a35*a45 + a15*a34*a55 + a14*a35*a55 + a13*a45*a55)
if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==2) & (r[6]==1))
psi <-
-2*a16*a35*a45 - 2*a15*a36*a45 - 2*a15*a35*a46 - a16*a34*a55 - a14*a36*a55 - 
 a13*a46*a55 - 2*a15*a34*a56 - 2*a14*a35*a56 - 2*a13*a45*a56
if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==2))
psi <-
-2*a16*a36*a45 - 2*a16*a35*a46 - 2*a15*a36*a46 - 2*a16*a34*a56 - 
 2*a14*a36*a56 - 2*a13*a46*a56 - a15*a34*a66 - a14*a35*a66 - a13*a45*a66
if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==3))
psi <-
-3*(2*a16*a36*a46 + a16*a34*a66 + a14*a36*a66 + a13*a46*a66)
if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==4) & (r[6]==0))
psi <-
-3*a55*(4*a15*a35 + a13*a55)
if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==3) & (r[6]==1))
psi <-
-3*(a16*a35*a55 + a15*a36*a55 + 2*a15*a35*a56 + a13*a55*a56)
if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==2))
psi <-
-2*a16*a36*a55 - 4*a16*a35*a56 - 4*a15*a36*a56 - 2*a13*a56^2 - 
 2*a15*a35*a66 - a13*a55*a66
if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==3))
psi <-
-3*(2*a16*a36*a56 + a16*a35*a66 + a15*a36*a66 + a13*a56*a66)
if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==4))
psi <-
-3*a66*(4*a16*a36 + a13*a66)
if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==5) 
& (r[5]==0) & (r[6]==0))
psi <-
-15*a14*a44^2
if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==4) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*a44*(a15*a44 + 4*a14*a45)
  if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==4) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*a44*(a16*a44 + 4*a14*a46)
if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==2) & (r[6]==0))
psi <-
-3*(2*a15*a44*a45 + 2*a14*a45^2 + a14*a44*a55)
if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==1) & (r[6]==1))
psi <-
-3*(a16*a44*a45 + a15*a44*a46 + 2*a14*a45*a46 + a14*a44*a56)
if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==2))
psi <-
-3*(2*a16*a44*a46 + 2*a14*a46^2 + a14*a44*a66)
if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==3) & (r[6]==0))
psi <-
-3*(2*a15*a45^2 + a15*a44*a55 + 2*a14*a45*a55)
if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==2) & (r[6]==1))
psi <-
-2*a16*a45^2 - 4*a15*a45*a46 - a16*a44*a55 - 2*a14*a46*a55 - 2*a15*a44*a56 - 
 4*a14*a45*a56
if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==2))
psi <-
-4*a16*a45*a46 - 2*a15*a46^2 - 2*a16*a44*a56 - 4*a14*a46*a56 - a15*a44*a66 - 
 2*a14*a45*a66
if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==3))
psi <-
-3*(2*a16*a46^2 + a16*a44*a66 + 2*a14*a46*a66)
if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==4) & (r[6]==0))
psi <-
-3*a55*(4*a15*a45 + a14*a55)
if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==3) & (r[6]==1))
psi <-
-3*(a16*a45*a55 + a15*a46*a55 + 2*a15*a45*a56 + a14*a55*a56)
if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==2))
psi <-
-2*a16*a46*a55 - 4*a16*a45*a56 - 4*a15*a46*a56 - 2*a14*a56^2 - 
 2*a15*a45*a66 - a14*a55*a66
if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==3))
psi <-
-3*(2*a16*a46*a56 + a16*a45*a66 + a15*a46*a66 + a14*a56*a66)
if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==4))
psi <-
-3*a66*(4*a16*a46 + a14*a66)
if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==5) & (r[6]==0))
psi <-
-15*a15*a55^2
if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==4) & (r[6]==1))
psi <-
-3*a55*(a16*a55 + 4*a15*a56)
if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==2))
psi <-
-3*(2*a16*a55*a56 + 2*a15*a56^2 + a15*a55*a66)
if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==3))
psi <-
-3*(2*a16*a56^2 + a16*a55*a66 + 2*a15*a56*a66)
if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==4))
psi <-
-3*a66*(4*a16*a56 + a15*a66)
if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==5))
psi <-
-15*a16*a66^2
if ((r[1]==0) & (r[2]==6) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
-15*a22^3
if ((r[1]==0) & (r[2]==5) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
-15*a22^2*a23
if ((r[1]==0) & (r[2]==5) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
-15*a22^2*a24
if ((r[1]==0) & (r[2]==5) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
-15*a22^2*a25
if ((r[1]==0) & (r[2]==5) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
-15*a22^2*a26
if ((r[1]==0) & (r[2]==4) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*a22*(4*a23^2 + a22*a33)
if ((r[1]==0) & (r[2]==4) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*a22*(4*a23*a24 + a22*a34)
if ((r[1]==0) & (r[2]==4) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*a22*(4*a23*a25 + a22*a35)
if ((r[1]==0) & (r[2]==4) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*a22*(4*a23*a26 + a22*a36)
if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*a22*(4*a24^2 + a22*a44)
if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*a22*(4*a24*a25 + a22*a45)
if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*a22*(4*a24*a26 + a22*a46)
if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
-3*a22*(4*a25^2 + a22*a55)
if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
-3*a22*(4*a25*a26 + a22*a56)
if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
-3*a22*(4*a26^2 + a22*a66)
if ((r[1]==0) & (r[2]==3) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
-6*a23^3 - 9*a22*a23*a33
if ((r[1]==0) & (r[2]==3) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a23^2*a24 + a22*a24*a33 + 2*a22*a23*a34)
if ((r[1]==0) & (r[2]==3) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*(2*a23^2*a25 + a22*a25*a33 + 2*a22*a23*a35)
if ((r[1]==0) & (r[2]==3) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*(2*a23^2*a26 + a22*a26*a33 + 2*a22*a23*a36)
if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a23*a24^2 + 2*a22*a24*a34 + a22*a23*a44)
if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*(2*a23*a24*a25 + a22*a25*a34 + a22*a24*a35 + a22*a23*a45)
if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*(2*a23*a24*a26 + a22*a26*a34 + a22*a24*a36 + a22*a23*a46)
if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
-3*(2*a23*a25^2 + 2*a22*a25*a35 + a22*a23*a55)
if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
-3*(2*a23*a25*a26 + a22*a26*a35 + a22*a25*a36 + a22*a23*a56)
if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
-3*(2*a23*a26^2 + 2*a22*a26*a36 + a22*a23*a66)
if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
-6*a24^3 - 9*a22*a24*a44
if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*(2*a24^2*a25 + a22*a25*a44 + 2*a22*a24*a45)
if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*(2*a24^2*a26 + a22*a26*a44 + 2*a22*a24*a46)
if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
-3*(2*a24*a25^2 + 2*a22*a25*a45 + a22*a24*a55)
if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
-3*(2*a24*a25*a26 + a22*a26*a45 + a22*a25*a46 + a22*a24*a56)
if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
-3*(2*a24*a26^2 + 2*a22*a26*a46 + a22*a24*a66)
if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
-6*a25^3 - 9*a22*a25*a55
if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
-3*(2*a25^2*a26 + a22*a26*a55 + 2*a22*a25*a56)
if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
-3*(2*a25*a26^2 + 2*a22*a26*a56 + a22*a25*a66)
if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
-6*a26^3 - 9*a22*a26*a66
if ((r[1]==0) & (r[2]==2) & (r[3]==4) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*a33*(4*a23^2 + a22*a33)
if ((r[1]==0) & (r[2]==2) & (r[3]==3) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a23*a24*a33 + 2*a23^2*a34 + a22*a33*a34)
if ((r[1]==0) & (r[2]==2) & (r[3]==3) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*(2*a23*a25*a33 + 2*a23^2*a35 + a22*a33*a35)
if ((r[1]==0) & (r[2]==2) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*(2*a23*a26*a33 + 2*a23^2*a36 + a22*a33*a36)
if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
-2*a24^2*a33 - 8*a23*a24*a34 - 2*a22*a34^2 - 2*a23^2*a44 - a22*a33*a44
if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
-2*a24*a25*a33 - 4*a23*a25*a34 - 4*a23*a24*a35 - 2*a22*a34*a35 - 
 2*a23^2*a45 - a22*a33*a45
if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
-2*a24*a26*a33 - 4*a23*a26*a34 - 4*a23*a24*a36 - 2*a22*a34*a36 - 
 2*a23^2*a46 - a22*a33*a46
if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
-2*a25^2*a33 - 8*a23*a25*a35 - 2*a22*a35^2 - 2*a23^2*a55 - a22*a33*a55
if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
-2*a25*a26*a33 - 4*a23*a26*a35 - 4*a23*a25*a36 - 2*a22*a35*a36 - 
 2*a23^2*a56 - a22*a33*a56
if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
-2*a26^2*a33 - 8*a23*a26*a36 - 2*a22*a36^2 - 2*a23^2*a66 - a22*a33*a66
if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a24^2*a34 + 2*a23*a24*a44 + a22*a34*a44)
if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
-4*a24*a25*a34 - 2*a24^2*a35 - 2*a23*a25*a44 - a22*a35*a44 - 4*a23*a24*a45 - 
 2*a22*a34*a45
if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
-4*a24*a26*a34 - 2*a24^2*a36 - 2*a23*a26*a44 - a22*a36*a44 - 4*a23*a24*a46 - 
 2*a22*a34*a46
if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
-2*a25^2*a34 - 4*a24*a25*a35 - 4*a23*a25*a45 - 2*a22*a35*a45 - 
 2*a23*a24*a55 - a22*a34*a55
if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
-2*a25*a26*a34 - 2*a24*a26*a35 - 2*a24*a25*a36 - 2*a23*a26*a45 - 
 a22*a36*a45 - 2*a23*a25*a46 - a22*a35*a46 - 2*a23*a24*a56 - a22*a34*a56
if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
-2*a26^2*a34 - 4*a24*a26*a36 - 4*a23*a26*a46 - 2*a22*a36*a46 - 
 2*a23*a24*a66 - a22*a34*a66
if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
-3*(2*a25^2*a35 + 2*a23*a25*a55 + a22*a35*a55)
if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
-4*a25*a26*a35 - 2*a25^2*a36 - 2*a23*a26*a55 - a22*a36*a55 - 4*a23*a25*a56 - 
 2*a22*a35*a56
if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
-2*a26^2*a35 - 4*a25*a26*a36 - 4*a23*a26*a56 - 2*a22*a36*a56 - 
 2*a23*a25*a66 - a22*a35*a66
if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
-3*(2*a26^2*a36 + 2*a23*a26*a66 + a22*a36*a66)
if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==4) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*a44*(4*a24^2 + a22*a44)
if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==3) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*(2*a24*a25*a44 + 2*a24^2*a45 + a22*a44*a45)
if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*(2*a24*a26*a44 + 2*a24^2*a46 + a22*a44*a46)
if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==2) 
& (r[5]==2) & (r[6]==0))
psi <-
-2*a25^2*a44 - 8*a24*a25*a45 - 2*a22*a45^2 - 2*a24^2*a55 - a22*a44*a55
if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==1))
psi <-
-2*a25*a26*a44 - 4*a24*a26*a45 - 4*a24*a25*a46 - 2*a22*a45*a46 - 
 2*a24^2*a56 - a22*a44*a56
if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==2))
psi <-
-2*a26^2*a44 - 8*a24*a26*a46 - 2*a22*a46^2 - 2*a24^2*a66 - a22*a44*a66
if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==3) & (r[6]==0))
psi <-
-3*(2*a25^2*a45 + 2*a24*a25*a55 + a22*a45*a55)
if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==1))
psi <-
-4*a25*a26*a45 - 2*a25^2*a46 - 2*a24*a26*a55 - a22*a46*a55 - 4*a24*a25*a56 - 
 2*a22*a45*a56
if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==2))
psi <-
-2*a26^2*a45 - 4*a25*a26*a46 - 4*a24*a26*a56 - 2*a22*a46*a56 - 
 2*a24*a25*a66 - a22*a45*a66
if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==3))
psi <-
-3*(2*a26^2*a46 + 2*a24*a26*a66 + a22*a46*a66)
if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==4) & (r[6]==0))
psi <-
-3*a55*(4*a25^2 + a22*a55)
if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==1))
psi <-
-3*(2*a25*a26*a55 + 2*a25^2*a56 + a22*a55*a56)
if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==2))
psi <-
-2*a26^2*a55 - 8*a25*a26*a56 - 2*a22*a56^2 - 2*a25^2*a66 - a22*a55*a66
if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==3))
psi <-
-3*(2*a26^2*a56 + 2*a25*a26*a66 + a22*a56*a66)
if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==4))
psi <-
-3*a66*(4*a26^2 + a22*a66)
if ((r[1]==0) & (r[2]==1) & (r[3]==5) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
-15*a23*a33^2
if ((r[1]==0) & (r[2]==1) & (r[3]==4) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*a33*(a24*a33 + 4*a23*a34)
if ((r[1]==0) & (r[2]==1) & (r[3]==4) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*a33*(a25*a33 + 4*a23*a35)
if ((r[1]==0) & (r[2]==1) & (r[3]==4) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*a33*(a26*a33 + 4*a23*a36)
if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a24*a33*a34 + 2*a23*a34^2 + a23*a33*a44)
if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*(a25*a33*a34 + a24*a33*a35 + 2*a23*a34*a35 + a23*a33*a45)
if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*(a26*a33*a34 + a24*a33*a36 + 2*a23*a34*a36 + a23*a33*a46)
if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
-3*(2*a25*a33*a35 + 2*a23*a35^2 + a23*a33*a55)
if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
-3*(a26*a33*a35 + a25*a33*a36 + 2*a23*a35*a36 + a23*a33*a56)
if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
-3*(2*a26*a33*a36 + 2*a23*a36^2 + a23*a33*a66)
if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*(2*a24*a34^2 + a24*a33*a44 + 2*a23*a34*a44)
if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
-2*a25*a34^2 - 4*a24*a34*a35 - a25*a33*a44 - 2*a23*a35*a44 - 2*a24*a33*a45 - 
 4*a23*a34*a45
if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
-2*a26*a34^2 - 4*a24*a34*a36 - a26*a33*a44 - 2*a23*a36*a44 - 2*a24*a33*a46 - 
 4*a23*a34*a46
if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
-4*a25*a34*a35 - 2*a24*a35^2 - 2*a25*a33*a45 - 4*a23*a35*a45 - a24*a33*a55 - 
 2*a23*a34*a55
if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
-2*a26*a34*a35 - 2*a25*a34*a36 - 2*a24*a35*a36 - a26*a33*a45 - 
 2*a23*a36*a45 - a25*a33*a46 - 2*a23*a35*a46 - a24*a33*a56 - 2*a23*a34*a56
if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
-4*a26*a34*a36 - 2*a24*a36^2 - 2*a26*a33*a46 - 4*a23*a36*a46 - a24*a33*a66 - 
 2*a23*a34*a66
if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
-3*(2*a25*a35^2 + a25*a33*a55 + 2*a23*a35*a55)
if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
-2*a26*a35^2 - 4*a25*a35*a36 - a26*a33*a55 - 2*a23*a36*a55 - 2*a25*a33*a56 - 
 4*a23*a35*a56
if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
-4*a26*a35*a36 - 2*a25*a36^2 - 2*a26*a33*a56 - 4*a23*a36*a56 - a25*a33*a66 - 
 2*a23*a35*a66
if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
-3*(2*a26*a36^2 + a26*a33*a66 + 2*a23*a36*a66)
if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==4) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*a44*(4*a24*a34 + a23*a44)
if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==3) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*(a25*a34*a44 + a24*a35*a44 + 2*a24*a34*a45 + a23*a44*a45)
if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==3) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*(a26*a34*a44 + a24*a36*a44 + 2*a24*a34*a46 + a23*a44*a46)
if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==2) 
& (r[5]==2) & (r[6]==0))
psi <-
-2*a25*a35*a44 - 4*a25*a34*a45 - 4*a24*a35*a45 - 2*a23*a45^2 - 
 2*a24*a34*a55 - a23*a44*a55
if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==2) 
& (r[5]==1) & (r[6]==1))
psi <-
-(a26*a35*a44) - a25*a36*a44 - 2*a26*a34*a45 - 2*a24*a36*a45 - 
 2*a25*a34*a46 - 2*a24*a35*a46 - 2*a23*a45*a46 - 2*a24*a34*a56 - a23*a44*a56
if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==2))
psi <-
-2*a26*a36*a44 - 4*a26*a34*a46 - 4*a24*a36*a46 - 2*a23*a46^2 - 
 2*a24*a34*a66 - a23*a44*a66
if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==3) & (r[6]==0))
psi <-
-3*(2*a25*a35*a45 + a25*a34*a55 + a24*a35*a55 + a23*a45*a55)
if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==2) & (r[6]==1))
psi <-
-2*a26*a35*a45 - 2*a25*a36*a45 - 2*a25*a35*a46 - a26*a34*a55 - a24*a36*a55 - 
 a23*a46*a55 - 2*a25*a34*a56 - 2*a24*a35*a56 - 2*a23*a45*a56
if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==2))
psi <-
-2*a26*a36*a45 - 2*a26*a35*a46 - 2*a25*a36*a46 - 2*a26*a34*a56 - 
 2*a24*a36*a56 - 2*a23*a46*a56 - a25*a34*a66 - a24*a35*a66 - a23*a45*a66
if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==3))
psi <-
-3*(2*a26*a36*a46 + a26*a34*a66 + a24*a36*a66 + a23*a46*a66)
if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==4) & (r[6]==0))
psi <-
-3*a55*(4*a25*a35 + a23*a55)
if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==3) & (r[6]==1))
psi <-
-3*(a26*a35*a55 + a25*a36*a55 + 2*a25*a35*a56 + a23*a55*a56)
if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==2))
psi <-
-2*a26*a36*a55 - 4*a26*a35*a56 - 4*a25*a36*a56 - 2*a23*a56^2 - 
 2*a25*a35*a66 - a23*a55*a66
if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==3))
psi <-
-3*(2*a26*a36*a56 + a26*a35*a66 + a25*a36*a66 + a23*a56*a66)
if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==4))
psi <-
-3*a66*(4*a26*a36 + a23*a66)
if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==5) 
& (r[5]==0) & (r[6]==0))
psi <-
-15*a24*a44^2
if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==4) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*a44*(a25*a44 + 4*a24*a45)
if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==4) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*a44*(a26*a44 + 4*a24*a46)
if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==3) 
& (r[5]==2) & (r[6]==0))
psi <-
-3*(2*a25*a44*a45 + 2*a24*a45^2 + a24*a44*a55)
if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==3) 
& (r[5]==1) & (r[6]==1))
psi <-
-3*(a26*a44*a45 + a25*a44*a46 + 2*a24*a45*a46 + a24*a44*a56)
if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==2))
psi <-
-3*(2*a26*a44*a46 + 2*a24*a46^2 + a24*a44*a66)
if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==3) & (r[6]==0))
psi <-
-3*(2*a25*a45^2 + a25*a44*a55 + 2*a24*a45*a55)
if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==2) & (r[6]==1))
psi <-
-2*a26*a45^2 - 4*a25*a45*a46 - a26*a44*a55 - 2*a24*a46*a55 - 2*a25*a44*a56 - 
 4*a24*a45*a56
if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==2))
psi <-
-4*a26*a45*a46 - 2*a25*a46^2 - 2*a26*a44*a56 - 4*a24*a46*a56 - a25*a44*a66 - 
 2*a24*a45*a66
if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==3))
psi <-
-3*(2*a26*a46^2 + a26*a44*a66 + 2*a24*a46*a66)
if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==4) & (r[6]==0))
psi <-
-3*a55*(4*a25*a45 + a24*a55)
if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==3) & (r[6]==1))
psi <-
-3*(a26*a45*a55 + a25*a46*a55 + 2*a25*a45*a56 + a24*a55*a56)
if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==2))
psi <-
-2*a26*a46*a55 - 4*a26*a45*a56 - 4*a25*a46*a56 - 2*a24*a56^2 - 
 2*a25*a45*a66 - a24*a55*a66
if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==3))
psi <-
-3*(2*a26*a46*a56 + a26*a45*a66 + a25*a46*a66 + a24*a56*a66)
if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==4))
psi <-
-3*a66*(4*a26*a46 + a24*a66)
if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==5) & (r[6]==0))
psi <-
-15*a25*a55^2
if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==4) & (r[6]==1))
psi <-
-3*a55*(a26*a55 + 4*a25*a56)
if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==2))
psi <-
-3*(2*a26*a55*a56 + 2*a25*a56^2 + a25*a55*a66)
if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==3))
psi <-
-3*(2*a26*a56^2 + a26*a55*a66 + 2*a25*a56*a66)
if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==4))
psi <-
-3*a66*(4*a26*a56 + a25*a66)
if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==5))
psi <-
-15*a26*a66^2
if ((r[1]==0) & (r[2]==0) & (r[3]==6) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
-15*a33^3
if ((r[1]==0) & (r[2]==0) & (r[3]==5) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
-15*a33^2*a34
if ((r[1]==0) & (r[2]==0) & (r[3]==5) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
-15*a33^2*a35
if ((r[1]==0) & (r[2]==0) & (r[3]==5) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
-15*a33^2*a36
if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*a33*(4*a34^2 + a33*a44)
if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*a33*(4*a34*a35 + a33*a45)
if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*a33*(4*a34*a36 + a33*a46)
if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
-3*a33*(4*a35^2 + a33*a55)
if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
-3*a33*(4*a35*a36 + a33*a56)
if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
-3*a33*(4*a36^2 + a33*a66)
if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
-6*a34^3 - 9*a33*a34*a44
if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*(2*a34^2*a35 + a33*a35*a44 + 2*a33*a34*a45)
if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*(2*a34^2*a36 + a33*a36*a44 + 2*a33*a34*a46)
if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
-3*(2*a34*a35^2 + 2*a33*a35*a45 + a33*a34*a55)
if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
-3*(2*a34*a35*a36 + a33*a36*a45 + a33*a35*a46 + a33*a34*a56)
if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
-3*(2*a34*a36^2 + 2*a33*a36*a46 + a33*a34*a66)
if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
-6*a35^3 - 9*a33*a35*a55
if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
-3*(2*a35^2*a36 + a33*a36*a55 + 2*a33*a35*a56)
if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
-3*(2*a35*a36^2 + 2*a33*a36*a56 + a33*a35*a66)
if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
-6*a36^3 - 9*a33*a36*a66
if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==4) 
& (r[5]==0) & (r[6]==0))
psi <-
-3*a44*(4*a34^2 + a33*a44)
if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==3) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*(2*a34*a35*a44 + 2*a34^2*a45 + a33*a44*a45)
if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==3) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*(2*a34*a36*a44 + 2*a34^2*a46 + a33*a44*a46)
if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==2) 
& (r[5]==2) & (r[6]==0))
psi <-
-2*a35^2*a44 - 8*a34*a35*a45 - 2*a33*a45^2 - 2*a34^2*a55 - a33*a44*a55
if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==2) 
& (r[5]==1) & (r[6]==1))
psi <-
-2*a35*a36*a44 - 4*a34*a36*a45 - 4*a34*a35*a46 - 2*a33*a45*a46 - 
 2*a34^2*a56 - a33*a44*a56
if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==2) 
& (r[5]==0) & (r[6]==2))
psi <-
-2*a36^2*a44 - 8*a34*a36*a46 - 2*a33*a46^2 - 2*a34^2*a66 - a33*a44*a66
if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==3) & (r[6]==0))
psi <-
-3*(2*a35^2*a45 + 2*a34*a35*a55 + a33*a45*a55)
if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==2) & (r[6]==1))
psi <-
-4*a35*a36*a45 - 2*a35^2*a46 - 2*a34*a36*a55 - a33*a46*a55 - 4*a34*a35*a56 - 
 2*a33*a45*a56
if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==1) & (r[6]==2))
psi <-
-2*a36^2*a45 - 4*a35*a36*a46 - 4*a34*a36*a56 - 2*a33*a46*a56 - 
 2*a34*a35*a66 - a33*a45*a66
if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==3))
psi <-
-3*(2*a36^2*a46 + 2*a34*a36*a66 + a33*a46*a66)
if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==4) & (r[6]==0))
psi <-
-3*a55*(4*a35^2 + a33*a55)
if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==3) & (r[6]==1))
psi <-
-3*(2*a35*a36*a55 + 2*a35^2*a56 + a33*a55*a56)
if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==2) & (r[6]==2))
psi <-
-2*a36^2*a55 - 8*a35*a36*a56 - 2*a33*a56^2 - 2*a35^2*a66 - a33*a55*a66
if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==3))
psi <-
-3*(2*a36^2*a56 + 2*a35*a36*a66 + a33*a56*a66)
if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==4))
psi <-
-3*a66*(4*a36^2 + a33*a66)
if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==5) 
& (r[5]==0) & (r[6]==0))
psi <-
-15*a34*a44^2
if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==4) 
& (r[5]==1) & (r[6]==0))
psi <-
-3*a44*(a35*a44 + 4*a34*a45)
if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==4) 
& (r[5]==0) & (r[6]==1))
psi <-
-3*a44*(a36*a44 + 4*a34*a46)
if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==3) 
& (r[5]==2) & (r[6]==0))
psi <-
-3*(2*a35*a44*a45 + 2*a34*a45^2 + a34*a44*a55)
if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==3) 
& (r[5]==1) & (r[6]==1))
psi <-
-3*(a36*a44*a45 + a35*a44*a46 + 2*a34*a45*a46 + a34*a44*a56)
if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==3) 
& (r[5]==0) & (r[6]==2))
psi <-
-3*(2*a36*a44*a46 + 2*a34*a46^2 + a34*a44*a66)
if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==3) & (r[6]==0))
psi <-
-3*(2*a35*a45^2 + a35*a44*a55 + 2*a34*a45*a55)
if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==2) & (r[6]==1))
psi <-
-2*a36*a45^2 - 4*a35*a45*a46 - a36*a44*a55 - 2*a34*a46*a55 - 2*a35*a44*a56 - 
 4*a34*a45*a56
if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==1) & (r[6]==2))
psi <-
-4*a36*a45*a46 - 2*a35*a46^2 - 2*a36*a44*a56 - 4*a34*a46*a56 - a35*a44*a66 - 
 2*a34*a45*a66
if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==3))
psi <-
-3*(2*a36*a46^2 + a36*a44*a66 + 2*a34*a46*a66)
if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==4) & (r[6]==0))
psi <-
-3*a55*(4*a35*a45 + a34*a55)
if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==3) & (r[6]==1))
psi <-
-3*(a36*a45*a55 + a35*a46*a55 + 2*a35*a45*a56 + a34*a55*a56)
if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==2) & (r[6]==2))
psi <-
-2*a36*a46*a55 - 4*a36*a45*a56 - 4*a35*a46*a56 - 2*a34*a56^2 - 
 2*a35*a45*a66 - a34*a55*a66
if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==3))
psi <-
-3*(2*a36*a46*a56 + a36*a45*a66 + a35*a46*a66 + a34*a56*a66)
if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==4))
psi <-
-3*a66*(4*a36*a46 + a34*a66)
if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==5) & (r[6]==0))
psi <-
-15*a35*a55^2
if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==4) & (r[6]==1))
psi <-
-3*a55*(a36*a55 + 4*a35*a56)
  if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==3) & (r[6]==2))
psi <-
-3*(2*a36*a55*a56 + 2*a35*a56^2 + a35*a55*a66)
if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==3))
psi <-
-3*(2*a36*a56^2 + a36*a55*a66 + 2*a35*a56*a66)
if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==4))
psi <-
-3*a66*(4*a36*a56 + a35*a66)
if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==5))
psi <-
-15*a36*a66^2
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==6) 
& (r[5]==0) & (r[6]==0))
psi <-
-15*a44^3
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==5) 
& (r[5]==1) & (r[6]==0))
psi <-
-15*a44^2*a45
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==5) 
& (r[5]==0) & (r[6]==1))
psi <-
-15*a44^2*a46
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==4) 
& (r[5]==2) & (r[6]==0))
psi <-
-3*a44*(4*a45^2 + a44*a55)
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==4) 
& (r[5]==1) & (r[6]==1))
psi <-
-3*a44*(4*a45*a46 + a44*a56)
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==4) 
& (r[5]==0) & (r[6]==2))
psi <-
-3*a44*(4*a46^2 + a44*a66)
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==3) & (r[6]==0))
psi <-
-6*a45^3 - 9*a44*a45*a55
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==2) & (r[6]==1))
psi <-
-3*(2*a45^2*a46 + a44*a46*a55 + 2*a44*a45*a56)
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==1) & (r[6]==2))
psi <-
-3*(2*a45*a46^2 + 2*a44*a46*a56 + a44*a45*a66)
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==3))
psi <-
-6*a46^3 - 9*a44*a46*a66
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==4) & (r[6]==0))
psi <-
-3*a55*(4*a45^2 + a44*a55)
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==3) & (r[6]==1))
psi <-
-3*(2*a45*a46*a55 + 2*a45^2*a56 + a44*a55*a56)
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==2) & (r[6]==2))
psi <-
-2*a46^2*a55 - 8*a45*a46*a56 - 2*a44*a56^2 - 2*a45^2*a66 - a44*a55*a66
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==3))
psi <-
-3*(2*a46^2*a56 + 2*a45*a46*a66 + a44*a56*a66)
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==4))
psi <-
-3*a66*(4*a46^2 + a44*a66)
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==5) & (r[6]==0))
psi <-
-15*a45*a55^2
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==4) & (r[6]==1))
psi <-
-3*a55*(a46*a55 + 4*a45*a56)
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==3) & (r[6]==2))
psi <-
-3*(2*a46*a55*a56 + 2*a45*a56^2 + a45*a55*a66)
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==1) & (r[5]==2) & (r[6]==3))
psi <-
-3*(2*a46*a56^2 + a46*a55*a66 + 2*a45*a56*a66)
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==1) & (r[5]==1) & (r[6]==4))
psi <-
-3*a66*(4*a46*a56 + a45*a66)
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==1) & (r[5]==0) & (r[6]==5))
psi <-
-15*a46*a66^2
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==6) & (r[6]==0))
psi <-
-15*a55^3
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==5) & (r[6]==1))
psi <-
-15*a55^2*a56
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==4) & (r[6]==2))
psi <-
-3*a55*(4*a56^2 + a55*a66)
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==3) & (r[6]==3))
psi <-
-6*a56^3 - 9*a55*a56*a66
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==2) & (r[6]==4))
psi <-
-3*a66*(4*a56^2 + a55*a66)
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==1) & (r[6]==5))
psi <-
-15*a56*a66^2
if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==0) & (r[6]==6))
psi <-
-15*a66^3

return(psi)
}






psins.6d.part81 <- function(r, a11, a22, a33, a44, a55, a66, a12, a13, a14, a15,
                           a16, a23, a24, a25, a26, a34, a35, a36, a45, a46, a56)
{
psi <- NA
if ((r[1]==8) & (r[2]==0) & (r[3]==0) & (r[4]==0)  & (r[5]==0) & (r[6]==0))
    psi <- 105*a11^4 
if ((r[1]==7) & (r[2]==1) & (r[3]==0) & (r[4]==0) & (r[5]==0) & (r[6]==0))
  psi <- 105*a11^3*a12
 if ((r[1]==7) & (r[2]==0) & (r[3]==1) & (r[4]==0) & (r[5]==0) & (r[6]==0))
psi <-
105*a11^3*a13
 if ((r[1]==7) & (r[2]==0) & (r[3]==0) & (r[4]==1) & (r[5]==0) & (r[6]==0))
psi <-
105*a11^3*a14
 if ((r[1]==7) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==1) & (r[6]==0))
psi <-105*a11^3*a15
 if ((r[1]==7) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==0) & (r[6]==1))
psi <-
105*a11^3*a16
 if ((r[1]==6) & (r[2]==2) & (r[3]==0) & (r[4]==0) & (r[5]==0) & (r[6]==0))
psi <-
15*a11^2*(6*a12^2 + a11*a22)
 if ((r[1]==6) & (r[2]==1) & (r[3]==1) & (r[4]==0) & (r[5]==0) & (r[6]==0))
psi <-
15*a11^2*(6*a12*a13 + a11*a23)
 if ((r[1]==6) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a11^2*(6*a12*a14 + a11*a24)
 if ((r[1]==6) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a11^2*(6*a12*a15 + a11*a25)
 if ((r[1]==6) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a11^2*(6*a12*a16 + a11*a26)
 if ((r[1]==6) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a11^2*(6*a13^2 + a11*a33)
 if ((r[1]==6) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a11^2*(6*a13*a14 + a11*a34)
 if ((r[1]==6) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a11^2*(6*a13*a15 + a11*a35)
 if ((r[1]==6) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a11^2*(6*a13*a16 + a11*a36)
 if ((r[1]==6) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a11^2*(6*a14^2 + a11*a44)
 if ((r[1]==6) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a11^2*(6*a14*a15 + a11*a45)
 if ((r[1]==6) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a11^2*(6*a14*a16 + a11*a46)
 if ((r[1]==6) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
15*a11^2*(6*a15^2 + a11*a55)
 if ((r[1]==6) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
15*a11^2*(6*a15*a16 + a11*a56)
 if ((r[1]==6) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
15*a11^2*(6*a16^2 + a11*a66)
 if ((r[1]==5) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a11*a12*(4*a12^2 + 3*a11*a22)
 if ((r[1]==5) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a11*(4*a12^2*a13 + a11*a13*a22 + 2*a11*a12*a23)
 if ((r[1]==5) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a11*(4*a12^2*a14 + a11*a14*a22 + 2*a11*a12*a24)
 if ((r[1]==5) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a11*(4*a12^2*a15 + a11*a15*a22 + 2*a11*a12*a25)
 if ((r[1]==5) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a11*(4*a12^2*a16 + a11*a16*a22 + 2*a11*a12*a26)
 if ((r[1]==5) & (r[2]==1) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a11*(4*a12*a13^2 + 2*a11*a13*a23 + a11*a12*a33)
 if ((r[1]==5) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a11*(4*a12*a13*a14 + a11*a14*a23 + a11*a13*a24 + a11*a12*a34)
 if ((r[1]==5) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a11*(4*a12*a13*a15 + a11*a15*a23 + a11*a13*a25 + a11*a12*a35)
 if ((r[1]==5) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a11*(4*a12*a13*a16 + a11*a16*a23 + a11*a13*a26 + a11*a12*a36)
 if ((r[1]==5) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a11*(4*a12*a14^2 + 2*a11*a14*a24 + a11*a12*a44)
 if ((r[1]==5) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a11*(4*a12*a14*a15 + a11*a15*a24 + a11*a14*a25 + a11*a12*a45)
 if ((r[1]==5) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a11*(4*a12*a14*a16 + a11*a16*a24 + a11*a14*a26 + a11*a12*a46)
 if ((r[1]==5) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
15*a11*(4*a12*a15^2 + 2*a11*a15*a25 + a11*a12*a55)
 if ((r[1]==5) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
15*a11*(4*a12*a15*a16 + a11*a16*a25 + a11*a15*a26 + a11*a12*a56)
 if ((r[1]==5) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
15*a11*(4*a12*a16^2 + 2*a11*a16*a26 + a11*a12*a66)
 if ((r[1]==5) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a11*a13*(4*a13^2 + 3*a11*a33)
 if ((r[1]==5) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a11*(4*a13^2*a14 + a11*a14*a33 + 2*a11*a13*a34)
 if ((r[1]==5) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a11*(4*a13^2*a15 + a11*a15*a33 + 2*a11*a13*a35)
 if ((r[1]==5) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a11*(4*a13^2*a16 + a11*a16*a33 + 2*a11*a13*a36)
 if ((r[1]==5) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a11*(4*a13*a14^2 + 2*a11*a14*a34 + a11*a13*a44)
 if ((r[1]==5) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a11*(4*a13*a14*a15 + a11*a15*a34 + a11*a14*a35 + a11*a13*a45)
 if ((r[1]==5) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a11*(4*a13*a14*a16 + a11*a16*a34 + a11*a14*a36 + a11*a13*a46)
 if ((r[1]==5) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
15*a11*(4*a13*a15^2 + 2*a11*a15*a35 + a11*a13*a55)
 if ((r[1]==5) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
15*a11*(4*a13*a15*a16 + a11*a16*a35 + a11*a15*a36 + a11*a13*a56)
if ((r[1]==5) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
15*a11*(4*a13*a16^2 + 2*a11*a16*a36 + a11*a13*a66)
 if ((r[1]==5) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a11*a14*(4*a14^2 + 3*a11*a44)
 if ((r[1]==5) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a11*(4*a14^2*a15 + a11*a15*a44 + 2*a11*a14*a45)
 if ((r[1]==5) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a11*(4*a14^2*a16 + a11*a16*a44 + 2*a11*a14*a46)
 if ((r[1]==5) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
15*a11*(4*a14*a15^2 + 2*a11*a15*a45 + a11*a14*a55)
 if ((r[1]==5) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
15*a11*(4*a14*a15*a16 + a11*a16*a45 + a11*a15*a46 + a11*a14*a56)
 if ((r[1]==5) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
15*a11*(4*a14*a16^2 + 2*a11*a16*a46 + a11*a14*a66)
 if ((r[1]==5) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
15*a11*a15*(4*a15^2 + 3*a11*a55)
 if ((r[1]==5) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
15*a11*(4*a15^2*a16 + a11*a16*a55 + 2*a11*a15*a56)
 if ((r[1]==5) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
15*a11*(4*a15*a16^2 + 2*a11*a16*a56 + a11*a15*a66)
 if ((r[1]==5) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
15*a11*a16*(4*a16^2 + 3*a11*a66)
 if ((r[1]==4) & (r[2]==4) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
24*a12^4 + 72*a11*a12^2*a22 + 9*a11^2*a22^2
 if ((r[1]==4) & (r[2]==3) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
24*a12^3*a13 + 36*a11*a12*a13*a22 + 36*a11*a12^2*a23 + 9*a11^2*a22*a23
 if ((r[1]==4) & (r[2]==3) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
24*a12^3*a14 + 36*a11*a12*a14*a22 + 36*a11*a12^2*a24 + 9*a11^2*a22*a24
 if ((r[1]==4) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
24*a12^3*a15 + 36*a11*a12*a15*a22 + 36*a11*a12^2*a25 + 9*a11^2*a22*a25
 if ((r[1]==4) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
24*a12^3*a16 + 36*a11*a12*a16*a22 + 36*a11*a12^2*a26 + 9*a11^2*a22*a26
 if ((r[1]==4) & (r[2]==2) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(16*a11*a12*a13*a23 + 4*a12^2*(2*a13^2 + a11*a33) + 
  a11*(4*a13^2*a22 + 2*a11*a23^2 + a11*a22*a33))
 if ((r[1]==4) & (r[2]==2) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(8*a11*a12*(a14*a23 + a13*a24) + 4*a12^2*(2*a13*a14 + a11*a34) + 
  a11*(4*a13*a14*a22 + 2*a11*a23*a24 + a11*a22*a34))
 if ((r[1]==4) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(8*a11*a12*(a15*a23 + a13*a25) + 4*a12^2*(2*a13*a15 + a11*a35) + 
  a11*(4*a13*a15*a22 + 2*a11*a23*a25 + a11*a22*a35))
 if ((r[1]==4) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(8*a11*a12*(a16*a23 + a13*a26) + 4*a12^2*(2*a13*a16 + a11*a36) + 
  a11*(4*a13*a16*a22 + 2*a11*a23*a26 + a11*a22*a36))
 if ((r[1]==4) & (r[2]==2) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(16*a11*a12*a14*a24 + 4*a12^2*(2*a14^2 + a11*a44) + 
  a11*(4*a14^2*a22 + 2*a11*a24^2 + a11*a22*a44))
 if ((r[1]==4) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(8*a11*a12*(a15*a24 + a14*a25) + 4*a12^2*(2*a14*a15 + a11*a45) + 
  a11*(4*a14*a15*a22 + 2*a11*a24*a25 + a11*a22*a45))
 if ((r[1]==4) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(8*a11*a12*(a16*a24 + a14*a26) + 4*a12^2*(2*a14*a16 + a11*a46) + 
  a11*(4*a14*a16*a22 + 2*a11*a24*a26 + a11*a22*a46))
 if ((r[1]==4) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(16*a11*a12*a15*a25 + 4*a12^2*(2*a15^2 + a11*a55) + 
  a11*(4*a15^2*a22 + 2*a11*a25^2 + a11*a22*a55))
 if ((r[1]==4) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(8*a11*a12*(a16*a25 + a15*a26) + 4*a12^2*(2*a15*a16 + a11*a56) + 
  a11*(4*a15*a16*a22 + 2*a11*a25*a26 + a11*a22*a56))
 if ((r[1]==4) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(16*a11*a12*a16*a26 + 4*a12^2*(2*a16^2 + a11*a66) + 
  a11*(4*a16^2*a22 + 2*a11*a26^2 + a11*a22*a66))
 if ((r[1]==4) & (r[2]==1) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
9*a11*a23*(4*a13^2 + a11*a33) + 12*a12*(2*a13^3 + 3*a11*a13*a33)
 if ((r[1]==4) & (r[2]==1) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(4*a12*(2*a13^2*a14 + a11*a14*a33 + 2*a11*a13*a34) + 
  a11*(8*a13*a14*a23 + 4*a13^2*a24 + a11*a24*a33 + 2*a11*a23*a34))
 if ((r[1]==4) & (r[2]==1) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(4*a12*(2*a13^2*a15 + a11*a15*a33 + 2*a11*a13*a35) + 
  a11*(8*a13*a15*a23 + 4*a13^2*a25 + a11*a25*a33 + 2*a11*a23*a35))
 if ((r[1]==4) & (r[2]==1) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(4*a12*(2*a13^2*a16 + a11*a16*a33 + 2*a11*a13*a36) + 
  a11*(8*a13*a16*a23 + 4*a13^2*a26 + a11*a26*a33 + 2*a11*a23*a36))
 if ((r[1]==4) & (r[2]==1) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(4*a12*(2*a13*a14^2 + 2*a11*a14*a34 + a11*a13*a44) + 
  a11*(4*a14^2*a23 + 8*a13*a14*a24 + 2*a11*a24*a34 + a11*a23*a44))
 if ((r[1]==4) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(4*a12*(2*a13*a14*a15 + a11*a15*a34 + a11*a14*a35 + a11*a13*a45) + 
  a11*(4*a13*a15*a24 + 4*a14*(a15*a23 + a13*a25) + a11*a25*a34 + 
    a11*a24*a35 + a11*a23*a45))
 if ((r[1]==4) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(4*a12*(2*a13*a14*a16 + a11*a16*a34 + a11*a14*a36 + a11*a13*a46) + 
  a11*(4*a13*a16*a24 + 4*a14*(a16*a23 + a13*a26) + a11*a26*a34 + 
    a11*a24*a36 + a11*a23*a46))
 if ((r[1]==4) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(4*a12*(2*a13*a15^2 + 2*a11*a15*a35 + a11*a13*a55) + 
  a11*(4*a15^2*a23 + 8*a13*a15*a25 + 2*a11*a25*a35 + a11*a23*a55))
 if ((r[1]==4) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(4*a12*(2*a13*a15*a16 + a11*a16*a35 + a11*a15*a36 + a11*a13*a56) + 
  a11*(4*a13*a16*a25 + 4*a15*(a16*a23 + a13*a26) + a11*a26*a35 + 
    a11*a25*a36 + a11*a23*a56))
 if ((r[1]==4) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(4*a12*(2*a13*a16^2 + 2*a11*a16*a36 + a11*a13*a66) + 
  a11*(4*a16^2*a23 + 8*a13*a16*a26 + 2*a11*a26*a36 + a11*a23*a66))
  if ((r[1]==4) & (r[2]==1) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
9*a11*a24*(4*a14^2 + a11*a44) + 12*a12*(2*a14^3 + 3*a11*a14*a44)
 if ((r[1]==4) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(4*a12*(2*a14^2*a15 + a11*a15*a44 + 2*a11*a14*a45) + 
  a11*(8*a14*a15*a24 + 4*a14^2*a25 + a11*a25*a44 + 2*a11*a24*a45))
 if ((r[1]==4) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(4*a12*(2*a14^2*a16 + a11*a16*a44 + 2*a11*a14*a46) + 
  a11*(8*a14*a16*a24 + 4*a14^2*a26 + a11*a26*a44 + 2*a11*a24*a46))
 if ((r[1]==4) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(4*a12*(2*a14*a15^2 + 2*a11*a15*a45 + a11*a14*a55) + 
  a11*(4*a15^2*a24 + 8*a14*a15*a25 + 2*a11*a25*a45 + a11*a24*a55))
 if ((r[1]==4) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(4*a12*(2*a14*a15*a16 + a11*a16*a45 + a11*a15*a46 + a11*a14*a56) + 
  a11*(4*a14*a16*a25 + 4*a15*(a16*a24 + a14*a26) + a11*a26*a45 + 
    a11*a25*a46 + a11*a24*a56))
 if ((r[1]==4) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(4*a12*(2*a14*a16^2 + 2*a11*a16*a46 + a11*a14*a66) + 
  a11*(4*a16^2*a24 + 8*a14*a16*a26 + 2*a11*a26*a46 + a11*a24*a66))
 if ((r[1]==4) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
9*a11*a25*(4*a15^2 + a11*a55) + 12*a12*(2*a15^3 + 3*a11*a15*a55)
 if ((r[1]==4) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(4*a12*(2*a15^2*a16 + a11*a16*a55 + 2*a11*a15*a56) + 
  a11*(8*a15*a16*a25 + 4*a15^2*a26 + a11*a26*a55 + 2*a11*a25*a56))
 if ((r[1]==4) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(4*a12*(2*a15*a16^2 + 2*a11*a16*a56 + a11*a15*a66) + 
  a11*(4*a16^2*a25 + 8*a15*a16*a26 + 2*a11*a26*a56 + a11*a25*a66))
 if ((r[1]==4) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
9*a11*a26*(4*a16^2 + a11*a66) + 12*a12*(2*a16^3 + 3*a11*a16*a66)
 if ((r[1]==4) & (r[2]==0) & (r[3]==4) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
24*a13^4 + 72*a11*a13^2*a33 + 9*a11^2*a33^2
 if ((r[1]==4) & (r[2]==0) & (r[3]==3) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
24*a13^3*a14 + 36*a11*a13*a14*a33 + 36*a11*a13^2*a34 + 9*a11^2*a33*a34
 if ((r[1]==4) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
24*a13^3*a15 + 36*a11*a13*a15*a33 + 36*a11*a13^2*a35 + 9*a11^2*a33*a35
 if ((r[1]==4) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
24*a13^3*a16 + 36*a11*a13*a16*a33 + 36*a11*a13^2*a36 + 9*a11^2*a33*a36
 if ((r[1]==4) & (r[2]==0) & (r[3]==2) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(16*a11*a13*a14*a34 + 4*a13^2*(2*a14^2 + a11*a44) + 
  a11*(4*a14^2*a33 + 2*a11*a34^2 + a11*a33*a44))
 if ((r[1]==4) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(8*a11*a13*(a15*a34 + a14*a35) + 4*a13^2*(2*a14*a15 + a11*a45) + 
  a11*(4*a14*a15*a33 + 2*a11*a34*a35 + a11*a33*a45))
 if ((r[1]==4) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(8*a11*a13*(a16*a34 + a14*a36) + 4*a13^2*(2*a14*a16 + a11*a46) + 
  a11*(4*a14*a16*a33 + 2*a11*a34*a36 + a11*a33*a46))
 if ((r[1]==4) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(16*a11*a13*a15*a35 + 4*a13^2*(2*a15^2 + a11*a55) + 
  a11*(4*a15^2*a33 + 2*a11*a35^2 + a11*a33*a55))
 if ((r[1]==4) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(8*a11*a13*(a16*a35 + a15*a36) + 4*a13^2*(2*a15*a16 + a11*a56) + 
  a11*(4*a15*a16*a33 + 2*a11*a35*a36 + a11*a33*a56))
 if ((r[1]==4) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(16*a11*a13*a16*a36 + 4*a13^2*(2*a16^2 + a11*a66) + 
  a11*(4*a16^2*a33 + 2*a11*a36^2 + a11*a33*a66))
 if ((r[1]==4) & (r[2]==0) & (r[3]==1) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
9*a11*a34*(4*a14^2 + a11*a44) + 12*a13*(2*a14^3 + 3*a11*a14*a44)
 if ((r[1]==4) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(4*a13*(2*a14^2*a15 + a11*a15*a44 + 2*a11*a14*a45) + 
  a11*(8*a14*a15*a34 + 4*a14^2*a35 + a11*a35*a44 + 2*a11*a34*a45))
 if ((r[1]==4) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(4*a13*(2*a14^2*a16 + a11*a16*a44 + 2*a11*a14*a46) + 
  a11*(8*a14*a16*a34 + 4*a14^2*a36 + a11*a36*a44 + 2*a11*a34*a46))
 if ((r[1]==4) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(4*a13*(2*a14*a15^2 + 2*a11*a15*a45 + a11*a14*a55) + 
  a11*(4*a15^2*a34 + 8*a14*a15*a35 + 2*a11*a35*a45 + a11*a34*a55))
 if ((r[1]==4) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(4*a13*(2*a14*a15*a16 + a11*a16*a45 + a11*a15*a46 + a11*a14*a56) + 
  a11*(4*a14*a16*a35 + 4*a15*(a16*a34 + a14*a36) + a11*a36*a45 + 
    a11*a35*a46 + a11*a34*a56))
 if ((r[1]==4) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(4*a13*(2*a14*a16^2 + 2*a11*a16*a46 + a11*a14*a66) + 
  a11*(4*a16^2*a34 + 8*a14*a16*a36 + 2*a11*a36*a46 + a11*a34*a66))
 if ((r[1]==4) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
9*a11*a35*(4*a15^2 + a11*a55) + 12*a13*(2*a15^3 + 3*a11*a15*a55)
 if ((r[1]==4) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(4*a13*(2*a15^2*a16 + a11*a16*a55 + 2*a11*a15*a56) + 
  a11*(8*a15*a16*a35 + 4*a15^2*a36 + a11*a36*a55 + 2*a11*a35*a56))
 if ((r[1]==4) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(4*a13*(2*a15*a16^2 + 2*a11*a16*a56 + a11*a15*a66) + 
  a11*(4*a16^2*a35 + 8*a15*a16*a36 + 2*a11*a36*a56 + a11*a35*a66))
 if ((r[1]==4) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
9*a11*a36*(4*a16^2 + a11*a66) + 12*a13*(2*a16^3 + 3*a11*a16*a66)
 if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==4) 
& (r[5]==0) & (r[6]==0))
psi <-
24*a14^4 + 72*a11*a14^2*a44 + 9*a11^2*a44^2
 if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==1) & (r[6]==0))
psi <-
24*a14^3*a15 + 36*a11*a14*a15*a44 + 36*a11*a14^2*a45 + 9*a11^2*a44*a45
 if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==1))
psi <-
24*a14^3*a16 + 36*a11*a14*a16*a44 + 36*a11*a14^2*a46 + 9*a11^2*a44*a46
 if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(16*a11*a14*a15*a45 + 4*a14^2*(2*a15^2 + a11*a55) + 
  a11*(4*a15^2*a44 + 2*a11*a45^2 + a11*a44*a55))
 if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(8*a11*a14*(a16*a45 + a15*a46) + 4*a14^2*(2*a15*a16 + a11*a56) + 
  a11*(4*a15*a16*a44 + 2*a11*a45*a46 + a11*a44*a56))
  if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(16*a11*a14*a16*a46 + 4*a14^2*(2*a16^2 + a11*a66) + 
  a11*(4*a16^2*a44 + 2*a11*a46^2 + a11*a44*a66))
 if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==3) & (r[6]==0))
psi <-
9*a11*a45*(4*a15^2 + a11*a55) + 12*a14*(2*a15^3 + 3*a11*a15*a55)
 if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(4*a14*(2*a15^2*a16 + a11*a16*a55 + 2*a11*a15*a56) + 
  a11*(8*a15*a16*a45 + 4*a15^2*a46 + a11*a46*a55 + 2*a11*a45*a56))
 if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(4*a14*(2*a15*a16^2 + 2*a11*a16*a56 + a11*a15*a66) + 
  a11*(4*a16^2*a45 + 8*a15*a16*a46 + 2*a11*a46*a56 + a11*a45*a66))
 if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==3))
psi <-
9*a11*a46*(4*a16^2 + a11*a66) + 12*a14*(2*a16^3 + 3*a11*a16*a66)
 if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==4) & (r[6]==0))
psi <-
24*a15^4 + 72*a11*a15^2*a55 + 9*a11^2*a55^2
 if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==1))
psi <-
24*a15^3*a16 + 36*a11*a15*a16*a55 + 36*a11*a15^2*a56 + 9*a11^2*a55*a56
 if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==2))
psi <-
3*(16*a11*a15*a16*a56 + 4*a15^2*(2*a16^2 + a11*a66) + 
  a11*(4*a16^2*a55 + 2*a11*a56^2 + a11*a55*a66))
 if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==3))
psi <-
9*a11*a56*(4*a16^2 + a11*a66) + 12*a15*(2*a16^3 + 3*a11*a16*a66)
 if ((r[1]==4) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==4))
psi <-
24*a16^4 + 72*a11*a16^2*a66 + 9*a11^2*a66^2
 if ((r[1]==3) & (r[2]==5) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a12*a22*(4*a12^2 + 3*a11*a22)
 if ((r[1]==3) & (r[2]==4) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
36*a12^2*a13*a22 + 9*a11*a13*a22^2 + 24*a12^3*a23 + 36*a11*a12*a22*a23
 if ((r[1]==3) & (r[2]==4) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
36*a12^2*a14*a22 + 9*a11*a14*a22^2 + 24*a12^3*a24 + 36*a11*a12*a22*a24
 if ((r[1]==3) & (r[2]==4) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
36*a12^2*a15*a22 + 9*a11*a15*a22^2 + 24*a12^3*a25 + 36*a11*a12*a22*a25
 if ((r[1]==3) & (r[2]==4) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
36*a12^2*a16*a22 + 9*a11*a16*a22^2 + 24*a12^3*a26 + 36*a11*a12*a22*a26
 if ((r[1]==3) & (r[2]==3) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(12*a12^2*a13*a23 + 6*a11*a13*a22*a23 + 2*a12^3*a33 + 
  3*a12*(2*a13^2*a22 + 2*a11*a23^2 + a11*a22*a33))
 if ((r[1]==3) & (r[2]==3) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(6*a12^2*(a14*a23 + a13*a24) + 3*a11*a22*(a14*a23 + a13*a24) + 
  2*a12^3*a34 + 3*a12*(2*a13*a14*a22 + 2*a11*a23*a24 + a11*a22*a34))
 if ((r[1]==3) & (r[2]==3) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(6*a12^2*(a15*a23 + a13*a25) + 3*a11*a22*(a15*a23 + a13*a25) + 
  2*a12^3*a35 + 3*a12*(2*a13*a15*a22 + 2*a11*a23*a25 + a11*a22*a35))
 if ((r[1]==3) & (r[2]==3) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(6*a12^2*(a16*a23 + a13*a26) + 3*a11*a22*(a16*a23 + a13*a26) + 
  2*a12^3*a36 + 3*a12*(2*a13*a16*a22 + 2*a11*a23*a26 + a11*a22*a36))
 if ((r[1]==3) & (r[2]==3) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(12*a12^2*a14*a24 + 6*a11*a14*a22*a24 + 2*a12^3*a44 + 
  3*a12*(2*a14^2*a22 + 2*a11*a24^2 + a11*a22*a44))
 if ((r[1]==3) & (r[2]==3) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(6*a12^2*(a15*a24 + a14*a25) + 3*a11*a22*(a15*a24 + a14*a25) + 
  2*a12^3*a45 + 3*a12*(2*a14*a15*a22 + 2*a11*a24*a25 + a11*a22*a45))
 if ((r[1]==3) & (r[2]==3) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(6*a12^2*(a16*a24 + a14*a26) + 3*a11*a22*(a16*a24 + a14*a26) + 
  2*a12^3*a46 + 3*a12*(2*a14*a16*a22 + 2*a11*a24*a26 + a11*a22*a46))
 if ((r[1]==3) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(12*a12^2*a15*a25 + 6*a11*a15*a22*a25 + 2*a12^3*a55 + 
  3*a12*(2*a15^2*a22 + 2*a11*a25^2 + a11*a22*a55))
 if ((r[1]==3) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(6*a12^2*(a16*a25 + a15*a26) + 3*a11*a22*(a16*a25 + a15*a26) + 
  2*a12^3*a56 + 3*a12*(2*a15*a16*a22 + 2*a11*a25*a26 + a11*a22*a56))
 if ((r[1]==3) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(12*a12^2*a16*a26 + 6*a11*a16*a22*a26 + 2*a12^3*a66 + 
  3*a12*(2*a16^2*a22 + 2*a11*a26^2 + a11*a22*a66))
 if ((r[1]==3) & (r[2]==2) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a13^3*a22 + 12*a12*a13^2*a23 + 6*a11*a12*a23*a33 + 
  3*a13*(2*a11*a23^2 + 2*a12^2*a33 + a11*a22*a33))
 if ((r[1]==3) & (r[2]==2) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a13^2*(a14*a22 + 2*a12*a24) + 2*a12^2*a14*a33 + 
  2*a13*(4*a12*a14*a23 + 2*a11*a23*a24 + 2*a12^2*a34 + a11*a22*a34) + 
  a11*(2*a14*a23^2 + a14*a22*a33 + 2*a12*a24*a33 + 4*a12*a23*a34))
 if ((r[1]==3) & (r[2]==2) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a13^2*(a15*a22 + 2*a12*a25) + 2*a12^2*a15*a33 + 
  2*a13*(4*a12*a15*a23 + 2*a11*a23*a25 + 2*a12^2*a35 + a11*a22*a35) + 
  a11*(2*a15*a23^2 + a15*a22*a33 + 2*a12*a25*a33 + 4*a12*a23*a35))
 if ((r[1]==3) & (r[2]==2) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a13^2*(a16*a22 + 2*a12*a26) + 2*a12^2*a16*a33 + 
  2*a13*(4*a12*a16*a23 + 2*a11*a23*a26 + 2*a12^2*a36 + a11*a22*a36) + 
  a11*(2*a16*a23^2 + a16*a22*a33 + 2*a12*a26*a33 + 4*a12*a23*a36))
 if ((r[1]==3) & (r[2]==2) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(a13*(2*a14^2*a22 + 8*a12*a14*a24 + 2*a11*a24^2 + 2*a12^2*a44 + 
    a11*a22*a44) + 2*(2*a12^2*a14*a34 + a11*a14*(2*a23*a24 + a22*a34) + 
    a12*(2*a14^2*a23 + 2*a11*a24*a34 + a11*a23*a44)))
 if ((r[1]==3) & (r[2]==2) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a12^2*(a15*a34 + a14*a35) + a11*(2*a15*a23*a24 + 2*a14*a23*a25 + 
    a15*a22*a34 + a14*a22*a35) + a13*(2*a14*a15*a22 + 4*a12*a15*a24 + 
    4*a12*a14*a25 + 2*a11*a24*a25 + 2*a12^2*a45 + a11*a22*a45) + 
  2*a12*(2*a14*a15*a23 + a11*a25*a34 + a11*a24*a35 + a11*a23*a45))
 if ((r[1]==3) & (r[2]==2) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a12^2*(a16*a34 + a14*a36) + a11*(2*a16*a23*a24 + 2*a14*a23*a26 + 
    a16*a22*a34 + a14*a22*a36) + a13*(2*a14*a16*a22 + 4*a12*a16*a24 + 
    4*a12*a14*a26 + 2*a11*a24*a26 + 2*a12^2*a46 + a11*a22*a46) + 
  2*a12*(2*a14*a16*a23 + a11*a26*a34 + a11*a24*a36 + a11*a23*a46))
 if ((r[1]==3) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(a13*(2*a15^2*a22 + 8*a12*a15*a25 + 2*a11*a25^2 + 2*a12^2*a55 + 
    a11*a22*a55) + 2*(2*a12^2*a15*a35 + a11*a15*(2*a23*a25 + a22*a35) + 
    a12*(2*a15^2*a23 + 2*a11*a25*a35 + a11*a23*a55)))
 if ((r[1]==3) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(2*a12^2*(a16*a35 + a15*a36) + a11*(2*a16*a23*a25 + 2*a15*a23*a26 + 
    a16*a22*a35 + a15*a22*a36) + a13*(2*a15*a16*a22 + 4*a12*a16*a25 + 
    4*a12*a15*a26 + 2*a11*a25*a26 + 2*a12^2*a56 + a11*a22*a56) + 
  2*a12*(2*a15*a16*a23 + a11*a26*a35 + a11*a25*a36 + a11*a23*a56))
  if ((r[1]==3) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(a13*(2*a16^2*a22 + 8*a12*a16*a26 + 2*a11*a26^2 + 2*a12^2*a66 + 
    a11*a22*a66) + 2*(2*a12^2*a16*a36 + a11*a16*(2*a23*a26 + a22*a36) + 
    a12*(2*a16^2*a23 + 2*a11*a26*a36 + a11*a23*a66)))
 if ((r[1]==3) & (r[2]==2) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a14^3*a22 + 12*a12*a14^2*a24 + 6*a11*a12*a24*a44 + 
  3*a14*(2*a11*a24^2 + 2*a12^2*a44 + a11*a22*a44))
 if ((r[1]==3) & (r[2]==2) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a14^2*(a15*a22 + 2*a12*a25) + 2*a12^2*a15*a44 + 
  2*a14*(4*a12*a15*a24 + 2*a11*a24*a25 + 2*a12^2*a45 + a11*a22*a45) + 
  a11*(2*a15*a24^2 + a15*a22*a44 + 2*a12*a25*a44 + 4*a12*a24*a45))
 if ((r[1]==3) & (r[2]==2) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a14^2*(a16*a22 + 2*a12*a26) + 2*a12^2*a16*a44 + 
  2*a14*(4*a12*a16*a24 + 2*a11*a24*a26 + 2*a12^2*a46 + a11*a22*a46) + 
  a11*(2*a16*a24^2 + a16*a22*a44 + 2*a12*a26*a44 + 4*a12*a24*a46))
 if ((r[1]==3) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(a14*(2*a15^2*a22 + 8*a12*a15*a25 + 2*a11*a25^2 + 2*a12^2*a55 + 
    a11*a22*a55) + 2*(2*a12^2*a15*a45 + a11*a15*(2*a24*a25 + a22*a45) + 
    a12*(2*a15^2*a24 + 2*a11*a25*a45 + a11*a24*a55)))
 if ((r[1]==3) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(2*a12^2*(a16*a45 + a15*a46) + a11*(2*a16*a24*a25 + 2*a15*a24*a26 + 
    a16*a22*a45 + a15*a22*a46) + a14*(2*a15*a16*a22 + 4*a12*a16*a25 + 
    4*a12*a15*a26 + 2*a11*a25*a26 + 2*a12^2*a56 + a11*a22*a56) + 
  2*a12*(2*a15*a16*a24 + a11*a26*a45 + a11*a25*a46 + a11*a24*a56))
 if ((r[1]==3) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(a14*(2*a16^2*a22 + 8*a12*a16*a26 + 2*a11*a26^2 + 2*a12^2*a66 + 
    a11*a22*a66) + 2*(2*a12^2*a16*a46 + a11*a16*(2*a24*a26 + a22*a46) + 
    a12*(2*a16^2*a24 + 2*a11*a26*a46 + a11*a24*a66)))
 if ((r[1]==3) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a15^3*a22 + 12*a12*a15^2*a25 + 6*a11*a12*a25*a55 + 
  3*a15*(2*a11*a25^2 + 2*a12^2*a55 + a11*a22*a55))
 if ((r[1]==3) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(2*a15^2*(a16*a22 + 2*a12*a26) + 2*a12^2*a16*a55 + 
  2*a15*(4*a12*a16*a25 + 2*a11*a25*a26 + 2*a12^2*a56 + a11*a22*a56) + 
  a11*(2*a16*a25^2 + a16*a22*a55 + 2*a12*a26*a55 + 4*a12*a25*a56))
 if ((r[1]==3) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(a15*(2*a16^2*a22 + 8*a12*a16*a26 + 2*a11*a26^2 + 2*a12^2*a66 + 
    a11*a22*a66) + 2*(2*a12^2*a16*a56 + a11*a16*(2*a25*a26 + a22*a56) + 
    a12*(2*a16^2*a25 + 2*a11*a26*a56 + a11*a25*a66)))
 if ((r[1]==3) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a16^3*a22 + 12*a12*a16^2*a26 + 6*a11*a12*a26*a66 + 
  3*a16*(2*a11*a26^2 + 2*a12^2*a66 + a11*a22*a66))
 if ((r[1]==3) & (r[2]==1) & (r[3]==4) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
24*a13^3*a23 + 36*a12*a13^2*a33 + 36*a11*a13*a23*a33 + 9*a11*a12*a33^2
 if ((r[1]==3) & (r[2]==1) & (r[3]==3) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a13^3*a24 + 6*a13^2*(a14*a23 + a12*a34) + 
  3*a11*a33*(a14*a23 + a12*a34) + 3*a13*(2*a12*a14*a33 + a11*a24*a33 + 
    2*a11*a23*a34))
 if ((r[1]==3) & (r[2]==1) & (r[3]==3) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a13^3*a25 + 6*a13^2*(a15*a23 + a12*a35) + 
  3*a11*a33*(a15*a23 + a12*a35) + 3*a13*(2*a12*a15*a33 + a11*a25*a33 + 
    2*a11*a23*a35))
 if ((r[1]==3) & (r[2]==1) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a13^3*a26 + 6*a13^2*(a16*a23 + a12*a36) + 
  3*a11*a33*(a16*a23 + a12*a36) + 3*a13*(2*a12*a16*a33 + a11*a26*a33 + 
    2*a11*a23*a36))
 if ((r[1]==3) & (r[2]==1) & (r[3]==2) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a11*a14*(a24*a33 + 2*a23*a34) + 2*a13^2*(2*a14*a24 + a12*a44) + 
  2*a13*(2*a14^2*a23 + 4*a12*a14*a34 + 2*a11*a24*a34 + a11*a23*a44) + 
  a12*(2*a14^2*a33 + 2*a11*a34^2 + a11*a33*a44))
 if ((r[1]==3) & (r[2]==1) & (r[3]==2) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(a11*(a15*a24*a33 + a14*a25*a33 + 2*a15*a23*a34 + 2*a14*a23*a35) + 
  2*a13^2*(a15*a24 + a14*a25 + a12*a45) + 
  2*a13*(2*a12*a15*a34 + a11*a25*a34 + a11*a24*a35 + 
    2*a14*(a15*a23 + a12*a35) + a11*a23*a45) + 
  a12*(2*a14*a15*a33 + 2*a11*a34*a35 + a11*a33*a45))
 if ((r[1]==3) & (r[2]==1) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(a11*(a16*a24*a33 + a14*a26*a33 + 2*a16*a23*a34 + 2*a14*a23*a36) + 
  2*a13^2*(a16*a24 + a14*a26 + a12*a46) + 
  2*a13*(2*a12*a16*a34 + a11*a26*a34 + a11*a24*a36 + 
    2*a14*(a16*a23 + a12*a36) + a11*a23*a46) + 
  a12*(2*a14*a16*a33 + 2*a11*a34*a36 + a11*a33*a46))
 if ((r[1]==3) & (r[2]==1) & (r[3]==2) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a11*a15*(a25*a33 + 2*a23*a35) + 2*a13^2*(2*a15*a25 + a12*a55) + 
  2*a13*(2*a15^2*a23 + 4*a12*a15*a35 + 2*a11*a25*a35 + a11*a23*a55) + 
  a12*(2*a15^2*a33 + 2*a11*a35^2 + a11*a33*a55))
 if ((r[1]==3) & (r[2]==1) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(a11*(a16*a25*a33 + a15*a26*a33 + 2*a16*a23*a35 + 2*a15*a23*a36) + 
  2*a13^2*(a16*a25 + a15*a26 + a12*a56) + 
  2*a13*(2*a12*a16*a35 + a11*a26*a35 + a11*a25*a36 + 
    2*a15*(a16*a23 + a12*a36) + a11*a23*a56) + 
  a12*(2*a15*a16*a33 + 2*a11*a35*a36 + a11*a33*a56))
 if ((r[1]==3) & (r[2]==1) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a11*a16*(a26*a33 + 2*a23*a36) + 2*a13^2*(2*a16*a26 + a12*a66) + 
  2*a13*(2*a16^2*a23 + 4*a12*a16*a36 + 2*a11*a26*a36 + a11*a23*a66) + 
  a12*(2*a16^2*a33 + 2*a11*a36^2 + a11*a33*a66))
 if ((r[1]==3) & (r[2]==1) & (r[3]==1) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a14^3*a23 + 6*a14^2*(a13*a24 + a12*a34) + 3*a11*(a13*a24 + a12*a34)*
   a44 + 3*a14*(2*a11*a24*a34 + 2*a12*a13*a44 + a11*a23*a44))
 if ((r[1]==3) & (r[2]==1) & (r[3]==1) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a14^2*(a15*a23 + a13*a25 + a12*a35) + 2*a12*a13*a15*a44 + 
  a11*(2*a15*a24*a34 + a15*a23*a44 + a13*a25*a44 + a12*a35*a44 + 
    2*a13*a24*a45 + 2*a12*a34*a45) + 
  2*a14*(2*a12*a15*a34 + a11*a25*a34 + a11*a24*a35 + a11*a23*a45 + 
    2*a13*(a15*a24 + a12*a45)))
 if ((r[1]==3) & (r[2]==1) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a14^2*(a16*a23 + a13*a26 + a12*a36) + 2*a12*a13*a16*a44 + 
  a11*(2*a16*a24*a34 + a16*a23*a44 + a13*a26*a44 + a12*a36*a44 + 
    2*a13*a24*a46 + 2*a12*a34*a46) + 
  2*a14*(2*a12*a16*a34 + a11*a26*a34 + a11*a24*a36 + a11*a23*a46 + 
    2*a13*(a16*a24 + a12*a46)))
 if ((r[1]==3) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a12*a15^2*a34 + 2*a11*a15*a25*a34 + 2*a11*a15*a24*a35 + 
  2*a11*a15*a23*a45 + 2*a11*a12*a35*a45 + a11*a12*a34*a55 + 
  a14*(2*a15^2*a23 + 2*a11*a25*a35 + 4*a15*(a13*a25 + a12*a35) + 
    2*a12*a13*a55 + a11*a23*a55) + a13*(2*a15^2*a24 + 4*a12*a15*a45 + 
    2*a11*a25*a45 + a11*a24*a55))
 if ((r[1]==3) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(2*a12*a15*a16*a34 + a11*a16*a25*a34 + a11*a15*a26*a34 + a11*a16*a24*a35 + 
  a11*a15*a24*a36 + a11*a16*a23*a45 + a11*a12*a36*a45 + a11*a15*a23*a46 + 
  a11*a12*a35*a46 + a11*a12*a34*a56 + 
  a14*(2*a13*a16*a25 + 2*a12*a16*a35 + a11*a26*a35 + a11*a25*a36 + 
    2*a15*(a16*a23 + a13*a26 + a12*a36) + 2*a12*a13*a56 + a11*a23*a56) + 
  a13*(2*a15*a16*a24 + 2*a12*a16*a45 + a11*a26*a45 + 2*a12*a15*a46 + 
    a11*a25*a46 + a11*a24*a56))
  if ((r[1]==3) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a12*a16^2*a34 + 2*a11*a16*a26*a34 + 2*a11*a16*a24*a36 + 
  2*a11*a16*a23*a46 + 2*a11*a12*a36*a46 + a11*a12*a34*a66 + 
  a14*(2*a16^2*a23 + 2*a11*a26*a36 + 4*a16*(a13*a26 + a12*a36) + 
    2*a12*a13*a66 + a11*a23*a66) + a13*(2*a16^2*a24 + 4*a12*a16*a46 + 
    2*a11*a26*a46 + a11*a24*a66))
 if ((r[1]==3) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a15^3*a23 + 6*a15^2*(a13*a25 + a12*a35) + 3*a11*(a13*a25 + a12*a35)*
   a55 + 3*a15*(2*a11*a25*a35 + 2*a12*a13*a55 + a11*a23*a55))
 if ((r[1]==3) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(2*a15^2*(a16*a23 + a13*a26 + a12*a36) + 2*a12*a13*a16*a55 + 
  a11*(2*a16*a25*a35 + a16*a23*a55 + a13*a26*a55 + a12*a36*a55 + 
    2*a13*a25*a56 + 2*a12*a35*a56) + 
  2*a15*(2*a12*a16*a35 + a11*a26*a35 + a11*a25*a36 + a11*a23*a56 + 
    2*a13*(a16*a25 + a12*a56)))
 if ((r[1]==3) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(2*a12*a16^2*a35 + 2*a11*a16*a26*a35 + 2*a11*a16*a25*a36 + 
  2*a11*a16*a23*a56 + 2*a11*a12*a36*a56 + a11*a12*a35*a66 + 
  a15*(2*a16^2*a23 + 2*a11*a26*a36 + 4*a16*(a13*a26 + a12*a36) + 
    2*a12*a13*a66 + a11*a23*a66) + a13*(2*a16^2*a25 + 4*a12*a16*a56 + 
    2*a11*a26*a56 + a11*a25*a66))
 if ((r[1]==3) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a16^3*a23 + 6*a16^2*(a13*a26 + a12*a36) + 3*a11*(a13*a26 + a12*a36)*
   a66 + 3*a16*(2*a11*a26*a36 + 2*a12*a13*a66 + a11*a23*a66))
 if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==4) 
& (r[5]==0) & (r[6]==0))
psi <-
24*a14^3*a24 + 36*a12*a14^2*a44 + 36*a11*a14*a24*a44 + 9*a11*a12*a44^2
 if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==3) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a14^3*a25 + 6*a14^2*(a15*a24 + a12*a45) + 
  3*a11*a44*(a15*a24 + a12*a45) + 3*a14*(2*a12*a15*a44 + a11*a25*a44 + 
    2*a11*a24*a45))
 if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a14^3*a26 + 6*a14^2*(a16*a24 + a12*a46) + 
  3*a11*a44*(a16*a24 + a12*a46) + 3*a14*(2*a12*a16*a44 + a11*a26*a44 + 
    2*a11*a24*a46))
 if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a11*a15*(a25*a44 + 2*a24*a45) + 2*a14^2*(2*a15*a25 + a12*a55) + 
  2*a14*(2*a15^2*a24 + 4*a12*a15*a45 + 2*a11*a25*a45 + a11*a24*a55) + 
  a12*(2*a15^2*a44 + 2*a11*a45^2 + a11*a44*a55))
 if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(a11*(a16*a25*a44 + a15*a26*a44 + 2*a16*a24*a45 + 2*a15*a24*a46) + 
  2*a14^2*(a16*a25 + a15*a26 + a12*a56) + 
  2*a14*(2*a12*a16*a45 + a11*a26*a45 + a11*a25*a46 + 
    2*a15*(a16*a24 + a12*a46) + a11*a24*a56) + 
  a12*(2*a15*a16*a44 + 2*a11*a45*a46 + a11*a44*a56))
 if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a11*a16*(a26*a44 + 2*a24*a46) + 2*a14^2*(2*a16*a26 + a12*a66) + 
  2*a14*(2*a16^2*a24 + 4*a12*a16*a46 + 2*a11*a26*a46 + a11*a24*a66) + 
  a12*(2*a16^2*a44 + 2*a11*a46^2 + a11*a44*a66))
 if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a15^3*a24 + 6*a15^2*(a14*a25 + a12*a45) + 3*a11*(a14*a25 + a12*a45)*
   a55 + 3*a15*(2*a11*a25*a45 + 2*a12*a14*a55 + a11*a24*a55))
 if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(2*a15^2*(a16*a24 + a14*a26 + a12*a46) + 2*a12*a14*a16*a55 + 
  a11*(2*a16*a25*a45 + a16*a24*a55 + a14*a26*a55 + a12*a46*a55 + 
    2*a14*a25*a56 + 2*a12*a45*a56) + 
  2*a15*(2*a12*a16*a45 + a11*a26*a45 + a11*a25*a46 + a11*a24*a56 + 
    2*a14*(a16*a25 + a12*a56)))
 if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(2*a12*a16^2*a45 + 2*a11*a16*a26*a45 + 2*a11*a16*a25*a46 + 
  2*a11*a16*a24*a56 + 2*a11*a12*a46*a56 + a11*a12*a45*a66 + 
  a15*(2*a16^2*a24 + 2*a11*a26*a46 + 4*a16*(a14*a26 + a12*a46) + 
    2*a12*a14*a66 + a11*a24*a66) + a14*(2*a16^2*a25 + 4*a12*a16*a56 + 
    2*a11*a26*a56 + a11*a25*a66))
 if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a16^3*a24 + 6*a16^2*(a14*a26 + a12*a46) + 3*a11*(a14*a26 + a12*a46)*
   a66 + 3*a16*(2*a11*a26*a46 + 2*a12*a14*a66 + a11*a24*a66))
 if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==4) & (r[6]==0))
psi <-
24*a15^3*a25 + 36*a12*a15^2*a55 + 36*a11*a15*a25*a55 + 9*a11*a12*a55^2
 if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==1))
psi <-
3*(2*a15^3*a26 + 6*a15^2*(a16*a25 + a12*a56) + 
  3*a11*a55*(a16*a25 + a12*a56) + 3*a15*(2*a12*a16*a55 + a11*a26*a55 + 
    2*a11*a25*a56))
 if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==2))
psi <-
3*(2*a11*a16*(a26*a55 + 2*a25*a56) + 2*a15^2*(2*a16*a26 + a12*a66) + 
  2*a15*(2*a16^2*a25 + 4*a12*a16*a56 + 2*a11*a26*a56 + a11*a25*a66) + 
  a12*(2*a16^2*a55 + 2*a11*a56^2 + a11*a55*a66))
 if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(2*a16^3*a25 + 6*a16^2*(a15*a26 + a12*a56) + 3*a11*(a15*a26 + a12*a56)*
   a66 + 3*a16*(2*a11*a26*a56 + 2*a12*a15*a66 + a11*a25*a66))
 if ((r[1]==3) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==4))
psi <-
24*a16^3*a26 + 36*a12*a16^2*a66 + 36*a11*a16*a26*a66 + 9*a11*a12*a66^2
 if ((r[1]==3) & (r[2]==0) & (r[3]==5) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a13*a33*(4*a13^2 + 3*a11*a33)
 if ((r[1]==3) & (r[2]==0) & (r[3]==4) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
36*a13^2*a14*a33 + 9*a11*a14*a33^2 + 24*a13^3*a34 + 36*a11*a13*a33*a34
 if ((r[1]==3) & (r[2]==0) & (r[3]==4) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
36*a13^2*a15*a33 + 9*a11*a15*a33^2 + 24*a13^3*a35 + 36*a11*a13*a33*a35
 if ((r[1]==3) & (r[2]==0) & (r[3]==4) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
36*a13^2*a16*a33 + 9*a11*a16*a33^2 + 24*a13^3*a36 + 36*a11*a13*a33*a36
 if ((r[1]==3) & (r[2]==0) & (r[3]==3) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(12*a13^2*a14*a34 + 6*a11*a14*a33*a34 + 2*a13^3*a44 + 
  3*a13*(2*a14^2*a33 + 2*a11*a34^2 + a11*a33*a44))
 if ((r[1]==3) & (r[2]==0) & (r[3]==3) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(6*a13^2*(a15*a34 + a14*a35) + 3*a11*a33*(a15*a34 + a14*a35) + 
  2*a13^3*a45 + 3*a13*(2*a14*a15*a33 + 2*a11*a34*a35 + a11*a33*a45))
 if ((r[1]==3) & (r[2]==0) & (r[3]==3) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(6*a13^2*(a16*a34 + a14*a36) + 3*a11*a33*(a16*a34 + a14*a36) + 
  2*a13^3*a46 + 3*a13*(2*a14*a16*a33 + 2*a11*a34*a36 + a11*a33*a46))
 if ((r[1]==3) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(12*a13^2*a15*a35 + 6*a11*a15*a33*a35 + 2*a13^3*a55 + 
  3*a13*(2*a15^2*a33 + 2*a11*a35^2 + a11*a33*a55))
 if ((r[1]==3) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(6*a13^2*(a16*a35 + a15*a36) + 3*a11*a33*(a16*a35 + a15*a36) + 
  2*a13^3*a56 + 3*a13*(2*a15*a16*a33 + 2*a11*a35*a36 + a11*a33*a56))
  if ((r[1]==3) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(12*a13^2*a16*a36 + 6*a11*a16*a33*a36 + 2*a13^3*a66 + 
  3*a13*(2*a16^2*a33 + 2*a11*a36^2 + a11*a33*a66))
 if ((r[1]==3) & (r[2]==0) & (r[3]==2) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a14^3*a33 + 12*a13*a14^2*a34 + 6*a11*a13*a34*a44 + 
  3*a14*(2*a11*a34^2 + 2*a13^2*a44 + a11*a33*a44))
 if ((r[1]==3) & (r[2]==0) & (r[3]==2) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a14^2*(a15*a33 + 2*a13*a35) + 2*a13^2*a15*a44 + 
  2*a14*(4*a13*a15*a34 + 2*a11*a34*a35 + 2*a13^2*a45 + a11*a33*a45) + 
  a11*(2*a15*a34^2 + a15*a33*a44 + 2*a13*a35*a44 + 4*a13*a34*a45))
 if ((r[1]==3) & (r[2]==0) & (r[3]==2) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a14^2*(a16*a33 + 2*a13*a36) + 2*a13^2*a16*a44 + 
  2*a14*(4*a13*a16*a34 + 2*a11*a34*a36 + 2*a13^2*a46 + a11*a33*a46) + 
  a11*(2*a16*a34^2 + a16*a33*a44 + 2*a13*a36*a44 + 4*a13*a34*a46))
 if ((r[1]==3) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(a14*(2*a15^2*a33 + 8*a13*a15*a35 + 2*a11*a35^2 + 2*a13^2*a55 + 
    a11*a33*a55) + 2*(2*a13^2*a15*a45 + a11*a15*(2*a34*a35 + a33*a45) + 
    a13*(2*a15^2*a34 + 2*a11*a35*a45 + a11*a34*a55)))
 if ((r[1]==3) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(2*a13^2*(a16*a45 + a15*a46) + a11*(2*a16*a34*a35 + 2*a15*a34*a36 + 
    a16*a33*a45 + a15*a33*a46) + a14*(2*a15*a16*a33 + 4*a13*a16*a35 + 
    4*a13*a15*a36 + 2*a11*a35*a36 + 2*a13^2*a56 + a11*a33*a56) + 
  2*a13*(2*a15*a16*a34 + a11*a36*a45 + a11*a35*a46 + a11*a34*a56))
 if ((r[1]==3) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(a14*(2*a16^2*a33 + 8*a13*a16*a36 + 2*a11*a36^2 + 2*a13^2*a66 + 
    a11*a33*a66) + 2*(2*a13^2*a16*a46 + a11*a16*(2*a34*a36 + a33*a46) + 
    a13*(2*a16^2*a34 + 2*a11*a36*a46 + a11*a34*a66)))
 if ((r[1]==3) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a15^3*a33 + 12*a13*a15^2*a35 + 6*a11*a13*a35*a55 + 
  3*a15*(2*a11*a35^2 + 2*a13^2*a55 + a11*a33*a55))
 if ((r[1]==3) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(2*a15^2*(a16*a33 + 2*a13*a36) + 2*a13^2*a16*a55 + 
  2*a15*(4*a13*a16*a35 + 2*a11*a35*a36 + 2*a13^2*a56 + a11*a33*a56) + 
  a11*(2*a16*a35^2 + a16*a33*a55 + 2*a13*a36*a55 + 4*a13*a35*a56))
 if ((r[1]==3) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(a15*(2*a16^2*a33 + 8*a13*a16*a36 + 2*a11*a36^2 + 2*a13^2*a66 + 
    a11*a33*a66) + 2*(2*a13^2*a16*a56 + a11*a16*(2*a35*a36 + a33*a56) + 
    a13*(2*a16^2*a35 + 2*a11*a36*a56 + a11*a35*a66)))
 if ((r[1]==3) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a16^3*a33 + 12*a13*a16^2*a36 + 6*a11*a13*a36*a66 + 
  3*a16*(2*a11*a36^2 + 2*a13^2*a66 + a11*a33*a66))
 if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==4) 
& (r[5]==0) & (r[6]==0))
psi <-
24*a14^3*a34 + 36*a13*a14^2*a44 + 36*a11*a14*a34*a44 + 9*a11*a13*a44^2
 if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==3) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a14^3*a35 + 6*a14^2*(a15*a34 + a13*a45) + 
  3*a11*a44*(a15*a34 + a13*a45) + 3*a14*(2*a13*a15*a44 + a11*a35*a44 + 
    2*a11*a34*a45))
 if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==3) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a14^3*a36 + 6*a14^2*(a16*a34 + a13*a46) + 
  3*a11*a44*(a16*a34 + a13*a46) + 3*a14*(2*a13*a16*a44 + a11*a36*a44 + 
    2*a11*a34*a46))
 if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a11*a15*(a35*a44 + 2*a34*a45) + 2*a14^2*(2*a15*a35 + a13*a55) + 
  2*a14*(2*a15^2*a34 + 4*a13*a15*a45 + 2*a11*a35*a45 + a11*a34*a55) + 
  a13*(2*a15^2*a44 + 2*a11*a45^2 + a11*a44*a55))
 if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(a11*(a16*a35*a44 + a15*a36*a44 + 2*a16*a34*a45 + 2*a15*a34*a46) + 
  2*a14^2*(a16*a35 + a15*a36 + a13*a56) + 
  2*a14*(2*a13*a16*a45 + a11*a36*a45 + a11*a35*a46 + 
    2*a15*(a16*a34 + a13*a46) + a11*a34*a56) + 
  a13*(2*a15*a16*a44 + 2*a11*a45*a46 + a11*a44*a56))
 if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a11*a16*(a36*a44 + 2*a34*a46) + 2*a14^2*(2*a16*a36 + a13*a66) + 
  2*a14*(2*a16^2*a34 + 4*a13*a16*a46 + 2*a11*a36*a46 + a11*a34*a66) + 
  a13*(2*a16^2*a44 + 2*a11*a46^2 + a11*a44*a66))
 if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a15^3*a34 + 6*a15^2*(a14*a35 + a13*a45) + 3*a11*(a14*a35 + a13*a45)*
   a55 + 3*a15*(2*a11*a35*a45 + 2*a13*a14*a55 + a11*a34*a55))
 if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(2*a15^2*(a16*a34 + a14*a36 + a13*a46) + 2*a13*a14*a16*a55 + 
  a11*(2*a16*a35*a45 + a16*a34*a55 + a14*a36*a55 + a13*a46*a55 + 
    2*a14*a35*a56 + 2*a13*a45*a56) + 
  2*a15*(2*a13*a16*a45 + a11*a36*a45 + a11*a35*a46 + a11*a34*a56 + 
    2*a14*(a16*a35 + a13*a56)))
 if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(2*a13*a16^2*a45 + 2*a11*a16*a36*a45 + 2*a11*a16*a35*a46 + 
  2*a11*a16*a34*a56 + 2*a11*a13*a46*a56 + a11*a13*a45*a66 + 
  a15*(2*a16^2*a34 + 2*a11*a36*a46 + 4*a16*(a14*a36 + a13*a46) + 
    2*a13*a14*a66 + a11*a34*a66) + a14*(2*a16^2*a35 + 4*a13*a16*a56 + 
    2*a11*a36*a56 + a11*a35*a66))
 if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a16^3*a34 + 6*a16^2*(a14*a36 + a13*a46) + 3*a11*(a14*a36 + a13*a46)*
   a66 + 3*a16*(2*a11*a36*a46 + 2*a13*a14*a66 + a11*a34*a66))
 if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==4) & (r[6]==0))
psi <-
24*a15^3*a35 + 36*a13*a15^2*a55 + 36*a11*a15*a35*a55 + 9*a11*a13*a55^2
 if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==3) & (r[6]==1))
psi <-
3*(2*a15^3*a36 + 6*a15^2*(a16*a35 + a13*a56) + 
  3*a11*a55*(a16*a35 + a13*a56) + 3*a15*(2*a13*a16*a55 + a11*a36*a55 + 
    2*a11*a35*a56))
 if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==2))
psi <-
3*(2*a11*a16*(a36*a55 + 2*a35*a56) + 2*a15^2*(2*a16*a36 + a13*a66) + 
  2*a15*(2*a16^2*a35 + 4*a13*a16*a56 + 2*a11*a36*a56 + a11*a35*a66) + 
  a13*(2*a16^2*a55 + 2*a11*a56^2 + a11*a55*a66))
 if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(2*a16^3*a35 + 6*a16^2*(a15*a36 + a13*a56) + 3*a11*(a15*a36 + a13*a56)*
   a66 + 3*a16*(2*a11*a36*a56 + 2*a13*a15*a66 + a11*a35*a66))
 if ((r[1]==3) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==4))
psi <-
24*a16^3*a36 + 36*a13*a16^2*a66 + 36*a11*a16*a36*a66 + 9*a11*a13*a66^2
 if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==5) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a14*a44*(4*a14^2 + 3*a11*a44)
 if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==4) 
& (r[5]==1) & (r[6]==0))
psi <-
36*a14^2*a15*a44 + 9*a11*a15*a44^2 + 24*a14^3*a45 + 36*a11*a14*a44*a45
 if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==4) 
& (r[5]==0) & (r[6]==1))
psi <-
36*a14^2*a16*a44 + 9*a11*a16*a44^2 + 24*a14^3*a46 + 36*a11*a14*a44*a46
  if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(12*a14^2*a15*a45 + 6*a11*a15*a44*a45 + 2*a14^3*a55 + 
  3*a14*(2*a15^2*a44 + 2*a11*a45^2 + a11*a44*a55))
 if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(6*a14^2*(a16*a45 + a15*a46) + 3*a11*a44*(a16*a45 + a15*a46) + 
  2*a14^3*a56 + 3*a14*(2*a15*a16*a44 + 2*a11*a45*a46 + a11*a44*a56))
 if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(12*a14^2*a16*a46 + 6*a11*a16*a44*a46 + 2*a14^3*a66 + 
  3*a14*(2*a16^2*a44 + 2*a11*a46^2 + a11*a44*a66))
 if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a15^3*a44 + 12*a14*a15^2*a45 + 6*a11*a14*a45*a55 + 
  3*a15*(2*a11*a45^2 + 2*a14^2*a55 + a11*a44*a55))
 if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(2*a15^2*(a16*a44 + 2*a14*a46) + 2*a14^2*a16*a55 + 
  2*a15*(4*a14*a16*a45 + 2*a11*a45*a46 + 2*a14^2*a56 + a11*a44*a56) + 
  a11*(2*a16*a45^2 + a16*a44*a55 + 2*a14*a46*a55 + 4*a14*a45*a56))
 if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(a15*(2*a16^2*a44 + 8*a14*a16*a46 + 2*a11*a46^2 + 2*a14^2*a66 + 
    a11*a44*a66) + 2*(2*a14^2*a16*a56 + a11*a16*(2*a45*a46 + a44*a56) + 
    a14*(2*a16^2*a45 + 2*a11*a46*a56 + a11*a45*a66)))
 if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a16^3*a44 + 12*a14*a16^2*a46 + 6*a11*a14*a46*a66 + 
  3*a16*(2*a11*a46^2 + 2*a14^2*a66 + a11*a44*a66))
 if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==4) & (r[6]==0))
psi <-
24*a15^3*a45 + 36*a14*a15^2*a55 + 36*a11*a15*a45*a55 + 9*a11*a14*a55^2
 if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==3) & (r[6]==1))
psi <-
3*(2*a15^3*a46 + 6*a15^2*(a16*a45 + a14*a56) + 
  3*a11*a55*(a16*a45 + a14*a56) + 3*a15*(2*a14*a16*a55 + a11*a46*a55 + 
    2*a11*a45*a56))
 if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==2))
psi <-
3*(2*a11*a16*(a46*a55 + 2*a45*a56) + 2*a15^2*(2*a16*a46 + a14*a66) + 
  2*a15*(2*a16^2*a45 + 4*a14*a16*a56 + 2*a11*a46*a56 + a11*a45*a66) + 
  a14*(2*a16^2*a55 + 2*a11*a56^2 + a11*a55*a66))
 if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(2*a16^3*a45 + 6*a16^2*(a15*a46 + a14*a56) + 3*a11*(a15*a46 + a14*a56)*
   a66 + 3*a16*(2*a11*a46*a56 + 2*a14*a15*a66 + a11*a45*a66))
 if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==4))
psi <-
24*a16^3*a46 + 36*a14*a16^2*a66 + 36*a11*a16*a46*a66 + 9*a11*a14*a66^2
 if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==5) & (r[6]==0))
psi <-
15*a15*a55*(4*a15^2 + 3*a11*a55)
 if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==4) & (r[6]==1))
psi <-
36*a15^2*a16*a55 + 9*a11*a16*a55^2 + 24*a15^3*a56 + 36*a11*a15*a55*a56
 if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==2))
psi <-
3*(12*a15^2*a16*a56 + 6*a11*a16*a55*a56 + 2*a15^3*a66 + 
  3*a15*(2*a16^2*a55 + 2*a11*a56^2 + a11*a55*a66))
 if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==3))
psi <-
3*(2*a16^3*a55 + 12*a15*a16^2*a56 + 6*a11*a15*a56*a66 + 
  3*a16*(2*a11*a56^2 + 2*a15^2*a66 + a11*a55*a66))
 if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==4))
psi <-
24*a16^3*a56 + 36*a15*a16^2*a66 + 36*a11*a16*a56*a66 + 9*a11*a15*a66^2
 if ((r[1]==3) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==5))
psi <-
15*a16*a66*(4*a16^2 + 3*a11*a66)
 if ((r[1]==2) & (r[2]==6) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a22^2*(6*a12^2 + a11*a22)
 if ((r[1]==2) & (r[2]==5) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a22*(2*a12*a13*a22 + 4*a12^2*a23 + a11*a22*a23)
 if ((r[1]==2) & (r[2]==5) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a22*(2*a12*a14*a22 + 4*a12^2*a24 + a11*a22*a24)
 if ((r[1]==2) & (r[2]==5) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a22*(2*a12*a15*a22 + 4*a12^2*a25 + a11*a22*a25)
 if ((r[1]==2) & (r[2]==5) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a22*(2*a12*a16*a22 + 4*a12^2*a26 + a11*a22*a26)
 if ((r[1]==2) & (r[2]==4) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a13^2*a22^2 + 16*a12*a13*a22*a23 + 4*a12^2*(2*a23^2 + a22*a33) + 
  a11*a22*(4*a23^2 + a22*a33))
 if ((r[1]==2) & (r[2]==4) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(8*a12*a14*a22*a23 + 2*a13*a22*(a14*a22 + 4*a12*a24) + 
  4*a12^2*(2*a23*a24 + a22*a34) + a11*a22*(4*a23*a24 + a22*a34))
 if ((r[1]==2) & (r[2]==4) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(8*a12*a15*a22*a23 + 2*a13*a22*(a15*a22 + 4*a12*a25) + 
  4*a12^2*(2*a23*a25 + a22*a35) + a11*a22*(4*a23*a25 + a22*a35))
 if ((r[1]==2) & (r[2]==4) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(8*a12*a16*a22*a23 + 2*a13*a22*(a16*a22 + 4*a12*a26) + 
  4*a12^2*(2*a23*a26 + a22*a36) + a11*a22*(4*a23*a26 + a22*a36))
 if ((r[1]==2) & (r[2]==4) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a14^2*a22^2 + 16*a12*a14*a22*a24 + 4*a12^2*(2*a24^2 + a22*a44) + 
  a11*a22*(4*a24^2 + a22*a44))
 if ((r[1]==2) & (r[2]==4) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(8*a12*a15*a22*a24 + 2*a14*a22*(a15*a22 + 4*a12*a25) + 
  4*a12^2*(2*a24*a25 + a22*a45) + a11*a22*(4*a24*a25 + a22*a45))
 if ((r[1]==2) & (r[2]==4) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(8*a12*a16*a22*a24 + 2*a14*a22*(a16*a22 + 4*a12*a26) + 
  4*a12^2*(2*a24*a26 + a22*a46) + a11*a22*(4*a24*a26 + a22*a46))
 if ((r[1]==2) & (r[2]==4) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a15^2*a22^2 + 16*a12*a15*a22*a25 + 4*a12^2*(2*a25^2 + a22*a55) + 
  a11*a22*(4*a25^2 + a22*a55))
 if ((r[1]==2) & (r[2]==4) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(8*a12*a16*a22*a25 + 2*a15*a22*(a16*a22 + 4*a12*a26) + 
  4*a12^2*(2*a25*a26 + a22*a56) + a11*a22*(4*a25*a26 + a22*a56))
 if ((r[1]==2) & (r[2]==4) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a16^2*a22^2 + 16*a12*a16*a22*a26 + 4*a12^2*(2*a26^2 + a22*a66) + 
  a11*a22*(4*a26^2 + a22*a66))
 if ((r[1]==2) & (r[2]==3) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(6*a13^2*a22*a23 + 6*a12*a13*(2*a23^2 + a22*a33) + 
  a23*(2*a11*a23^2 + 6*a12^2*a33 + 3*a11*a22*a33))
 if ((r[1]==2) & (r[2]==3) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a13^2*a22*a24 + 2*a12*a14*(2*a23^2 + a22*a33) + 
  4*a13*(a14*a22*a23 + 2*a12*a23*a24 + a12*a22*a34) + 
  2*a12^2*(a24*a33 + 2*a23*a34) + a11*(2*a23^2*a24 + a22*a24*a33 + 
    2*a22*a23*a34))
  if ((r[1]==2) & (r[2]==3) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a13^2*a22*a25 + 2*a12*a15*(2*a23^2 + a22*a33) + 
  4*a13*(a15*a22*a23 + 2*a12*a23*a25 + a12*a22*a35) + 
  2*a12^2*(a25*a33 + 2*a23*a35) + a11*(2*a23^2*a25 + a22*a25*a33 + 
    2*a22*a23*a35))
 if ((r[1]==2) & (r[2]==3) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a13^2*a22*a26 + 2*a12*a16*(2*a23^2 + a22*a33) + 
  4*a13*(a16*a22*a23 + 2*a12*a23*a26 + a12*a22*a36) + 
  2*a12^2*(a26*a33 + 2*a23*a36) + a11*(2*a23^2*a26 + a22*a26*a33 + 
    2*a22*a23*a36))
 if ((r[1]==2) & (r[2]==3) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a14^2*a22*a23 + 4*a14*(a13*a22*a24 + 2*a12*a23*a24 + a12*a22*a34) + 
  2*a12*a13*(2*a24^2 + a22*a44) + 2*a12^2*(2*a24*a34 + a23*a44) + 
  a11*(2*a23*a24^2 + 2*a22*a24*a34 + a22*a23*a44))
 if ((r[1]==2) & (r[2]==3) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(4*a12*a15*a23*a24 + 2*a11*a23*a24*a25 + 2*a12*a15*a22*a34 + 
  2*a12^2*a25*a34 + a11*a22*a25*a34 + 2*a12^2*a24*a35 + a11*a22*a24*a35 + 
  2*a14*(a15*a22*a23 + a13*a22*a25 + 2*a12*a23*a25 + a12*a22*a35) + 
  2*a12^2*a23*a45 + a11*a22*a23*a45 + 
  2*a13*(a15*a22*a24 + 2*a12*a24*a25 + a12*a22*a45))
 if ((r[1]==2) & (r[2]==3) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(4*a12*a16*a23*a24 + 2*a11*a23*a24*a26 + 2*a12*a16*a22*a34 + 
  2*a12^2*a26*a34 + a11*a22*a26*a34 + 2*a12^2*a24*a36 + a11*a22*a24*a36 + 
  2*a14*(a16*a22*a23 + a13*a22*a26 + 2*a12*a23*a26 + a12*a22*a36) + 
  2*a12^2*a23*a46 + a11*a22*a23*a46 + 
  2*a13*(a16*a22*a24 + 2*a12*a24*a26 + a12*a22*a46))
 if ((r[1]==2) & (r[2]==3) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a15^2*a22*a23 + 4*a15*(a13*a22*a25 + 2*a12*a23*a25 + a12*a22*a35) + 
  2*a12*a13*(2*a25^2 + a22*a55) + 2*a12^2*(2*a25*a35 + a23*a55) + 
  a11*(2*a23*a25^2 + 2*a22*a25*a35 + a22*a23*a55))
 if ((r[1]==2) & (r[2]==3) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(4*a12*a16*a23*a25 + 2*a11*a23*a25*a26 + 2*a12*a16*a22*a35 + 
  2*a12^2*a26*a35 + a11*a22*a26*a35 + 2*a12^2*a25*a36 + a11*a22*a25*a36 + 
  2*a15*(a16*a22*a23 + a13*a22*a26 + 2*a12*a23*a26 + a12*a22*a36) + 
  2*a12^2*a23*a56 + a11*a22*a23*a56 + 
  2*a13*(a16*a22*a25 + 2*a12*a25*a26 + a12*a22*a56))
 if ((r[1]==2) & (r[2]==3) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a16^2*a22*a23 + 4*a16*(a13*a22*a26 + 2*a12*a23*a26 + a12*a22*a36) + 
  2*a12*a13*(2*a26^2 + a22*a66) + 2*a12^2*(2*a26*a36 + a23*a66) + 
  a11*(2*a23*a26^2 + 2*a22*a26*a36 + a22*a23*a66))
 if ((r[1]==2) & (r[2]==3) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(6*a14^2*a22*a24 + 6*a12*a14*(2*a24^2 + a22*a44) + 
  a24*(2*a11*a24^2 + 6*a12^2*a44 + 3*a11*a22*a44))
 if ((r[1]==2) & (r[2]==3) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a14^2*a22*a25 + 2*a12*a15*(2*a24^2 + a22*a44) + 
  4*a14*(a15*a22*a24 + 2*a12*a24*a25 + a12*a22*a45) + 
  2*a12^2*(a25*a44 + 2*a24*a45) + a11*(2*a24^2*a25 + a22*a25*a44 + 
    2*a22*a24*a45))
 if ((r[1]==2) & (r[2]==3) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a14^2*a22*a26 + 2*a12*a16*(2*a24^2 + a22*a44) + 
  4*a14*(a16*a22*a24 + 2*a12*a24*a26 + a12*a22*a46) + 
  2*a12^2*(a26*a44 + 2*a24*a46) + a11*(2*a24^2*a26 + a22*a26*a44 + 
    2*a22*a24*a46))
 if ((r[1]==2) & (r[2]==3) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a15^2*a22*a24 + 4*a15*(a14*a22*a25 + 2*a12*a24*a25 + a12*a22*a45) + 
  2*a12*a14*(2*a25^2 + a22*a55) + 2*a12^2*(2*a25*a45 + a24*a55) + 
  a11*(2*a24*a25^2 + 2*a22*a25*a45 + a22*a24*a55))
 if ((r[1]==2) & (r[2]==3) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(4*a12*a16*a24*a25 + 2*a11*a24*a25*a26 + 2*a12*a16*a22*a45 + 
  2*a12^2*a26*a45 + a11*a22*a26*a45 + 2*a12^2*a25*a46 + a11*a22*a25*a46 + 
  2*a15*(a16*a22*a24 + a14*a22*a26 + 2*a12*a24*a26 + a12*a22*a46) + 
  2*a12^2*a24*a56 + a11*a22*a24*a56 + 
  2*a14*(a16*a22*a25 + 2*a12*a25*a26 + a12*a22*a56))
 if ((r[1]==2) & (r[2]==3) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a16^2*a22*a24 + 4*a16*(a14*a22*a26 + 2*a12*a24*a26 + a12*a22*a46) + 
  2*a12*a14*(2*a26^2 + a22*a66) + 2*a12^2*(2*a26*a46 + a24*a66) + 
  a11*(2*a24*a26^2 + 2*a22*a26*a46 + a22*a24*a66))
 if ((r[1]==2) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(6*a15^2*a22*a25 + 6*a12*a15*(2*a25^2 + a22*a55) + 
  a25*(2*a11*a25^2 + 6*a12^2*a55 + 3*a11*a22*a55))
 if ((r[1]==2) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(2*a15^2*a22*a26 + 2*a12*a16*(2*a25^2 + a22*a55) + 
  4*a15*(a16*a22*a25 + 2*a12*a25*a26 + a12*a22*a56) + 
  2*a12^2*(a26*a55 + 2*a25*a56) + a11*(2*a25^2*a26 + a22*a26*a55 + 
    2*a22*a25*a56))
 if ((r[1]==2) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(2*a16^2*a22*a25 + 4*a16*(a15*a22*a26 + 2*a12*a25*a26 + a12*a22*a56) + 
  2*a12*a15*(2*a26^2 + a22*a66) + 2*a12^2*(2*a26*a56 + a25*a66) + 
  a11*(2*a25*a26^2 + 2*a22*a26*a56 + a22*a25*a66))
 if ((r[1]==2) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(6*a16^2*a22*a26 + 6*a12*a16*(2*a26^2 + a22*a66) + 
  a26*(2*a11*a26^2 + 6*a12^2*a66 + 3*a11*a22*a66))
 if ((r[1]==2) & (r[2]==2) & (r[3]==4) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(16*a12*a13*a23*a33 + 4*a13^2*(2*a23^2 + a22*a33) + 
  a33*(4*a11*a23^2 + 2*a12^2*a33 + a11*a22*a33))
 if ((r[1]==2) & (r[2]==2) & (r[3]==3) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(4*a12*a14*a23*a33 + 2*a12^2*a33*a34 + 2*a13^2*(2*a23*a24 + a22*a34) + 
  2*a13*(2*a14*a23^2 + a14*a22*a33 + 2*a12*a24*a33 + 4*a12*a23*a34) + 
  a11*(2*a23*a24*a33 + 2*a23^2*a34 + a22*a33*a34))
 if ((r[1]==2) & (r[2]==2) & (r[3]==3) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(4*a12*a15*a23*a33 + 2*a12^2*a33*a35 + 2*a13^2*(2*a23*a25 + a22*a35) + 
  2*a13*(2*a15*a23^2 + a15*a22*a33 + 2*a12*a25*a33 + 4*a12*a23*a35) + 
  a11*(2*a23*a25*a33 + 2*a23^2*a35 + a22*a33*a35))
 if ((r[1]==2) & (r[2]==2) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(4*a12*a16*a23*a33 + 2*a12^2*a33*a36 + 2*a13^2*(2*a23*a26 + a22*a36) + 
  2*a13*(2*a16*a23^2 + a16*a22*a33 + 2*a12*a26*a33 + 4*a12*a23*a36) + 
  a11*(2*a23*a26*a33 + 2*a23^2*a36 + a22*a33*a36))
 if ((r[1]==2) & (r[2]==2) & (r[3]==2) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
2*a11*a24^2*a33 + 2*a14^2*(2*a23^2 + a22*a33) + 8*a11*a23*a24*a34 + 
 4*a12^2*a34^2 + 2*a11*a22*a34^2 + 
 8*a14*(2*a13*a23*a24 + a12*a24*a33 + a13*a22*a34 + 2*a12*a23*a34) + 
 2*a11*a23^2*a44 + 2*a12^2*a33*a44 + a11*a22*a33*a44 + 
 2*a13^2*(2*a24^2 + a22*a44) + 8*a12*a13*(2*a24*a34 + a23*a44)
 if ((r[1]==2) & (r[2]==2) & (r[3]==2) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
4*a12*a15*a24*a33 + 2*a11*a24*a25*a33 + 8*a12*a15*a23*a34 + 
 4*a11*a23*a25*a34 + 4*a11*a23*a24*a35 + 4*a12^2*a34*a35 + 
 2*a11*a22*a34*a35 + 2*a14*(2*a15*a23^2 + 4*a13*a23*a25 + a15*a22*a33 + 
   2*a12*a25*a33 + 2*a13*a22*a35 + 4*a12*a23*a35) + 2*a11*a23^2*a45 + 
 2*a12^2*a33*a45 + a11*a22*a33*a45 + 2*a13^2*(2*a24*a25 + a22*a45) + 
 4*a13*(a15*(2*a23*a24 + a22*a34) + 2*a12*(a25*a34 + a24*a35 + a23*a45))
 if ((r[1]==2) & (r[2]==2) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
4*a12*a16*a24*a33 + 2*a11*a24*a26*a33 + 8*a12*a16*a23*a34 + 
 4*a11*a23*a26*a34 + 4*a11*a23*a24*a36 + 4*a12^2*a34*a36 + 
 2*a11*a22*a34*a36 + 2*a14*(2*a16*a23^2 + 4*a13*a23*a26 + a16*a22*a33 + 
   2*a12*a26*a33 + 2*a13*a22*a36 + 4*a12*a23*a36) + 2*a11*a23^2*a46 + 
 2*a12^2*a33*a46 + a11*a22*a33*a46 + 2*a13^2*(2*a24*a26 + a22*a46) + 
 4*a13*(a16*(2*a23*a24 + a22*a34) + 2*a12*(a26*a34 + a24*a36 + a23*a46))
  if ((r[1]==2) & (r[2]==2) & (r[3]==2) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
2*a11*a25^2*a33 + 2*a15^2*(2*a23^2 + a22*a33) + 8*a11*a23*a25*a35 + 
 4*a12^2*a35^2 + 2*a11*a22*a35^2 + 
 8*a15*(2*a13*a23*a25 + a12*a25*a33 + a13*a22*a35 + 2*a12*a23*a35) + 
 2*a11*a23^2*a55 + 2*a12^2*a33*a55 + a11*a22*a33*a55 + 
 2*a13^2*(2*a25^2 + a22*a55) + 8*a12*a13*(2*a25*a35 + a23*a55)
 if ((r[1]==2) & (r[2]==2) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
4*a12*a16*a25*a33 + 2*a11*a25*a26*a33 + 8*a12*a16*a23*a35 + 
 4*a11*a23*a26*a35 + 4*a11*a23*a25*a36 + 4*a12^2*a35*a36 + 
 2*a11*a22*a35*a36 + 2*a15*(2*a16*a23^2 + 4*a13*a23*a26 + a16*a22*a33 + 
   2*a12*a26*a33 + 2*a13*a22*a36 + 4*a12*a23*a36) + 2*a11*a23^2*a56 + 
 2*a12^2*a33*a56 + a11*a22*a33*a56 + 2*a13^2*(2*a25*a26 + a22*a56) + 
 4*a13*(a16*(2*a23*a25 + a22*a35) + 2*a12*(a26*a35 + a25*a36 + a23*a56))
 if ((r[1]==2) & (r[2]==2) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
2*a11*a26^2*a33 + 2*a16^2*(2*a23^2 + a22*a33) + 8*a11*a23*a26*a36 + 
 4*a12^2*a36^2 + 2*a11*a22*a36^2 + 
 8*a16*(2*a13*a23*a26 + a12*a26*a33 + a13*a22*a36 + 2*a12*a23*a36) + 
 2*a11*a23^2*a66 + 2*a12^2*a33*a66 + a11*a22*a33*a66 + 
 2*a13^2*(2*a26^2 + a22*a66) + 8*a12*a13*(2*a26*a36 + a23*a66)
 if ((r[1]==2) & (r[2]==2) & (r[3]==1) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a14^2*(2*a23*a24 + a22*a34) + 2*a12*(2*a13*a24 + a12*a34)*a44 + 
  2*a14*(2*a13*a24^2 + 4*a12*a24*a34 + a13*a22*a44 + 2*a12*a23*a44) + 
  a11*(2*a24^2*a34 + 2*a23*a24*a44 + a22*a34*a44))
 if ((r[1]==2) & (r[2]==2) & (r[3]==1) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
8*a12*a15*a24*a34 + 4*a11*a24*a25*a34 + 2*a11*a24^2*a35 + 
 2*a14^2*(2*a23*a25 + a22*a35) + 4*a12*a15*a23*a44 + 2*a11*a23*a25*a44 + 
 2*a12^2*a35*a44 + a11*a22*a35*a44 + 4*a11*a23*a24*a45 + 4*a12^2*a34*a45 + 
 2*a11*a22*a34*a45 + 4*a14*(2*a15*a23*a24 + 2*a13*a24*a25 + a15*a22*a34 + 
   2*a12*a25*a34 + 2*a12*a24*a35 + a13*a22*a45 + 2*a12*a23*a45) + 
 2*a13*(2*a15*a24^2 + a15*a22*a44 + 2*a12*a25*a44 + 4*a12*a24*a45)
 if ((r[1]==2) & (r[2]==2) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
8*a12*a16*a24*a34 + 4*a11*a24*a26*a34 + 2*a11*a24^2*a36 + 
 2*a14^2*(2*a23*a26 + a22*a36) + 4*a12*a16*a23*a44 + 2*a11*a23*a26*a44 + 
 2*a12^2*a36*a44 + a11*a22*a36*a44 + 4*a11*a23*a24*a46 + 4*a12^2*a34*a46 + 
 2*a11*a22*a34*a46 + 4*a14*(2*a16*a23*a24 + 2*a13*a24*a26 + a16*a22*a34 + 
   2*a12*a26*a34 + 2*a12*a24*a36 + a13*a22*a46 + 2*a12*a23*a46) + 
 2*a13*(2*a16*a24^2 + a16*a22*a44 + 2*a12*a26*a44 + 4*a12*a24*a46)
 if ((r[1]==2) & (r[2]==2) & (r[3]==1) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
2*a11*a25^2*a34 + 2*a15^2*(2*a23*a24 + a22*a34) + 8*a12*a14*a25*a35 + 
 4*a11*a24*a25*a35 + 4*a11*a23*a25*a45 + 4*a12^2*a35*a45 + 
 2*a11*a22*a35*a45 + 4*a15*(2*a14*a23*a25 + 2*a13*a24*a25 + 2*a12*a25*a34 + 
   a14*a22*a35 + 2*a12*a24*a35 + a13*a22*a45 + 2*a12*a23*a45) + 
 4*a12*a14*a23*a55 + 2*a11*a23*a24*a55 + 2*a12^2*a34*a55 + a11*a22*a34*a55 + 
 2*a13*(2*a14*a25^2 + 4*a12*a25*a45 + a14*a22*a55 + 2*a12*a24*a55)
 if ((r[1]==2) & (r[2]==2) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
4*a13*a16*a24*a25 + 4*a12*a16*a25*a34 + 2*a11*a25*a26*a34 + 
 4*a12*a16*a24*a35 + 2*a11*a24*a26*a35 + 2*a11*a24*a25*a36 + 
 2*a13*a16*a22*a45 + 4*a12*a16*a23*a45 + 4*a12*a13*a26*a45 + 
 2*a11*a23*a26*a45 + 2*a12^2*a36*a45 + a11*a22*a36*a45 + 4*a12*a13*a25*a46 + 
 2*a11*a23*a25*a46 + 2*a12^2*a35*a46 + a11*a22*a35*a46 + 
 2*a15*(2*a16*a23*a24 + 2*a14*a23*a26 + 2*a13*a24*a26 + a16*a22*a34 + 
   2*a12*a26*a34 + a14*a22*a36 + 2*a12*a24*a36 + a13*a22*a46 + 
   2*a12*a23*a46) + 4*a12*a13*a24*a56 + 2*a11*a23*a24*a56 + 2*a12^2*a34*a56 + 
 a11*a22*a34*a56 + 2*a14*(2*a16*a23*a25 + 2*a13*a25*a26 + a16*a22*a35 + 
   2*a12*a26*a35 + 2*a12*a25*a36 + a13*a22*a56 + 2*a12*a23*a56)
 if ((r[1]==2) & (r[2]==2) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
2*a11*a26^2*a34 + 2*a16^2*(2*a23*a24 + a22*a34) + 8*a12*a14*a26*a36 + 
 4*a11*a24*a26*a36 + 4*a11*a23*a26*a46 + 4*a12^2*a36*a46 + 
 2*a11*a22*a36*a46 + 4*a16*(2*a14*a23*a26 + 2*a13*a24*a26 + 2*a12*a26*a34 + 
   a14*a22*a36 + 2*a12*a24*a36 + a13*a22*a46 + 2*a12*a23*a46) + 
 4*a12*a14*a23*a66 + 2*a11*a23*a24*a66 + 2*a12^2*a34*a66 + a11*a22*a34*a66 + 
 2*a13*(2*a14*a26^2 + 4*a12*a26*a46 + a14*a22*a66 + 2*a12*a24*a66)
 if ((r[1]==2) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a15^2*(2*a23*a25 + a22*a35) + 2*a12*(2*a13*a25 + a12*a35)*a55 + 
  2*a15*(2*a13*a25^2 + 4*a12*a25*a35 + a13*a22*a55 + 2*a12*a23*a55) + 
  a11*(2*a25^2*a35 + 2*a23*a25*a55 + a22*a35*a55))
 if ((r[1]==2) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
8*a12*a16*a25*a35 + 4*a11*a25*a26*a35 + 2*a11*a25^2*a36 + 
 2*a15^2*(2*a23*a26 + a22*a36) + 4*a12*a16*a23*a55 + 2*a11*a23*a26*a55 + 
 2*a12^2*a36*a55 + a11*a22*a36*a55 + 4*a11*a23*a25*a56 + 4*a12^2*a35*a56 + 
 2*a11*a22*a35*a56 + 4*a15*(2*a16*a23*a25 + 2*a13*a25*a26 + a16*a22*a35 + 
   2*a12*a26*a35 + 2*a12*a25*a36 + a13*a22*a56 + 2*a12*a23*a56) + 
 2*a13*(2*a16*a25^2 + a16*a22*a55 + 2*a12*a26*a55 + 4*a12*a25*a56)
 if ((r[1]==2) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
2*a11*a26^2*a35 + 2*a16^2*(2*a23*a25 + a22*a35) + 8*a12*a15*a26*a36 + 
 4*a11*a25*a26*a36 + 4*a11*a23*a26*a56 + 4*a12^2*a36*a56 + 
 2*a11*a22*a36*a56 + 4*a16*(2*a15*a23*a26 + 2*a13*a25*a26 + 2*a12*a26*a35 + 
   a15*a22*a36 + 2*a12*a25*a36 + a13*a22*a56 + 2*a12*a23*a56) + 
 4*a12*a15*a23*a66 + 2*a11*a23*a25*a66 + 2*a12^2*a35*a66 + a11*a22*a35*a66 + 
 2*a13*(2*a15*a26^2 + 4*a12*a26*a56 + a15*a22*a66 + 2*a12*a25*a66)
 if ((r[1]==2) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a16^2*(2*a23*a26 + a22*a36) + 2*a12*(2*a13*a26 + a12*a36)*a66 + 
  2*a16*(2*a13*a26^2 + 4*a12*a26*a36 + a13*a22*a66 + 2*a12*a23*a66) + 
  a11*(2*a26^2*a36 + 2*a23*a26*a66 + a22*a36*a66))
 if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==4) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(16*a12*a14*a24*a44 + 4*a14^2*(2*a24^2 + a22*a44) + 
  a44*(4*a11*a24^2 + 2*a12^2*a44 + a11*a22*a44))
 if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==3) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(4*a12*a15*a24*a44 + 2*a12^2*a44*a45 + 2*a14^2*(2*a24*a25 + a22*a45) + 
  2*a14*(2*a15*a24^2 + a15*a22*a44 + 2*a12*a25*a44 + 4*a12*a24*a45) + 
  a11*(2*a24*a25*a44 + 2*a24^2*a45 + a22*a44*a45))
 if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(4*a12*a16*a24*a44 + 2*a12^2*a44*a46 + 2*a14^2*(2*a24*a26 + a22*a46) + 
  2*a14*(2*a16*a24^2 + a16*a22*a44 + 2*a12*a26*a44 + 4*a12*a24*a46) + 
  a11*(2*a24*a26*a44 + 2*a24^2*a46 + a22*a44*a46))
 if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==2) 
& (r[5]==2) & (r[6]==0))
psi <-
2*a11*a25^2*a44 + 2*a15^2*(2*a24^2 + a22*a44) + 8*a11*a24*a25*a45 + 
 4*a12^2*a45^2 + 2*a11*a22*a45^2 + 
 8*a15*(2*a14*a24*a25 + a12*a25*a44 + a14*a22*a45 + 2*a12*a24*a45) + 
 2*a11*a24^2*a55 + 2*a12^2*a44*a55 + a11*a22*a44*a55 + 
 2*a14^2*(2*a25^2 + a22*a55) + 8*a12*a14*(2*a25*a45 + a24*a55)
 if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==1))
psi <-
4*a12*a16*a25*a44 + 2*a11*a25*a26*a44 + 8*a12*a16*a24*a45 + 
 4*a11*a24*a26*a45 + 4*a11*a24*a25*a46 + 4*a12^2*a45*a46 + 
 2*a11*a22*a45*a46 + 2*a15*(2*a16*a24^2 + 4*a14*a24*a26 + a16*a22*a44 + 
   2*a12*a26*a44 + 2*a14*a22*a46 + 4*a12*a24*a46) + 2*a11*a24^2*a56 + 
 2*a12^2*a44*a56 + a11*a22*a44*a56 + 2*a14^2*(2*a25*a26 + a22*a56) + 
 4*a14*(a16*(2*a24*a25 + a22*a45) + 2*a12*(a26*a45 + a25*a46 + a24*a56))
 if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==2))
psi <-
2*a11*a26^2*a44 + 2*a16^2*(2*a24^2 + a22*a44) + 8*a11*a24*a26*a46 + 
 4*a12^2*a46^2 + 2*a11*a22*a46^2 + 
 8*a16*(2*a14*a24*a26 + a12*a26*a44 + a14*a22*a46 + 2*a12*a24*a46) + 
 2*a11*a24^2*a66 + 2*a12^2*a44*a66 + a11*a22*a44*a66 + 
 2*a14^2*(2*a26^2 + a22*a66) + 8*a12*a14*(2*a26*a46 + a24*a66)
 if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a15^2*(2*a24*a25 + a22*a45) + 2*a12*(2*a14*a25 + a12*a45)*a55 + 
  2*a15*(2*a14*a25^2 + 4*a12*a25*a45 + a14*a22*a55 + 2*a12*a24*a55) + 
  a11*(2*a25^2*a45 + 2*a24*a25*a55 + a22*a45*a55))
  if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==1))
psi <-
8*a12*a16*a25*a45 + 4*a11*a25*a26*a45 + 2*a11*a25^2*a46 + 
 2*a15^2*(2*a24*a26 + a22*a46) + 4*a12*a16*a24*a55 + 2*a11*a24*a26*a55 + 
 2*a12^2*a46*a55 + a11*a22*a46*a55 + 4*a11*a24*a25*a56 + 4*a12^2*a45*a56 + 
 2*a11*a22*a45*a56 + 4*a15*(2*a16*a24*a25 + 2*a14*a25*a26 + a16*a22*a45 + 
   2*a12*a26*a45 + 2*a12*a25*a46 + a14*a22*a56 + 2*a12*a24*a56) + 
 2*a14*(2*a16*a25^2 + a16*a22*a55 + 2*a12*a26*a55 + 4*a12*a25*a56)
 if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==2))
psi <-
2*a11*a26^2*a45 + 2*a16^2*(2*a24*a25 + a22*a45) + 8*a12*a15*a26*a46 + 
 4*a11*a25*a26*a46 + 4*a11*a24*a26*a56 + 4*a12^2*a46*a56 + 
 2*a11*a22*a46*a56 + 4*a16*(2*a15*a24*a26 + 2*a14*a25*a26 + 2*a12*a26*a45 + 
   a15*a22*a46 + 2*a12*a25*a46 + a14*a22*a56 + 2*a12*a24*a56) + 
 4*a12*a15*a24*a66 + 2*a11*a24*a25*a66 + 2*a12^2*a45*a66 + a11*a22*a45*a66 + 
 2*a14*(2*a15*a26^2 + 4*a12*a26*a56 + a15*a22*a66 + 2*a12*a25*a66)
 if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a16^2*(2*a24*a26 + a22*a46) + 2*a12*(2*a14*a26 + a12*a46)*a66 + 
  2*a16*(2*a14*a26^2 + 4*a12*a26*a46 + a14*a22*a66 + 2*a12*a24*a66) + 
  a11*(2*a26^2*a46 + 2*a24*a26*a66 + a22*a46*a66))
 if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==4) & (r[6]==0))
psi <-
3*(16*a12*a15*a25*a55 + 4*a15^2*(2*a25^2 + a22*a55) + 
  a55*(4*a11*a25^2 + 2*a12^2*a55 + a11*a22*a55))
 if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==0) & (r[5]==3) & (r[6]==1))
psi <-
3*(4*a12*a16*a25*a55 + 2*a12^2*a55*a56 + 2*a15^2*(2*a25*a26 + a22*a56) + 
  2*a15*(2*a16*a25^2 + a16*a22*a55 + 2*a12*a26*a55 + 4*a12*a25*a56) + 
  a11*(2*a25*a26*a55 + 2*a25^2*a56 + a22*a55*a56))


return(psi)
}






psins.6d.part82 <- function(r, a11, a22, a33, a44, a55, a66, a12, a13, a14, a15,
                           a16, a23, a24, a25, a26, a34, a35, a36, a45, a46, a56)
{
  psi <- NA
if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==0) & (r[5]==2) & (r[6]==2))
psi <-
2*a11*a26^2*a55 + 2*a16^2*(2*a25^2 + a22*a55) + 8*a11*a25*a26*a56 + 
 4*a12^2*a56^2 + 2*a11*a22*a56^2 + 
 8*a16*(2*a15*a25*a26 + a12*a26*a55 + a15*a22*a56 + 2*a12*a25*a56) + 
 2*a11*a25^2*a66 + 2*a12^2*a55*a66 + a11*a22*a55*a66 + 
 2*a15^2*(2*a26^2 + a22*a66) + 8*a12*a15*(2*a26*a56 + a25*a66)
 if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(2*a16^2*(2*a25*a26 + a22*a56) + 2*a12*(2*a15*a26 + a12*a56)*a66 + 
  2*a16*(2*a15*a26^2 + 4*a12*a26*a56 + a15*a22*a66 + 2*a12*a25*a66) + 
  a11*(2*a26^2*a56 + 2*a25*a26*a66 + a22*a56*a66))
 if ((r[1]==2) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==4))
psi <-
3*(16*a12*a16*a26*a66 + 4*a16^2*(2*a26^2 + a22*a66) + 
  a66*(4*a11*a26^2 + 2*a12^2*a66 + a11*a22*a66))
 if ((r[1]==2) & (r[2]==1) & (r[3]==5) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a33*(4*a13^2*a23 + 2*a12*a13*a33 + a11*a23*a33)
 if ((r[1]==2) & (r[2]==1) & (r[3]==4) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(8*a13*a33*(a14*a23 + a12*a34) + 4*a13^2*(a24*a33 + 2*a23*a34) + 
  a33*(2*a12*a14*a33 + a11*a24*a33 + 4*a11*a23*a34))
 if ((r[1]==2) & (r[2]==1) & (r[3]==4) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(8*a13*a33*(a15*a23 + a12*a35) + 4*a13^2*(a25*a33 + 2*a23*a35) + 
  a33*(2*a12*a15*a33 + a11*a25*a33 + 4*a11*a23*a35))
 if ((r[1]==2) & (r[2]==1) & (r[3]==4) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(8*a13*a33*(a16*a23 + a12*a36) + 4*a13^2*(a26*a33 + 2*a23*a36) + 
  a33*(2*a12*a16*a33 + a11*a26*a33 + 4*a11*a23*a36))
 if ((r[1]==2) & (r[2]==1) & (r[3]==3) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a14^2*a23*a33 + 4*a14*(a13*a24*a33 + 2*a13*a23*a34 + a12*a33*a34) + 
  2*a13^2*(2*a24*a34 + a23*a44) + 2*a12*a13*(2*a34^2 + a33*a44) + 
  a11*(2*a24*a33*a34 + 2*a23*a34^2 + a23*a33*a44))
 if ((r[1]==2) & (r[2]==1) & (r[3]==3) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a12*a15*a33*a34 + a11*a25*a33*a34 + a11*a24*a33*a35 + 
  2*a11*a23*a34*a35 + 2*a14*(a15*a23*a33 + a13*a25*a33 + 2*a13*a23*a35 + 
    a12*a33*a35) + a11*a23*a33*a45 + 2*a13^2*(a25*a34 + a24*a35 + a23*a45) + 
  2*a13*(a15*a24*a33 + 2*a15*a23*a34 + 2*a12*a34*a35 + a12*a33*a45))
 if ((r[1]==2) & (r[2]==1) & (r[3]==3) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a12*a16*a33*a34 + a11*a26*a33*a34 + a11*a24*a33*a36 + 
  2*a11*a23*a34*a36 + 2*a14*(a16*a23*a33 + a13*a26*a33 + 2*a13*a23*a36 + 
    a12*a33*a36) + a11*a23*a33*a46 + 2*a13^2*(a26*a34 + a24*a36 + a23*a46) + 
  2*a13*(a16*a24*a33 + 2*a16*a23*a34 + 2*a12*a34*a36 + a12*a33*a46))
 if ((r[1]==2) & (r[2]==1) & (r[3]==3) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a15^2*a23*a33 + 4*a15*(a13*a25*a33 + 2*a13*a23*a35 + a12*a33*a35) + 
  2*a13^2*(2*a25*a35 + a23*a55) + 2*a12*a13*(2*a35^2 + a33*a55) + 
  a11*(2*a25*a33*a35 + 2*a23*a35^2 + a23*a33*a55))
 if ((r[1]==2) & (r[2]==1) & (r[3]==3) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(2*a12*a16*a33*a35 + a11*a26*a33*a35 + a11*a25*a33*a36 + 
  2*a11*a23*a35*a36 + 2*a15*(a16*a23*a33 + a13*a26*a33 + 2*a13*a23*a36 + 
    a12*a33*a36) + a11*a23*a33*a56 + 2*a13^2*(a26*a35 + a25*a36 + a23*a56) + 
  2*a13*(a16*a25*a33 + 2*a16*a23*a35 + 2*a12*a35*a36 + a12*a33*a56))
 if ((r[1]==2) & (r[2]==1) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a16^2*a23*a33 + 4*a16*(a13*a26*a33 + 2*a13*a23*a36 + a12*a33*a36) + 
  2*a13^2*(2*a26*a36 + a23*a66) + 2*a12*a13*(2*a36^2 + a33*a66) + 
  a11*(2*a26*a33*a36 + 2*a23*a36^2 + a23*a33*a66))
 if ((r[1]==2) & (r[2]==1) & (r[3]==2) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a14^2*(a24*a33 + 2*a23*a34) + 2*a13*(a13*a24 + 2*a12*a34)*a44 + 
  2*a14*(4*a13*a24*a34 + 2*a12*a34^2 + 2*a13*a23*a44 + a12*a33*a44) + 
  a11*(2*a24*a34^2 + a24*a33*a44 + 2*a23*a34*a44))
 if ((r[1]==2) & (r[2]==1) & (r[3]==2) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
4*a12*a15*a34^2 + 2*a11*a25*a34^2 + 4*a11*a24*a34*a35 + 
 2*a14^2*(a25*a33 + 2*a23*a35) + 2*a12*a15*a33*a44 + a11*a25*a33*a44 + 
 2*a11*a23*a35*a44 + 2*a11*a24*a33*a45 + 4*a11*a23*a34*a45 + 
 2*a13^2*(a25*a44 + 2*a24*a45) + 4*a14*(a15*a24*a33 + 2*a15*a23*a34 + 
   2*a13*a25*a34 + 2*a13*a24*a35 + 2*a12*a34*a35 + 2*a13*a23*a45 + 
   a12*a33*a45) + 4*a13*(2*a15*a24*a34 + a15*a23*a44 + a12*a35*a44 + 
   2*a12*a34*a45)
 if ((r[1]==2) & (r[2]==1) & (r[3]==2) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
4*a12*a16*a34^2 + 2*a11*a26*a34^2 + 4*a11*a24*a34*a36 + 
 2*a14^2*(a26*a33 + 2*a23*a36) + 2*a12*a16*a33*a44 + a11*a26*a33*a44 + 
 2*a11*a23*a36*a44 + 2*a11*a24*a33*a46 + 4*a11*a23*a34*a46 + 
 2*a13^2*(a26*a44 + 2*a24*a46) + 4*a14*(a16*a24*a33 + 2*a16*a23*a34 + 
   2*a13*a26*a34 + 2*a13*a24*a36 + 2*a12*a34*a36 + 2*a13*a23*a46 + 
   a12*a33*a46) + 4*a13*(2*a16*a24*a34 + a16*a23*a44 + a12*a36*a44 + 
   2*a12*a34*a46)
 if ((r[1]==2) & (r[2]==1) & (r[3]==2) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
2*a15^2*(a24*a33 + 2*a23*a34) + 4*a11*a25*a34*a35 + 4*a12*a14*a35^2 + 
 2*a11*a24*a35^2 + 2*a11*a25*a33*a45 + 4*a11*a23*a35*a45 + 
 4*a15*(a14*a25*a33 + 2*a13*a25*a34 + 2*a14*a23*a35 + 2*a13*a24*a35 + 
   2*a12*a34*a35 + 2*a13*a23*a45 + a12*a33*a45) + 2*a12*a14*a33*a55 + 
 a11*a24*a33*a55 + 2*a11*a23*a34*a55 + 2*a13^2*(2*a25*a45 + a24*a55) + 
 4*a13*(2*a14*a25*a35 + 2*a12*a35*a45 + a14*a23*a55 + a12*a34*a55)
 if ((r[1]==2) & (r[2]==1) & (r[3]==2) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
4*a13*a16*a25*a34 + 4*a13*a16*a24*a35 + 4*a12*a16*a34*a35 + 
 2*a11*a26*a34*a35 + 2*a11*a25*a34*a36 + 2*a11*a24*a35*a36 + 
 4*a13*a16*a23*a45 + 2*a13^2*a26*a45 + 2*a12*a16*a33*a45 + a11*a26*a33*a45 + 
 4*a12*a13*a36*a45 + 2*a11*a23*a36*a45 + 2*a13^2*a25*a46 + a11*a25*a33*a46 + 
 4*a12*a13*a35*a46 + 2*a11*a23*a35*a46 + 
 2*a15*(a16*a24*a33 + a14*a26*a33 + 2*a16*a23*a34 + 2*a13*a26*a34 + 
   2*a14*a23*a36 + 2*a13*a24*a36 + 2*a12*a34*a36 + 2*a13*a23*a46 + 
   a12*a33*a46) + 2*a13^2*a24*a56 + a11*a24*a33*a56 + 4*a12*a13*a34*a56 + 
 2*a11*a23*a34*a56 + 2*a14*(a16*a25*a33 + 2*a16*a23*a35 + 2*a13*a26*a35 + 
   2*a13*a25*a36 + 2*a12*a35*a36 + 2*a13*a23*a56 + a12*a33*a56)
 if ((r[1]==2) & (r[2]==1) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
2*a16^2*(a24*a33 + 2*a23*a34) + 4*a11*a26*a34*a36 + 4*a12*a14*a36^2 + 
 2*a11*a24*a36^2 + 2*a11*a26*a33*a46 + 4*a11*a23*a36*a46 + 
 4*a16*(a14*a26*a33 + 2*a13*a26*a34 + 2*a14*a23*a36 + 2*a13*a24*a36 + 
   2*a12*a34*a36 + 2*a13*a23*a46 + a12*a33*a46) + 2*a12*a14*a33*a66 + 
 a11*a24*a33*a66 + 2*a11*a23*a34*a66 + 2*a13^2*(2*a26*a46 + a24*a66) + 
 4*a13*(2*a14*a26*a36 + 2*a12*a36*a46 + a14*a23*a66 + a12*a34*a66)
  if ((r[1]==2) & (r[2]==1) & (r[3]==2) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a15^2*(a25*a33 + 2*a23*a35) + 2*a13*(a13*a25 + 2*a12*a35)*a55 + 
  2*a15*(4*a13*a25*a35 + 2*a12*a35^2 + 2*a13*a23*a55 + a12*a33*a55) + 
  a11*(2*a25*a35^2 + a25*a33*a55 + 2*a23*a35*a55))
 if ((r[1]==2) & (r[2]==1) & (r[3]==2) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
4*a12*a16*a35^2 + 2*a11*a26*a35^2 + 4*a11*a25*a35*a36 + 
 2*a15^2*(a26*a33 + 2*a23*a36) + 2*a12*a16*a33*a55 + a11*a26*a33*a55 + 
 2*a11*a23*a36*a55 + 2*a11*a25*a33*a56 + 4*a11*a23*a35*a56 + 
 2*a13^2*(a26*a55 + 2*a25*a56) + 4*a15*(a16*a25*a33 + 2*a16*a23*a35 + 
   2*a13*a26*a35 + 2*a13*a25*a36 + 2*a12*a35*a36 + 2*a13*a23*a56 + 
   a12*a33*a56) + 4*a13*(2*a16*a25*a35 + a16*a23*a55 + a12*a36*a55 + 
   2*a12*a35*a56)
 if ((r[1]==2) & (r[2]==1) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
2*a16^2*(a25*a33 + 2*a23*a35) + 4*a11*a26*a35*a36 + 4*a12*a15*a36^2 + 
 2*a11*a25*a36^2 + 2*a11*a26*a33*a56 + 4*a11*a23*a36*a56 + 
 4*a16*(a15*a26*a33 + 2*a13*a26*a35 + 2*a15*a23*a36 + 2*a13*a25*a36 + 
   2*a12*a35*a36 + 2*a13*a23*a56 + a12*a33*a56) + 2*a12*a15*a33*a66 + 
 a11*a25*a33*a66 + 2*a11*a23*a35*a66 + 2*a13^2*(2*a26*a56 + a25*a66) + 
 4*a13*(2*a15*a26*a36 + 2*a12*a36*a56 + a15*a23*a66 + a12*a35*a66)
 if ((r[1]==2) & (r[2]==1) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a16^2*(a26*a33 + 2*a23*a36) + 2*a13*(a13*a26 + 2*a12*a36)*a66 + 
  2*a16*(4*a13*a26*a36 + 2*a12*a36^2 + 2*a13*a23*a66 + a12*a33*a66) + 
  a11*(2*a26*a36^2 + a26*a33*a66 + 2*a23*a36*a66))
 if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==4) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(8*a14*(a13*a24 + a12*a34)*a44 + 4*a14^2*(2*a24*a34 + a23*a44) + 
  a44*(4*a11*a24*a34 + 2*a12*a13*a44 + a11*a23*a44))
 if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==3) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a12*a15*a34*a44 + a11*a25*a34*a44 + a11*a24*a35*a44 + 
  2*a11*a24*a34*a45 + a11*a23*a44*a45 + 2*a13*a44*(a15*a24 + a12*a45) + 
  2*a14^2*(a25*a34 + a24*a35 + a23*a45) + 
  2*a14*(2*a15*a24*a34 + a15*a23*a44 + a13*a25*a44 + a12*a35*a44 + 
    2*a13*a24*a45 + 2*a12*a34*a45))
 if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==3) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a12*a16*a34*a44 + a11*a26*a34*a44 + a11*a24*a36*a44 + 
  2*a11*a24*a34*a46 + a11*a23*a44*a46 + 2*a13*a44*(a16*a24 + a12*a46) + 
  2*a14^2*(a26*a34 + a24*a36 + a23*a46) + 
  2*a14*(2*a16*a24*a34 + a16*a23*a44 + a13*a26*a44 + a12*a36*a44 + 
    2*a13*a24*a46 + 2*a12*a34*a46))
 if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==2) 
& (r[5]==2) & (r[6]==0))
psi <-
2*a11*a25*a35*a44 + 2*a15^2*(2*a24*a34 + a23*a44) + 4*a11*a25*a34*a45 + 
 4*a11*a24*a35*a45 + 4*a12*a13*a45^2 + 2*a11*a23*a45^2 + 
 4*a15*(a13*a25*a44 + a12*a35*a44 + 2*a13*a24*a45 + 2*a12*a34*a45 + 
   2*a14*(a25*a34 + a24*a35 + a23*a45)) + 2*a11*a24*a34*a55 + 
 2*a12*a13*a44*a55 + a11*a23*a44*a55 + 2*a14^2*(2*a25*a35 + a23*a55) + 
 4*a14*(2*a13*a25*a45 + 2*a12*a35*a45 + a13*a24*a55 + a12*a34*a55)
 if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==2) 
& (r[5]==1) & (r[6]==1))
psi <-
2*a13*a16*a25*a44 + 2*a12*a16*a35*a44 + a11*a26*a35*a44 + a11*a25*a36*a44 + 
 4*a13*a16*a24*a45 + 4*a12*a16*a34*a45 + 2*a11*a26*a34*a45 + 
 2*a11*a24*a36*a45 + 2*a11*a25*a34*a46 + 2*a11*a24*a35*a46 + 
 4*a12*a13*a45*a46 + 2*a11*a23*a45*a46 + 
 2*a15*(a13*a26*a44 + a12*a36*a44 + a16*(2*a24*a34 + a23*a44) + 
   2*a13*a24*a46 + 2*a12*a34*a46 + 2*a14*(a26*a34 + a24*a36 + a23*a46)) + 
 2*a11*a24*a34*a56 + 2*a12*a13*a44*a56 + a11*a23*a44*a56 + 
 2*a14^2*(a26*a35 + a25*a36 + a23*a56) + 
 4*a14*(a16*(a25*a34 + a24*a35 + a23*a45) + 
   a13*(a26*a45 + a25*a46 + a24*a56) + a12*(a36*a45 + a35*a46 + a34*a56))
 if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==2))
psi <-
2*a11*a26*a36*a44 + 2*a16^2*(2*a24*a34 + a23*a44) + 4*a11*a26*a34*a46 + 
 4*a11*a24*a36*a46 + 4*a12*a13*a46^2 + 2*a11*a23*a46^2 + 
 4*a16*(a13*a26*a44 + a12*a36*a44 + 2*a13*a24*a46 + 2*a12*a34*a46 + 
   2*a14*(a26*a34 + a24*a36 + a23*a46)) + 2*a11*a24*a34*a66 + 
 2*a12*a13*a44*a66 + a11*a23*a44*a66 + 2*a14^2*(2*a26*a36 + a23*a66) + 
 4*a14*(2*a13*a26*a46 + 2*a12*a36*a46 + a13*a24*a66 + a12*a34*a66)
 if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a15^2*(a25*a34 + a24*a35 + a23*a45) + 
  2*(a13*a14*a25 + a12*a14*a35 + a12*a13*a45)*a55 + 
  2*a15*(2*a14*a25*a35 + 2*a13*a25*a45 + 2*a12*a35*a45 + a14*a23*a55 + 
    a13*a24*a55 + a12*a34*a55) + a11*(2*a25*a35*a45 + a25*a34*a55 + 
    a24*a35*a55 + a23*a45*a55))
 if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==2) & (r[6]==1))
psi <-
4*a13*a16*a25*a45 + 4*a12*a16*a35*a45 + 2*a11*a26*a35*a45 + 
 2*a11*a25*a36*a45 + 2*a11*a25*a35*a46 + 
 2*a15^2*(a26*a34 + a24*a36 + a23*a46) + 2*a13*a16*a24*a55 + 
 2*a12*a16*a34*a55 + a11*a26*a34*a55 + a11*a24*a36*a55 + 2*a12*a13*a46*a55 + 
 a11*a23*a46*a55 + 2*a11*a25*a34*a56 + 2*a11*a24*a35*a56 + 
 4*a12*a13*a45*a56 + 2*a11*a23*a45*a56 + 
 4*a15*(a14*a26*a35 + a14*a25*a36 + a13*a26*a45 + a12*a36*a45 + 
   a16*(a25*a34 + a24*a35 + a23*a45) + a13*a25*a46 + a12*a35*a46 + 
   a14*a23*a56 + a13*a24*a56 + a12*a34*a56) + 
 2*a14*(2*a16*a25*a35 + a16*a23*a55 + a13*a26*a55 + a12*a36*a55 + 
   2*a13*a25*a56 + 2*a12*a35*a56)
 if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==2))
psi <-
2*a11*a26*a36*a45 + 2*a16^2*(a25*a34 + a24*a35 + a23*a45) + 
 4*a13*a15*a26*a46 + 2*a11*a26*a35*a46 + 4*a12*a15*a36*a46 + 
 2*a11*a25*a36*a46 + 2*a11*a26*a34*a56 + 2*a11*a24*a36*a56 + 
 4*a12*a13*a46*a56 + 2*a11*a23*a46*a56 + 
 4*a16*(a14*a26*a35 + a14*a25*a36 + a13*a26*a45 + a12*a36*a45 + a13*a25*a46 + 
   a12*a35*a46 + a15*(a26*a34 + a24*a36 + a23*a46) + a14*a23*a56 + 
   a13*a24*a56 + a12*a34*a56) + 2*a13*a15*a24*a66 + 2*a12*a15*a34*a66 + 
 a11*a25*a34*a66 + a11*a24*a35*a66 + 2*a12*a13*a45*a66 + a11*a23*a45*a66 + 
 2*a14*(2*a15*a26*a36 + 2*a13*a26*a56 + 2*a12*a36*a56 + a15*a23*a66 + 
   a13*a25*a66 + a12*a35*a66)
 if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a16^2*(a26*a34 + a24*a36 + a23*a46) + 
  2*(a13*a14*a26 + a12*a14*a36 + a12*a13*a46)*a66 + 
  2*a16*(2*a14*a26*a36 + 2*a13*a26*a46 + 2*a12*a36*a46 + a14*a23*a66 + 
    a13*a24*a66 + a12*a34*a66) + a11*(2*a26*a36*a46 + a26*a34*a66 + 
    a24*a36*a66 + a23*a46*a66))
 if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==4) & (r[6]==0))
psi <-
3*(8*a15*(a13*a25 + a12*a35)*a55 + 4*a15^2*(2*a25*a35 + a23*a55) + 
  a55*(4*a11*a25*a35 + 2*a12*a13*a55 + a11*a23*a55))
 if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==3) & (r[6]==1))
psi <-
3*(2*a12*a16*a35*a55 + a11*a26*a35*a55 + a11*a25*a36*a55 + 
  2*a11*a25*a35*a56 + a11*a23*a55*a56 + 2*a13*a55*(a16*a25 + a12*a56) + 
  2*a15^2*(a26*a35 + a25*a36 + a23*a56) + 
  2*a15*(2*a16*a25*a35 + a16*a23*a55 + a13*a26*a55 + a12*a36*a55 + 
    2*a13*a25*a56 + 2*a12*a35*a56))
 if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==2))
psi <-
2*a11*a26*a36*a55 + 2*a16^2*(2*a25*a35 + a23*a55) + 4*a11*a26*a35*a56 + 
 4*a11*a25*a36*a56 + 4*a12*a13*a56^2 + 2*a11*a23*a56^2 + 
 4*a16*(a13*a26*a55 + a12*a36*a55 + 2*a13*a25*a56 + 2*a12*a35*a56 + 
   2*a15*(a26*a35 + a25*a36 + a23*a56)) + 2*a11*a25*a35*a66 + 
 2*a12*a13*a55*a66 + a11*a23*a55*a66 + 2*a15^2*(2*a26*a36 + a23*a66) + 
 4*a15*(2*a13*a26*a56 + 2*a12*a36*a56 + a13*a25*a66 + a12*a35*a66)
 if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(2*a16^2*(a26*a35 + a25*a36 + a23*a56) + 
  2*(a13*a15*a26 + a12*a15*a36 + a12*a13*a56)*a66 + 
  2*a16*(2*a15*a26*a36 + 2*a13*a26*a56 + 2*a12*a36*a56 + a15*a23*a66 + 
    a13*a25*a66 + a12*a35*a66) + a11*(2*a26*a36*a56 + a26*a35*a66 + 
    a25*a36*a66 + a23*a56*a66))
 if ((r[1]==2) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==4))
psi <-
3*(8*a16*(a13*a26 + a12*a36)*a66 + 4*a16^2*(2*a26*a36 + a23*a66) + 
  a66*(4*a11*a26*a36 + 2*a12*a13*a66 + a11*a23*a66))
  if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==5) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a44*(4*a14^2*a24 + 2*a12*a14*a44 + a11*a24*a44)
 if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==4) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(8*a14*a44*(a15*a24 + a12*a45) + 4*a14^2*(a25*a44 + 2*a24*a45) + 
  a44*(2*a12*a15*a44 + a11*a25*a44 + 4*a11*a24*a45))
 if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==4) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(8*a14*a44*(a16*a24 + a12*a46) + 4*a14^2*(a26*a44 + 2*a24*a46) + 
  a44*(2*a12*a16*a44 + a11*a26*a44 + 4*a11*a24*a46))
 if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==3) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a15^2*a24*a44 + 4*a15*(a14*a25*a44 + 2*a14*a24*a45 + a12*a44*a45) + 
  2*a14^2*(2*a25*a45 + a24*a55) + 2*a12*a14*(2*a45^2 + a44*a55) + 
  a11*(2*a25*a44*a45 + 2*a24*a45^2 + a24*a44*a55))
 if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==3) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(2*a12*a16*a44*a45 + a11*a26*a44*a45 + a11*a25*a44*a46 + 
  2*a11*a24*a45*a46 + 2*a15*(a16*a24*a44 + a14*a26*a44 + 2*a14*a24*a46 + 
    a12*a44*a46) + a11*a24*a44*a56 + 2*a14^2*(a26*a45 + a25*a46 + a24*a56) + 
  2*a14*(a16*a25*a44 + 2*a16*a24*a45 + 2*a12*a45*a46 + a12*a44*a56))
 if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a16^2*a24*a44 + 4*a16*(a14*a26*a44 + 2*a14*a24*a46 + a12*a44*a46) + 
  2*a14^2*(2*a26*a46 + a24*a66) + 2*a12*a14*(2*a46^2 + a44*a66) + 
  a11*(2*a26*a44*a46 + 2*a24*a46^2 + a24*a44*a66))
 if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a15^2*(a25*a44 + 2*a24*a45) + 2*a14*(a14*a25 + 2*a12*a45)*a55 + 
  2*a15*(4*a14*a25*a45 + 2*a12*a45^2 + 2*a14*a24*a55 + a12*a44*a55) + 
  a11*(2*a25*a45^2 + a25*a44*a55 + 2*a24*a45*a55))
 if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==2) & (r[6]==1))
psi <-
4*a12*a16*a45^2 + 2*a11*a26*a45^2 + 4*a11*a25*a45*a46 + 
 2*a15^2*(a26*a44 + 2*a24*a46) + 2*a12*a16*a44*a55 + a11*a26*a44*a55 + 
 2*a11*a24*a46*a55 + 2*a11*a25*a44*a56 + 4*a11*a24*a45*a56 + 
 2*a14^2*(a26*a55 + 2*a25*a56) + 4*a15*(a16*a25*a44 + 2*a16*a24*a45 + 
   2*a14*a26*a45 + 2*a14*a25*a46 + 2*a12*a45*a46 + 2*a14*a24*a56 + 
   a12*a44*a56) + 4*a14*(2*a16*a25*a45 + a16*a24*a55 + a12*a46*a55 + 
   2*a12*a45*a56)
 if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==2))
psi <-
2*a16^2*(a25*a44 + 2*a24*a45) + 4*a11*a26*a45*a46 + 4*a12*a15*a46^2 + 
 2*a11*a25*a46^2 + 2*a11*a26*a44*a56 + 4*a11*a24*a46*a56 + 
 4*a16*(a15*a26*a44 + 2*a14*a26*a45 + 2*a15*a24*a46 + 2*a14*a25*a46 + 
   2*a12*a45*a46 + 2*a14*a24*a56 + a12*a44*a56) + 2*a12*a15*a44*a66 + 
 a11*a25*a44*a66 + 2*a11*a24*a45*a66 + 2*a14^2*(2*a26*a56 + a25*a66) + 
 4*a14*(2*a15*a26*a46 + 2*a12*a46*a56 + a15*a24*a66 + a12*a45*a66)
 if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a16^2*(a26*a44 + 2*a24*a46) + 2*a14*(a14*a26 + 2*a12*a46)*a66 + 
  2*a16*(4*a14*a26*a46 + 2*a12*a46^2 + 2*a14*a24*a66 + a12*a44*a66) + 
  a11*(2*a26*a46^2 + a26*a44*a66 + 2*a24*a46*a66))
 if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==4) & (r[6]==0))
psi <-
3*(8*a15*(a14*a25 + a12*a45)*a55 + 4*a15^2*(2*a25*a45 + a24*a55) + 
  a55*(4*a11*a25*a45 + 2*a12*a14*a55 + a11*a24*a55))
 if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==3) & (r[6]==1))
psi <-
3*(2*a12*a16*a45*a55 + a11*a26*a45*a55 + a11*a25*a46*a55 + 
  2*a11*a25*a45*a56 + a11*a24*a55*a56 + 2*a14*a55*(a16*a25 + a12*a56) + 
  2*a15^2*(a26*a45 + a25*a46 + a24*a56) + 
  2*a15*(2*a16*a25*a45 + a16*a24*a55 + a14*a26*a55 + a12*a46*a55 + 
    2*a14*a25*a56 + 2*a12*a45*a56))
 if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==2))
psi <-
2*a11*a26*a46*a55 + 2*a16^2*(2*a25*a45 + a24*a55) + 4*a11*a26*a45*a56 + 
 4*a11*a25*a46*a56 + 4*a12*a14*a56^2 + 2*a11*a24*a56^2 + 
 4*a16*(a14*a26*a55 + a12*a46*a55 + 2*a14*a25*a56 + 2*a12*a45*a56 + 
   2*a15*(a26*a45 + a25*a46 + a24*a56)) + 2*a11*a25*a45*a66 + 
 2*a12*a14*a55*a66 + a11*a24*a55*a66 + 2*a15^2*(2*a26*a46 + a24*a66) + 
 4*a15*(2*a14*a26*a56 + 2*a12*a46*a56 + a14*a25*a66 + a12*a45*a66)
 if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(2*a16^2*(a26*a45 + a25*a46 + a24*a56) + 
  2*(a14*a15*a26 + a12*a15*a46 + a12*a14*a56)*a66 + 
  2*a16*(2*a15*a26*a46 + 2*a14*a26*a56 + 2*a12*a46*a56 + a15*a24*a66 + 
    a14*a25*a66 + a12*a45*a66) + a11*(2*a26*a46*a56 + a26*a45*a66 + 
    a25*a46*a66 + a24*a56*a66))
 if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==4))
psi <-
3*(8*a16*(a14*a26 + a12*a46)*a66 + 4*a16^2*(2*a26*a46 + a24*a66) + 
  a66*(4*a11*a26*a46 + 2*a12*a14*a66 + a11*a24*a66))
 if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==5) & (r[6]==0))
psi <-
15*a55*(4*a15^2*a25 + 2*a12*a15*a55 + a11*a25*a55)
 if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==4) & (r[6]==1))
psi <-
3*(8*a15*a55*(a16*a25 + a12*a56) + 4*a15^2*(a26*a55 + 2*a25*a56) + 
  a55*(2*a12*a16*a55 + a11*a26*a55 + 4*a11*a25*a56))
 if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==2))
psi <-
3*(2*a16^2*a25*a55 + 4*a16*(a15*a26*a55 + 2*a15*a25*a56 + a12*a55*a56) + 
  2*a15^2*(2*a26*a56 + a25*a66) + 2*a12*a15*(2*a56^2 + a55*a66) + 
  a11*(2*a26*a55*a56 + 2*a25*a56^2 + a25*a55*a66))
 if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==3))
psi <-
3*(2*a16^2*(a26*a55 + 2*a25*a56) + 2*a15*(a15*a26 + 2*a12*a56)*a66 + 
  2*a16*(4*a15*a26*a56 + 2*a12*a56^2 + 2*a15*a25*a66 + a12*a55*a66) + 
  a11*(2*a26*a56^2 + a26*a55*a66 + 2*a25*a56*a66))
 if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==4))
psi <-
3*(8*a16*(a15*a26 + a12*a56)*a66 + 4*a16^2*(2*a26*a56 + a25*a66) + 
  a66*(4*a11*a26*a56 + 2*a12*a15*a66 + a11*a25*a66))
 if ((r[1]==2) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==5))
psi <-
15*a66*(4*a16^2*a26 + 2*a12*a16*a66 + a11*a26*a66)
 if ((r[1]==2) & (r[2]==0) & (r[3]==6) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a33^2*(6*a13^2 + a11*a33)
 if ((r[1]==2) & (r[2]==0) & (r[3]==5) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a33*(2*a13*a14*a33 + 4*a13^2*a34 + a11*a33*a34)
 if ((r[1]==2) & (r[2]==0) & (r[3]==5) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a33*(2*a13*a15*a33 + 4*a13^2*a35 + a11*a33*a35)
 if ((r[1]==2) & (r[2]==0) & (r[3]==5) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a33*(2*a13*a16*a33 + 4*a13^2*a36 + a11*a33*a36)
 if ((r[1]==2) & (r[2]==0) & (r[3]==4) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a14^2*a33^2 + 16*a13*a14*a33*a34 + 4*a13^2*(2*a34^2 + a33*a44) + 
  a11*a33*(4*a34^2 + a33*a44))
 if ((r[1]==2) & (r[2]==0) & (r[3]==4) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(8*a13*a15*a33*a34 + 2*a14*a33*(a15*a33 + 4*a13*a35) + 
  4*a13^2*(2*a34*a35 + a33*a45) + a11*a33*(4*a34*a35 + a33*a45))
 if ((r[1]==2) & (r[2]==0) & (r[3]==4) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(8*a13*a16*a33*a34 + 2*a14*a33*(a16*a33 + 4*a13*a36) + 
  4*a13^2*(2*a34*a36 + a33*a46) + a11*a33*(4*a34*a36 + a33*a46))
 if ((r[1]==2) & (r[2]==0) & (r[3]==4) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a15^2*a33^2 + 16*a13*a15*a33*a35 + 4*a13^2*(2*a35^2 + a33*a55) + 
  a11*a33*(4*a35^2 + a33*a55))
  if ((r[1]==2) & (r[2]==0) & (r[3]==4) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(8*a13*a16*a33*a35 + 2*a15*a33*(a16*a33 + 4*a13*a36) + 
  4*a13^2*(2*a35*a36 + a33*a56) + a11*a33*(4*a35*a36 + a33*a56))
 if ((r[1]==2) & (r[2]==0) & (r[3]==4) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a16^2*a33^2 + 16*a13*a16*a33*a36 + 4*a13^2*(2*a36^2 + a33*a66) + 
  a11*a33*(4*a36^2 + a33*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==3) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(6*a14^2*a33*a34 + 6*a13*a14*(2*a34^2 + a33*a44) + 
  a34*(2*a11*a34^2 + 6*a13^2*a44 + 3*a11*a33*a44))
 if ((r[1]==2) & (r[2]==0) & (r[3]==3) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a14^2*a33*a35 + 2*a13*a15*(2*a34^2 + a33*a44) + 
  4*a14*(a15*a33*a34 + 2*a13*a34*a35 + a13*a33*a45) + 
  2*a13^2*(a35*a44 + 2*a34*a45) + a11*(2*a34^2*a35 + a33*a35*a44 + 
    2*a33*a34*a45))
 if ((r[1]==2) & (r[2]==0) & (r[3]==3) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a14^2*a33*a36 + 2*a13*a16*(2*a34^2 + a33*a44) + 
  4*a14*(a16*a33*a34 + 2*a13*a34*a36 + a13*a33*a46) + 
  2*a13^2*(a36*a44 + 2*a34*a46) + a11*(2*a34^2*a36 + a33*a36*a44 + 
    2*a33*a34*a46))
 if ((r[1]==2) & (r[2]==0) & (r[3]==3) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a15^2*a33*a34 + 4*a15*(a14*a33*a35 + 2*a13*a34*a35 + a13*a33*a45) + 
  2*a13*a14*(2*a35^2 + a33*a55) + 2*a13^2*(2*a35*a45 + a34*a55) + 
  a11*(2*a34*a35^2 + 2*a33*a35*a45 + a33*a34*a55))
 if ((r[1]==2) & (r[2]==0) & (r[3]==3) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(4*a13*a16*a34*a35 + 2*a11*a34*a35*a36 + 2*a13*a16*a33*a45 + 
  2*a13^2*a36*a45 + a11*a33*a36*a45 + 2*a13^2*a35*a46 + a11*a33*a35*a46 + 
  2*a15*(a16*a33*a34 + a14*a33*a36 + 2*a13*a34*a36 + a13*a33*a46) + 
  2*a13^2*a34*a56 + a11*a33*a34*a56 + 
  2*a14*(a16*a33*a35 + 2*a13*a35*a36 + a13*a33*a56))
 if ((r[1]==2) & (r[2]==0) & (r[3]==3) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a16^2*a33*a34 + 4*a16*(a14*a33*a36 + 2*a13*a34*a36 + a13*a33*a46) + 
  2*a13*a14*(2*a36^2 + a33*a66) + 2*a13^2*(2*a36*a46 + a34*a66) + 
  a11*(2*a34*a36^2 + 2*a33*a36*a46 + a33*a34*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(6*a15^2*a33*a35 + 6*a13*a15*(2*a35^2 + a33*a55) + 
  a35*(2*a11*a35^2 + 6*a13^2*a55 + 3*a11*a33*a55))
 if ((r[1]==2) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(2*a15^2*a33*a36 + 2*a13*a16*(2*a35^2 + a33*a55) + 
  4*a15*(a16*a33*a35 + 2*a13*a35*a36 + a13*a33*a56) + 
  2*a13^2*(a36*a55 + 2*a35*a56) + a11*(2*a35^2*a36 + a33*a36*a55 + 
    2*a33*a35*a56))
 if ((r[1]==2) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(2*a16^2*a33*a35 + 4*a16*(a15*a33*a36 + 2*a13*a35*a36 + a13*a33*a56) + 
  2*a13*a15*(2*a36^2 + a33*a66) + 2*a13^2*(2*a36*a56 + a35*a66) + 
  a11*(2*a35*a36^2 + 2*a33*a36*a56 + a33*a35*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(6*a16^2*a33*a36 + 6*a13*a16*(2*a36^2 + a33*a66) + 
  a36*(2*a11*a36^2 + 6*a13^2*a66 + 3*a11*a33*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==4) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(16*a13*a14*a34*a44 + 4*a14^2*(2*a34^2 + a33*a44) + 
  a44*(4*a11*a34^2 + 2*a13^2*a44 + a11*a33*a44))
 if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==3) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(4*a13*a15*a34*a44 + 2*a13^2*a44*a45 + 2*a14^2*(2*a34*a35 + a33*a45) + 
  2*a14*(2*a15*a34^2 + a15*a33*a44 + 2*a13*a35*a44 + 4*a13*a34*a45) + 
  a11*(2*a34*a35*a44 + 2*a34^2*a45 + a33*a44*a45))
 if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==3) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(4*a13*a16*a34*a44 + 2*a13^2*a44*a46 + 2*a14^2*(2*a34*a36 + a33*a46) + 
  2*a14*(2*a16*a34^2 + a16*a33*a44 + 2*a13*a36*a44 + 4*a13*a34*a46) + 
  a11*(2*a34*a36*a44 + 2*a34^2*a46 + a33*a44*a46))
 if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==2) 
& (r[5]==2) & (r[6]==0))
psi <-
2*a11*a35^2*a44 + 2*a15^2*(2*a34^2 + a33*a44) + 8*a11*a34*a35*a45 + 
 4*a13^2*a45^2 + 2*a11*a33*a45^2 + 
 8*a15*(2*a14*a34*a35 + a13*a35*a44 + a14*a33*a45 + 2*a13*a34*a45) + 
 2*a11*a34^2*a55 + 2*a13^2*a44*a55 + a11*a33*a44*a55 + 
 2*a14^2*(2*a35^2 + a33*a55) + 8*a13*a14*(2*a35*a45 + a34*a55)
 if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==2) 
& (r[5]==1) & (r[6]==1))
psi <-
4*a13*a16*a35*a44 + 2*a11*a35*a36*a44 + 8*a13*a16*a34*a45 + 
 4*a11*a34*a36*a45 + 4*a11*a34*a35*a46 + 4*a13^2*a45*a46 + 
 2*a11*a33*a45*a46 + 2*a15*(2*a16*a34^2 + 4*a14*a34*a36 + a16*a33*a44 + 
   2*a13*a36*a44 + 2*a14*a33*a46 + 4*a13*a34*a46) + 2*a11*a34^2*a56 + 
 2*a13^2*a44*a56 + a11*a33*a44*a56 + 2*a14^2*(2*a35*a36 + a33*a56) + 
 4*a14*(a16*(2*a34*a35 + a33*a45) + 2*a13*(a36*a45 + a35*a46 + a34*a56))
 if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==2) 
& (r[5]==0) & (r[6]==2))
psi <-
2*a11*a36^2*a44 + 2*a16^2*(2*a34^2 + a33*a44) + 8*a11*a34*a36*a46 + 
 4*a13^2*a46^2 + 2*a11*a33*a46^2 + 
 8*a16*(2*a14*a34*a36 + a13*a36*a44 + a14*a33*a46 + 2*a13*a34*a46) + 
 2*a11*a34^2*a66 + 2*a13^2*a44*a66 + a11*a33*a44*a66 + 
 2*a14^2*(2*a36^2 + a33*a66) + 8*a13*a14*(2*a36*a46 + a34*a66)
 if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a15^2*(2*a34*a35 + a33*a45) + 2*a13*(2*a14*a35 + a13*a45)*a55 + 
  2*a15*(2*a14*a35^2 + 4*a13*a35*a45 + a14*a33*a55 + 2*a13*a34*a55) + 
  a11*(2*a35^2*a45 + 2*a34*a35*a55 + a33*a45*a55))
 if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==2) & (r[6]==1))
psi <-
8*a13*a16*a35*a45 + 4*a11*a35*a36*a45 + 2*a11*a35^2*a46 + 
 2*a15^2*(2*a34*a36 + a33*a46) + 4*a13*a16*a34*a55 + 2*a11*a34*a36*a55 + 
 2*a13^2*a46*a55 + a11*a33*a46*a55 + 4*a11*a34*a35*a56 + 4*a13^2*a45*a56 + 
 2*a11*a33*a45*a56 + 4*a15*(2*a16*a34*a35 + 2*a14*a35*a36 + a16*a33*a45 + 
   2*a13*a36*a45 + 2*a13*a35*a46 + a14*a33*a56 + 2*a13*a34*a56) + 
 2*a14*(2*a16*a35^2 + a16*a33*a55 + 2*a13*a36*a55 + 4*a13*a35*a56)
 if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==1) & (r[6]==2))
psi <-
2*a11*a36^2*a45 + 2*a16^2*(2*a34*a35 + a33*a45) + 8*a13*a15*a36*a46 + 
 4*a11*a35*a36*a46 + 4*a11*a34*a36*a56 + 4*a13^2*a46*a56 + 
 2*a11*a33*a46*a56 + 4*a16*(2*a15*a34*a36 + 2*a14*a35*a36 + 2*a13*a36*a45 + 
   a15*a33*a46 + 2*a13*a35*a46 + a14*a33*a56 + 2*a13*a34*a56) + 
 4*a13*a15*a34*a66 + 2*a11*a34*a35*a66 + 2*a13^2*a45*a66 + a11*a33*a45*a66 + 
 2*a14*(2*a15*a36^2 + 4*a13*a36*a56 + a15*a33*a66 + 2*a13*a35*a66)
 if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a16^2*(2*a34*a36 + a33*a46) + 2*a13*(2*a14*a36 + a13*a46)*a66 + 
  2*a16*(2*a14*a36^2 + 4*a13*a36*a46 + a14*a33*a66 + 2*a13*a34*a66) + 
  a11*(2*a36^2*a46 + 2*a34*a36*a66 + a33*a46*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==4) & (r[6]==0))
psi <-
3*(16*a13*a15*a35*a55 + 4*a15^2*(2*a35^2 + a33*a55) + 
  a55*(4*a11*a35^2 + 2*a13^2*a55 + a11*a33*a55))
 if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==3) & (r[6]==1))
psi <-
3*(4*a13*a16*a35*a55 + 2*a13^2*a55*a56 + 2*a15^2*(2*a35*a36 + a33*a56) + 
  2*a15*(2*a16*a35^2 + a16*a33*a55 + 2*a13*a36*a55 + 4*a13*a35*a56) + 
  a11*(2*a35*a36*a55 + 2*a35^2*a56 + a33*a55*a56))
 if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==2) & (r[6]==2))
psi <-
2*a11*a36^2*a55 + 2*a16^2*(2*a35^2 + a33*a55) + 8*a11*a35*a36*a56 + 
 4*a13^2*a56^2 + 2*a11*a33*a56^2 + 
 8*a16*(2*a15*a35*a36 + a13*a36*a55 + a15*a33*a56 + 2*a13*a35*a56) + 
 2*a11*a35^2*a66 + 2*a13^2*a55*a66 + a11*a33*a55*a66 + 
 2*a15^2*(2*a36^2 + a33*a66) + 8*a13*a15*(2*a36*a56 + a35*a66)
 if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(2*a16^2*(2*a35*a36 + a33*a56) + 2*a13*(2*a15*a36 + a13*a56)*a66 + 
  2*a16*(2*a15*a36^2 + 4*a13*a36*a56 + a15*a33*a66 + 2*a13*a35*a66) + 
  a11*(2*a36^2*a56 + 2*a35*a36*a66 + a33*a56*a66))
  if ((r[1]==2) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==4))
psi <-
3*(16*a13*a16*a36*a66 + 4*a16^2*(2*a36^2 + a33*a66) + 
  a66*(4*a11*a36^2 + 2*a13^2*a66 + a11*a33*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==5) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a44*(4*a14^2*a34 + 2*a13*a14*a44 + a11*a34*a44)
 if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==4) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(8*a14*a44*(a15*a34 + a13*a45) + 4*a14^2*(a35*a44 + 2*a34*a45) + 
  a44*(2*a13*a15*a44 + a11*a35*a44 + 4*a11*a34*a45))
 if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==4) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(8*a14*a44*(a16*a34 + a13*a46) + 4*a14^2*(a36*a44 + 2*a34*a46) + 
  a44*(2*a13*a16*a44 + a11*a36*a44 + 4*a11*a34*a46))
 if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==3) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a15^2*a34*a44 + 4*a15*(a14*a35*a44 + 2*a14*a34*a45 + a13*a44*a45) + 
  2*a14^2*(2*a35*a45 + a34*a55) + 2*a13*a14*(2*a45^2 + a44*a55) + 
  a11*(2*a35*a44*a45 + 2*a34*a45^2 + a34*a44*a55))
 if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==3) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(2*a13*a16*a44*a45 + a11*a36*a44*a45 + a11*a35*a44*a46 + 
  2*a11*a34*a45*a46 + 2*a15*(a16*a34*a44 + a14*a36*a44 + 2*a14*a34*a46 + 
    a13*a44*a46) + a11*a34*a44*a56 + 2*a14^2*(a36*a45 + a35*a46 + a34*a56) + 
  2*a14*(a16*a35*a44 + 2*a16*a34*a45 + 2*a13*a45*a46 + a13*a44*a56))
 if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==3) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a16^2*a34*a44 + 4*a16*(a14*a36*a44 + 2*a14*a34*a46 + a13*a44*a46) + 
  2*a14^2*(2*a36*a46 + a34*a66) + 2*a13*a14*(2*a46^2 + a44*a66) + 
  a11*(2*a36*a44*a46 + 2*a34*a46^2 + a34*a44*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a15^2*(a35*a44 + 2*a34*a45) + 2*a14*(a14*a35 + 2*a13*a45)*a55 + 
  2*a15*(4*a14*a35*a45 + 2*a13*a45^2 + 2*a14*a34*a55 + a13*a44*a55) + 
  a11*(2*a35*a45^2 + a35*a44*a55 + 2*a34*a45*a55))
 if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==2) & (r[6]==1))
psi <-
4*a13*a16*a45^2 + 2*a11*a36*a45^2 + 4*a11*a35*a45*a46 + 
 2*a15^2*(a36*a44 + 2*a34*a46) + 2*a13*a16*a44*a55 + a11*a36*a44*a55 + 
 2*a11*a34*a46*a55 + 2*a11*a35*a44*a56 + 4*a11*a34*a45*a56 + 
 2*a14^2*(a36*a55 + 2*a35*a56) + 4*a15*(a16*a35*a44 + 2*a16*a34*a45 + 
   2*a14*a36*a45 + 2*a14*a35*a46 + 2*a13*a45*a46 + 2*a14*a34*a56 + 
   a13*a44*a56) + 4*a14*(2*a16*a35*a45 + a16*a34*a55 + a13*a46*a55 + 
   2*a13*a45*a56)
 if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==1) & (r[6]==2))
psi <-
2*a16^2*(a35*a44 + 2*a34*a45) + 4*a11*a36*a45*a46 + 4*a13*a15*a46^2 + 
 2*a11*a35*a46^2 + 2*a11*a36*a44*a56 + 4*a11*a34*a46*a56 + 
 4*a16*(a15*a36*a44 + 2*a14*a36*a45 + 2*a15*a34*a46 + 2*a14*a35*a46 + 
   2*a13*a45*a46 + 2*a14*a34*a56 + a13*a44*a56) + 2*a13*a15*a44*a66 + 
 a11*a35*a44*a66 + 2*a11*a34*a45*a66 + 2*a14^2*(2*a36*a56 + a35*a66) + 
 4*a14*(2*a15*a36*a46 + 2*a13*a46*a56 + a15*a34*a66 + a13*a45*a66)
 if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a16^2*(a36*a44 + 2*a34*a46) + 2*a14*(a14*a36 + 2*a13*a46)*a66 + 
  2*a16*(4*a14*a36*a46 + 2*a13*a46^2 + 2*a14*a34*a66 + a13*a44*a66) + 
  a11*(2*a36*a46^2 + a36*a44*a66 + 2*a34*a46*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==4) & (r[6]==0))
psi <-
3*(8*a15*(a14*a35 + a13*a45)*a55 + 4*a15^2*(2*a35*a45 + a34*a55) + 
  a55*(4*a11*a35*a45 + 2*a13*a14*a55 + a11*a34*a55))
 if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==3) & (r[6]==1))
psi <-
3*(2*a13*a16*a45*a55 + a11*a36*a45*a55 + a11*a35*a46*a55 + 
  2*a11*a35*a45*a56 + a11*a34*a55*a56 + 2*a14*a55*(a16*a35 + a13*a56) + 
  2*a15^2*(a36*a45 + a35*a46 + a34*a56) + 
  2*a15*(2*a16*a35*a45 + a16*a34*a55 + a14*a36*a55 + a13*a46*a55 + 
    2*a14*a35*a56 + 2*a13*a45*a56))
 if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==2) & (r[6]==2))
psi <-
2*a11*a36*a46*a55 + 2*a16^2*(2*a35*a45 + a34*a55) + 4*a11*a36*a45*a56 + 
 4*a11*a35*a46*a56 + 4*a13*a14*a56^2 + 2*a11*a34*a56^2 + 
 4*a16*(a14*a36*a55 + a13*a46*a55 + 2*a14*a35*a56 + 2*a13*a45*a56 + 
   2*a15*(a36*a45 + a35*a46 + a34*a56)) + 2*a11*a35*a45*a66 + 
 2*a13*a14*a55*a66 + a11*a34*a55*a66 + 2*a15^2*(2*a36*a46 + a34*a66) + 
 4*a15*(2*a14*a36*a56 + 2*a13*a46*a56 + a14*a35*a66 + a13*a45*a66)
 if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(2*a16^2*(a36*a45 + a35*a46 + a34*a56) + 
  2*(a14*a15*a36 + a13*a15*a46 + a13*a14*a56)*a66 + 
  2*a16*(2*a15*a36*a46 + 2*a14*a36*a56 + 2*a13*a46*a56 + a15*a34*a66 + 
    a14*a35*a66 + a13*a45*a66) + a11*(2*a36*a46*a56 + a36*a45*a66 + 
    a35*a46*a66 + a34*a56*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==4))
psi <-
3*(8*a16*(a14*a36 + a13*a46)*a66 + 4*a16^2*(2*a36*a46 + a34*a66) + 
  a66*(4*a11*a36*a46 + 2*a13*a14*a66 + a11*a34*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==5) & (r[6]==0))
psi <-
15*a55*(4*a15^2*a35 + 2*a13*a15*a55 + a11*a35*a55)
 if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==4) & (r[6]==1))
psi <-
3*(8*a15*a55*(a16*a35 + a13*a56) + 4*a15^2*(a36*a55 + 2*a35*a56) + 
  a55*(2*a13*a16*a55 + a11*a36*a55 + 4*a11*a35*a56))
 if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==3) & (r[6]==2))
psi <-
3*(2*a16^2*a35*a55 + 4*a16*(a15*a36*a55 + 2*a15*a35*a56 + a13*a55*a56) + 
  2*a15^2*(2*a36*a56 + a35*a66) + 2*a13*a15*(2*a56^2 + a55*a66) + 
  a11*(2*a36*a55*a56 + 2*a35*a56^2 + a35*a55*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==3))
psi <-
3*(2*a16^2*(a36*a55 + 2*a35*a56) + 2*a15*(a15*a36 + 2*a13*a56)*a66 + 
  2*a16*(4*a15*a36*a56 + 2*a13*a56^2 + 2*a15*a35*a66 + a13*a55*a66) + 
  a11*(2*a36*a56^2 + a36*a55*a66 + 2*a35*a56*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==4))
psi <-
3*(8*a16*(a15*a36 + a13*a56)*a66 + 4*a16^2*(2*a36*a56 + a35*a66) + 
  a66*(4*a11*a36*a56 + 2*a13*a15*a66 + a11*a35*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==5))
psi <-
15*a66*(4*a16^2*a36 + 2*a13*a16*a66 + a11*a36*a66)
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==6) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a44^2*(6*a14^2 + a11*a44)
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==5) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a44*(2*a14*a15*a44 + 4*a14^2*a45 + a11*a44*a45)
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==5) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a44*(2*a14*a16*a44 + 4*a14^2*a46 + a11*a44*a46)
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==4) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a15^2*a44^2 + 16*a14*a15*a44*a45 + 4*a14^2*(2*a45^2 + a44*a55) + 
  a11*a44*(4*a45^2 + a44*a55))
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==4) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(8*a14*a16*a44*a45 + 2*a15*a44*(a16*a44 + 4*a14*a46) + 
  4*a14^2*(2*a45*a46 + a44*a56) + a11*a44*(4*a45*a46 + a44*a56))
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==4) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a16^2*a44^2 + 16*a14*a16*a44*a46 + 4*a14^2*(2*a46^2 + a44*a66) + 
  a11*a44*(4*a46^2 + a44*a66))
  if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(6*a15^2*a44*a45 + 6*a14*a15*(2*a45^2 + a44*a55) + 
  a45*(2*a11*a45^2 + 6*a14^2*a55 + 3*a11*a44*a55))
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(2*a15^2*a44*a46 + 2*a14*a16*(2*a45^2 + a44*a55) + 
  4*a15*(a16*a44*a45 + 2*a14*a45*a46 + a14*a44*a56) + 
  2*a14^2*(a46*a55 + 2*a45*a56) + a11*(2*a45^2*a46 + a44*a46*a55 + 
    2*a44*a45*a56))
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(2*a16^2*a44*a45 + 4*a16*(a15*a44*a46 + 2*a14*a45*a46 + a14*a44*a56) + 
  2*a14*a15*(2*a46^2 + a44*a66) + 2*a14^2*(2*a46*a56 + a45*a66) + 
  a11*(2*a45*a46^2 + 2*a44*a46*a56 + a44*a45*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(6*a16^2*a44*a46 + 6*a14*a16*(2*a46^2 + a44*a66) + 
  a46*(2*a11*a46^2 + 6*a14^2*a66 + 3*a11*a44*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==4) & (r[6]==0))
psi <-
3*(16*a14*a15*a45*a55 + 4*a15^2*(2*a45^2 + a44*a55) + 
  a55*(4*a11*a45^2 + 2*a14^2*a55 + a11*a44*a55))
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==3) & (r[6]==1))
psi <-
3*(4*a14*a16*a45*a55 + 2*a14^2*a55*a56 + 2*a15^2*(2*a45*a46 + a44*a56) + 
  2*a15*(2*a16*a45^2 + a16*a44*a55 + 2*a14*a46*a55 + 4*a14*a45*a56) + 
  a11*(2*a45*a46*a55 + 2*a45^2*a56 + a44*a55*a56))
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==2) & (r[6]==2))
psi <-
2*a11*a46^2*a55 + 2*a16^2*(2*a45^2 + a44*a55) + 8*a11*a45*a46*a56 + 
 4*a14^2*a56^2 + 2*a11*a44*a56^2 + 
 8*a16*(2*a15*a45*a46 + a14*a46*a55 + a15*a44*a56 + 2*a14*a45*a56) + 
 2*a11*a45^2*a66 + 2*a14^2*a55*a66 + a11*a44*a55*a66 + 
 2*a15^2*(2*a46^2 + a44*a66) + 8*a14*a15*(2*a46*a56 + a45*a66)
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(2*a16^2*(2*a45*a46 + a44*a56) + 2*a14*(2*a15*a46 + a14*a56)*a66 + 
  2*a16*(2*a15*a46^2 + 4*a14*a46*a56 + a15*a44*a66 + 2*a14*a45*a66) + 
  a11*(2*a46^2*a56 + 2*a45*a46*a66 + a44*a56*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==4))
psi <-
3*(16*a14*a16*a46*a66 + 4*a16^2*(2*a46^2 + a44*a66) + 
  a66*(4*a11*a46^2 + 2*a14^2*a66 + a11*a44*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==5) & (r[6]==0))
psi <-
15*a55*(4*a15^2*a45 + 2*a14*a15*a55 + a11*a45*a55)
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==4) & (r[6]==1))
psi <-
3*(8*a15*a55*(a16*a45 + a14*a56) + 4*a15^2*(a46*a55 + 2*a45*a56) + 
  a55*(2*a14*a16*a55 + a11*a46*a55 + 4*a11*a45*a56))
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==3) & (r[6]==2))
psi <-
3*(2*a16^2*a45*a55 + 4*a16*(a15*a46*a55 + 2*a15*a45*a56 + a14*a55*a56) + 
  2*a15^2*(2*a46*a56 + a45*a66) + 2*a14*a15*(2*a56^2 + a55*a66) + 
  a11*(2*a46*a55*a56 + 2*a45*a56^2 + a45*a55*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==3))
psi <-
3*(2*a16^2*(a46*a55 + 2*a45*a56) + 2*a15*(a15*a46 + 2*a14*a56)*a66 + 
  2*a16*(4*a15*a46*a56 + 2*a14*a56^2 + 2*a15*a45*a66 + a14*a55*a66) + 
  a11*(2*a46*a56^2 + a46*a55*a66 + 2*a45*a56*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==4))
psi <-
3*(8*a16*(a15*a46 + a14*a56)*a66 + 4*a16^2*(2*a46*a56 + a45*a66) + 
  a66*(4*a11*a46*a56 + 2*a14*a15*a66 + a11*a45*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==5))
psi <-
15*a66*(4*a16^2*a46 + 2*a14*a16*a66 + a11*a46*a66)
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==6) & (r[6]==0))
psi <-
15*a55^2*(6*a15^2 + a11*a55)
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==5) & (r[6]==1))
psi <-
15*a55*(2*a15*a16*a55 + 4*a15^2*a56 + a11*a55*a56)
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==4) & (r[6]==2))
psi <-
3*(2*a16^2*a55^2 + 16*a15*a16*a55*a56 + 4*a15^2*(2*a56^2 + a55*a66) + 
  a11*a55*(4*a56^2 + a55*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==3))
psi <-
3*(6*a16^2*a55*a56 + 6*a15*a16*(2*a56^2 + a55*a66) + 
  a56*(2*a11*a56^2 + 6*a15^2*a66 + 3*a11*a55*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==4))
psi <-
3*(16*a15*a16*a56*a66 + 4*a16^2*(2*a56^2 + a55*a66) + 
  a66*(4*a11*a56^2 + 2*a15^2*a66 + a11*a55*a66))
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==5))
psi <-
15*a66*(4*a16^2*a56 + 2*a15*a16*a66 + a11*a56*a66)
 if ((r[1]==2) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==6))
psi <-
15*a66^2*(6*a16^2 + a11*a66)
 if ((r[1]==1) & (r[2]==7) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
105*a12*a22^3
 if ((r[1]==1) & (r[2]==6) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a22^2*(a13*a22 + 6*a12*a23)
 if ((r[1]==1) & (r[2]==6) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a22^2*(a14*a22 + 6*a12*a24)
 if ((r[1]==1) & (r[2]==6) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a22^2*(a15*a22 + 6*a12*a25)
 if ((r[1]==1) & (r[2]==6) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a22^2*(a16*a22 + 6*a12*a26)
 if ((r[1]==1) & (r[2]==5) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a22*(2*a13*a22*a23 + 4*a12*a23^2 + a12*a22*a33)
 if ((r[1]==1) & (r[2]==5) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a22*(a14*a22*a23 + a13*a22*a24 + 4*a12*a23*a24 + a12*a22*a34)
 if ((r[1]==1) & (r[2]==5) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a22*(a15*a22*a23 + a13*a22*a25 + 4*a12*a23*a25 + a12*a22*a35)
 if ((r[1]==1) & (r[2]==5) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a22*(a16*a22*a23 + a13*a22*a26 + 4*a12*a23*a26 + a12*a22*a36)
 if ((r[1]==1) & (r[2]==5) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a22*(2*a14*a22*a24 + 4*a12*a24^2 + a12*a22*a44)
 if ((r[1]==1) & (r[2]==5) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a22*(a15*a22*a24 + a14*a22*a25 + 4*a12*a24*a25 + a12*a22*a45)
 if ((r[1]==1) & (r[2]==5) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a22*(a16*a22*a24 + a14*a22*a26 + 4*a12*a24*a26 + a12*a22*a46)
 if ((r[1]==1) & (r[2]==5) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
15*a22*(2*a15*a22*a25 + 4*a12*a25^2 + a12*a22*a55)
  if ((r[1]==1) & (r[2]==5) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
15*a22*(a16*a22*a25 + a15*a22*a26 + 4*a12*a25*a26 + a12*a22*a56)
 if ((r[1]==1) & (r[2]==5) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
15*a22*(2*a16*a22*a26 + 4*a12*a26^2 + a12*a22*a66)
 if ((r[1]==1) & (r[2]==4) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
9*a13*a22*(4*a23^2 + a22*a33) + 12*a12*a23*(2*a23^2 + 3*a22*a33)
 if ((r[1]==1) & (r[2]==4) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(a14*a22*(4*a23^2 + a22*a33) + 2*a13*a22*(4*a23*a24 + a22*a34) + 
  4*a12*(2*a23^2*a24 + a22*a24*a33 + 2*a22*a23*a34))
 if ((r[1]==1) & (r[2]==4) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(a15*a22*(4*a23^2 + a22*a33) + 2*a13*a22*(4*a23*a25 + a22*a35) + 
  4*a12*(2*a23^2*a25 + a22*a25*a33 + 2*a22*a23*a35))
 if ((r[1]==1) & (r[2]==4) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(a16*a22*(4*a23^2 + a22*a33) + 2*a13*a22*(4*a23*a26 + a22*a36) + 
  4*a12*(2*a23^2*a26 + a22*a26*a33 + 2*a22*a23*a36))
 if ((r[1]==1) & (r[2]==4) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a14*a22*(4*a23*a24 + a22*a34) + a13*a22*(4*a24^2 + a22*a44) + 
  4*a12*(2*a23*a24^2 + 2*a22*a24*a34 + a22*a23*a44))
 if ((r[1]==1) & (r[2]==4) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(4*a13*a22*a24*a25 + 8*a12*a23*a24*a25 + 4*a12*a22*a25*a34 + 
  a15*a22*(4*a23*a24 + a22*a34) + 4*a12*a22*a24*a35 + 
  a14*a22*(4*a23*a25 + a22*a35) + a13*a22^2*a45 + 4*a12*a22*a23*a45)
 if ((r[1]==1) & (r[2]==4) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(4*a13*a22*a24*a26 + 8*a12*a23*a24*a26 + 4*a12*a22*a26*a34 + 
  a16*a22*(4*a23*a24 + a22*a34) + 4*a12*a22*a24*a36 + 
  a14*a22*(4*a23*a26 + a22*a36) + a13*a22^2*a46 + 4*a12*a22*a23*a46)
 if ((r[1]==1) & (r[2]==4) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a15*a22*(4*a23*a25 + a22*a35) + a13*a22*(4*a25^2 + a22*a55) + 
  4*a12*(2*a23*a25^2 + 2*a22*a25*a35 + a22*a23*a55))
 if ((r[1]==1) & (r[2]==4) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(4*a13*a22*a25*a26 + 8*a12*a23*a25*a26 + 4*a12*a22*a26*a35 + 
  a16*a22*(4*a23*a25 + a22*a35) + 4*a12*a22*a25*a36 + 
  a15*a22*(4*a23*a26 + a22*a36) + a13*a22^2*a56 + 4*a12*a22*a23*a56)
 if ((r[1]==1) & (r[2]==4) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a16*a22*(4*a23*a26 + a22*a36) + a13*a22*(4*a26^2 + a22*a66) + 
  4*a12*(2*a23*a26^2 + 2*a22*a26*a36 + a22*a23*a66))
 if ((r[1]==1) & (r[2]==4) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
9*a14*a22*(4*a24^2 + a22*a44) + 12*a12*a24*(2*a24^2 + 3*a22*a44)
 if ((r[1]==1) & (r[2]==4) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(a15*a22*(4*a24^2 + a22*a44) + 2*a14*a22*(4*a24*a25 + a22*a45) + 
  4*a12*(2*a24^2*a25 + a22*a25*a44 + 2*a22*a24*a45))
 if ((r[1]==1) & (r[2]==4) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(a16*a22*(4*a24^2 + a22*a44) + 2*a14*a22*(4*a24*a26 + a22*a46) + 
  4*a12*(2*a24^2*a26 + a22*a26*a44 + 2*a22*a24*a46))
 if ((r[1]==1) & (r[2]==4) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a15*a22*(4*a24*a25 + a22*a45) + a14*a22*(4*a25^2 + a22*a55) + 
  4*a12*(2*a24*a25^2 + 2*a22*a25*a45 + a22*a24*a55))
 if ((r[1]==1) & (r[2]==4) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(4*a14*a22*a25*a26 + 8*a12*a24*a25*a26 + 4*a12*a22*a26*a45 + 
  a16*a22*(4*a24*a25 + a22*a45) + 4*a12*a22*a25*a46 + 
  a15*a22*(4*a24*a26 + a22*a46) + a14*a22^2*a56 + 4*a12*a22*a24*a56)
 if ((r[1]==1) & (r[2]==4) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a16*a22*(4*a24*a26 + a22*a46) + a14*a22*(4*a26^2 + a22*a66) + 
  4*a12*(2*a24*a26^2 + 2*a22*a26*a46 + a22*a24*a66))
 if ((r[1]==1) & (r[2]==4) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
9*a15*a22*(4*a25^2 + a22*a55) + 12*a12*a25*(2*a25^2 + 3*a22*a55)
 if ((r[1]==1) & (r[2]==4) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(a16*a22*(4*a25^2 + a22*a55) + 2*a15*a22*(4*a25*a26 + a22*a56) + 
  4*a12*(2*a25^2*a26 + a22*a26*a55 + 2*a22*a25*a56))
 if ((r[1]==1) & (r[2]==4) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(2*a16*a22*(4*a25*a26 + a22*a56) + a15*a22*(4*a26^2 + a22*a66) + 
  4*a12*(2*a25*a26^2 + 2*a22*a26*a56 + a22*a25*a66))
 if ((r[1]==1) & (r[2]==4) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
9*a16*a22*(4*a26^2 + a22*a66) + 12*a12*a26*(2*a26^2 + 3*a22*a66)
 if ((r[1]==1) & (r[2]==3) & (r[3]==4) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
9*a12*a33*(4*a23^2 + a22*a33) + 12*a13*(2*a23^3 + 3*a22*a23*a33)
 if ((r[1]==1) & (r[2]==3) & (r[3]==3) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a14*a23^3 + 6*a13*a23^2*a24 + 3*a14*a22*a23*a33 + 3*a13*a22*a24*a33 + 
  6*a12*a23*a24*a33 + 6*a13*a22*a23*a34 + 6*a12*a23^2*a34 + 3*a12*a22*a33*a34)
 if ((r[1]==1) & (r[2]==3) & (r[3]==3) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a15*a23^3 + 6*a13*a23^2*a25 + 3*a15*a22*a23*a33 + 3*a13*a22*a25*a33 + 
  6*a12*a23*a25*a33 + 6*a13*a22*a23*a35 + 6*a12*a23^2*a35 + 3*a12*a22*a33*a35)
 if ((r[1]==1) & (r[2]==3) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a16*a23^3 + 6*a13*a23^2*a26 + 3*a16*a22*a23*a33 + 3*a13*a22*a26*a33 + 
  6*a12*a23*a26*a33 + 6*a13*a22*a23*a36 + 6*a12*a23^2*a36 + 3*a12*a22*a33*a36)
 if ((r[1]==1) & (r[2]==3) & (r[3]==2) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a14*(2*a23^2*a24 + a22*a24*a33 + 2*a22*a23*a34) + 
  2*a13*(2*a23*a24^2 + 2*a22*a24*a34 + a22*a23*a44) + 
  a12*(2*a24^2*a33 + 8*a23*a24*a34 + 2*a22*a34^2 + 2*a23^2*a44 + a22*a33*a44))
 if ((r[1]==1) & (r[2]==3) & (r[3]==2) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(4*a13*a23*a24*a25 + 2*a12*a24*a25*a33 + 2*a13*a22*a25*a34 + 
  4*a12*a23*a25*a34 + a15*(2*a23^2*a24 + a22*a24*a33 + 2*a22*a23*a34) + 
  2*a13*a22*a24*a35 + 4*a12*a23*a24*a35 + 2*a12*a22*a34*a35 + 
  a14*(2*a23^2*a25 + a22*a25*a33 + 2*a22*a23*a35) + 2*a13*a22*a23*a45 + 
  2*a12*a23^2*a45 + a12*a22*a33*a45)
 if ((r[1]==1) & (r[2]==3) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(4*a13*a23*a24*a26 + 2*a12*a24*a26*a33 + 2*a13*a22*a26*a34 + 
  4*a12*a23*a26*a34 + a16*(2*a23^2*a24 + a22*a24*a33 + 2*a22*a23*a34) + 
  2*a13*a22*a24*a36 + 4*a12*a23*a24*a36 + 2*a12*a22*a34*a36 + 
  a14*(2*a23^2*a26 + a22*a26*a33 + 2*a22*a23*a36) + 2*a13*a22*a23*a46 + 
  2*a12*a23^2*a46 + a12*a22*a33*a46)
 if ((r[1]==1) & (r[2]==3) & (r[3]==2) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a15*(2*a23^2*a25 + a22*a25*a33 + 2*a22*a23*a35) + 
  2*a13*(2*a23*a25^2 + 2*a22*a25*a35 + a22*a23*a55) + 
  a12*(2*a25^2*a33 + 8*a23*a25*a35 + 2*a22*a35^2 + 2*a23^2*a55 + a22*a33*a55))
 if ((r[1]==1) & (r[2]==3) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(4*a13*a23*a25*a26 + 2*a12*a25*a26*a33 + 2*a13*a22*a26*a35 + 
  4*a12*a23*a26*a35 + a16*(2*a23^2*a25 + a22*a25*a33 + 2*a22*a23*a35) + 
  2*a13*a22*a25*a36 + 4*a12*a23*a25*a36 + 2*a12*a22*a35*a36 + 
  a15*(2*a23^2*a26 + a22*a26*a33 + 2*a22*a23*a36) + 2*a13*a22*a23*a56 + 
  2*a12*a23^2*a56 + a12*a22*a33*a56)
  if ((r[1]==1) & (r[2]==3) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a16*(2*a23^2*a26 + a22*a26*a33 + 2*a22*a23*a36) + 
  2*a13*(2*a23*a26^2 + 2*a22*a26*a36 + a22*a23*a66) + 
  a12*(2*a26^2*a33 + 8*a23*a26*a36 + 2*a22*a36^2 + 2*a23^2*a66 + a22*a33*a66))
 if ((r[1]==1) & (r[2]==3) & (r[3]==1) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a13*a24^3 + 6*a12*a24^2*a34 + 3*a13*a22*a24*a44 + 6*a12*a23*a24*a44 + 
  3*a12*a22*a34*a44 + 3*a14*(2*a23*a24^2 + 2*a22*a24*a34 + a22*a23*a44))
 if ((r[1]==1) & (r[2]==3) & (r[3]==1) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a13*a24^2*a25 + 4*a12*a24*a25*a34 + 2*a12*a24^2*a35 + a13*a22*a25*a44 + 
  2*a12*a23*a25*a44 + a12*a22*a35*a44 + 
  a15*(2*a23*a24^2 + 2*a22*a24*a34 + a22*a23*a44) + 2*a13*a22*a24*a45 + 
  4*a12*a23*a24*a45 + 2*a12*a22*a34*a45 + 
  2*a14*(2*a23*a24*a25 + a22*a25*a34 + a22*a24*a35 + a22*a23*a45))
 if ((r[1]==1) & (r[2]==3) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a13*a24^2*a26 + 4*a12*a24*a26*a34 + 2*a12*a24^2*a36 + a13*a22*a26*a44 + 
  2*a12*a23*a26*a44 + a12*a22*a36*a44 + 
  a16*(2*a23*a24^2 + 2*a22*a24*a34 + a22*a23*a44) + 2*a13*a22*a24*a46 + 
  4*a12*a23*a24*a46 + 2*a12*a22*a34*a46 + 
  2*a14*(2*a23*a24*a26 + a22*a26*a34 + a22*a24*a36 + a22*a23*a46))
 if ((r[1]==1) & (r[2]==3) & (r[3]==1) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a13*a24*a25^2 + 2*a12*a25^2*a34 + 4*a12*a24*a25*a35 + 
  2*a13*a22*a25*a45 + 4*a12*a23*a25*a45 + 2*a12*a22*a35*a45 + 
  2*a15*(2*a23*a24*a25 + a22*a25*a34 + a22*a24*a35 + a22*a23*a45) + 
  a13*a22*a24*a55 + 2*a12*a23*a24*a55 + a12*a22*a34*a55 + 
  a14*(2*a23*a25^2 + 2*a22*a25*a35 + a22*a23*a55))
 if ((r[1]==1) & (r[2]==3) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(2*a14*a23*a25*a26 + 2*a13*a24*a25*a26 + 2*a12*a25*a26*a34 + 
  a14*a22*a26*a35 + 2*a12*a24*a26*a35 + a14*a22*a25*a36 + 2*a12*a24*a25*a36 + 
  a13*a22*a26*a45 + 2*a12*a23*a26*a45 + a12*a22*a36*a45 + 
  a16*(2*a23*a24*a25 + a22*a25*a34 + a22*a24*a35 + a22*a23*a45) + 
  a13*a22*a25*a46 + 2*a12*a23*a25*a46 + a12*a22*a35*a46 + 
  a15*(2*a23*a24*a26 + a22*a26*a34 + a22*a24*a36 + a22*a23*a46) + 
  a14*a22*a23*a56 + a13*a22*a24*a56 + 2*a12*a23*a24*a56 + a12*a22*a34*a56)
 if ((r[1]==1) & (r[2]==3) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a13*a24*a26^2 + 2*a12*a26^2*a34 + 4*a12*a24*a26*a36 + 
  2*a13*a22*a26*a46 + 4*a12*a23*a26*a46 + 2*a12*a22*a36*a46 + 
  2*a16*(2*a23*a24*a26 + a22*a26*a34 + a22*a24*a36 + a22*a23*a46) + 
  a13*a22*a24*a66 + 2*a12*a23*a24*a66 + a12*a22*a34*a66 + 
  a14*(2*a23*a26^2 + 2*a22*a26*a36 + a22*a23*a66))
 if ((r[1]==1) & (r[2]==3) & (r[3]==1) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a13*a25^3 + 6*a12*a25^2*a35 + 3*a13*a22*a25*a55 + 6*a12*a23*a25*a55 + 
  3*a12*a22*a35*a55 + 3*a15*(2*a23*a25^2 + 2*a22*a25*a35 + a22*a23*a55))
 if ((r[1]==1) & (r[2]==3) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(2*a13*a25^2*a26 + 4*a12*a25*a26*a35 + 2*a12*a25^2*a36 + a13*a22*a26*a55 + 
  2*a12*a23*a26*a55 + a12*a22*a36*a55 + 
  a16*(2*a23*a25^2 + 2*a22*a25*a35 + a22*a23*a55) + 2*a13*a22*a25*a56 + 
  4*a12*a23*a25*a56 + 2*a12*a22*a35*a56 + 
  2*a15*(2*a23*a25*a26 + a22*a26*a35 + a22*a25*a36 + a22*a23*a56))
 if ((r[1]==1) & (r[2]==3) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(2*a13*a25*a26^2 + 2*a12*a26^2*a35 + 4*a12*a25*a26*a36 + 
  2*a13*a22*a26*a56 + 4*a12*a23*a26*a56 + 2*a12*a22*a36*a56 + 
  2*a16*(2*a23*a25*a26 + a22*a26*a35 + a22*a25*a36 + a22*a23*a56) + 
  a13*a22*a25*a66 + 2*a12*a23*a25*a66 + a12*a22*a35*a66 + 
  a15*(2*a23*a26^2 + 2*a22*a26*a36 + a22*a23*a66))
 if ((r[1]==1) & (r[2]==3) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a13*a26^3 + 6*a12*a26^2*a36 + 3*a13*a22*a26*a66 + 6*a12*a23*a26*a66 + 
  3*a12*a22*a36*a66 + 3*a16*(2*a23*a26^2 + 2*a22*a26*a36 + a22*a23*a66))
 if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==4) 
& (r[5]==0) & (r[6]==0))
psi <-
9*a12*a44*(4*a24^2 + a22*a44) + 12*a14*(2*a24^3 + 3*a22*a24*a44)
 if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==3) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a15*a24^3 + 6*a14*a24^2*a25 + 3*a15*a22*a24*a44 + 3*a14*a22*a25*a44 + 
  6*a12*a24*a25*a44 + 6*a14*a22*a24*a45 + 6*a12*a24^2*a45 + 3*a12*a22*a44*a45)
 if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a16*a24^3 + 6*a14*a24^2*a26 + 3*a16*a22*a24*a44 + 3*a14*a22*a26*a44 + 
  6*a12*a24*a26*a44 + 6*a14*a22*a24*a46 + 6*a12*a24^2*a46 + 3*a12*a22*a44*a46)
 if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==2) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a15*(2*a24^2*a25 + a22*a25*a44 + 2*a22*a24*a45) + 
  2*a14*(2*a24*a25^2 + 2*a22*a25*a45 + a22*a24*a55) + 
  a12*(2*a25^2*a44 + 8*a24*a25*a45 + 2*a22*a45^2 + 2*a24^2*a55 + a22*a44*a55))
 if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(4*a14*a24*a25*a26 + 2*a12*a25*a26*a44 + 2*a14*a22*a26*a45 + 
  4*a12*a24*a26*a45 + a16*(2*a24^2*a25 + a22*a25*a44 + 2*a22*a24*a45) + 
  2*a14*a22*a25*a46 + 4*a12*a24*a25*a46 + 2*a12*a22*a45*a46 + 
  a15*(2*a24^2*a26 + a22*a26*a44 + 2*a22*a24*a46) + 2*a14*a22*a24*a56 + 
  2*a12*a24^2*a56 + a12*a22*a44*a56)
 if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a16*(2*a24^2*a26 + a22*a26*a44 + 2*a22*a24*a46) + 
  2*a14*(2*a24*a26^2 + 2*a22*a26*a46 + a22*a24*a66) + 
  a12*(2*a26^2*a44 + 8*a24*a26*a46 + 2*a22*a46^2 + 2*a24^2*a66 + a22*a44*a66))
 if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==1) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a14*a25^3 + 6*a12*a25^2*a45 + 3*a14*a22*a25*a55 + 6*a12*a24*a25*a55 + 
  3*a12*a22*a45*a55 + 3*a15*(2*a24*a25^2 + 2*a22*a25*a45 + a22*a24*a55))
 if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(2*a14*a25^2*a26 + 4*a12*a25*a26*a45 + 2*a12*a25^2*a46 + a14*a22*a26*a55 + 
  2*a12*a24*a26*a55 + a12*a22*a46*a55 + 
  a16*(2*a24*a25^2 + 2*a22*a25*a45 + a22*a24*a55) + 2*a14*a22*a25*a56 + 
  4*a12*a24*a25*a56 + 2*a12*a22*a45*a56 + 
  2*a15*(2*a24*a25*a26 + a22*a26*a45 + a22*a25*a46 + a22*a24*a56))
 if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(2*a14*a25*a26^2 + 2*a12*a26^2*a45 + 4*a12*a25*a26*a46 + 
  2*a14*a22*a26*a56 + 4*a12*a24*a26*a56 + 2*a12*a22*a46*a56 + 
  2*a16*(2*a24*a25*a26 + a22*a26*a45 + a22*a25*a46 + a22*a24*a56) + 
  a14*a22*a25*a66 + 2*a12*a24*a25*a66 + a12*a22*a45*a66 + 
  a15*(2*a24*a26^2 + 2*a22*a26*a46 + a22*a24*a66))
 if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a14*a26^3 + 6*a12*a26^2*a46 + 3*a14*a22*a26*a66 + 6*a12*a24*a26*a66 + 
  3*a12*a22*a46*a66 + 3*a16*(2*a24*a26^2 + 2*a22*a26*a46 + a22*a24*a66))
 if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==4) & (r[6]==0))
psi <-
9*a12*a55*(4*a25^2 + a22*a55) + 12*a15*(2*a25^3 + 3*a22*a25*a55)
 if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==1))
psi <-
3*(2*a16*a25^3 + 6*a15*a25^2*a26 + 3*a16*a22*a25*a55 + 3*a15*a22*a26*a55 + 
  6*a12*a25*a26*a55 + 6*a15*a22*a25*a56 + 6*a12*a25^2*a56 + 3*a12*a22*a55*a56)
 if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==2))
psi <-
3*(2*a16*(2*a25^2*a26 + a22*a26*a55 + 2*a22*a25*a56) + 
  2*a15*(2*a25*a26^2 + 2*a22*a26*a56 + a22*a25*a66) + 
  a12*(2*a26^2*a55 + 8*a25*a26*a56 + 2*a22*a56^2 + 2*a25^2*a66 + a22*a55*a66))
 if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(2*a15*a26^3 + 6*a12*a26^2*a56 + 3*a15*a22*a26*a66 + 6*a12*a25*a26*a66 + 
  3*a12*a22*a56*a66 + 3*a16*(2*a25*a26^2 + 2*a22*a26*a56 + a22*a25*a66))
 if ((r[1]==1) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==4))
psi <-
9*a12*a66*(4*a26^2 + a22*a66) + 12*a16*(2*a26^3 + 3*a22*a26*a66)
 if ((r[1]==1) & (r[2]==2) & (r[3]==5) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a33*(4*a13*a23^2 + a13*a22*a33 + 2*a12*a23*a33)
  if ((r[1]==1) & (r[2]==2) & (r[3]==4) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(a14*a33*(4*a23^2 + a22*a33) + 2*a12*a33*(a24*a33 + 4*a23*a34) + 
  4*a13*(2*a23*a24*a33 + 2*a23^2*a34 + a22*a33*a34))
 if ((r[1]==1) & (r[2]==2) & (r[3]==4) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(a15*a33*(4*a23^2 + a22*a33) + 2*a12*a33*(a25*a33 + 4*a23*a35) + 
  4*a13*(2*a23*a25*a33 + 2*a23^2*a35 + a22*a33*a35))
 if ((r[1]==1) & (r[2]==2) & (r[3]==4) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(a16*a33*(4*a23^2 + a22*a33) + 2*a12*a33*(a26*a33 + 4*a23*a36) + 
  4*a13*(2*a23*a26*a33 + 2*a23^2*a36 + a22*a33*a36))
 if ((r[1]==1) & (r[2]==2) & (r[3]==3) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a14*(2*a23*a24*a33 + 2*a23^2*a34 + a22*a33*a34) + 
  a13*(2*a24^2*a33 + 8*a23*a24*a34 + 2*a22*a34^2 + 2*a23^2*a44 + 
    a22*a33*a44) + 2*a12*(2*a24*a33*a34 + 2*a23*a34^2 + a23*a33*a44))
 if ((r[1]==1) & (r[2]==2) & (r[3]==3) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a13*a24*a25*a33 + 4*a13*a23*a25*a34 + 2*a12*a25*a33*a34 + 
  a15*(2*a23*a24*a33 + 2*a23^2*a34 + a22*a33*a34) + 4*a13*a23*a24*a35 + 
  2*a12*a24*a33*a35 + 2*a13*a22*a34*a35 + 4*a12*a23*a34*a35 + 
  a14*(2*a23*a25*a33 + 2*a23^2*a35 + a22*a33*a35) + 2*a13*a23^2*a45 + 
  a13*a22*a33*a45 + 2*a12*a23*a33*a45)
 if ((r[1]==1) & (r[2]==2) & (r[3]==3) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a13*a24*a26*a33 + 4*a13*a23*a26*a34 + 2*a12*a26*a33*a34 + 
  a16*(2*a23*a24*a33 + 2*a23^2*a34 + a22*a33*a34) + 4*a13*a23*a24*a36 + 
  2*a12*a24*a33*a36 + 2*a13*a22*a34*a36 + 4*a12*a23*a34*a36 + 
  a14*(2*a23*a26*a33 + 2*a23^2*a36 + a22*a33*a36) + 2*a13*a23^2*a46 + 
  a13*a22*a33*a46 + 2*a12*a23*a33*a46)
 if ((r[1]==1) & (r[2]==2) & (r[3]==3) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a15*(2*a23*a25*a33 + 2*a23^2*a35 + a22*a33*a35) + 
  a13*(2*a25^2*a33 + 8*a23*a25*a35 + 2*a22*a35^2 + 2*a23^2*a55 + 
    a22*a33*a55) + 2*a12*(2*a25*a33*a35 + 2*a23*a35^2 + a23*a33*a55))
 if ((r[1]==1) & (r[2]==2) & (r[3]==3) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(2*a13*a25*a26*a33 + 4*a13*a23*a26*a35 + 2*a12*a26*a33*a35 + 
  a16*(2*a23*a25*a33 + 2*a23^2*a35 + a22*a33*a35) + 4*a13*a23*a25*a36 + 
  2*a12*a25*a33*a36 + 2*a13*a22*a35*a36 + 4*a12*a23*a35*a36 + 
  a15*(2*a23*a26*a33 + 2*a23^2*a36 + a22*a33*a36) + 2*a13*a23^2*a56 + 
  a13*a22*a33*a56 + 2*a12*a23*a33*a56)
 if ((r[1]==1) & (r[2]==2) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a16*(2*a23*a26*a33 + 2*a23^2*a36 + a22*a33*a36) + 
  a13*(2*a26^2*a33 + 8*a23*a26*a36 + 2*a22*a36^2 + 2*a23^2*a66 + 
    a22*a33*a66) + 2*a12*(2*a26*a33*a36 + 2*a23*a36^2 + a23*a33*a66))
 if ((r[1]==1) & (r[2]==2) & (r[3]==2) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(a14*(2*a24^2*a33 + 8*a23*a24*a34 + 2*a22*a34^2 + 2*a23^2*a44 + 
    a22*a33*a44) + 2*(a13*(2*a24^2*a34 + 2*a23*a24*a44 + a22*a34*a44) + 
    a12*(2*a24*a34^2 + a24*a33*a44 + 2*a23*a34*a44)))
 if ((r[1]==1) & (r[2]==2) & (r[3]==2) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
a15*(2*a24^2*a33 + 8*a23*a24*a34 + 2*a22*a34^2 + 2*a23^2*a44 + a22*a33*a44) + 
 2*(a14*(2*a24*a25*a33 + 4*a23*a25*a34 + 4*a23*a24*a35 + 2*a22*a34*a35 + 
     2*a23^2*a45 + a22*a33*a45) + a13*(4*a24*a25*a34 + 2*a24^2*a35 + 
     2*a23*a25*a44 + a22*a35*a44 + 4*a23*a24*a45 + 2*a22*a34*a45) + 
   a12*(2*a25*a34^2 + 4*a24*a34*a35 + a25*a33*a44 + 2*a23*a35*a44 + 
     2*a24*a33*a45 + 4*a23*a34*a45))
 if ((r[1]==1) & (r[2]==2) & (r[3]==2) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
a16*(2*a24^2*a33 + 8*a23*a24*a34 + 2*a22*a34^2 + 2*a23^2*a44 + a22*a33*a44) + 
 2*(a14*(2*a24*a26*a33 + 4*a23*a26*a34 + 4*a23*a24*a36 + 2*a22*a34*a36 + 
     2*a23^2*a46 + a22*a33*a46) + a13*(4*a24*a26*a34 + 2*a24^2*a36 + 
     2*a23*a26*a44 + a22*a36*a44 + 4*a23*a24*a46 + 2*a22*a34*a46) + 
   a12*(2*a26*a34^2 + 4*a24*a34*a36 + a26*a33*a44 + 2*a23*a36*a44 + 
     2*a24*a33*a46 + 4*a23*a34*a46))
 if ((r[1]==1) & (r[2]==2) & (r[3]==2) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
2*a15*(2*a24*a25*a33 + 4*a23*a25*a34 + 4*a23*a24*a35 + 2*a22*a34*a35 + 
   2*a23^2*a45 + a22*a33*a45) + a14*(2*a25^2*a33 + 8*a23*a25*a35 + 
   2*a22*a35^2 + 2*a23^2*a55 + a22*a33*a55) + 
 2*(a13*(2*a25^2*a34 + 4*a24*a25*a35 + 4*a23*a25*a45 + 2*a22*a35*a45 + 
     2*a23*a24*a55 + a22*a34*a55) + a12*(4*a25*a34*a35 + 2*a24*a35^2 + 
     2*a25*a33*a45 + 4*a23*a35*a45 + a24*a33*a55 + 2*a23*a34*a55))
 if ((r[1]==1) & (r[2]==2) & (r[3]==2) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
2*a14*a25*a26*a33 + 4*a13*a25*a26*a34 + 4*a14*a23*a26*a35 + 
 4*a13*a24*a26*a35 + 4*a12*a26*a34*a35 + 4*a14*a23*a25*a36 + 
 4*a13*a24*a25*a36 + 4*a12*a25*a34*a36 + 2*a14*a22*a35*a36 + 
 4*a12*a24*a35*a36 + 4*a13*a23*a26*a45 + 2*a12*a26*a33*a45 + 
 2*a13*a22*a36*a45 + 4*a12*a23*a36*a45 + 
 a16*(2*a24*a25*a33 + 4*a23*a25*a34 + 4*a23*a24*a35 + 2*a22*a34*a35 + 
   2*a23^2*a45 + a22*a33*a45) + 4*a13*a23*a25*a46 + 2*a12*a25*a33*a46 + 
 2*a13*a22*a35*a46 + 4*a12*a23*a35*a46 + 
 a15*(2*a24*a26*a33 + 4*a23*a26*a34 + 4*a23*a24*a36 + 2*a22*a34*a36 + 
   2*a23^2*a46 + a22*a33*a46) + 2*a14*a23^2*a56 + 4*a13*a23*a24*a56 + 
 a14*a22*a33*a56 + 2*a12*a24*a33*a56 + 2*a13*a22*a34*a56 + 4*a12*a23*a34*a56
 if ((r[1]==1) & (r[2]==2) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
2*a16*(2*a24*a26*a33 + 4*a23*a26*a34 + 4*a23*a24*a36 + 2*a22*a34*a36 + 
   2*a23^2*a46 + a22*a33*a46) + a14*(2*a26^2*a33 + 8*a23*a26*a36 + 
   2*a22*a36^2 + 2*a23^2*a66 + a22*a33*a66) + 
 2*(a13*(2*a26^2*a34 + 4*a24*a26*a36 + 4*a23*a26*a46 + 2*a22*a36*a46 + 
     2*a23*a24*a66 + a22*a34*a66) + a12*(4*a26*a34*a36 + 2*a24*a36^2 + 
     2*a26*a33*a46 + 4*a23*a36*a46 + a24*a33*a66 + 2*a23*a34*a66))
 if ((r[1]==1) & (r[2]==2) & (r[3]==2) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(a15*(2*a25^2*a33 + 8*a23*a25*a35 + 2*a22*a35^2 + 2*a23^2*a55 + 
    a22*a33*a55) + 2*(a13*(2*a25^2*a35 + 2*a23*a25*a55 + a22*a35*a55) + 
    a12*(2*a25*a35^2 + a25*a33*a55 + 2*a23*a35*a55)))
 if ((r[1]==1) & (r[2]==2) & (r[3]==2) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
a16*(2*a25^2*a33 + 8*a23*a25*a35 + 2*a22*a35^2 + 2*a23^2*a55 + a22*a33*a55) + 
 2*(a15*(2*a25*a26*a33 + 4*a23*a26*a35 + 4*a23*a25*a36 + 2*a22*a35*a36 + 
     2*a23^2*a56 + a22*a33*a56) + a13*(4*a25*a26*a35 + 2*a25^2*a36 + 
     2*a23*a26*a55 + a22*a36*a55 + 4*a23*a25*a56 + 2*a22*a35*a56) + 
   a12*(2*a26*a35^2 + 4*a25*a35*a36 + a26*a33*a55 + 2*a23*a36*a55 + 
     2*a25*a33*a56 + 4*a23*a35*a56))
 if ((r[1]==1) & (r[2]==2) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
2*a16*(2*a25*a26*a33 + 4*a23*a26*a35 + 4*a23*a25*a36 + 2*a22*a35*a36 + 
   2*a23^2*a56 + a22*a33*a56) + a15*(2*a26^2*a33 + 8*a23*a26*a36 + 
   2*a22*a36^2 + 2*a23^2*a66 + a22*a33*a66) + 
 2*(a13*(2*a26^2*a35 + 4*a25*a26*a36 + 4*a23*a26*a56 + 2*a22*a36*a56 + 
     2*a23*a25*a66 + a22*a35*a66) + a12*(4*a26*a35*a36 + 2*a25*a36^2 + 
     2*a26*a33*a56 + 4*a23*a36*a56 + a25*a33*a66 + 2*a23*a35*a66))
 if ((r[1]==1) & (r[2]==2) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(a16*(2*a26^2*a33 + 8*a23*a26*a36 + 2*a22*a36^2 + 2*a23^2*a66 + 
    a22*a33*a66) + 2*(a13*(2*a26^2*a36 + 2*a23*a26*a66 + a22*a36*a66) + 
    a12*(2*a26*a36^2 + a26*a33*a66 + 2*a23*a36*a66)))
 if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==4) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(a44*(4*a13*a24^2 + 8*a12*a24*a34 + a13*a22*a44 + 2*a12*a23*a44) + 
  4*a14*(2*a24^2*a34 + 2*a23*a24*a44 + a22*a34*a44))
 if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==3) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a13*a24*a25*a44 + 2*a12*a25*a34*a44 + 2*a12*a24*a35*a44 + 
  a15*(2*a24^2*a34 + 2*a23*a24*a44 + a22*a34*a44) + 2*a13*a24^2*a45 + 
  4*a12*a24*a34*a45 + a13*a22*a44*a45 + 2*a12*a23*a44*a45 + 
  a14*(4*a24*a25*a34 + 2*a24^2*a35 + 2*a23*a25*a44 + a22*a35*a44 + 
    4*a23*a24*a45 + 2*a22*a34*a45))
 if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==3) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a13*a24*a26*a44 + 2*a12*a26*a34*a44 + 2*a12*a24*a36*a44 + 
  a16*(2*a24^2*a34 + 2*a23*a24*a44 + a22*a34*a44) + 2*a13*a24^2*a46 + 
  4*a12*a24*a34*a46 + a13*a22*a44*a46 + 2*a12*a23*a44*a46 + 
  a14*(4*a24*a26*a34 + 2*a24^2*a36 + 2*a23*a26*a44 + a22*a36*a44 + 
    4*a23*a24*a46 + 2*a22*a34*a46))
  if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==2) 
& (r[5]==2) & (r[6]==0))
psi <-
2*a13*a25^2*a44 + 4*a12*a25*a35*a44 + 8*a13*a24*a25*a45 + 8*a12*a25*a34*a45 + 
 8*a12*a24*a35*a45 + 2*a13*a22*a45^2 + 4*a12*a23*a45^2 + 
 2*a15*(2*a24^2*a35 + 2*a23*a25*a44 + a22*a35*a44 + 2*a22*a34*a45 + 
   4*a24*(a25*a34 + a23*a45)) + 2*a13*a24^2*a55 + 4*a12*a24*a34*a55 + 
 a13*a22*a44*a55 + 2*a12*a23*a44*a55 + 
 2*a14*(2*a25^2*a34 + 4*a24*a25*a35 + 4*a23*a25*a45 + 2*a22*a35*a45 + 
   2*a23*a24*a55 + a22*a34*a55)
 if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==2) 
& (r[5]==1) & (r[6]==1))
psi <-
4*a14*a25*a26*a34 + 4*a14*a24*a26*a35 + 4*a14*a24*a25*a36 + 
 2*a13*a25*a26*a44 + 2*a12*a26*a35*a44 + 2*a12*a25*a36*a44 + 
 4*a14*a23*a26*a45 + 4*a13*a24*a26*a45 + 4*a12*a26*a34*a45 + 
 2*a14*a22*a36*a45 + 4*a12*a24*a36*a45 + 
 a16*(2*a24^2*a35 + 2*a23*a25*a44 + a22*a35*a44 + 2*a22*a34*a45 + 
   4*a24*(a25*a34 + a23*a45)) + 4*a14*a23*a25*a46 + 4*a13*a24*a25*a46 + 
 4*a12*a25*a34*a46 + 2*a14*a22*a35*a46 + 4*a12*a24*a35*a46 + 
 2*a13*a22*a45*a46 + 4*a12*a23*a45*a46 + 
 a15*(4*a24*a26*a34 + 2*a24^2*a36 + 2*a23*a26*a44 + a22*a36*a44 + 
   4*a23*a24*a46 + 2*a22*a34*a46) + 4*a14*a23*a24*a56 + 2*a13*a24^2*a56 + 
 2*a14*a22*a34*a56 + 4*a12*a24*a34*a56 + a13*a22*a44*a56 + 2*a12*a23*a44*a56
 if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==2))
psi <-
2*a13*a26^2*a44 + 4*a12*a26*a36*a44 + 8*a13*a24*a26*a46 + 8*a12*a26*a34*a46 + 
 8*a12*a24*a36*a46 + 2*a13*a22*a46^2 + 4*a12*a23*a46^2 + 
 2*a16*(2*a24^2*a36 + 2*a23*a26*a44 + a22*a36*a44 + 2*a22*a34*a46 + 
   4*a24*(a26*a34 + a23*a46)) + 2*a13*a24^2*a66 + 4*a12*a24*a34*a66 + 
 a13*a22*a44*a66 + 2*a12*a23*a44*a66 + 
 2*a14*(2*a26^2*a34 + 4*a24*a26*a36 + 4*a23*a26*a46 + 2*a22*a36*a46 + 
   2*a23*a24*a66 + a22*a34*a66)
 if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==1) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a13*a25^2*a45 + 4*a12*a25*a35*a45 + 2*a13*a24*a25*a55 + 
  2*a12*a25*a34*a55 + 2*a12*a24*a35*a55 + a13*a22*a45*a55 + 
  2*a12*a23*a45*a55 + a15*(2*a25^2*a34 + 2*a22*a35*a45 + 
    4*a25*(a24*a35 + a23*a45) + 2*a23*a24*a55 + a22*a34*a55) + 
  a14*(2*a25^2*a35 + 2*a23*a25*a55 + a22*a35*a55))
 if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==1) 
& (r[5]==2) & (r[6]==1))
psi <-
4*a14*a25*a26*a35 + 2*a14*a25^2*a36 + 4*a13*a25*a26*a45 + 4*a12*a26*a35*a45 + 
 4*a12*a25*a36*a45 + 2*a13*a25^2*a46 + 4*a12*a25*a35*a46 + 
 2*a14*a23*a26*a55 + 2*a13*a24*a26*a55 + 2*a12*a26*a34*a55 + 
 a14*a22*a36*a55 + 2*a12*a24*a36*a55 + a13*a22*a46*a55 + 2*a12*a23*a46*a55 + 
 a16*(2*a25^2*a34 + 2*a22*a35*a45 + 4*a25*(a24*a35 + a23*a45) + 
   2*a23*a24*a55 + a22*a34*a55) + 4*a14*a23*a25*a56 + 4*a13*a24*a25*a56 + 
 4*a12*a25*a34*a56 + 2*a14*a22*a35*a56 + 4*a12*a24*a35*a56 + 
 2*a13*a22*a45*a56 + 4*a12*a23*a45*a56 + 
 2*a15*(2*a24*a26*a35 + 2*a23*a26*a45 + a22*a36*a45 + a22*a35*a46 + 
   2*a25*(a26*a34 + a24*a36 + a23*a46) + 2*a23*a24*a56 + a22*a34*a56)
 if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==2))
psi <-
2*a14*a26^2*a35 + 4*a14*a25*a26*a36 + 2*a13*a26^2*a45 + 4*a12*a26*a36*a45 + 
 4*a13*a25*a26*a46 + 4*a12*a26*a35*a46 + 4*a12*a25*a36*a46 + 
 4*a14*a23*a26*a56 + 4*a13*a24*a26*a56 + 4*a12*a26*a34*a56 + 
 2*a14*a22*a36*a56 + 4*a12*a24*a36*a56 + 2*a13*a22*a46*a56 + 
 4*a12*a23*a46*a56 + 2*a16*(2*a24*a26*a35 + 2*a23*a26*a45 + a22*a36*a45 + 
   a22*a35*a46 + 2*a25*(a26*a34 + a24*a36 + a23*a46) + 2*a23*a24*a56 + 
   a22*a34*a56) + 2*a14*a23*a25*a66 + 2*a13*a24*a25*a66 + 2*a12*a25*a34*a66 + 
 a14*a22*a35*a66 + 2*a12*a24*a35*a66 + a13*a22*a45*a66 + 2*a12*a23*a45*a66 + 
 a15*(2*a26^2*a34 + 4*a24*a26*a36 + 4*a23*a26*a46 + 2*a22*a36*a46 + 
   2*a23*a24*a66 + a22*a34*a66)
 if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a13*a26^2*a46 + 4*a12*a26*a36*a46 + 2*a13*a24*a26*a66 + 
  2*a12*a26*a34*a66 + 2*a12*a24*a36*a66 + a13*a22*a46*a66 + 
  2*a12*a23*a46*a66 + a16*(2*a26^2*a34 + 2*a22*a36*a46 + 
    4*a26*(a24*a36 + a23*a46) + 2*a23*a24*a66 + a22*a34*a66) + 
  a14*(2*a26^2*a36 + 2*a23*a26*a66 + a22*a36*a66))
 if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==4) & (r[6]==0))
psi <-
3*(a55*(4*a13*a25^2 + 8*a12*a25*a35 + a13*a22*a55 + 2*a12*a23*a55) + 
  4*a15*(2*a25^2*a35 + 2*a23*a25*a55 + a22*a35*a55))
 if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==3) & (r[6]==1))
psi <-
3*(2*a13*a25*a26*a55 + 2*a12*a26*a35*a55 + 2*a12*a25*a36*a55 + 
  a16*(2*a25^2*a35 + 2*a23*a25*a55 + a22*a35*a55) + 2*a13*a25^2*a56 + 
  4*a12*a25*a35*a56 + a13*a22*a55*a56 + 2*a12*a23*a55*a56 + 
  a15*(4*a25*a26*a35 + 2*a25^2*a36 + 2*a23*a26*a55 + a22*a36*a55 + 
    4*a23*a25*a56 + 2*a22*a35*a56))
 if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==2))
psi <-
2*a13*a26^2*a55 + 4*a12*a26*a36*a55 + 8*a13*a25*a26*a56 + 8*a12*a26*a35*a56 + 
 8*a12*a25*a36*a56 + 2*a13*a22*a56^2 + 4*a12*a23*a56^2 + 
 2*a16*(2*a25^2*a36 + 2*a23*a26*a55 + a22*a36*a55 + 2*a22*a35*a56 + 
   4*a25*(a26*a35 + a23*a56)) + 2*a13*a25^2*a66 + 4*a12*a25*a35*a66 + 
 a13*a22*a55*a66 + 2*a12*a23*a55*a66 + 
 2*a15*(2*a26^2*a35 + 4*a25*a26*a36 + 4*a23*a26*a56 + 2*a22*a36*a56 + 
   2*a23*a25*a66 + a22*a35*a66)
 if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(2*a13*a26^2*a56 + 4*a12*a26*a36*a56 + 2*a13*a25*a26*a66 + 
  2*a12*a26*a35*a66 + 2*a12*a25*a36*a66 + a13*a22*a56*a66 + 
  2*a12*a23*a56*a66 + a16*(2*a26^2*a35 + 2*a22*a36*a56 + 
    4*a26*(a25*a36 + a23*a56) + 2*a23*a25*a66 + a22*a35*a66) + 
  a15*(2*a26^2*a36 + 2*a23*a26*a66 + a22*a36*a66))
 if ((r[1]==1) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==4))
psi <-
3*(a66*(4*a13*a26^2 + 8*a12*a26*a36 + a13*a22*a66 + 2*a12*a23*a66) + 
  4*a16*(2*a26^2*a36 + 2*a23*a26*a66 + a22*a36*a66))
 if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==5) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a44*(4*a14*a24^2 + a14*a22*a44 + 2*a12*a24*a44)
 if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==4) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(a15*a44*(4*a24^2 + a22*a44) + 2*a12*a44*(a25*a44 + 4*a24*a45) + 
  4*a14*(2*a24*a25*a44 + 2*a24^2*a45 + a22*a44*a45))
 if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==4) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(a16*a44*(4*a24^2 + a22*a44) + 2*a12*a44*(a26*a44 + 4*a24*a46) + 
  4*a14*(2*a24*a26*a44 + 2*a24^2*a46 + a22*a44*a46))
 if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==3) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a15*(2*a24*a25*a44 + 2*a24^2*a45 + a22*a44*a45) + 
  a14*(2*a25^2*a44 + 8*a24*a25*a45 + 2*a22*a45^2 + 2*a24^2*a55 + 
    a22*a44*a55) + 2*a12*(2*a25*a44*a45 + 2*a24*a45^2 + a24*a44*a55))
 if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==3) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(2*a14*a25*a26*a44 + 4*a14*a24*a26*a45 + 2*a12*a26*a44*a45 + 
  a16*(2*a24*a25*a44 + 2*a24^2*a45 + a22*a44*a45) + 4*a14*a24*a25*a46 + 
  2*a12*a25*a44*a46 + 2*a14*a22*a45*a46 + 4*a12*a24*a45*a46 + 
  a15*(2*a24*a26*a44 + 2*a24^2*a46 + a22*a44*a46) + 2*a14*a24^2*a56 + 
  a14*a22*a44*a56 + 2*a12*a24*a44*a56)
 if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a16*(2*a24*a26*a44 + 2*a24^2*a46 + a22*a44*a46) + 
  a14*(2*a26^2*a44 + 8*a24*a26*a46 + 2*a22*a46^2 + 2*a24^2*a66 + 
    a22*a44*a66) + 2*a12*(2*a26*a44*a46 + 2*a24*a46^2 + a24*a44*a66))
 if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==2) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(a15*(2*a25^2*a44 + 8*a24*a25*a45 + 2*a22*a45^2 + 2*a24^2*a55 + 
    a22*a44*a55) + 2*(a14*(2*a25^2*a45 + 2*a24*a25*a55 + a22*a45*a55) + 
    a12*(2*a25*a45^2 + a25*a44*a55 + 2*a24*a45*a55)))
 if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==2) 
& (r[5]==2) & (r[6]==1))
psi <-
a16*(2*a25^2*a44 + 8*a24*a25*a45 + 2*a22*a45^2 + 2*a24^2*a55 + a22*a44*a55) + 
 2*(a15*(2*a25*a26*a44 + 4*a24*a26*a45 + 4*a24*a25*a46 + 2*a22*a45*a46 + 
     2*a24^2*a56 + a22*a44*a56) + a14*(4*a25*a26*a45 + 2*a25^2*a46 + 
     2*a24*a26*a55 + a22*a46*a55 + 4*a24*a25*a56 + 2*a22*a45*a56) + 
   a12*(2*a26*a45^2 + 4*a25*a45*a46 + a26*a44*a55 + 2*a24*a46*a55 + 
     2*a25*a44*a56 + 4*a24*a45*a56))
 if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==2) & (r[5]==1) & (r[6]==2))
psi <-
2*a16*(2*a25*a26*a44 + 4*a24*a26*a45 + 4*a24*a25*a46 + 2*a22*a45*a46 + 
   2*a24^2*a56 + a22*a44*a56) + a15*(2*a26^2*a44 + 8*a24*a26*a46 + 
   2*a22*a46^2 + 2*a24^2*a66 + a22*a44*a66) + 
 2*(a14*(2*a26^2*a45 + 4*a25*a26*a46 + 4*a24*a26*a56 + 2*a22*a46*a56 + 
     2*a24*a25*a66 + a22*a45*a66) + a12*(4*a26*a45*a46 + 2*a25*a46^2 + 
     2*a26*a44*a56 + 4*a24*a46*a56 + a25*a44*a66 + 2*a24*a45*a66))

return(psi)
}

psins.6d.part83 <- function(r, a11, a22, a33, a44, a55, a66, a12, a13, a14, a15,
                           a16, a23, a24, a25, a26, a34, a35, a36, a45, a46, a56)
{
psi <- NA
  if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==2) & (r[5]==0) & (r[6]==3))
psi <-
3*(a16*(2*a26^2*a44 + 8*a24*a26*a46 + 2*a22*a46^2 + 2*a24^2*a66 + 
    a22*a44*a66) + 2*(a14*(2*a26^2*a46 + 2*a24*a26*a66 + a22*a46*a66) + 
    a12*(2*a26*a46^2 + a26*a44*a66 + 2*a24*a46*a66)))
 if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==4) & (r[6]==0))
psi <-
3*(a55*(4*a14*a25^2 + 8*a12*a25*a45 + a14*a22*a55 + 2*a12*a24*a55) + 
  4*a15*(2*a25^2*a45 + 2*a24*a25*a55 + a22*a45*a55))
 if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==3) & (r[6]==1))
psi <-
3*(2*a14*a25*a26*a55 + 2*a12*a26*a45*a55 + 2*a12*a25*a46*a55 + 
  a16*(2*a25^2*a45 + 2*a24*a25*a55 + a22*a45*a55) + 2*a14*a25^2*a56 + 
  4*a12*a25*a45*a56 + a14*a22*a55*a56 + 2*a12*a24*a55*a56 + 
  a15*(4*a25*a26*a45 + 2*a25^2*a46 + 2*a24*a26*a55 + a22*a46*a55 + 
    4*a24*a25*a56 + 2*a22*a45*a56))
 if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==2))
psi <-
2*a14*a26^2*a55 + 4*a12*a26*a46*a55 + 8*a14*a25*a26*a56 + 8*a12*a26*a45*a56 + 
 8*a12*a25*a46*a56 + 2*a14*a22*a56^2 + 4*a12*a24*a56^2 + 
 2*a16*(2*a25^2*a46 + 2*a24*a26*a55 + a22*a46*a55 + 2*a22*a45*a56 + 
   4*a25*(a26*a45 + a24*a56)) + 2*a14*a25^2*a66 + 4*a12*a25*a45*a66 + 
 a14*a22*a55*a66 + 2*a12*a24*a55*a66 + 
 2*a15*(2*a26^2*a45 + 4*a25*a26*a46 + 4*a24*a26*a56 + 2*a22*a46*a56 + 
   2*a24*a25*a66 + a22*a45*a66)
 if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(2*a14*a26^2*a56 + 4*a12*a26*a46*a56 + 2*a14*a25*a26*a66 + 
  2*a12*a26*a45*a66 + 2*a12*a25*a46*a66 + a14*a22*a56*a66 + 
  2*a12*a24*a56*a66 + a16*(2*a26^2*a45 + 2*a22*a46*a56 + 
    4*a26*(a25*a46 + a24*a56) + 2*a24*a25*a66 + a22*a45*a66) + 
  a15*(2*a26^2*a46 + 2*a24*a26*a66 + a22*a46*a66))
 if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==4))
psi <-
3*(a66*(4*a14*a26^2 + 8*a12*a26*a46 + a14*a22*a66 + 2*a12*a24*a66) + 
  4*a16*(2*a26^2*a46 + 2*a24*a26*a66 + a22*a46*a66))
 if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==5) & (r[6]==0))
psi <-
15*a55*(4*a15*a25^2 + a15*a22*a55 + 2*a12*a25*a55)
 if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==4) & (r[6]==1))
psi <-
3*(a16*a55*(4*a25^2 + a22*a55) + 2*a12*a55*(a26*a55 + 4*a25*a56) + 
  4*a15*(2*a25*a26*a55 + 2*a25^2*a56 + a22*a55*a56))
 if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==2))
psi <-
3*(2*a16*(2*a25*a26*a55 + 2*a25^2*a56 + a22*a55*a56) + 
  a15*(2*a26^2*a55 + 8*a25*a26*a56 + 2*a22*a56^2 + 2*a25^2*a66 + 
    a22*a55*a66) + 2*a12*(2*a26*a55*a56 + 2*a25*a56^2 + a25*a55*a66))
 if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==3))
psi <-
3*(a16*(2*a26^2*a55 + 8*a25*a26*a56 + 2*a22*a56^2 + 2*a25^2*a66 + 
    a22*a55*a66) + 2*(a15*(2*a26^2*a56 + 2*a25*a26*a66 + a22*a56*a66) + 
    a12*(2*a26*a56^2 + a26*a55*a66 + 2*a25*a56*a66)))
 if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==4))
psi <-
3*(a66*(4*a15*a26^2 + 8*a12*a26*a56 + a15*a22*a66 + 2*a12*a25*a66) + 
  4*a16*(2*a26^2*a56 + 2*a25*a26*a66 + a22*a56*a66))
 if ((r[1]==1) & (r[2]==2) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==5))
psi <-
15*a66*(4*a16*a26^2 + a16*a22*a66 + 2*a12*a26*a66)
 if ((r[1]==1) & (r[2]==1) & (r[3]==6) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a33^2*(6*a13*a23 + a12*a33)
 if ((r[1]==1) & (r[2]==1) & (r[3]==5) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a33*(a14*a23*a33 + a13*a24*a33 + 4*a13*a23*a34 + a12*a33*a34)
 if ((r[1]==1) & (r[2]==1) & (r[3]==5) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a33*(a15*a23*a33 + a13*a25*a33 + 4*a13*a23*a35 + a12*a33*a35)
 if ((r[1]==1) & (r[2]==1) & (r[3]==5) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a33*(a16*a23*a33 + a13*a26*a33 + 4*a13*a23*a36 + a12*a33*a36)
 if ((r[1]==1) & (r[2]==1) & (r[3]==4) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a14*a33*(a24*a33 + 4*a23*a34) + a12*a33*(4*a34^2 + a33*a44) + 
  4*a13*(2*a24*a33*a34 + 2*a23*a34^2 + a23*a33*a44))
 if ((r[1]==1) & (r[2]==1) & (r[3]==4) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(4*a13*a25*a33*a34 + a15*a33*(a24*a33 + 4*a23*a34) + 4*a13*a24*a33*a35 + 
  8*a13*a23*a34*a35 + 4*a12*a33*a34*a35 + a14*a33*(a25*a33 + 4*a23*a35) + 
  4*a13*a23*a33*a45 + a12*a33^2*a45)
 if ((r[1]==1) & (r[2]==1) & (r[3]==4) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(4*a13*a26*a33*a34 + a16*a33*(a24*a33 + 4*a23*a34) + 4*a13*a24*a33*a36 + 
  8*a13*a23*a34*a36 + 4*a12*a33*a34*a36 + a14*a33*(a26*a33 + 4*a23*a36) + 
  4*a13*a23*a33*a46 + a12*a33^2*a46)
 if ((r[1]==1) & (r[2]==1) & (r[3]==4) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a15*a33*(a25*a33 + 4*a23*a35) + a12*a33*(4*a35^2 + a33*a55) + 
  4*a13*(2*a25*a33*a35 + 2*a23*a35^2 + a23*a33*a55))
 if ((r[1]==1) & (r[2]==1) & (r[3]==4) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(4*a13*a26*a33*a35 + a16*a33*(a25*a33 + 4*a23*a35) + 4*a13*a25*a33*a36 + 
  8*a13*a23*a35*a36 + 4*a12*a33*a35*a36 + a15*a33*(a26*a33 + 4*a23*a36) + 
  4*a13*a23*a33*a56 + a12*a33^2*a56)
 if ((r[1]==1) & (r[2]==1) & (r[3]==4) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a16*a33*(a26*a33 + 4*a23*a36) + a12*a33*(4*a36^2 + a33*a66) + 
  4*a13*(2*a26*a33*a36 + 2*a23*a36^2 + a23*a33*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==3) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(6*a13*a24*a34^2 + 2*a12*a34^3 + 3*a13*a24*a33*a44 + 6*a13*a23*a34*a44 + 
  3*a12*a33*a34*a44 + 3*a14*(2*a24*a33*a34 + 2*a23*a34^2 + a23*a33*a44))
 if ((r[1]==1) & (r[2]==1) & (r[3]==3) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a13*a25*a34^2 + 4*a13*a24*a34*a35 + 2*a12*a34^2*a35 + a13*a25*a33*a44 + 
  2*a13*a23*a35*a44 + a12*a33*a35*a44 + 
  a15*(2*a24*a33*a34 + 2*a23*a34^2 + a23*a33*a44) + 2*a13*a24*a33*a45 + 
  4*a13*a23*a34*a45 + 2*a12*a33*a34*a45 + 
  2*a14*(a25*a33*a34 + a24*a33*a35 + 2*a23*a34*a35 + a23*a33*a45))
 if ((r[1]==1) & (r[2]==1) & (r[3]==3) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a13*a26*a34^2 + 4*a13*a24*a34*a36 + 2*a12*a34^2*a36 + a13*a26*a33*a44 + 
  2*a13*a23*a36*a44 + a12*a33*a36*a44 + 
  a16*(2*a24*a33*a34 + 2*a23*a34^2 + a23*a33*a44) + 2*a13*a24*a33*a46 + 
  4*a13*a23*a34*a46 + 2*a12*a33*a34*a46 + 
  2*a14*(a26*a33*a34 + a24*a33*a36 + 2*a23*a34*a36 + a23*a33*a46))
 if ((r[1]==1) & (r[2]==1) & (r[3]==3) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(4*a13*a25*a34*a35 + 2*a13*a24*a35^2 + 2*a12*a34*a35^2 + 
  2*a13*a25*a33*a45 + 4*a13*a23*a35*a45 + 2*a12*a33*a35*a45 + 
  2*a15*(a25*a33*a34 + a24*a33*a35 + 2*a23*a34*a35 + a23*a33*a45) + 
  a13*a24*a33*a55 + 2*a13*a23*a34*a55 + a12*a33*a34*a55 + 
  a14*(2*a25*a33*a35 + 2*a23*a35^2 + a23*a33*a55))
 if ((r[1]==1) & (r[2]==1) & (r[3]==3) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(a14*a26*a33*a35 + 2*a13*a26*a34*a35 + a14*a25*a33*a36 + 
  2*a13*a25*a34*a36 + 2*a14*a23*a35*a36 + 2*a13*a24*a35*a36 + 
  2*a12*a34*a35*a36 + a13*a26*a33*a45 + 2*a13*a23*a36*a45 + a12*a33*a36*a45 + 
  a16*(a25*a33*a34 + a24*a33*a35 + 2*a23*a34*a35 + a23*a33*a45) + 
  a13*a25*a33*a46 + 2*a13*a23*a35*a46 + a12*a33*a35*a46 + 
  a15*(a26*a33*a34 + a24*a33*a36 + 2*a23*a34*a36 + a23*a33*a46) + 
  a14*a23*a33*a56 + a13*a24*a33*a56 + 2*a13*a23*a34*a56 + a12*a33*a34*a56)
  if ((r[1]==1) & (r[2]==1) & (r[3]==3) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(4*a13*a26*a34*a36 + 2*a13*a24*a36^2 + 2*a12*a34*a36^2 + 
  2*a13*a26*a33*a46 + 4*a13*a23*a36*a46 + 2*a12*a33*a36*a46 + 
  2*a16*(a26*a33*a34 + a24*a33*a36 + 2*a23*a34*a36 + a23*a33*a46) + 
  a13*a24*a33*a66 + 2*a13*a23*a34*a66 + a12*a33*a34*a66 + 
  a14*(2*a26*a33*a36 + 2*a23*a36^2 + a23*a33*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==3) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(6*a13*a25*a35^2 + 2*a12*a35^3 + 3*a13*a25*a33*a55 + 6*a13*a23*a35*a55 + 
  3*a12*a33*a35*a55 + 3*a15*(2*a25*a33*a35 + 2*a23*a35^2 + a23*a33*a55))
 if ((r[1]==1) & (r[2]==1) & (r[3]==3) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(2*a13*a26*a35^2 + 4*a13*a25*a35*a36 + 2*a12*a35^2*a36 + a13*a26*a33*a55 + 
  2*a13*a23*a36*a55 + a12*a33*a36*a55 + 
  a16*(2*a25*a33*a35 + 2*a23*a35^2 + a23*a33*a55) + 2*a13*a25*a33*a56 + 
  4*a13*a23*a35*a56 + 2*a12*a33*a35*a56 + 
  2*a15*(a26*a33*a35 + a25*a33*a36 + 2*a23*a35*a36 + a23*a33*a56))
 if ((r[1]==1) & (r[2]==1) & (r[3]==3) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(4*a13*a26*a35*a36 + 2*a13*a25*a36^2 + 2*a12*a35*a36^2 + 
  2*a13*a26*a33*a56 + 4*a13*a23*a36*a56 + 2*a12*a33*a36*a56 + 
  2*a16*(a26*a33*a35 + a25*a33*a36 + 2*a23*a35*a36 + a23*a33*a56) + 
  a13*a25*a33*a66 + 2*a13*a23*a35*a66 + a12*a33*a35*a66 + 
  a15*(2*a26*a33*a36 + 2*a23*a36^2 + a23*a33*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(6*a13*a26*a36^2 + 2*a12*a36^3 + 3*a13*a26*a33*a66 + 6*a13*a23*a36*a66 + 
  3*a12*a33*a36*a66 + 3*a16*(2*a26*a33*a36 + 2*a23*a36^2 + a23*a33*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==4) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(a44*(8*a13*a24*a34 + 4*a12*a34^2 + 2*a13*a23*a44 + a12*a33*a44) + 
  4*a14*(2*a24*a34^2 + a24*a33*a44 + 2*a23*a34*a44))
 if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==3) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a13*a25*a34*a44 + 2*a13*a24*a35*a44 + 2*a12*a34*a35*a44 + 
  a15*(2*a24*a34^2 + a24*a33*a44 + 2*a23*a34*a44) + 4*a13*a24*a34*a45 + 
  2*a12*a34^2*a45 + 2*a13*a23*a44*a45 + a12*a33*a44*a45 + 
  a14*(2*a25*a34^2 + 4*a24*a34*a35 + a25*a33*a44 + 2*a23*a35*a44 + 
    2*a24*a33*a45 + 4*a23*a34*a45))
 if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==3) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a13*a26*a34*a44 + 2*a13*a24*a36*a44 + 2*a12*a34*a36*a44 + 
  a16*(2*a24*a34^2 + a24*a33*a44 + 2*a23*a34*a44) + 4*a13*a24*a34*a46 + 
  2*a12*a34^2*a46 + 2*a13*a23*a44*a46 + a12*a33*a44*a46 + 
  a14*(2*a26*a34^2 + 4*a24*a34*a36 + a26*a33*a44 + 2*a23*a36*a44 + 
    2*a24*a33*a46 + 4*a23*a34*a46))
 if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==2) 
& (r[5]==2) & (r[6]==0))
psi <-
4*a13*a25*a35*a44 + 2*a12*a35^2*a44 + 8*a13*a25*a34*a45 + 8*a13*a24*a35*a45 + 
 8*a12*a34*a35*a45 + 4*a13*a23*a45^2 + 2*a12*a33*a45^2 + 
 2*a15*(2*a25*a34^2 + 4*a24*a34*a35 + a25*a33*a44 + 2*a23*a35*a44 + 
   2*a24*a33*a45 + 4*a23*a34*a45) + 4*a13*a24*a34*a55 + 2*a12*a34^2*a55 + 
 2*a13*a23*a44*a55 + a12*a33*a44*a55 + 
 2*a14*(4*a25*a34*a35 + 2*a24*a35^2 + 2*a25*a33*a45 + 4*a23*a35*a45 + 
   a24*a33*a55 + 2*a23*a34*a55)
 if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==2) 
& (r[5]==1) & (r[6]==1))
psi <-
4*a14*a26*a34*a35 + 4*a14*a25*a34*a36 + 4*a14*a24*a35*a36 + 
 2*a13*a26*a35*a44 + 2*a13*a25*a36*a44 + 2*a12*a35*a36*a44 + 
 2*a14*a26*a33*a45 + 4*a13*a26*a34*a45 + 4*a14*a23*a36*a45 + 
 4*a13*a24*a36*a45 + 4*a12*a34*a36*a45 + 
 a16*(2*a25*a34^2 + 4*a24*a34*a35 + a25*a33*a44 + 2*a23*a35*a44 + 
   2*a24*a33*a45 + 4*a23*a34*a45) + 2*a14*a25*a33*a46 + 4*a13*a25*a34*a46 + 
 4*a14*a23*a35*a46 + 4*a13*a24*a35*a46 + 4*a12*a34*a35*a46 + 
 4*a13*a23*a45*a46 + 2*a12*a33*a45*a46 + 
 a15*(2*a26*a34^2 + 4*a24*a34*a36 + a26*a33*a44 + 2*a23*a36*a44 + 
   2*a24*a33*a46 + 4*a23*a34*a46) + 2*a14*a24*a33*a56 + 4*a14*a23*a34*a56 + 
 4*a13*a24*a34*a56 + 2*a12*a34^2*a56 + 2*a13*a23*a44*a56 + a12*a33*a44*a56
 if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==2) 
& (r[5]==0) & (r[6]==2))
psi <-
4*a13*a26*a36*a44 + 2*a12*a36^2*a44 + 8*a13*a26*a34*a46 + 8*a13*a24*a36*a46 + 
 8*a12*a34*a36*a46 + 4*a13*a23*a46^2 + 2*a12*a33*a46^2 + 
 2*a16*(2*a26*a34^2 + 4*a24*a34*a36 + a26*a33*a44 + 2*a23*a36*a44 + 
   2*a24*a33*a46 + 4*a23*a34*a46) + 4*a13*a24*a34*a66 + 2*a12*a34^2*a66 + 
 2*a13*a23*a44*a66 + a12*a33*a44*a66 + 
 2*a14*(4*a26*a34*a36 + 2*a24*a36^2 + 2*a26*a33*a46 + 4*a23*a36*a46 + 
   a24*a33*a66 + 2*a23*a34*a66)
 if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==1) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(4*a13*a25*a35*a45 + 2*a12*a35^2*a45 + 2*a13*a25*a34*a55 + 
  2*a13*a24*a35*a55 + 2*a12*a34*a35*a55 + 2*a13*a23*a45*a55 + 
  a12*a33*a45*a55 + a15*(4*a25*a34*a35 + 2*a24*a35^2 + 2*a25*a33*a45 + 
    4*a23*a35*a45 + a24*a33*a55 + 2*a23*a34*a55) + 
  a14*(2*a25*a35^2 + a25*a33*a55 + 2*a23*a35*a55))
 if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==1) 
& (r[5]==2) & (r[6]==1))
psi <-
2*a14*a26*a35^2 + 4*a14*a25*a35*a36 + 4*a13*a26*a35*a45 + 4*a13*a25*a36*a45 + 
 4*a12*a35*a36*a45 + 4*a13*a25*a35*a46 + 2*a12*a35^2*a46 + a14*a26*a33*a55 + 
 2*a13*a26*a34*a55 + 2*a14*a23*a36*a55 + 2*a13*a24*a36*a55 + 
 2*a12*a34*a36*a55 + 2*a13*a23*a46*a55 + a12*a33*a46*a55 + 
 a16*(4*a25*a34*a35 + 2*a24*a35^2 + 2*a25*a33*a45 + 4*a23*a35*a45 + 
   a24*a33*a55 + 2*a23*a34*a55) + 2*a14*a25*a33*a56 + 4*a13*a25*a34*a56 + 
 4*a14*a23*a35*a56 + 4*a13*a24*a35*a56 + 4*a12*a34*a35*a56 + 
 4*a13*a23*a45*a56 + 2*a12*a33*a45*a56 + 
 2*a15*(2*a26*a34*a35 + 2*a25*a34*a36 + 2*a24*a35*a36 + a26*a33*a45 + 
   2*a23*a36*a45 + a25*a33*a46 + 2*a23*a35*a46 + a24*a33*a56 + 2*a23*a34*a56)
 if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==1) 
& (r[5]==1) & (r[6]==2))
psi <-
4*a14*a26*a35*a36 + 2*a14*a25*a36^2 + 4*a13*a26*a36*a45 + 2*a12*a36^2*a45 + 
 4*a13*a26*a35*a46 + 4*a13*a25*a36*a46 + 4*a12*a35*a36*a46 + 
 2*a14*a26*a33*a56 + 4*a13*a26*a34*a56 + 4*a14*a23*a36*a56 + 
 4*a13*a24*a36*a56 + 4*a12*a34*a36*a56 + 4*a13*a23*a46*a56 + 
 2*a12*a33*a46*a56 + 2*a16*(2*a26*a34*a35 + 2*a25*a34*a36 + 2*a24*a35*a36 + 
   a26*a33*a45 + 2*a23*a36*a45 + a25*a33*a46 + 2*a23*a35*a46 + a24*a33*a56 + 
   2*a23*a34*a56) + a14*a25*a33*a66 + 2*a13*a25*a34*a66 + 2*a14*a23*a35*a66 + 
 2*a13*a24*a35*a66 + 2*a12*a34*a35*a66 + 2*a13*a23*a45*a66 + 
 a12*a33*a45*a66 + a15*(4*a26*a34*a36 + 2*a24*a36^2 + 2*a26*a33*a46 + 
   4*a23*a36*a46 + a24*a33*a66 + 2*a23*a34*a66)
 if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(4*a13*a26*a36*a46 + 2*a12*a36^2*a46 + 2*a13*a26*a34*a66 + 
  2*a13*a24*a36*a66 + 2*a12*a34*a36*a66 + 2*a13*a23*a46*a66 + 
  a12*a33*a46*a66 + a16*(4*a26*a34*a36 + 2*a24*a36^2 + 2*a26*a33*a46 + 
    4*a23*a36*a46 + a24*a33*a66 + 2*a23*a34*a66) + 
  a14*(2*a26*a36^2 + a26*a33*a66 + 2*a23*a36*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==0) 
& (r[5]==4) & (r[6]==0))
psi <-
3*(a55*(8*a13*a25*a35 + 4*a12*a35^2 + 2*a13*a23*a55 + a12*a33*a55) + 
  4*a15*(2*a25*a35^2 + a25*a33*a55 + 2*a23*a35*a55))
 if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==0) 
& (r[5]==3) & (r[6]==1))
psi <-
3*(2*a13*a26*a35*a55 + 2*a13*a25*a36*a55 + 2*a12*a35*a36*a55 + 
  a16*(2*a25*a35^2 + a25*a33*a55 + 2*a23*a35*a55) + 4*a13*a25*a35*a56 + 
  2*a12*a35^2*a56 + 2*a13*a23*a55*a56 + a12*a33*a55*a56 + 
  a15*(2*a26*a35^2 + 4*a25*a35*a36 + a26*a33*a55 + 2*a23*a36*a55 + 
    2*a25*a33*a56 + 4*a23*a35*a56))
 if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==0) 
& (r[5]==2) & (r[6]==2))
psi <-
4*a13*a26*a36*a55 + 2*a12*a36^2*a55 + 8*a13*a26*a35*a56 + 8*a13*a25*a36*a56 + 
 8*a12*a35*a36*a56 + 4*a13*a23*a56^2 + 2*a12*a33*a56^2 + 
 2*a16*(2*a26*a35^2 + 4*a25*a35*a36 + a26*a33*a55 + 2*a23*a36*a55 + 
   2*a25*a33*a56 + 4*a23*a35*a56) + 4*a13*a25*a35*a66 + 2*a12*a35^2*a66 + 
 2*a13*a23*a55*a66 + a12*a33*a55*a66 + 
 2*a15*(4*a26*a35*a36 + 2*a25*a36^2 + 2*a26*a33*a56 + 4*a23*a36*a56 + 
   a25*a33*a66 + 2*a23*a35*a66)
 if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(4*a13*a26*a36*a56 + 2*a12*a36^2*a56 + 2*a13*a26*a35*a66 + 
  2*a13*a25*a36*a66 + 2*a12*a35*a36*a66 + 2*a13*a23*a56*a66 + 
  a12*a33*a56*a66 + a16*(4*a26*a35*a36 + 2*a25*a36^2 + 2*a26*a33*a56 + 
    4*a23*a36*a56 + a25*a33*a66 + 2*a23*a35*a66) + 
  a15*(2*a26*a36^2 + a26*a33*a66 + 2*a23*a36*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==4))
psi <-
3*(a66*(8*a13*a26*a36 + 4*a12*a36^2 + 2*a13*a23*a66 + a12*a33*a66) + 
  4*a16*(2*a26*a36^2 + a26*a33*a66 + 2*a23*a36*a66))
  if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==5) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a44*(4*a14*a24*a34 + a14*a23*a44 + a13*a24*a44 + a12*a34*a44)
 if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==4) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(a15*a44*(4*a24*a34 + a23*a44) + a44*(a13*a25*a44 + a12*a35*a44 + 
    4*a13*a24*a45 + 4*a12*a34*a45) + 
  4*a14*(a25*a34*a44 + a24*a35*a44 + 2*a24*a34*a45 + a23*a44*a45))
 if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==4) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(a16*a44*(4*a24*a34 + a23*a44) + a44*(a13*a26*a44 + a12*a36*a44 + 
    4*a13*a24*a46 + 4*a12*a34*a46) + 
  4*a14*(a26*a34*a44 + a24*a36*a44 + 2*a24*a34*a46 + a23*a44*a46))
 if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==3) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a13*a25*a44*a45 + 2*a12*a35*a44*a45 + 2*a13*a24*a45^2 + 
  2*a12*a34*a45^2 + 2*a15*(a25*a34*a44 + a24*a35*a44 + 2*a24*a34*a45 + 
    a23*a44*a45) + a13*a24*a44*a55 + a12*a34*a44*a55 + 
  a14*(2*a25*a35*a44 + 4*a25*a34*a45 + 4*a24*a35*a45 + 2*a23*a45^2 + 
    2*a24*a34*a55 + a23*a44*a55))
 if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==3) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(a14*a26*a35*a44 + a14*a25*a36*a44 + 2*a14*a26*a34*a45 + 
  2*a14*a24*a36*a45 + a13*a26*a44*a45 + a12*a36*a44*a45 + 
  a16*(a25*a34*a44 + a24*a35*a44 + 2*a24*a34*a45 + a23*a44*a45) + 
  2*a14*a25*a34*a46 + 2*a14*a24*a35*a46 + a13*a25*a44*a46 + a12*a35*a44*a46 + 
  2*a14*a23*a45*a46 + 2*a13*a24*a45*a46 + 2*a12*a34*a45*a46 + 
  a15*(a26*a34*a44 + a24*a36*a44 + 2*a24*a34*a46 + a23*a44*a46) + 
  2*a14*a24*a34*a56 + a14*a23*a44*a56 + a13*a24*a44*a56 + a12*a34*a44*a56)
 if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==3) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a13*a26*a44*a46 + 2*a12*a36*a44*a46 + 2*a13*a24*a46^2 + 
  2*a12*a34*a46^2 + 2*a16*(a26*a34*a44 + a24*a36*a44 + 2*a24*a34*a46 + 
    a23*a44*a46) + a13*a24*a44*a66 + a12*a34*a44*a66 + 
  a14*(2*a26*a36*a44 + 4*a26*a34*a46 + 4*a24*a36*a46 + 2*a23*a46^2 + 
    2*a24*a34*a66 + a23*a44*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==2) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a13*a25*a45^2 + 2*a12*a35*a45^2 + a13*a25*a44*a55 + a12*a35*a44*a55 + 
  2*a13*a24*a45*a55 + 2*a12*a34*a45*a55 + 
  a15*(2*a25*a35*a44 + 4*a25*a34*a45 + 4*a24*a35*a45 + 2*a23*a45^2 + 
    2*a24*a34*a55 + a23*a44*a55) + 2*a14*(2*a25*a35*a45 + a25*a34*a55 + 
    a24*a35*a55 + a23*a45*a55))
 if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==2) 
& (r[5]==2) & (r[6]==1))
psi <-
4*a14*a26*a35*a45 + 4*a14*a25*a36*a45 + 2*a13*a26*a45^2 + 2*a12*a36*a45^2 + 
 4*a14*a25*a35*a46 + 4*a13*a25*a45*a46 + 4*a12*a35*a45*a46 + 
 2*a14*a26*a34*a55 + 2*a14*a24*a36*a55 + a13*a26*a44*a55 + a12*a36*a44*a55 + 
 2*a14*a23*a46*a55 + 2*a13*a24*a46*a55 + 2*a12*a34*a46*a55 + 
 a16*(2*a25*a35*a44 + 4*a25*a34*a45 + 4*a24*a35*a45 + 2*a23*a45^2 + 
   2*a24*a34*a55 + a23*a44*a55) + 4*a14*a25*a34*a56 + 4*a14*a24*a35*a56 + 
 2*a13*a25*a44*a56 + 2*a12*a35*a44*a56 + 4*a14*a23*a45*a56 + 
 4*a13*a24*a45*a56 + 4*a12*a34*a45*a56 + 
 2*a15*(a26*a35*a44 + a25*a36*a44 + 2*a26*a34*a45 + 2*a24*a36*a45 + 
   2*a25*a34*a46 + 2*a24*a35*a46 + 2*a23*a45*a46 + 2*a24*a34*a56 + 
   a23*a44*a56)
 if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==2) 
& (r[5]==1) & (r[6]==2))
psi <-
4*a14*a26*a36*a45 + 4*a14*a26*a35*a46 + 4*a14*a25*a36*a46 + 
 4*a13*a26*a45*a46 + 4*a12*a36*a45*a46 + 2*a13*a25*a46^2 + 2*a12*a35*a46^2 + 
 4*a14*a26*a34*a56 + 4*a14*a24*a36*a56 + 2*a13*a26*a44*a56 + 
 2*a12*a36*a44*a56 + 4*a14*a23*a46*a56 + 4*a13*a24*a46*a56 + 
 4*a12*a34*a46*a56 + 2*a16*(a26*a35*a44 + a25*a36*a44 + 2*a26*a34*a45 + 
   2*a24*a36*a45 + 2*a25*a34*a46 + 2*a24*a35*a46 + 2*a23*a45*a46 + 
   2*a24*a34*a56 + a23*a44*a56) + 2*a14*a25*a34*a66 + 2*a14*a24*a35*a66 + 
 a13*a25*a44*a66 + a12*a35*a44*a66 + 2*a14*a23*a45*a66 + 2*a13*a24*a45*a66 + 
 2*a12*a34*a45*a66 + a15*(2*a26*a36*a44 + 4*a26*a34*a46 + 4*a24*a36*a46 + 
   2*a23*a46^2 + 2*a24*a34*a66 + a23*a44*a66)
 if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a13*a26*a46^2 + 2*a12*a36*a46^2 + a13*a26*a44*a66 + a12*a36*a44*a66 + 
  2*a13*a24*a46*a66 + 2*a12*a34*a46*a66 + 
  a16*(2*a26*a36*a44 + 4*a26*a34*a46 + 4*a24*a36*a46 + 2*a23*a46^2 + 
    2*a24*a34*a66 + a23*a44*a66) + 2*a14*(2*a26*a36*a46 + a26*a34*a66 + 
    a24*a36*a66 + a23*a46*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==4) & (r[6]==0))
psi <-
3*(a55*(4*a14*a25*a35 + 4*a13*a25*a45 + 4*a12*a35*a45 + a14*a23*a55 + 
    a13*a24*a55 + a12*a34*a55) + 4*a15*(2*a25*a35*a45 + a25*a34*a55 + 
    a24*a35*a55 + a23*a45*a55))
 if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==3) & (r[6]==1))
psi <-
3*(a14*a26*a35*a55 + a14*a25*a36*a55 + a13*a26*a45*a55 + a12*a36*a45*a55 + 
  a13*a25*a46*a55 + a12*a35*a46*a55 + 
  a16*(2*a25*a35*a45 + a25*a34*a55 + a24*a35*a55 + a23*a45*a55) + 
  2*a14*a25*a35*a56 + 2*a13*a25*a45*a56 + 2*a12*a35*a45*a56 + 
  a14*a23*a55*a56 + a13*a24*a55*a56 + a12*a34*a55*a56 + 
  a15*(a24*a36*a55 + a23*a46*a55 + a26*(2*a35*a45 + a34*a55) + 
    2*a24*a35*a56 + 2*a23*a45*a56 + 2*a25*(a36*a45 + a35*a46 + a34*a56)))
 if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==2) & (r[6]==2))
psi <-
2*a14*a26*a36*a55 + 2*a13*a26*a46*a55 + 2*a12*a36*a46*a55 + 
 4*a14*a26*a35*a56 + 4*a14*a25*a36*a56 + 4*a13*a26*a45*a56 + 
 4*a12*a36*a45*a56 + 4*a13*a25*a46*a56 + 4*a12*a35*a46*a56 + 
 2*a14*a23*a56^2 + 2*a13*a24*a56^2 + 2*a12*a34*a56^2 + 
 2*a16*(a24*a36*a55 + a23*a46*a55 + a26*(2*a35*a45 + a34*a55) + 
   2*a24*a35*a56 + 2*a23*a45*a56 + 2*a25*(a36*a45 + a35*a46 + a34*a56)) + 
 2*a14*a25*a35*a66 + 2*a13*a25*a45*a66 + 2*a12*a35*a45*a66 + 
 a14*a23*a55*a66 + a13*a24*a55*a66 + a12*a34*a55*a66 + 
 2*a15*(2*a25*a36*a46 + 2*a24*a36*a56 + 2*a23*a46*a56 + 
   2*a26*(a36*a45 + a35*a46 + a34*a56) + a25*a34*a66 + a24*a35*a66 + 
   a23*a45*a66)
 if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(2*a14*a26*a36*a56 + 2*a13*a26*a46*a56 + 2*a12*a36*a46*a56 + 
  a14*a26*a35*a66 + a14*a25*a36*a66 + a13*a26*a45*a66 + a12*a36*a45*a66 + 
  a13*a25*a46*a66 + a12*a35*a46*a66 + a14*a23*a56*a66 + a13*a24*a56*a66 + 
  a12*a34*a56*a66 + a16*(2*a25*a36*a46 + 2*a24*a36*a56 + 2*a23*a46*a56 + 
    2*a26*(a36*a45 + a35*a46 + a34*a56) + a25*a34*a66 + a24*a35*a66 + 
    a23*a45*a66) + a15*(2*a26*a36*a46 + a26*a34*a66 + a24*a36*a66 + 
    a23*a46*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==4))
psi <-
3*(a66*(4*a14*a26*a36 + 4*a13*a26*a46 + 4*a12*a36*a46 + a14*a23*a66 + 
    a13*a24*a66 + a12*a34*a66) + 4*a16*(2*a26*a36*a46 + a26*a34*a66 + 
    a24*a36*a66 + a23*a46*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==5) & (r[6]==0))
psi <-
15*a55*(4*a15*a25*a35 + a15*a23*a55 + a13*a25*a55 + a12*a35*a55)
 if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==4) & (r[6]==1))
psi <-
3*(a16*a55*(4*a25*a35 + a23*a55) + a55*(a13*a26*a55 + a12*a36*a55 + 
    4*a13*a25*a56 + 4*a12*a35*a56) + 
  4*a15*(a26*a35*a55 + a25*a36*a55 + 2*a25*a35*a56 + a23*a55*a56))
 if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==3) & (r[6]==2))
psi <-
3*(2*a13*a26*a55*a56 + 2*a12*a36*a55*a56 + 2*a13*a25*a56^2 + 
  2*a12*a35*a56^2 + 2*a16*(a26*a35*a55 + a25*a36*a55 + 2*a25*a35*a56 + 
    a23*a55*a56) + a13*a25*a55*a66 + a12*a35*a55*a66 + 
  a15*(2*a26*a36*a55 + 4*a26*a35*a56 + 4*a25*a36*a56 + 2*a23*a56^2 + 
    2*a25*a35*a66 + a23*a55*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==3))
psi <-
3*(2*a13*a26*a56^2 + 2*a12*a36*a56^2 + a13*a26*a55*a66 + a12*a36*a55*a66 + 
  2*a13*a25*a56*a66 + 2*a12*a35*a56*a66 + 
  a16*(2*a26*a36*a55 + 4*a26*a35*a56 + 4*a25*a36*a56 + 2*a23*a56^2 + 
    2*a25*a35*a66 + a23*a55*a66) + 2*a15*(2*a26*a36*a56 + a26*a35*a66 + 
    a25*a36*a66 + a23*a56*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==4))
psi <-
3*(a66*(4*a15*a26*a36 + 4*a13*a26*a56 + 4*a12*a36*a56 + a15*a23*a66 + 
    a13*a25*a66 + a12*a35*a66) + 4*a16*(2*a26*a36*a56 + a26*a35*a66 + 
    a25*a36*a66 + a23*a56*a66))
  if ((r[1]==1) & (r[2]==1) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==5))
psi <-
15*a66*(4*a16*a26*a36 + a16*a23*a66 + a13*a26*a66 + a12*a36*a66)
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==6) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a44^2*(6*a14*a24 + a12*a44)
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==5) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a44*(a15*a24*a44 + a14*a25*a44 + 4*a14*a24*a45 + a12*a44*a45)
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==5) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a44*(a16*a24*a44 + a14*a26*a44 + 4*a14*a24*a46 + a12*a44*a46)
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==4) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a15*a44*(a25*a44 + 4*a24*a45) + a12*a44*(4*a45^2 + a44*a55) + 
  4*a14*(2*a25*a44*a45 + 2*a24*a45^2 + a24*a44*a55))
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==4) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(4*a14*a26*a44*a45 + a16*a44*(a25*a44 + 4*a24*a45) + 4*a14*a25*a44*a46 + 
  8*a14*a24*a45*a46 + 4*a12*a44*a45*a46 + a15*a44*(a26*a44 + 4*a24*a46) + 
  4*a14*a24*a44*a56 + a12*a44^2*a56)
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==4) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a16*a44*(a26*a44 + 4*a24*a46) + a12*a44*(4*a46^2 + a44*a66) + 
  4*a14*(2*a26*a44*a46 + 2*a24*a46^2 + a24*a44*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==3) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(6*a14*a25*a45^2 + 2*a12*a45^3 + 3*a14*a25*a44*a55 + 6*a14*a24*a45*a55 + 
  3*a12*a44*a45*a55 + 3*a15*(2*a25*a44*a45 + 2*a24*a45^2 + a24*a44*a55))
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==3) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(2*a14*a26*a45^2 + 4*a14*a25*a45*a46 + 2*a12*a45^2*a46 + a14*a26*a44*a55 + 
  2*a14*a24*a46*a55 + a12*a44*a46*a55 + 
  a16*(2*a25*a44*a45 + 2*a24*a45^2 + a24*a44*a55) + 2*a14*a25*a44*a56 + 
  4*a14*a24*a45*a56 + 2*a12*a44*a45*a56 + 
  2*a15*(a26*a44*a45 + a25*a44*a46 + 2*a24*a45*a46 + a24*a44*a56))
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==3) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(4*a14*a26*a45*a46 + 2*a14*a25*a46^2 + 2*a12*a45*a46^2 + 
  2*a14*a26*a44*a56 + 4*a14*a24*a46*a56 + 2*a12*a44*a46*a56 + 
  2*a16*(a26*a44*a45 + a25*a44*a46 + 2*a24*a45*a46 + a24*a44*a56) + 
  a14*a25*a44*a66 + 2*a14*a24*a45*a66 + a12*a44*a45*a66 + 
  a15*(2*a26*a44*a46 + 2*a24*a46^2 + a24*a44*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(6*a14*a26*a46^2 + 2*a12*a46^3 + 3*a14*a26*a44*a66 + 6*a14*a24*a46*a66 + 
  3*a12*a44*a46*a66 + 3*a16*(2*a26*a44*a46 + 2*a24*a46^2 + a24*a44*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==4) & (r[6]==0))
psi <-
3*(a55*(8*a14*a25*a45 + 4*a12*a45^2 + 2*a14*a24*a55 + a12*a44*a55) + 
  4*a15*(2*a25*a45^2 + a25*a44*a55 + 2*a24*a45*a55))
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==3) & (r[6]==1))
psi <-
3*(2*a14*a26*a45*a55 + 2*a14*a25*a46*a55 + 2*a12*a45*a46*a55 + 
  a16*(2*a25*a45^2 + a25*a44*a55 + 2*a24*a45*a55) + 4*a14*a25*a45*a56 + 
  2*a12*a45^2*a56 + 2*a14*a24*a55*a56 + a12*a44*a55*a56 + 
  a15*(2*a26*a45^2 + 4*a25*a45*a46 + a26*a44*a55 + 2*a24*a46*a55 + 
    2*a25*a44*a56 + 4*a24*a45*a56))
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==2) & (r[6]==2))
psi <-
4*a14*a26*a46*a55 + 2*a12*a46^2*a55 + 8*a14*a26*a45*a56 + 8*a14*a25*a46*a56 + 
 8*a12*a45*a46*a56 + 4*a14*a24*a56^2 + 2*a12*a44*a56^2 + 
 2*a16*(2*a26*a45^2 + 4*a25*a45*a46 + a26*a44*a55 + 2*a24*a46*a55 + 
   2*a25*a44*a56 + 4*a24*a45*a56) + 4*a14*a25*a45*a66 + 2*a12*a45^2*a66 + 
 2*a14*a24*a55*a66 + a12*a44*a55*a66 + 
 2*a15*(4*a26*a45*a46 + 2*a25*a46^2 + 2*a26*a44*a56 + 4*a24*a46*a56 + 
   a25*a44*a66 + 2*a24*a45*a66)
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(4*a14*a26*a46*a56 + 2*a12*a46^2*a56 + 2*a14*a26*a45*a66 + 
  2*a14*a25*a46*a66 + 2*a12*a45*a46*a66 + 2*a14*a24*a56*a66 + 
  a12*a44*a56*a66 + a16*(4*a26*a45*a46 + 2*a25*a46^2 + 2*a26*a44*a56 + 
    4*a24*a46*a56 + a25*a44*a66 + 2*a24*a45*a66) + 
  a15*(2*a26*a46^2 + a26*a44*a66 + 2*a24*a46*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==4))
psi <-
3*(a66*(8*a14*a26*a46 + 4*a12*a46^2 + 2*a14*a24*a66 + a12*a44*a66) + 
  4*a16*(2*a26*a46^2 + a26*a44*a66 + 2*a24*a46*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==5) & (r[6]==0))
psi <-
15*a55*(4*a15*a25*a45 + a15*a24*a55 + a14*a25*a55 + a12*a45*a55)
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==4) & (r[6]==1))
psi <-
3*(a16*a55*(4*a25*a45 + a24*a55) + a55*(a14*a26*a55 + a12*a46*a55 + 
    4*a14*a25*a56 + 4*a12*a45*a56) + 
  4*a15*(a26*a45*a55 + a25*a46*a55 + 2*a25*a45*a56 + a24*a55*a56))
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==3) & (r[6]==2))
psi <-
3*(2*a14*a26*a55*a56 + 2*a12*a46*a55*a56 + 2*a14*a25*a56^2 + 
  2*a12*a45*a56^2 + 2*a16*(a26*a45*a55 + a25*a46*a55 + 2*a25*a45*a56 + 
    a24*a55*a56) + a14*a25*a55*a66 + a12*a45*a55*a66 + 
  a15*(2*a26*a46*a55 + 4*a26*a45*a56 + 4*a25*a46*a56 + 2*a24*a56^2 + 
    2*a25*a45*a66 + a24*a55*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==3))
psi <-
3*(2*a14*a26*a56^2 + 2*a12*a46*a56^2 + a14*a26*a55*a66 + a12*a46*a55*a66 + 
  2*a14*a25*a56*a66 + 2*a12*a45*a56*a66 + 
  a16*(2*a26*a46*a55 + 4*a26*a45*a56 + 4*a25*a46*a56 + 2*a24*a56^2 + 
    2*a25*a45*a66 + a24*a55*a66) + 2*a15*(2*a26*a46*a56 + a26*a45*a66 + 
    a25*a46*a66 + a24*a56*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==4))
psi <-
3*(a66*(4*a15*a26*a46 + 4*a14*a26*a56 + 4*a12*a46*a56 + a15*a24*a66 + 
    a14*a25*a66 + a12*a45*a66) + 4*a16*(2*a26*a46*a56 + a26*a45*a66 + 
    a25*a46*a66 + a24*a56*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==5))
psi <-
15*a66*(4*a16*a26*a46 + a16*a24*a66 + a14*a26*a66 + a12*a46*a66)
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==6) & (r[6]==0))
psi <-
15*a55^2*(6*a15*a25 + a12*a55)
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==5) & (r[6]==1))
psi <-
15*a55*(a16*a25*a55 + a15*a26*a55 + 4*a15*a25*a56 + a12*a55*a56)
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==4) & (r[6]==2))
psi <-
3*(2*a16*a55*(a26*a55 + 4*a25*a56) + a12*a55*(4*a56^2 + a55*a66) + 
  4*a15*(2*a26*a55*a56 + 2*a25*a56^2 + a25*a55*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==3))
psi <-
3*(6*a15*a26*a56^2 + 2*a12*a56^3 + 3*a15*a26*a55*a66 + 6*a15*a25*a56*a66 + 
  3*a12*a55*a56*a66 + 3*a16*(2*a26*a55*a56 + 2*a25*a56^2 + a25*a55*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==4))
psi <-
3*(a66*(8*a15*a26*a56 + 4*a12*a56^2 + 2*a15*a25*a66 + a12*a55*a66) + 
  4*a16*(2*a26*a56^2 + a26*a55*a66 + 2*a25*a56*a66))
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==5))
psi <-
15*a66*(4*a16*a26*a56 + a16*a25*a66 + a15*a26*a66 + a12*a56*a66)
 if ((r[1]==1) & (r[2]==1) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==6))
psi <-
15*a66^2*(6*a16*a26 + a12*a66)
 if ((r[1]==1) & (r[2]==0) & (r[3]==7) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
105*a13*a33^3
  if ((r[1]==1) & (r[2]==0) & (r[3]==6) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a33^2*(a14*a33 + 6*a13*a34)
 if ((r[1]==1) & (r[2]==0) & (r[3]==6) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a33^2*(a15*a33 + 6*a13*a35)
 if ((r[1]==1) & (r[2]==0) & (r[3]==6) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a33^2*(a16*a33 + 6*a13*a36)
 if ((r[1]==1) & (r[2]==0) & (r[3]==5) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a33*(2*a14*a33*a34 + 4*a13*a34^2 + a13*a33*a44)
 if ((r[1]==1) & (r[2]==0) & (r[3]==5) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a33*(a15*a33*a34 + a14*a33*a35 + 4*a13*a34*a35 + a13*a33*a45)
 if ((r[1]==1) & (r[2]==0) & (r[3]==5) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a33*(a16*a33*a34 + a14*a33*a36 + 4*a13*a34*a36 + a13*a33*a46)
 if ((r[1]==1) & (r[2]==0) & (r[3]==5) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
15*a33*(2*a15*a33*a35 + 4*a13*a35^2 + a13*a33*a55)
 if ((r[1]==1) & (r[2]==0) & (r[3]==5) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
15*a33*(a16*a33*a35 + a15*a33*a36 + 4*a13*a35*a36 + a13*a33*a56)
 if ((r[1]==1) & (r[2]==0) & (r[3]==5) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
15*a33*(2*a16*a33*a36 + 4*a13*a36^2 + a13*a33*a66)
 if ((r[1]==1) & (r[2]==0) & (r[3]==4) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
9*a14*a33*(4*a34^2 + a33*a44) + 12*a13*a34*(2*a34^2 + 3*a33*a44)
 if ((r[1]==1) & (r[2]==0) & (r[3]==4) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(a15*a33*(4*a34^2 + a33*a44) + 2*a14*a33*(4*a34*a35 + a33*a45) + 
  4*a13*(2*a34^2*a35 + a33*a35*a44 + 2*a33*a34*a45))
 if ((r[1]==1) & (r[2]==0) & (r[3]==4) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(a16*a33*(4*a34^2 + a33*a44) + 2*a14*a33*(4*a34*a36 + a33*a46) + 
  4*a13*(2*a34^2*a36 + a33*a36*a44 + 2*a33*a34*a46))
 if ((r[1]==1) & (r[2]==0) & (r[3]==4) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a15*a33*(4*a34*a35 + a33*a45) + a14*a33*(4*a35^2 + a33*a55) + 
  4*a13*(2*a34*a35^2 + 2*a33*a35*a45 + a33*a34*a55))
 if ((r[1]==1) & (r[2]==0) & (r[3]==4) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(4*a14*a33*a35*a36 + 8*a13*a34*a35*a36 + 4*a13*a33*a36*a45 + 
  a16*a33*(4*a34*a35 + a33*a45) + 4*a13*a33*a35*a46 + 
  a15*a33*(4*a34*a36 + a33*a46) + a14*a33^2*a56 + 4*a13*a33*a34*a56)
 if ((r[1]==1) & (r[2]==0) & (r[3]==4) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a16*a33*(4*a34*a36 + a33*a46) + a14*a33*(4*a36^2 + a33*a66) + 
  4*a13*(2*a34*a36^2 + 2*a33*a36*a46 + a33*a34*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==4) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
9*a15*a33*(4*a35^2 + a33*a55) + 12*a13*a35*(2*a35^2 + 3*a33*a55)
 if ((r[1]==1) & (r[2]==0) & (r[3]==4) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(a16*a33*(4*a35^2 + a33*a55) + 2*a15*a33*(4*a35*a36 + a33*a56) + 
  4*a13*(2*a35^2*a36 + a33*a36*a55 + 2*a33*a35*a56))
 if ((r[1]==1) & (r[2]==0) & (r[3]==4) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(2*a16*a33*(4*a35*a36 + a33*a56) + a15*a33*(4*a36^2 + a33*a66) + 
  4*a13*(2*a35*a36^2 + 2*a33*a36*a56 + a33*a35*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==4) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
9*a16*a33*(4*a36^2 + a33*a66) + 12*a13*a36*(2*a36^2 + 3*a33*a66)
 if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==4) 
& (r[5]==0) & (r[6]==0))
psi <-
9*a13*a44*(4*a34^2 + a33*a44) + 12*a14*(2*a34^3 + 3*a33*a34*a44)
 if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==3) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a15*a34^3 + 6*a14*a34^2*a35 + 3*a15*a33*a34*a44 + 3*a14*a33*a35*a44 + 
  6*a13*a34*a35*a44 + 6*a14*a33*a34*a45 + 6*a13*a34^2*a45 + 3*a13*a33*a44*a45)
 if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==3) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a16*a34^3 + 6*a14*a34^2*a36 + 3*a16*a33*a34*a44 + 3*a14*a33*a36*a44 + 
  6*a13*a34*a36*a44 + 6*a14*a33*a34*a46 + 6*a13*a34^2*a46 + 3*a13*a33*a44*a46)
 if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==2) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a15*(2*a34^2*a35 + a33*a35*a44 + 2*a33*a34*a45) + 
  2*a14*(2*a34*a35^2 + 2*a33*a35*a45 + a33*a34*a55) + 
  a13*(2*a35^2*a44 + 8*a34*a35*a45 + 2*a33*a45^2 + 2*a34^2*a55 + a33*a44*a55))
 if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==2) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(4*a14*a34*a35*a36 + 2*a13*a35*a36*a44 + 2*a14*a33*a36*a45 + 
  4*a13*a34*a36*a45 + a16*(2*a34^2*a35 + a33*a35*a44 + 2*a33*a34*a45) + 
  2*a14*a33*a35*a46 + 4*a13*a34*a35*a46 + 2*a13*a33*a45*a46 + 
  a15*(2*a34^2*a36 + a33*a36*a44 + 2*a33*a34*a46) + 2*a14*a33*a34*a56 + 
  2*a13*a34^2*a56 + a13*a33*a44*a56)
 if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==2) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a16*(2*a34^2*a36 + a33*a36*a44 + 2*a33*a34*a46) + 
  2*a14*(2*a34*a36^2 + 2*a33*a36*a46 + a33*a34*a66) + 
  a13*(2*a36^2*a44 + 8*a34*a36*a46 + 2*a33*a46^2 + 2*a34^2*a66 + a33*a44*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==1) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a14*a35^3 + 6*a13*a35^2*a45 + 3*a14*a33*a35*a55 + 6*a13*a34*a35*a55 + 
  3*a13*a33*a45*a55 + 3*a15*(2*a34*a35^2 + 2*a33*a35*a45 + a33*a34*a55))
 if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==1) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(2*a14*a35^2*a36 + 4*a13*a35*a36*a45 + 2*a13*a35^2*a46 + a14*a33*a36*a55 + 
  2*a13*a34*a36*a55 + a13*a33*a46*a55 + 
  a16*(2*a34*a35^2 + 2*a33*a35*a45 + a33*a34*a55) + 2*a14*a33*a35*a56 + 
  4*a13*a34*a35*a56 + 2*a13*a33*a45*a56 + 
  2*a15*(2*a34*a35*a36 + a33*a36*a45 + a33*a35*a46 + a33*a34*a56))
 if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==1) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(2*a14*a35*a36^2 + 2*a13*a36^2*a45 + 4*a13*a35*a36*a46 + 
  2*a14*a33*a36*a56 + 4*a13*a34*a36*a56 + 2*a13*a33*a46*a56 + 
  2*a16*(2*a34*a35*a36 + a33*a36*a45 + a33*a35*a46 + a33*a34*a56) + 
  a14*a33*a35*a66 + 2*a13*a34*a35*a66 + a13*a33*a45*a66 + 
  a15*(2*a34*a36^2 + 2*a33*a36*a46 + a33*a34*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==1) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a14*a36^3 + 6*a13*a36^2*a46 + 3*a14*a33*a36*a66 + 6*a13*a34*a36*a66 + 
  3*a13*a33*a46*a66 + 3*a16*(2*a34*a36^2 + 2*a33*a36*a46 + a33*a34*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==4) & (r[6]==0))
psi <-
9*a13*a55*(4*a35^2 + a33*a55) + 12*a15*(2*a35^3 + 3*a33*a35*a55)
 if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==3) & (r[6]==1))
psi <-
3*(2*a16*a35^3 + 6*a15*a35^2*a36 + 3*a16*a33*a35*a55 + 3*a15*a33*a36*a55 + 
  6*a13*a35*a36*a55 + 6*a15*a33*a35*a56 + 6*a13*a35^2*a56 + 3*a13*a33*a55*a56)
 if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==2) & (r[6]==2))
psi <-
3*(2*a16*(2*a35^2*a36 + a33*a36*a55 + 2*a33*a35*a56) + 
  2*a15*(2*a35*a36^2 + 2*a33*a36*a56 + a33*a35*a66) + 
  a13*(2*a36^2*a55 + 8*a35*a36*a56 + 2*a33*a56^2 + 2*a35^2*a66 + a33*a55*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(2*a15*a36^3 + 6*a13*a36^2*a56 + 3*a15*a33*a36*a66 + 6*a13*a35*a36*a66 + 
  3*a13*a33*a56*a66 + 3*a16*(2*a35*a36^2 + 2*a33*a36*a56 + a33*a35*a66))
  if ((r[1]==1) & (r[2]==0) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==4))
psi <-
9*a13*a66*(4*a36^2 + a33*a66) + 12*a16*(2*a36^3 + 3*a33*a36*a66)
 if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==5) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a44*(4*a14*a34^2 + a14*a33*a44 + 2*a13*a34*a44)
 if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==4) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(a15*a44*(4*a34^2 + a33*a44) + 2*a13*a44*(a35*a44 + 4*a34*a45) + 
  4*a14*(2*a34*a35*a44 + 2*a34^2*a45 + a33*a44*a45))
 if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==4) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(a16*a44*(4*a34^2 + a33*a44) + 2*a13*a44*(a36*a44 + 4*a34*a46) + 
  4*a14*(2*a34*a36*a44 + 2*a34^2*a46 + a33*a44*a46))
 if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==3) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a15*(2*a34*a35*a44 + 2*a34^2*a45 + a33*a44*a45) + 
  a14*(2*a35^2*a44 + 8*a34*a35*a45 + 2*a33*a45^2 + 2*a34^2*a55 + 
    a33*a44*a55) + 2*a13*(2*a35*a44*a45 + 2*a34*a45^2 + a34*a44*a55))
 if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==3) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(2*a14*a35*a36*a44 + 4*a14*a34*a36*a45 + 2*a13*a36*a44*a45 + 
  a16*(2*a34*a35*a44 + 2*a34^2*a45 + a33*a44*a45) + 4*a14*a34*a35*a46 + 
  2*a13*a35*a44*a46 + 2*a14*a33*a45*a46 + 4*a13*a34*a45*a46 + 
  a15*(2*a34*a36*a44 + 2*a34^2*a46 + a33*a44*a46) + 2*a14*a34^2*a56 + 
  a14*a33*a44*a56 + 2*a13*a34*a44*a56)
 if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==3) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a16*(2*a34*a36*a44 + 2*a34^2*a46 + a33*a44*a46) + 
  a14*(2*a36^2*a44 + 8*a34*a36*a46 + 2*a33*a46^2 + 2*a34^2*a66 + 
    a33*a44*a66) + 2*a13*(2*a36*a44*a46 + 2*a34*a46^2 + a34*a44*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==2) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(a15*(2*a35^2*a44 + 8*a34*a35*a45 + 2*a33*a45^2 + 2*a34^2*a55 + 
    a33*a44*a55) + 2*(a14*(2*a35^2*a45 + 2*a34*a35*a55 + a33*a45*a55) + 
    a13*(2*a35*a45^2 + a35*a44*a55 + 2*a34*a45*a55)))
 if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==2) 
& (r[5]==2) & (r[6]==1))
psi <-
a16*(2*a35^2*a44 + 8*a34*a35*a45 + 2*a33*a45^2 + 2*a34^2*a55 + a33*a44*a55) + 
 2*(a15*(2*a35*a36*a44 + 4*a34*a36*a45 + 4*a34*a35*a46 + 2*a33*a45*a46 + 
     2*a34^2*a56 + a33*a44*a56) + a14*(4*a35*a36*a45 + 2*a35^2*a46 + 
     2*a34*a36*a55 + a33*a46*a55 + 4*a34*a35*a56 + 2*a33*a45*a56) + 
   a13*(2*a36*a45^2 + 4*a35*a45*a46 + a36*a44*a55 + 2*a34*a46*a55 + 
     2*a35*a44*a56 + 4*a34*a45*a56))
 if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==2) 
& (r[5]==1) & (r[6]==2))
psi <-
2*a16*(2*a35*a36*a44 + 4*a34*a36*a45 + 4*a34*a35*a46 + 2*a33*a45*a46 + 
   2*a34^2*a56 + a33*a44*a56) + a15*(2*a36^2*a44 + 8*a34*a36*a46 + 
   2*a33*a46^2 + 2*a34^2*a66 + a33*a44*a66) + 
 2*(a14*(2*a36^2*a45 + 4*a35*a36*a46 + 4*a34*a36*a56 + 2*a33*a46*a56 + 
     2*a34*a35*a66 + a33*a45*a66) + a13*(4*a36*a45*a46 + 2*a35*a46^2 + 
     2*a36*a44*a56 + 4*a34*a46*a56 + a35*a44*a66 + 2*a34*a45*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==2) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(a16*(2*a36^2*a44 + 8*a34*a36*a46 + 2*a33*a46^2 + 2*a34^2*a66 + 
    a33*a44*a66) + 2*(a14*(2*a36^2*a46 + 2*a34*a36*a66 + a33*a46*a66) + 
    a13*(2*a36*a46^2 + a36*a44*a66 + 2*a34*a46*a66)))
 if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==4) & (r[6]==0))
psi <-
3*(a55*(4*a14*a35^2 + 8*a13*a35*a45 + a14*a33*a55 + 2*a13*a34*a55) + 
  4*a15*(2*a35^2*a45 + 2*a34*a35*a55 + a33*a45*a55))
 if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==3) & (r[6]==1))
psi <-
3*(2*a14*a35*a36*a55 + 2*a13*a36*a45*a55 + 2*a13*a35*a46*a55 + 
  a16*(2*a35^2*a45 + 2*a34*a35*a55 + a33*a45*a55) + 2*a14*a35^2*a56 + 
  4*a13*a35*a45*a56 + a14*a33*a55*a56 + 2*a13*a34*a55*a56 + 
  a15*(4*a35*a36*a45 + 2*a35^2*a46 + 2*a34*a36*a55 + a33*a46*a55 + 
    4*a34*a35*a56 + 2*a33*a45*a56))
 if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==2) & (r[6]==2))
psi <-
2*a14*a36^2*a55 + 4*a13*a36*a46*a55 + 8*a14*a35*a36*a56 + 8*a13*a36*a45*a56 + 
 8*a13*a35*a46*a56 + 2*a14*a33*a56^2 + 4*a13*a34*a56^2 + 
 2*a16*(2*a35^2*a46 + 2*a34*a36*a55 + a33*a46*a55 + 2*a33*a45*a56 + 
   4*a35*(a36*a45 + a34*a56)) + 2*a14*a35^2*a66 + 4*a13*a35*a45*a66 + 
 a14*a33*a55*a66 + 2*a13*a34*a55*a66 + 
 2*a15*(2*a36^2*a45 + 4*a35*a36*a46 + 4*a34*a36*a56 + 2*a33*a46*a56 + 
   2*a34*a35*a66 + a33*a45*a66)
 if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(2*a14*a36^2*a56 + 4*a13*a36*a46*a56 + 2*a14*a35*a36*a66 + 
  2*a13*a36*a45*a66 + 2*a13*a35*a46*a66 + a14*a33*a56*a66 + 
  2*a13*a34*a56*a66 + a16*(2*a36^2*a45 + 2*a33*a46*a56 + 
    4*a36*(a35*a46 + a34*a56) + 2*a34*a35*a66 + a33*a45*a66) + 
  a15*(2*a36^2*a46 + 2*a34*a36*a66 + a33*a46*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==4))
psi <-
3*(a66*(4*a14*a36^2 + 8*a13*a36*a46 + a14*a33*a66 + 2*a13*a34*a66) + 
  4*a16*(2*a36^2*a46 + 2*a34*a36*a66 + a33*a46*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==5) & (r[6]==0))
psi <-
15*a55*(4*a15*a35^2 + a15*a33*a55 + 2*a13*a35*a55)
 if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==4) & (r[6]==1))
psi <-
3*(a16*a55*(4*a35^2 + a33*a55) + 2*a13*a55*(a36*a55 + 4*a35*a56) + 
  4*a15*(2*a35*a36*a55 + 2*a35^2*a56 + a33*a55*a56))
 if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==3) & (r[6]==2))
psi <-
3*(2*a16*(2*a35*a36*a55 + 2*a35^2*a56 + a33*a55*a56) + 
  a15*(2*a36^2*a55 + 8*a35*a36*a56 + 2*a33*a56^2 + 2*a35^2*a66 + 
    a33*a55*a66) + 2*a13*(2*a36*a55*a56 + 2*a35*a56^2 + a35*a55*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==2) & (r[6]==3))
psi <-
3*(a16*(2*a36^2*a55 + 8*a35*a36*a56 + 2*a33*a56^2 + 2*a35^2*a66 + 
    a33*a55*a66) + 2*(a15*(2*a36^2*a56 + 2*a35*a36*a66 + a33*a56*a66) + 
    a13*(2*a36*a56^2 + a36*a55*a66 + 2*a35*a56*a66)))
 if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==4))
psi <-
3*(a66*(4*a15*a36^2 + 8*a13*a36*a56 + a15*a33*a66 + 2*a13*a35*a66) + 
  4*a16*(2*a36^2*a56 + 2*a35*a36*a66 + a33*a56*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==5))
psi <-
15*a66*(4*a16*a36^2 + a16*a33*a66 + 2*a13*a36*a66)
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==6) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a44^2*(6*a14*a34 + a13*a44)
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==5) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a44*(a15*a34*a44 + a14*a35*a44 + 4*a14*a34*a45 + a13*a44*a45)
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==5) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a44*(a16*a34*a44 + a14*a36*a44 + 4*a14*a34*a46 + a13*a44*a46)
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==4) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a15*a44*(a35*a44 + 4*a34*a45) + a13*a44*(4*a45^2 + a44*a55) + 
  4*a14*(2*a35*a44*a45 + 2*a34*a45^2 + a34*a44*a55))
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==4) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(4*a14*a36*a44*a45 + a16*a44*(a35*a44 + 4*a34*a45) + 4*a14*a35*a44*a46 + 
  8*a14*a34*a45*a46 + 4*a13*a44*a45*a46 + a15*a44*(a36*a44 + 4*a34*a46) + 
  4*a14*a34*a44*a56 + a13*a44^2*a56)
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==4) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a16*a44*(a36*a44 + 4*a34*a46) + a13*a44*(4*a46^2 + a44*a66) + 
  4*a14*(2*a36*a44*a46 + 2*a34*a46^2 + a34*a44*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==3) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(6*a14*a35*a45^2 + 2*a13*a45^3 + 3*a14*a35*a44*a55 + 6*a14*a34*a45*a55 + 
  3*a13*a44*a45*a55 + 3*a15*(2*a35*a44*a45 + 2*a34*a45^2 + a34*a44*a55))
  if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==3) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(2*a14*a36*a45^2 + 4*a14*a35*a45*a46 + 2*a13*a45^2*a46 + a14*a36*a44*a55 + 
  2*a14*a34*a46*a55 + a13*a44*a46*a55 + 
  a16*(2*a35*a44*a45 + 2*a34*a45^2 + a34*a44*a55) + 2*a14*a35*a44*a56 + 
  4*a14*a34*a45*a56 + 2*a13*a44*a45*a56 + 
  2*a15*(a36*a44*a45 + a35*a44*a46 + 2*a34*a45*a46 + a34*a44*a56))
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==3) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(4*a14*a36*a45*a46 + 2*a14*a35*a46^2 + 2*a13*a45*a46^2 + 
  2*a14*a36*a44*a56 + 4*a14*a34*a46*a56 + 2*a13*a44*a46*a56 + 
  2*a16*(a36*a44*a45 + a35*a44*a46 + 2*a34*a45*a46 + a34*a44*a56) + 
  a14*a35*a44*a66 + 2*a14*a34*a45*a66 + a13*a44*a45*a66 + 
  a15*(2*a36*a44*a46 + 2*a34*a46^2 + a34*a44*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==3) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(6*a14*a36*a46^2 + 2*a13*a46^3 + 3*a14*a36*a44*a66 + 6*a14*a34*a46*a66 + 
  3*a13*a44*a46*a66 + 3*a16*(2*a36*a44*a46 + 2*a34*a46^2 + a34*a44*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==4) & (r[6]==0))
psi <-
3*(a55*(8*a14*a35*a45 + 4*a13*a45^2 + 2*a14*a34*a55 + a13*a44*a55) + 
  4*a15*(2*a35*a45^2 + a35*a44*a55 + 2*a34*a45*a55))
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==3) & (r[6]==1))
psi <-
3*(2*a14*a36*a45*a55 + 2*a14*a35*a46*a55 + 2*a13*a45*a46*a55 + 
  a16*(2*a35*a45^2 + a35*a44*a55 + 2*a34*a45*a55) + 4*a14*a35*a45*a56 + 
  2*a13*a45^2*a56 + 2*a14*a34*a55*a56 + a13*a44*a55*a56 + 
  a15*(2*a36*a45^2 + 4*a35*a45*a46 + a36*a44*a55 + 2*a34*a46*a55 + 
    2*a35*a44*a56 + 4*a34*a45*a56))
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==2) & (r[6]==2))
psi <-
4*a14*a36*a46*a55 + 2*a13*a46^2*a55 + 8*a14*a36*a45*a56 + 8*a14*a35*a46*a56 + 
 8*a13*a45*a46*a56 + 4*a14*a34*a56^2 + 2*a13*a44*a56^2 + 
 2*a16*(2*a36*a45^2 + 4*a35*a45*a46 + a36*a44*a55 + 2*a34*a46*a55 + 
   2*a35*a44*a56 + 4*a34*a45*a56) + 4*a14*a35*a45*a66 + 2*a13*a45^2*a66 + 
 2*a14*a34*a55*a66 + a13*a44*a55*a66 + 
 2*a15*(4*a36*a45*a46 + 2*a35*a46^2 + 2*a36*a44*a56 + 4*a34*a46*a56 + 
   a35*a44*a66 + 2*a34*a45*a66)
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(4*a14*a36*a46*a56 + 2*a13*a46^2*a56 + 2*a14*a36*a45*a66 + 
  2*a14*a35*a46*a66 + 2*a13*a45*a46*a66 + 2*a14*a34*a56*a66 + 
  a13*a44*a56*a66 + a16*(4*a36*a45*a46 + 2*a35*a46^2 + 2*a36*a44*a56 + 
    4*a34*a46*a56 + a35*a44*a66 + 2*a34*a45*a66) + 
  a15*(2*a36*a46^2 + a36*a44*a66 + 2*a34*a46*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==4))
psi <-
3*(a66*(8*a14*a36*a46 + 4*a13*a46^2 + 2*a14*a34*a66 + a13*a44*a66) + 
  4*a16*(2*a36*a46^2 + a36*a44*a66 + 2*a34*a46*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==5) & (r[6]==0))
psi <-
15*a55*(4*a15*a35*a45 + a15*a34*a55 + a14*a35*a55 + a13*a45*a55)
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==4) & (r[6]==1))
psi <-
3*(a16*a55*(4*a35*a45 + a34*a55) + a55*(a14*a36*a55 + a13*a46*a55 + 
    4*a14*a35*a56 + 4*a13*a45*a56) + 
  4*a15*(a36*a45*a55 + a35*a46*a55 + 2*a35*a45*a56 + a34*a55*a56))
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==3) & (r[6]==2))
psi <-
3*(2*a14*a36*a55*a56 + 2*a13*a46*a55*a56 + 2*a14*a35*a56^2 + 
  2*a13*a45*a56^2 + 2*a16*(a36*a45*a55 + a35*a46*a55 + 2*a35*a45*a56 + 
    a34*a55*a56) + a14*a35*a55*a66 + a13*a45*a55*a66 + 
  a15*(2*a36*a46*a55 + 4*a36*a45*a56 + 4*a35*a46*a56 + 2*a34*a56^2 + 
    2*a35*a45*a66 + a34*a55*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==2) & (r[6]==3))
psi <-
3*(2*a14*a36*a56^2 + 2*a13*a46*a56^2 + a14*a36*a55*a66 + a13*a46*a55*a66 + 
  2*a14*a35*a56*a66 + 2*a13*a45*a56*a66 + 
  a16*(2*a36*a46*a55 + 4*a36*a45*a56 + 4*a35*a46*a56 + 2*a34*a56^2 + 
    2*a35*a45*a66 + a34*a55*a66) + 2*a15*(2*a36*a46*a56 + a36*a45*a66 + 
    a35*a46*a66 + a34*a56*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==4))
psi <-
3*(a66*(4*a15*a36*a46 + 4*a14*a36*a56 + 4*a13*a46*a56 + a15*a34*a66 + 
    a14*a35*a66 + a13*a45*a66) + 4*a16*(2*a36*a46*a56 + a36*a45*a66 + 
    a35*a46*a66 + a34*a56*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==5))
psi <-
15*a66*(4*a16*a36*a46 + a16*a34*a66 + a14*a36*a66 + a13*a46*a66)
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==6) & (r[6]==0))
psi <-
15*a55^2*(6*a15*a35 + a13*a55)
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==5) & (r[6]==1))
psi <-
15*a55*(a16*a35*a55 + a15*a36*a55 + 4*a15*a35*a56 + a13*a55*a56)
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==4) & (r[6]==2))
psi <-
3*(2*a16*a55*(a36*a55 + 4*a35*a56) + a13*a55*(4*a56^2 + a55*a66) + 
  4*a15*(2*a36*a55*a56 + 2*a35*a56^2 + a35*a55*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==3) & (r[6]==3))
psi <-
3*(6*a15*a36*a56^2 + 2*a13*a56^3 + 3*a15*a36*a55*a66 + 6*a15*a35*a56*a66 + 
  3*a13*a55*a56*a66 + 3*a16*(2*a36*a55*a56 + 2*a35*a56^2 + a35*a55*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==4))
psi <-
3*(a66*(8*a15*a36*a56 + 4*a13*a56^2 + 2*a15*a35*a66 + a13*a55*a66) + 
  4*a16*(2*a36*a56^2 + a36*a55*a66 + 2*a35*a56*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==5))
psi <-
15*a66*(4*a16*a36*a56 + a16*a35*a66 + a15*a36*a66 + a13*a56*a66)
 if ((r[1]==1) & (r[2]==0) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==6))
psi <-
15*a66^2*(6*a16*a36 + a13*a66)
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==7) 
& (r[5]==0) & (r[6]==0))
psi <-
105*a14*a44^3
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==6) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a44^2*(a15*a44 + 6*a14*a45)
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==6) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a44^2*(a16*a44 + 6*a14*a46)
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==5) 
& (r[5]==2) & (r[6]==0))
psi <-
15*a44*(2*a15*a44*a45 + 4*a14*a45^2 + a14*a44*a55)
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==5) 
& (r[5]==1) & (r[6]==1))
psi <-
15*a44*(a16*a44*a45 + a15*a44*a46 + 4*a14*a45*a46 + a14*a44*a56)
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==5) 
& (r[5]==0) & (r[6]==2))
psi <-
15*a44*(2*a16*a44*a46 + 4*a14*a46^2 + a14*a44*a66)
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==4) 
& (r[5]==3) & (r[6]==0))
psi <-
9*a15*a44*(4*a45^2 + a44*a55) + 12*a14*a45*(2*a45^2 + 3*a44*a55)
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==4) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(a16*a44*(4*a45^2 + a44*a55) + 2*a15*a44*(4*a45*a46 + a44*a56) + 
  4*a14*(2*a45^2*a46 + a44*a46*a55 + 2*a44*a45*a56))
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==4) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(2*a16*a44*(4*a45*a46 + a44*a56) + a15*a44*(4*a46^2 + a44*a66) + 
  4*a14*(2*a45*a46^2 + 2*a44*a46*a56 + a44*a45*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==4) 
& (r[5]==0) & (r[6]==3))
psi <-
9*a16*a44*(4*a46^2 + a44*a66) + 12*a14*a46*(2*a46^2 + 3*a44*a66)
  if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==4) & (r[6]==0))
psi <-
9*a14*a55*(4*a45^2 + a44*a55) + 12*a15*(2*a45^3 + 3*a44*a45*a55)
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==3) & (r[6]==1))
psi <-
3*(2*a16*a45^3 + 6*a15*a45^2*a46 + 3*a16*a44*a45*a55 + 3*a15*a44*a46*a55 + 
  6*a14*a45*a46*a55 + 6*a15*a44*a45*a56 + 6*a14*a45^2*a56 + 3*a14*a44*a55*a56)
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==2) & (r[6]==2))
psi <-
3*(2*a16*(2*a45^2*a46 + a44*a46*a55 + 2*a44*a45*a56) + 
  2*a15*(2*a45*a46^2 + 2*a44*a46*a56 + a44*a45*a66) + 
  a14*(2*a46^2*a55 + 8*a45*a46*a56 + 2*a44*a56^2 + 2*a45^2*a66 + a44*a55*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(2*a15*a46^3 + 6*a14*a46^2*a56 + 3*a15*a44*a46*a66 + 6*a14*a45*a46*a66 + 
  3*a14*a44*a56*a66 + 3*a16*(2*a45*a46^2 + 2*a44*a46*a56 + a44*a45*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==4))
psi <-
9*a14*a66*(4*a46^2 + a44*a66) + 12*a16*(2*a46^3 + 3*a44*a46*a66)
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==5) & (r[6]==0))
psi <-
15*a55*(4*a15*a45^2 + a15*a44*a55 + 2*a14*a45*a55)
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==4) & (r[6]==1))
psi <-
3*(a16*a55*(4*a45^2 + a44*a55) + 2*a14*a55*(a46*a55 + 4*a45*a56) + 
  4*a15*(2*a45*a46*a55 + 2*a45^2*a56 + a44*a55*a56))
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==3) & (r[6]==2))
psi <-
3*(2*a16*(2*a45*a46*a55 + 2*a45^2*a56 + a44*a55*a56) + 
  a15*(2*a46^2*a55 + 8*a45*a46*a56 + 2*a44*a56^2 + 2*a45^2*a66 + 
    a44*a55*a66) + 2*a14*(2*a46*a55*a56 + 2*a45*a56^2 + a45*a55*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==2) & (r[6]==3))
psi <-
3*(a16*(2*a46^2*a55 + 8*a45*a46*a56 + 2*a44*a56^2 + 2*a45^2*a66 + 
    a44*a55*a66) + 2*(a15*(2*a46^2*a56 + 2*a45*a46*a66 + a44*a56*a66) + 
    a14*(2*a46*a56^2 + a46*a55*a66 + 2*a45*a56*a66)))
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==4))
psi <-
3*(a66*(4*a15*a46^2 + 8*a14*a46*a56 + a15*a44*a66 + 2*a14*a45*a66) + 
  4*a16*(2*a46^2*a56 + 2*a45*a46*a66 + a44*a56*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==5))
psi <-
15*a66*(4*a16*a46^2 + a16*a44*a66 + 2*a14*a46*a66)
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==6) & (r[6]==0))
psi <-
15*a55^2*(6*a15*a45 + a14*a55)
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==5) & (r[6]==1))
psi <-
15*a55*(a16*a45*a55 + a15*a46*a55 + 4*a15*a45*a56 + a14*a55*a56)
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==4) & (r[6]==2))
psi <-
3*(2*a16*a55*(a46*a55 + 4*a45*a56) + a14*a55*(4*a56^2 + a55*a66) + 
  4*a15*(2*a46*a55*a56 + 2*a45*a56^2 + a45*a55*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==3) & (r[6]==3))
psi <-
3*(6*a15*a46*a56^2 + 2*a14*a56^3 + 3*a15*a46*a55*a66 + 6*a15*a45*a56*a66 + 
  3*a14*a55*a56*a66 + 3*a16*(2*a46*a55*a56 + 2*a45*a56^2 + a45*a55*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==4))
psi <-
3*(a66*(8*a15*a46*a56 + 4*a14*a56^2 + 2*a15*a45*a66 + a14*a55*a66) + 
  4*a16*(2*a46*a56^2 + a46*a55*a66 + 2*a45*a56*a66))
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==5))
psi <-
15*a66*(4*a16*a46*a56 + a16*a45*a66 + a15*a46*a66 + a14*a56*a66)
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==6))
psi <-
15*a66^2*(6*a16*a46 + a14*a66)
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==7) & (r[6]==0))
psi <-
105*a15*a55^3
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==6) & (r[6]==1))
psi <-
15*a55^2*(a16*a55 + 6*a15*a56)
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==5) & (r[6]==2))
psi <-
15*a55*(2*a16*a55*a56 + 4*a15*a56^2 + a15*a55*a66)
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==4) & (r[6]==3))
psi <-
9*a16*a55*(4*a56^2 + a55*a66) + 12*a15*a56*(2*a56^2 + 3*a55*a66)
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==4))
psi <-
9*a15*a66*(4*a56^2 + a55*a66) + 12*a16*(2*a56^3 + 3*a55*a56*a66)
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==5))
psi <-
15*a66*(4*a16*a56^2 + a16*a55*a66 + 2*a15*a56*a66)
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==6))
psi <-
15*a66^2*(6*a16*a56 + a15*a66)
 if ((r[1]==1) & (r[2]==0) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==7))
psi <-
105*a16*a66^3
 if ((r[1]==0) & (r[2]==8) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
105*a22^4
 if ((r[1]==0) & (r[2]==7) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
105*a22^3*a23
 if ((r[1]==0) & (r[2]==7) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
105*a22^3*a24
 if ((r[1]==0) & (r[2]==7) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
105*a22^3*a25
 if ((r[1]==0) & (r[2]==7) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
105*a22^3*a26
 if ((r[1]==0) & (r[2]==6) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a22^2*(6*a23^2 + a22*a33)
 if ((r[1]==0) & (r[2]==6) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a22^2*(6*a23*a24 + a22*a34)
 if ((r[1]==0) & (r[2]==6) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a22^2*(6*a23*a25 + a22*a35)
 if ((r[1]==0) & (r[2]==6) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a22^2*(6*a23*a26 + a22*a36)
 if ((r[1]==0) & (r[2]==6) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a22^2*(6*a24^2 + a22*a44)
 if ((r[1]==0) & (r[2]==6) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a22^2*(6*a24*a25 + a22*a45)
 if ((r[1]==0) & (r[2]==6) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a22^2*(6*a24*a26 + a22*a46)
  if ((r[1]==0) & (r[2]==6) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
15*a22^2*(6*a25^2 + a22*a55)
 if ((r[1]==0) & (r[2]==6) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
15*a22^2*(6*a25*a26 + a22*a56)
 if ((r[1]==0) & (r[2]==6) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
15*a22^2*(6*a26^2 + a22*a66)
 if ((r[1]==0) & (r[2]==5) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a22*a23*(4*a23^2 + 3*a22*a33)
 if ((r[1]==0) & (r[2]==5) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a22*(4*a23^2*a24 + a22*a24*a33 + 2*a22*a23*a34)
 if ((r[1]==0) & (r[2]==5) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a22*(4*a23^2*a25 + a22*a25*a33 + 2*a22*a23*a35)
 if ((r[1]==0) & (r[2]==5) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a22*(4*a23^2*a26 + a22*a26*a33 + 2*a22*a23*a36)
 if ((r[1]==0) & (r[2]==5) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a22*(4*a23*a24^2 + 2*a22*a24*a34 + a22*a23*a44)
 if ((r[1]==0) & (r[2]==5) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a22*(4*a23*a24*a25 + a22*a25*a34 + a22*a24*a35 + a22*a23*a45)
 if ((r[1]==0) & (r[2]==5) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a22*(4*a23*a24*a26 + a22*a26*a34 + a22*a24*a36 + a22*a23*a46)
 if ((r[1]==0) & (r[2]==5) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
15*a22*(4*a23*a25^2 + 2*a22*a25*a35 + a22*a23*a55)
 if ((r[1]==0) & (r[2]==5) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
15*a22*(4*a23*a25*a26 + a22*a26*a35 + a22*a25*a36 + a22*a23*a56)
 if ((r[1]==0) & (r[2]==5) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
15*a22*(4*a23*a26^2 + 2*a22*a26*a36 + a22*a23*a66)
 if ((r[1]==0) & (r[2]==5) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a22*a24*(4*a24^2 + 3*a22*a44)
 if ((r[1]==0) & (r[2]==5) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a22*(4*a24^2*a25 + a22*a25*a44 + 2*a22*a24*a45)
 if ((r[1]==0) & (r[2]==5) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a22*(4*a24^2*a26 + a22*a26*a44 + 2*a22*a24*a46)
 if ((r[1]==0) & (r[2]==5) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
15*a22*(4*a24*a25^2 + 2*a22*a25*a45 + a22*a24*a55)
 if ((r[1]==0) & (r[2]==5) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
15*a22*(4*a24*a25*a26 + a22*a26*a45 + a22*a25*a46 + a22*a24*a56)
 if ((r[1]==0) & (r[2]==5) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
15*a22*(4*a24*a26^2 + 2*a22*a26*a46 + a22*a24*a66)
 if ((r[1]==0) & (r[2]==5) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
15*a22*a25*(4*a25^2 + 3*a22*a55)
 if ((r[1]==0) & (r[2]==5) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
15*a22*(4*a25^2*a26 + a22*a26*a55 + 2*a22*a25*a56)
 if ((r[1]==0) & (r[2]==5) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
15*a22*(4*a25*a26^2 + 2*a22*a26*a56 + a22*a25*a66)
 if ((r[1]==0) & (r[2]==5) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
15*a22*a26*(4*a26^2 + 3*a22*a66)
 if ((r[1]==0) & (r[2]==4) & (r[3]==4) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
24*a23^4 + 72*a22*a23^2*a33 + 9*a22^2*a33^2
 if ((r[1]==0) & (r[2]==4) & (r[3]==3) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
24*a23^3*a24 + 36*a22*a23*a24*a33 + 36*a22*a23^2*a34 + 9*a22^2*a33*a34
 if ((r[1]==0) & (r[2]==4) & (r[3]==3) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
24*a23^3*a25 + 36*a22*a23*a25*a33 + 36*a22*a23^2*a35 + 9*a22^2*a33*a35
 if ((r[1]==0) & (r[2]==4) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
24*a23^3*a26 + 36*a22*a23*a26*a33 + 36*a22*a23^2*a36 + 9*a22^2*a33*a36
 if ((r[1]==0) & (r[2]==4) & (r[3]==2) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(16*a22*a23*a24*a34 + 4*a23^2*(2*a24^2 + a22*a44) + 
  a22*(4*a24^2*a33 + 2*a22*a34^2 + a22*a33*a44))
 if ((r[1]==0) & (r[2]==4) & (r[3]==2) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(8*a22*a23*(a25*a34 + a24*a35) + 4*a23^2*(2*a24*a25 + a22*a45) + 
  a22*(4*a24*a25*a33 + 2*a22*a34*a35 + a22*a33*a45))
 if ((r[1]==0) & (r[2]==4) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(8*a22*a23*(a26*a34 + a24*a36) + 4*a23^2*(2*a24*a26 + a22*a46) + 
  a22*(4*a24*a26*a33 + 2*a22*a34*a36 + a22*a33*a46))
 if ((r[1]==0) & (r[2]==4) & (r[3]==2) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(16*a22*a23*a25*a35 + 4*a23^2*(2*a25^2 + a22*a55) + 
  a22*(4*a25^2*a33 + 2*a22*a35^2 + a22*a33*a55))
 if ((r[1]==0) & (r[2]==4) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(8*a22*a23*(a26*a35 + a25*a36) + 4*a23^2*(2*a25*a26 + a22*a56) + 
  a22*(4*a25*a26*a33 + 2*a22*a35*a36 + a22*a33*a56))
 if ((r[1]==0) & (r[2]==4) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(16*a22*a23*a26*a36 + 4*a23^2*(2*a26^2 + a22*a66) + 
  a22*(4*a26^2*a33 + 2*a22*a36^2 + a22*a33*a66))
 if ((r[1]==0) & (r[2]==4) & (r[3]==1) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
9*a22*a34*(4*a24^2 + a22*a44) + 12*a23*(2*a24^3 + 3*a22*a24*a44)
 if ((r[1]==0) & (r[2]==4) & (r[3]==1) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(4*a23*(2*a24^2*a25 + a22*a25*a44 + 2*a22*a24*a45) + 
  a22*(8*a24*a25*a34 + 4*a24^2*a35 + a22*a35*a44 + 2*a22*a34*a45))
 if ((r[1]==0) & (r[2]==4) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(4*a23*(2*a24^2*a26 + a22*a26*a44 + 2*a22*a24*a46) + 
  a22*(8*a24*a26*a34 + 4*a24^2*a36 + a22*a36*a44 + 2*a22*a34*a46))
 if ((r[1]==0) & (r[2]==4) & (r[3]==1) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(4*a23*(2*a24*a25^2 + 2*a22*a25*a45 + a22*a24*a55) + 
  a22*(4*a25^2*a34 + 8*a24*a25*a35 + 2*a22*a35*a45 + a22*a34*a55))
 if ((r[1]==0) & (r[2]==4) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(4*a23*(2*a24*a25*a26 + a22*a26*a45 + a22*a25*a46 + a22*a24*a56) + 
  a22*(4*a24*a26*a35 + 4*a25*(a26*a34 + a24*a36) + a22*a36*a45 + 
    a22*a35*a46 + a22*a34*a56))
  if ((r[1]==0) & (r[2]==4) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(4*a23*(2*a24*a26^2 + 2*a22*a26*a46 + a22*a24*a66) + 
  a22*(4*a26^2*a34 + 8*a24*a26*a36 + 2*a22*a36*a46 + a22*a34*a66))
 if ((r[1]==0) & (r[2]==4) & (r[3]==1) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
9*a22*a35*(4*a25^2 + a22*a55) + 12*a23*(2*a25^3 + 3*a22*a25*a55)
 if ((r[1]==0) & (r[2]==4) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(4*a23*(2*a25^2*a26 + a22*a26*a55 + 2*a22*a25*a56) + 
  a22*(8*a25*a26*a35 + 4*a25^2*a36 + a22*a36*a55 + 2*a22*a35*a56))
 if ((r[1]==0) & (r[2]==4) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(4*a23*(2*a25*a26^2 + 2*a22*a26*a56 + a22*a25*a66) + 
  a22*(4*a26^2*a35 + 8*a25*a26*a36 + 2*a22*a36*a56 + a22*a35*a66))
 if ((r[1]==0) & (r[2]==4) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
9*a22*a36*(4*a26^2 + a22*a66) + 12*a23*(2*a26^3 + 3*a22*a26*a66)
 if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==4) 
& (r[5]==0) & (r[6]==0))
psi <-
24*a24^4 + 72*a22*a24^2*a44 + 9*a22^2*a44^2
 if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==3) 
& (r[5]==1) & (r[6]==0))
psi <-
24*a24^3*a25 + 36*a22*a24*a25*a44 + 36*a22*a24^2*a45 + 9*a22^2*a44*a45
 if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==1))
psi <-
24*a24^3*a26 + 36*a22*a24*a26*a44 + 36*a22*a24^2*a46 + 9*a22^2*a44*a46
 if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==2) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(16*a22*a24*a25*a45 + 4*a24^2*(2*a25^2 + a22*a55) + 
  a22*(4*a25^2*a44 + 2*a22*a45^2 + a22*a44*a55))
 if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==2) & (r[5]==1) & (r[6]==1))
psi <-
3*(8*a22*a24*(a26*a45 + a25*a46) + 4*a24^2*(2*a25*a26 + a22*a56) + 
  a22*(4*a25*a26*a44 + 2*a22*a45*a46 + a22*a44*a56))
 if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==2) & (r[5]==0) & (r[6]==2))
psi <-
3*(16*a22*a24*a26*a46 + 4*a24^2*(2*a26^2 + a22*a66) + 
  a22*(4*a26^2*a44 + 2*a22*a46^2 + a22*a44*a66))
 if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==1) 
& (r[5]==3) & (r[6]==0))
psi <-
9*a22*a45*(4*a25^2 + a22*a55) + 12*a24*(2*a25^3 + 3*a22*a25*a55)
 if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(4*a24*(2*a25^2*a26 + a22*a26*a55 + 2*a22*a25*a56) + 
  a22*(8*a25*a26*a45 + 4*a25^2*a46 + a22*a46*a55 + 2*a22*a45*a56))
 if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(4*a24*(2*a25*a26^2 + 2*a22*a26*a56 + a22*a25*a66) + 
  a22*(4*a26^2*a45 + 8*a25*a26*a46 + 2*a22*a46*a56 + a22*a45*a66))
 if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==3))
psi <-
9*a22*a46*(4*a26^2 + a22*a66) + 12*a24*(2*a26^3 + 3*a22*a26*a66)
 if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==0) 
& (r[5]==4) & (r[6]==0))
psi <-
24*a25^4 + 72*a22*a25^2*a55 + 9*a22^2*a55^2
 if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==1))
psi <-
24*a25^3*a26 + 36*a22*a25*a26*a55 + 36*a22*a25^2*a56 + 9*a22^2*a55*a56
 if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==2))
psi <-
3*(16*a22*a25*a26*a56 + 4*a25^2*(2*a26^2 + a22*a66) + 
  a22*(4*a26^2*a55 + 2*a22*a56^2 + a22*a55*a66))
 if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==3))
psi <-
9*a22*a56*(4*a26^2 + a22*a66) + 12*a25*(2*a26^3 + 3*a22*a26*a66)
 if ((r[1]==0) & (r[2]==4) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==4))
psi <-
24*a26^4 + 72*a22*a26^2*a66 + 9*a22^2*a66^2
 if ((r[1]==0) & (r[2]==3) & (r[3]==5) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a23*a33*(4*a23^2 + 3*a22*a33)
 if ((r[1]==0) & (r[2]==3) & (r[3]==4) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
36*a23^2*a24*a33 + 9*a22*a24*a33^2 + 24*a23^3*a34 + 36*a22*a23*a33*a34
 if ((r[1]==0) & (r[2]==3) & (r[3]==4) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
36*a23^2*a25*a33 + 9*a22*a25*a33^2 + 24*a23^3*a35 + 36*a22*a23*a33*a35
 if ((r[1]==0) & (r[2]==3) & (r[3]==4) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
36*a23^2*a26*a33 + 9*a22*a26*a33^2 + 24*a23^3*a36 + 36*a22*a23*a33*a36
 if ((r[1]==0) & (r[2]==3) & (r[3]==3) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(12*a23^2*a24*a34 + 6*a22*a24*a33*a34 + 2*a23^3*a44 + 
  3*a23*(2*a24^2*a33 + 2*a22*a34^2 + a22*a33*a44))
 if ((r[1]==0) & (r[2]==3) & (r[3]==3) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(6*a23^2*(a25*a34 + a24*a35) + 3*a22*a33*(a25*a34 + a24*a35) + 
  2*a23^3*a45 + 3*a23*(2*a24*a25*a33 + 2*a22*a34*a35 + a22*a33*a45))
 if ((r[1]==0) & (r[2]==3) & (r[3]==3) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(6*a23^2*(a26*a34 + a24*a36) + 3*a22*a33*(a26*a34 + a24*a36) + 
  2*a23^3*a46 + 3*a23*(2*a24*a26*a33 + 2*a22*a34*a36 + a22*a33*a46))
 if ((r[1]==0) & (r[2]==3) & (r[3]==3) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(12*a23^2*a25*a35 + 6*a22*a25*a33*a35 + 2*a23^3*a55 + 
  3*a23*(2*a25^2*a33 + 2*a22*a35^2 + a22*a33*a55))
 if ((r[1]==0) & (r[2]==3) & (r[3]==3) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(6*a23^2*(a26*a35 + a25*a36) + 3*a22*a33*(a26*a35 + a25*a36) + 
  2*a23^3*a56 + 3*a23*(2*a25*a26*a33 + 2*a22*a35*a36 + a22*a33*a56))
 if ((r[1]==0) & (r[2]==3) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(12*a23^2*a26*a36 + 6*a22*a26*a33*a36 + 2*a23^3*a66 + 
  3*a23*(2*a26^2*a33 + 2*a22*a36^2 + a22*a33*a66))
 if ((r[1]==0) & (r[2]==3) & (r[3]==2) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a24^3*a33 + 12*a23*a24^2*a34 + 6*a22*a23*a34*a44 + 
  3*a24*(2*a22*a34^2 + 2*a23^2*a44 + a22*a33*a44))
 if ((r[1]==0) & (r[2]==3) & (r[3]==2) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a24^2*(a25*a33 + 2*a23*a35) + 2*a23^2*a25*a44 + 
  2*a24*(4*a23*a25*a34 + 2*a22*a34*a35 + 2*a23^2*a45 + a22*a33*a45) + 
  a22*(2*a25*a34^2 + a25*a33*a44 + 2*a23*a35*a44 + 4*a23*a34*a45))
 if ((r[1]==0) & (r[2]==3) & (r[3]==2) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a24^2*(a26*a33 + 2*a23*a36) + 2*a23^2*a26*a44 + 
  2*a24*(4*a23*a26*a34 + 2*a22*a34*a36 + 2*a23^2*a46 + a22*a33*a46) + 
  a22*(2*a26*a34^2 + a26*a33*a44 + 2*a23*a36*a44 + 4*a23*a34*a46))


return(psi)
}

psins.6d.part84 <- function(r, a11, a22, a33, a44, a55, a66, a12, a13, a14, a15,
                           a16, a23, a24, a25, a26, a34, a35, a36, a45, a46, a56)
{
psi <- NA
if ((r[1]==0) & (r[2]==3) & (r[3]==2) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(a24*(2*a25^2*a33 + 8*a23*a25*a35 + 2*a22*a35^2 + 2*a23^2*a55 + 
    a22*a33*a55) + 2*(2*a23^2*a25*a45 + a22*a25*(2*a34*a35 + a33*a45) + 
    a23*(2*a25^2*a34 + 2*a22*a35*a45 + a22*a34*a55)))
 if ((r[1]==0) & (r[2]==3) & (r[3]==2) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(2*a23^2*(a26*a45 + a25*a46) + a22*(2*a26*a34*a35 + 2*a25*a34*a36 + 
    a26*a33*a45 + a25*a33*a46) + a24*(2*a25*a26*a33 + 4*a23*a26*a35 + 
    4*a23*a25*a36 + 2*a22*a35*a36 + 2*a23^2*a56 + a22*a33*a56) + 
  2*a23*(2*a25*a26*a34 + a22*a36*a45 + a22*a35*a46 + a22*a34*a56))
  if ((r[1]==0) & (r[2]==3) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(a24*(2*a26^2*a33 + 8*a23*a26*a36 + 2*a22*a36^2 + 2*a23^2*a66 + 
    a22*a33*a66) + 2*(2*a23^2*a26*a46 + a22*a26*(2*a34*a36 + a33*a46) + 
    a23*(2*a26^2*a34 + 2*a22*a36*a46 + a22*a34*a66)))
 if ((r[1]==0) & (r[2]==3) & (r[3]==2) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a25^3*a33 + 12*a23*a25^2*a35 + 6*a22*a23*a35*a55 + 
  3*a25*(2*a22*a35^2 + 2*a23^2*a55 + a22*a33*a55))
 if ((r[1]==0) & (r[2]==3) & (r[3]==2) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(2*a25^2*(a26*a33 + 2*a23*a36) + 2*a23^2*a26*a55 + 
  2*a25*(4*a23*a26*a35 + 2*a22*a35*a36 + 2*a23^2*a56 + a22*a33*a56) + 
  a22*(2*a26*a35^2 + a26*a33*a55 + 2*a23*a36*a55 + 4*a23*a35*a56))
 if ((r[1]==0) & (r[2]==3) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(a25*(2*a26^2*a33 + 8*a23*a26*a36 + 2*a22*a36^2 + 2*a23^2*a66 + 
    a22*a33*a66) + 2*(2*a23^2*a26*a56 + a22*a26*(2*a35*a36 + a33*a56) + 
    a23*(2*a26^2*a35 + 2*a22*a36*a56 + a22*a35*a66)))
 if ((r[1]==0) & (r[2]==3) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a26^3*a33 + 12*a23*a26^2*a36 + 6*a22*a23*a36*a66 + 
  3*a26*(2*a22*a36^2 + 2*a23^2*a66 + a22*a33*a66))
 if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==4) 
& (r[5]==0) & (r[6]==0))
psi <-
24*a24^3*a34 + 36*a23*a24^2*a44 + 36*a22*a24*a34*a44 + 9*a22*a23*a44^2
 if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==3) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a24^3*a35 + 6*a24^2*(a25*a34 + a23*a45) + 
  3*a22*a44*(a25*a34 + a23*a45) + 3*a24*(2*a23*a25*a44 + a22*a35*a44 + 
    2*a22*a34*a45))
 if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==3) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a24^3*a36 + 6*a24^2*(a26*a34 + a23*a46) + 
  3*a22*a44*(a26*a34 + a23*a46) + 3*a24*(2*a23*a26*a44 + a22*a36*a44 + 
    2*a22*a34*a46))
 if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==2) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a22*a25*(a35*a44 + 2*a34*a45) + 2*a24^2*(2*a25*a35 + a23*a55) + 
  2*a24*(2*a25^2*a34 + 4*a23*a25*a45 + 2*a22*a35*a45 + a22*a34*a55) + 
  a23*(2*a25^2*a44 + 2*a22*a45^2 + a22*a44*a55))
 if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==2) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(a22*(a26*a35*a44 + a25*a36*a44 + 2*a26*a34*a45 + 2*a25*a34*a46) + 
  2*a24^2*(a26*a35 + a25*a36 + a23*a56) + 
  2*a24*(2*a23*a26*a45 + a22*a36*a45 + a22*a35*a46 + 
    2*a25*(a26*a34 + a23*a46) + a22*a34*a56) + 
  a23*(2*a25*a26*a44 + 2*a22*a45*a46 + a22*a44*a56))
 if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a22*a26*(a36*a44 + 2*a34*a46) + 2*a24^2*(2*a26*a36 + a23*a66) + 
  2*a24*(2*a26^2*a34 + 4*a23*a26*a46 + 2*a22*a36*a46 + a22*a34*a66) + 
  a23*(2*a26^2*a44 + 2*a22*a46^2 + a22*a44*a66))
 if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==1) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a25^3*a34 + 6*a25^2*(a24*a35 + a23*a45) + 3*a22*(a24*a35 + a23*a45)*
   a55 + 3*a25*(2*a22*a35*a45 + 2*a23*a24*a55 + a22*a34*a55))
 if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==1) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(2*a25^2*(a26*a34 + a24*a36 + a23*a46) + 2*a23*a24*a26*a55 + 
  a22*(2*a26*a35*a45 + a26*a34*a55 + a24*a36*a55 + a23*a46*a55 + 
    2*a24*a35*a56 + 2*a23*a45*a56) + 
  2*a25*(2*a23*a26*a45 + a22*a36*a45 + a22*a35*a46 + a22*a34*a56 + 
    2*a24*(a26*a35 + a23*a56)))
 if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(2*a23*a26^2*a45 + 2*a22*a26*a36*a45 + 2*a22*a26*a35*a46 + 
  2*a22*a26*a34*a56 + 2*a22*a23*a46*a56 + a22*a23*a45*a66 + 
  a25*(2*a26^2*a34 + 2*a22*a36*a46 + 4*a26*(a24*a36 + a23*a46) + 
    2*a23*a24*a66 + a22*a34*a66) + a24*(2*a26^2*a35 + 4*a23*a26*a56 + 
    2*a22*a36*a56 + a22*a35*a66))
 if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a26^3*a34 + 6*a26^2*(a24*a36 + a23*a46) + 3*a22*(a24*a36 + a23*a46)*
   a66 + 3*a26*(2*a22*a36*a46 + 2*a23*a24*a66 + a22*a34*a66))
 if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==0) 
& (r[5]==4) & (r[6]==0))
psi <-
24*a25^3*a35 + 36*a23*a25^2*a55 + 36*a22*a25*a35*a55 + 9*a22*a23*a55^2
 if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==0) 
& (r[5]==3) & (r[6]==1))
psi <-
3*(2*a25^3*a36 + 6*a25^2*(a26*a35 + a23*a56) + 
  3*a22*a55*(a26*a35 + a23*a56) + 3*a25*(2*a23*a26*a55 + a22*a36*a55 + 
    2*a22*a35*a56))
 if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==2))
psi <-
3*(2*a22*a26*(a36*a55 + 2*a35*a56) + 2*a25^2*(2*a26*a36 + a23*a66) + 
  2*a25*(2*a26^2*a35 + 4*a23*a26*a56 + 2*a22*a36*a56 + a22*a35*a66) + 
  a23*(2*a26^2*a55 + 2*a22*a56^2 + a22*a55*a66))
 if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(2*a26^3*a35 + 6*a26^2*(a25*a36 + a23*a56) + 3*a22*(a25*a36 + a23*a56)*
   a66 + 3*a26*(2*a22*a36*a56 + 2*a23*a25*a66 + a22*a35*a66))
 if ((r[1]==0) & (r[2]==3) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==4))
psi <-
24*a26^3*a36 + 36*a23*a26^2*a66 + 36*a22*a26*a36*a66 + 9*a22*a23*a66^2
 if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==5) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a24*a44*(4*a24^2 + 3*a22*a44)
 if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==4) 
& (r[5]==1) & (r[6]==0))
psi <-
36*a24^2*a25*a44 + 9*a22*a25*a44^2 + 24*a24^3*a45 + 36*a22*a24*a44*a45
 if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==4) 
& (r[5]==0) & (r[6]==1))
psi <-
36*a24^2*a26*a44 + 9*a22*a26*a44^2 + 24*a24^3*a46 + 36*a22*a24*a44*a46
 if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==3) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(12*a24^2*a25*a45 + 6*a22*a25*a44*a45 + 2*a24^3*a55 + 
  3*a24*(2*a25^2*a44 + 2*a22*a45^2 + a22*a44*a55))
 if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==3) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(6*a24^2*(a26*a45 + a25*a46) + 3*a22*a44*(a26*a45 + a25*a46) + 
  2*a24^3*a56 + 3*a24*(2*a25*a26*a44 + 2*a22*a45*a46 + a22*a44*a56))
 if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==3) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(12*a24^2*a26*a46 + 6*a22*a26*a44*a46 + 2*a24^3*a66 + 
  3*a24*(2*a26^2*a44 + 2*a22*a46^2 + a22*a44*a66))
 if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==2) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a25^3*a44 + 12*a24*a25^2*a45 + 6*a22*a24*a45*a55 + 
  3*a25*(2*a22*a45^2 + 2*a24^2*a55 + a22*a44*a55))
 if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==2) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(2*a25^2*(a26*a44 + 2*a24*a46) + 2*a24^2*a26*a55 + 
  2*a25*(4*a24*a26*a45 + 2*a22*a45*a46 + 2*a24^2*a56 + a22*a44*a56) + 
  a22*(2*a26*a45^2 + a26*a44*a55 + 2*a24*a46*a55 + 4*a24*a45*a56))
 if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==2) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(a25*(2*a26^2*a44 + 8*a24*a26*a46 + 2*a22*a46^2 + 2*a24^2*a66 + 
    a22*a44*a66) + 2*(2*a24^2*a26*a56 + a22*a26*(2*a45*a46 + a44*a56) + 
    a24*(2*a26^2*a45 + 2*a22*a46*a56 + a22*a45*a66)))
 if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==2) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a26^3*a44 + 12*a24*a26^2*a46 + 6*a22*a24*a46*a66 + 
  3*a26*(2*a22*a46^2 + 2*a24^2*a66 + a22*a44*a66))
 if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==1) 
& (r[5]==4) & (r[6]==0))
psi <-
24*a25^3*a45 + 36*a24*a25^2*a55 + 36*a22*a25*a45*a55 + 9*a22*a24*a55^2
  if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==1) 
& (r[5]==3) & (r[6]==1))
psi <-
3*(2*a25^3*a46 + 6*a25^2*(a26*a45 + a24*a56) + 
  3*a22*a55*(a26*a45 + a24*a56) + 3*a25*(2*a24*a26*a55 + a22*a46*a55 + 
    2*a22*a45*a56))
 if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==1) 
& (r[5]==2) & (r[6]==2))
psi <-
3*(2*a22*a26*(a46*a55 + 2*a45*a56) + 2*a25^2*(2*a26*a46 + a24*a66) + 
  2*a25*(2*a26^2*a45 + 4*a24*a26*a56 + 2*a22*a46*a56 + a22*a45*a66) + 
  a24*(2*a26^2*a55 + 2*a22*a56^2 + a22*a55*a66))
 if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==1) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(2*a26^3*a45 + 6*a26^2*(a25*a46 + a24*a56) + 3*a22*(a25*a46 + a24*a56)*
   a66 + 3*a26*(2*a22*a46*a56 + 2*a24*a25*a66 + a22*a45*a66))
 if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==4))
psi <-
24*a26^3*a46 + 36*a24*a26^2*a66 + 36*a22*a26*a46*a66 + 9*a22*a24*a66^2
 if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==5) & (r[6]==0))
psi <-
15*a25*a55*(4*a25^2 + 3*a22*a55)
 if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==4) & (r[6]==1))
psi <-
36*a25^2*a26*a55 + 9*a22*a26*a55^2 + 24*a25^3*a56 + 36*a22*a25*a55*a56
 if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==3) & (r[6]==2))
psi <-
3*(12*a25^2*a26*a56 + 6*a22*a26*a55*a56 + 2*a25^3*a66 + 
  3*a25*(2*a26^2*a55 + 2*a22*a56^2 + a22*a55*a66))
 if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==2) & (r[6]==3))
psi <-
3*(2*a26^3*a55 + 12*a25*a26^2*a56 + 6*a22*a25*a56*a66 + 
  3*a26*(2*a22*a56^2 + 2*a25^2*a66 + a22*a55*a66))
 if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==1) & (r[6]==4))
psi <-
24*a26^3*a56 + 36*a25*a26^2*a66 + 36*a22*a26*a56*a66 + 9*a22*a25*a66^2
 if ((r[1]==0) & (r[2]==3) & (r[3]==0) & (r[4]==0) 
& (r[5]==0) & (r[6]==5))
psi <-
15*a26*a66*(4*a26^2 + 3*a22*a66)
 if ((r[1]==0) & (r[2]==2) & (r[3]==6) & (r[4]==0) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a33^2*(6*a23^2 + a22*a33)
 if ((r[1]==0) & (r[2]==2) & (r[3]==5) & (r[4]==1) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a33*(2*a23*a24*a33 + 4*a23^2*a34 + a22*a33*a34)
 if ((r[1]==0) & (r[2]==2) & (r[3]==5) & (r[4]==0) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a33*(2*a23*a25*a33 + 4*a23^2*a35 + a22*a33*a35)
 if ((r[1]==0) & (r[2]==2) & (r[3]==5) & (r[4]==0) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a33*(2*a23*a26*a33 + 4*a23^2*a36 + a22*a33*a36)
 if ((r[1]==0) & (r[2]==2) & (r[3]==4) & (r[4]==2) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(2*a24^2*a33^2 + 16*a23*a24*a33*a34 + 4*a23^2*(2*a34^2 + a33*a44) + 
  a22*a33*(4*a34^2 + a33*a44))
 if ((r[1]==0) & (r[2]==2) & (r[3]==4) & (r[4]==1) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(8*a23*a25*a33*a34 + 2*a24*a33*(a25*a33 + 4*a23*a35) + 
  4*a23^2*(2*a34*a35 + a33*a45) + a22*a33*(4*a34*a35 + a33*a45))
 if ((r[1]==0) & (r[2]==2) & (r[3]==4) & (r[4]==1) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(8*a23*a26*a33*a34 + 2*a24*a33*(a26*a33 + 4*a23*a36) + 
  4*a23^2*(2*a34*a36 + a33*a46) + a22*a33*(4*a34*a36 + a33*a46))
 if ((r[1]==0) & (r[2]==2) & (r[3]==4) & (r[4]==0) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a25^2*a33^2 + 16*a23*a25*a33*a35 + 4*a23^2*(2*a35^2 + a33*a55) + 
  a22*a33*(4*a35^2 + a33*a55))
 if ((r[1]==0) & (r[2]==2) & (r[3]==4) & (r[4]==0) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(8*a23*a26*a33*a35 + 2*a25*a33*(a26*a33 + 4*a23*a36) + 
  4*a23^2*(2*a35*a36 + a33*a56) + a22*a33*(4*a35*a36 + a33*a56))
 if ((r[1]==0) & (r[2]==2) & (r[3]==4) & (r[4]==0) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a26^2*a33^2 + 16*a23*a26*a33*a36 + 4*a23^2*(2*a36^2 + a33*a66) + 
  a22*a33*(4*a36^2 + a33*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==3) & (r[4]==3) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(6*a24^2*a33*a34 + 6*a23*a24*(2*a34^2 + a33*a44) + 
  a34*(2*a22*a34^2 + 6*a23^2*a44 + 3*a22*a33*a44))
 if ((r[1]==0) & (r[2]==2) & (r[3]==3) & (r[4]==2) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(2*a24^2*a33*a35 + 2*a23*a25*(2*a34^2 + a33*a44) + 
  4*a24*(a25*a33*a34 + 2*a23*a34*a35 + a23*a33*a45) + 
  2*a23^2*(a35*a44 + 2*a34*a45) + a22*(2*a34^2*a35 + a33*a35*a44 + 
    2*a33*a34*a45))
 if ((r[1]==0) & (r[2]==2) & (r[3]==3) & (r[4]==2) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(2*a24^2*a33*a36 + 2*a23*a26*(2*a34^2 + a33*a44) + 
  4*a24*(a26*a33*a34 + 2*a23*a34*a36 + a23*a33*a46) + 
  2*a23^2*(a36*a44 + 2*a34*a46) + a22*(2*a34^2*a36 + a33*a36*a44 + 
    2*a33*a34*a46))
 if ((r[1]==0) & (r[2]==2) & (r[3]==3) & (r[4]==1) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a25^2*a33*a34 + 4*a25*(a24*a33*a35 + 2*a23*a34*a35 + a23*a33*a45) + 
  2*a23*a24*(2*a35^2 + a33*a55) + 2*a23^2*(2*a35*a45 + a34*a55) + 
  a22*(2*a34*a35^2 + 2*a33*a35*a45 + a33*a34*a55))
 if ((r[1]==0) & (r[2]==2) & (r[3]==3) & (r[4]==1) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(4*a23*a26*a34*a35 + 2*a22*a34*a35*a36 + 2*a23*a26*a33*a45 + 
  2*a23^2*a36*a45 + a22*a33*a36*a45 + 2*a23^2*a35*a46 + a22*a33*a35*a46 + 
  2*a25*(a26*a33*a34 + a24*a33*a36 + 2*a23*a34*a36 + a23*a33*a46) + 
  2*a23^2*a34*a56 + a22*a33*a34*a56 + 
  2*a24*(a26*a33*a35 + 2*a23*a35*a36 + a23*a33*a56))
 if ((r[1]==0) & (r[2]==2) & (r[3]==3) & (r[4]==1) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a26^2*a33*a34 + 4*a26*(a24*a33*a36 + 2*a23*a34*a36 + a23*a33*a46) + 
  2*a23*a24*(2*a36^2 + a33*a66) + 2*a23^2*(2*a36*a46 + a34*a66) + 
  a22*(2*a34*a36^2 + 2*a33*a36*a46 + a33*a34*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==3) & (r[4]==0) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(6*a25^2*a33*a35 + 6*a23*a25*(2*a35^2 + a33*a55) + 
  a35*(2*a22*a35^2 + 6*a23^2*a55 + 3*a22*a33*a55))
 if ((r[1]==0) & (r[2]==2) & (r[3]==3) & (r[4]==0) 
& (r[5]==2) & (r[6]==1))
psi <-
3*(2*a25^2*a33*a36 + 2*a23*a26*(2*a35^2 + a33*a55) + 
  4*a25*(a26*a33*a35 + 2*a23*a35*a36 + a23*a33*a56) + 
  2*a23^2*(a36*a55 + 2*a35*a56) + a22*(2*a35^2*a36 + a33*a36*a55 + 
    2*a33*a35*a56))
 if ((r[1]==0) & (r[2]==2) & (r[3]==3) & (r[4]==0) 
& (r[5]==1) & (r[6]==2))
psi <-
3*(2*a26^2*a33*a35 + 4*a26*(a25*a33*a36 + 2*a23*a35*a36 + a23*a33*a56) + 
  2*a23*a25*(2*a36^2 + a33*a66) + 2*a23^2*(2*a36*a56 + a35*a66) + 
  a22*(2*a35*a36^2 + 2*a33*a36*a56 + a33*a35*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==3) & (r[4]==0) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(6*a26^2*a33*a36 + 6*a23*a26*(2*a36^2 + a33*a66) + 
  a36*(2*a22*a36^2 + 6*a23^2*a66 + 3*a22*a33*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==4) 
& (r[5]==0) & (r[6]==0))
psi <-
3*(16*a23*a24*a34*a44 + 4*a24^2*(2*a34^2 + a33*a44) + 
  a44*(4*a22*a34^2 + 2*a23^2*a44 + a22*a33*a44))
 if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==3) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(4*a23*a25*a34*a44 + 2*a23^2*a44*a45 + 2*a24^2*(2*a34*a35 + a33*a45) + 
  2*a24*(2*a25*a34^2 + a25*a33*a44 + 2*a23*a35*a44 + 4*a23*a34*a45) + 
  a22*(2*a34*a35*a44 + 2*a34^2*a45 + a33*a44*a45))
  if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==3) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(4*a23*a26*a34*a44 + 2*a23^2*a44*a46 + 2*a24^2*(2*a34*a36 + a33*a46) + 
  2*a24*(2*a26*a34^2 + a26*a33*a44 + 2*a23*a36*a44 + 4*a23*a34*a46) + 
  a22*(2*a34*a36*a44 + 2*a34^2*a46 + a33*a44*a46))
 if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==2) 
& (r[5]==2) & (r[6]==0))
psi <-
2*a22*a35^2*a44 + 2*a25^2*(2*a34^2 + a33*a44) + 8*a22*a34*a35*a45 + 
 4*a23^2*a45^2 + 2*a22*a33*a45^2 + 
 8*a25*(2*a24*a34*a35 + a23*a35*a44 + a24*a33*a45 + 2*a23*a34*a45) + 
 2*a22*a34^2*a55 + 2*a23^2*a44*a55 + a22*a33*a44*a55 + 
 2*a24^2*(2*a35^2 + a33*a55) + 8*a23*a24*(2*a35*a45 + a34*a55)
 if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==2) 
& (r[5]==1) & (r[6]==1))
psi <-
4*a23*a26*a35*a44 + 2*a22*a35*a36*a44 + 8*a23*a26*a34*a45 + 
 4*a22*a34*a36*a45 + 4*a22*a34*a35*a46 + 4*a23^2*a45*a46 + 
 2*a22*a33*a45*a46 + 2*a25*(2*a26*a34^2 + 4*a24*a34*a36 + a26*a33*a44 + 
   2*a23*a36*a44 + 2*a24*a33*a46 + 4*a23*a34*a46) + 2*a22*a34^2*a56 + 
 2*a23^2*a44*a56 + a22*a33*a44*a56 + 2*a24^2*(2*a35*a36 + a33*a56) + 
 4*a24*(a26*(2*a34*a35 + a33*a45) + 2*a23*(a36*a45 + a35*a46 + a34*a56))
 if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==2) 
& (r[5]==0) & (r[6]==2))
psi <-
2*a22*a36^2*a44 + 2*a26^2*(2*a34^2 + a33*a44) + 8*a22*a34*a36*a46 + 
 4*a23^2*a46^2 + 2*a22*a33*a46^2 + 
 8*a26*(2*a24*a34*a36 + a23*a36*a44 + a24*a33*a46 + 2*a23*a34*a46) + 
 2*a22*a34^2*a66 + 2*a23^2*a44*a66 + a22*a33*a44*a66 + 
 2*a24^2*(2*a36^2 + a33*a66) + 8*a23*a24*(2*a36*a46 + a34*a66)
 if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==1) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a25^2*(2*a34*a35 + a33*a45) + 2*a23*(2*a24*a35 + a23*a45)*a55 + 
  2*a25*(2*a24*a35^2 + 4*a23*a35*a45 + a24*a33*a55 + 2*a23*a34*a55) + 
  a22*(2*a35^2*a45 + 2*a34*a35*a55 + a33*a45*a55))
 if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==1) 
& (r[5]==2) & (r[6]==1))
psi <-
8*a23*a26*a35*a45 + 4*a22*a35*a36*a45 + 2*a22*a35^2*a46 + 
 2*a25^2*(2*a34*a36 + a33*a46) + 4*a23*a26*a34*a55 + 2*a22*a34*a36*a55 + 
 2*a23^2*a46*a55 + a22*a33*a46*a55 + 4*a22*a34*a35*a56 + 4*a23^2*a45*a56 + 
 2*a22*a33*a45*a56 + 4*a25*(2*a26*a34*a35 + 2*a24*a35*a36 + a26*a33*a45 + 
   2*a23*a36*a45 + 2*a23*a35*a46 + a24*a33*a56 + 2*a23*a34*a56) + 
 2*a24*(2*a26*a35^2 + a26*a33*a55 + 2*a23*a36*a55 + 4*a23*a35*a56)
 if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==1) 
& (r[5]==1) & (r[6]==2))
psi <-
2*a22*a36^2*a45 + 2*a26^2*(2*a34*a35 + a33*a45) + 8*a23*a25*a36*a46 + 
 4*a22*a35*a36*a46 + 4*a22*a34*a36*a56 + 4*a23^2*a46*a56 + 
 2*a22*a33*a46*a56 + 4*a26*(2*a25*a34*a36 + 2*a24*a35*a36 + 2*a23*a36*a45 + 
   a25*a33*a46 + 2*a23*a35*a46 + a24*a33*a56 + 2*a23*a34*a56) + 
 4*a23*a25*a34*a66 + 2*a22*a34*a35*a66 + 2*a23^2*a45*a66 + a22*a33*a45*a66 + 
 2*a24*(2*a25*a36^2 + 4*a23*a36*a56 + a25*a33*a66 + 2*a23*a35*a66)
 if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==1) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a26^2*(2*a34*a36 + a33*a46) + 2*a23*(2*a24*a36 + a23*a46)*a66 + 
  2*a26*(2*a24*a36^2 + 4*a23*a36*a46 + a24*a33*a66 + 2*a23*a34*a66) + 
  a22*(2*a36^2*a46 + 2*a34*a36*a66 + a33*a46*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==0) 
& (r[5]==4) & (r[6]==0))
psi <-
3*(16*a23*a25*a35*a55 + 4*a25^2*(2*a35^2 + a33*a55) + 
  a55*(4*a22*a35^2 + 2*a23^2*a55 + a22*a33*a55))
 if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==0) 
& (r[5]==3) & (r[6]==1))
psi <-
3*(4*a23*a26*a35*a55 + 2*a23^2*a55*a56 + 2*a25^2*(2*a35*a36 + a33*a56) + 
  2*a25*(2*a26*a35^2 + a26*a33*a55 + 2*a23*a36*a55 + 4*a23*a35*a56) + 
  a22*(2*a35*a36*a55 + 2*a35^2*a56 + a33*a55*a56))
 if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==0) 
& (r[5]==2) & (r[6]==2))
psi <-
2*a22*a36^2*a55 + 2*a26^2*(2*a35^2 + a33*a55) + 8*a22*a35*a36*a56 + 
 4*a23^2*a56^2 + 2*a22*a33*a56^2 + 
 8*a26*(2*a25*a35*a36 + a23*a36*a55 + a25*a33*a56 + 2*a23*a35*a56) + 
 2*a22*a35^2*a66 + 2*a23^2*a55*a66 + a22*a33*a55*a66 + 
 2*a25^2*(2*a36^2 + a33*a66) + 8*a23*a25*(2*a36*a56 + a35*a66)
 if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==0) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(2*a26^2*(2*a35*a36 + a33*a56) + 2*a23*(2*a25*a36 + a23*a56)*a66 + 
  2*a26*(2*a25*a36^2 + 4*a23*a36*a56 + a25*a33*a66 + 2*a23*a35*a66) + 
  a22*(2*a36^2*a56 + 2*a35*a36*a66 + a33*a56*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==2) & (r[4]==0) 
& (r[5]==0) & (r[6]==4))
psi <-
3*(16*a23*a26*a36*a66 + 4*a26^2*(2*a36^2 + a33*a66) + 
  a66*(4*a22*a36^2 + 2*a23^2*a66 + a22*a33*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==5) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a44*(4*a24^2*a34 + 2*a23*a24*a44 + a22*a34*a44)
 if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==4) 
& (r[5]==1) & (r[6]==0))
psi <-
3*(8*a24*a44*(a25*a34 + a23*a45) + 4*a24^2*(a35*a44 + 2*a34*a45) + 
  a44*(2*a23*a25*a44 + a22*a35*a44 + 4*a22*a34*a45))
 if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==4) 
& (r[5]==0) & (r[6]==1))
psi <-
3*(8*a24*a44*(a26*a34 + a23*a46) + 4*a24^2*(a36*a44 + 2*a34*a46) + 
  a44*(2*a23*a26*a44 + a22*a36*a44 + 4*a22*a34*a46))
 if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==3) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a25^2*a34*a44 + 4*a25*(a24*a35*a44 + 2*a24*a34*a45 + a23*a44*a45) + 
  2*a24^2*(2*a35*a45 + a34*a55) + 2*a23*a24*(2*a45^2 + a44*a55) + 
  a22*(2*a35*a44*a45 + 2*a34*a45^2 + a34*a44*a55))
 if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==3) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(2*a23*a26*a44*a45 + a22*a36*a44*a45 + a22*a35*a44*a46 + 
  2*a22*a34*a45*a46 + 2*a25*(a26*a34*a44 + a24*a36*a44 + 2*a24*a34*a46 + 
    a23*a44*a46) + a22*a34*a44*a56 + 2*a24^2*(a36*a45 + a35*a46 + a34*a56) + 
  2*a24*(a26*a35*a44 + 2*a26*a34*a45 + 2*a23*a45*a46 + a23*a44*a56))
 if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==3) 
& (r[5]==0) & (r[6]==2))
psi <-
3*(2*a26^2*a34*a44 + 4*a26*(a24*a36*a44 + 2*a24*a34*a46 + a23*a44*a46) + 
  2*a24^2*(2*a36*a46 + a34*a66) + 2*a23*a24*(2*a46^2 + a44*a66) + 
  a22*(2*a36*a44*a46 + 2*a34*a46^2 + a34*a44*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==2) 
& (r[5]==3) & (r[6]==0))
psi <-
3*(2*a25^2*(a35*a44 + 2*a34*a45) + 2*a24*(a24*a35 + 2*a23*a45)*a55 + 
  2*a25*(4*a24*a35*a45 + 2*a23*a45^2 + 2*a24*a34*a55 + a23*a44*a55) + 
  a22*(2*a35*a45^2 + a35*a44*a55 + 2*a34*a45*a55))
 if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==2) 
& (r[5]==2) & (r[6]==1))
psi <-
4*a23*a26*a45^2 + 2*a22*a36*a45^2 + 4*a22*a35*a45*a46 + 
 2*a25^2*(a36*a44 + 2*a34*a46) + 2*a23*a26*a44*a55 + a22*a36*a44*a55 + 
 2*a22*a34*a46*a55 + 2*a22*a35*a44*a56 + 4*a22*a34*a45*a56 + 
 2*a24^2*(a36*a55 + 2*a35*a56) + 4*a25*(a26*a35*a44 + 2*a26*a34*a45 + 
   2*a24*a36*a45 + 2*a24*a35*a46 + 2*a23*a45*a46 + 2*a24*a34*a56 + 
   a23*a44*a56) + 4*a24*(2*a26*a35*a45 + a26*a34*a55 + a23*a46*a55 + 
   2*a23*a45*a56)
 if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==2) 
& (r[5]==1) & (r[6]==2))
psi <-
2*a26^2*(a35*a44 + 2*a34*a45) + 4*a22*a36*a45*a46 + 4*a23*a25*a46^2 + 
 2*a22*a35*a46^2 + 2*a22*a36*a44*a56 + 4*a22*a34*a46*a56 + 
 4*a26*(a25*a36*a44 + 2*a24*a36*a45 + 2*a25*a34*a46 + 2*a24*a35*a46 + 
   2*a23*a45*a46 + 2*a24*a34*a56 + a23*a44*a56) + 2*a23*a25*a44*a66 + 
 a22*a35*a44*a66 + 2*a22*a34*a45*a66 + 2*a24^2*(2*a36*a56 + a35*a66) + 
 4*a24*(2*a25*a36*a46 + 2*a23*a46*a56 + a25*a34*a66 + a23*a45*a66)
 if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==2) 
& (r[5]==0) & (r[6]==3))
psi <-
3*(2*a26^2*(a36*a44 + 2*a34*a46) + 2*a24*(a24*a36 + 2*a23*a46)*a66 + 
  2*a26*(4*a24*a36*a46 + 2*a23*a46^2 + 2*a24*a34*a66 + a23*a44*a66) + 
  a22*(2*a36*a46^2 + a36*a44*a66 + 2*a34*a46*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==1) 
& (r[5]==4) & (r[6]==0))
psi <-
3*(8*a25*(a24*a35 + a23*a45)*a55 + 4*a25^2*(2*a35*a45 + a34*a55) + 
  a55*(4*a22*a35*a45 + 2*a23*a24*a55 + a22*a34*a55))
  if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==1) 
& (r[5]==3) & (r[6]==1))
psi <-
3*(2*a23*a26*a45*a55 + a22*a36*a45*a55 + a22*a35*a46*a55 + 
  2*a22*a35*a45*a56 + a22*a34*a55*a56 + 2*a24*a55*(a26*a35 + a23*a56) + 
  2*a25^2*(a36*a45 + a35*a46 + a34*a56) + 
  2*a25*(2*a26*a35*a45 + a26*a34*a55 + a24*a36*a55 + a23*a46*a55 + 
    2*a24*a35*a56 + 2*a23*a45*a56))
 if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==1) 
& (r[5]==2) & (r[6]==2))
psi <-
2*a22*a36*a46*a55 + 2*a26^2*(2*a35*a45 + a34*a55) + 4*a22*a36*a45*a56 + 
 4*a22*a35*a46*a56 + 4*a23*a24*a56^2 + 2*a22*a34*a56^2 + 
 4*a26*(a24*a36*a55 + a23*a46*a55 + 2*a24*a35*a56 + 2*a23*a45*a56 + 
   2*a25*(a36*a45 + a35*a46 + a34*a56)) + 2*a22*a35*a45*a66 + 
 2*a23*a24*a55*a66 + a22*a34*a55*a66 + 2*a25^2*(2*a36*a46 + a34*a66) + 
 4*a25*(2*a24*a36*a56 + 2*a23*a46*a56 + a24*a35*a66 + a23*a45*a66)
 if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==1) 
& (r[5]==1) & (r[6]==3))
psi <-
3*(2*a26^2*(a36*a45 + a35*a46 + a34*a56) + 
  2*(a24*a25*a36 + a23*a25*a46 + a23*a24*a56)*a66 + 
  2*a26*(2*a25*a36*a46 + 2*a24*a36*a56 + 2*a23*a46*a56 + a25*a34*a66 + 
    a24*a35*a66 + a23*a45*a66) + a22*(2*a36*a46*a56 + a36*a45*a66 + 
    a35*a46*a66 + a34*a56*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==1) 
& (r[5]==0) & (r[6]==4))
psi <-
3*(8*a26*(a24*a36 + a23*a46)*a66 + 4*a26^2*(2*a36*a46 + a34*a66) + 
  a66*(4*a22*a36*a46 + 2*a23*a24*a66 + a22*a34*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==5) & (r[6]==0))
psi <-
15*a55*(4*a25^2*a35 + 2*a23*a25*a55 + a22*a35*a55)
 if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==4) & (r[6]==1))
psi <-
3*(8*a25*a55*(a26*a35 + a23*a56) + 4*a25^2*(a36*a55 + 2*a35*a56) + 
  a55*(2*a23*a26*a55 + a22*a36*a55 + 4*a22*a35*a56))
 if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==3) & (r[6]==2))
psi <-
3*(2*a26^2*a35*a55 + 4*a26*(a25*a36*a55 + 2*a25*a35*a56 + a23*a55*a56) + 
  2*a25^2*(2*a36*a56 + a35*a66) + 2*a23*a25*(2*a56^2 + a55*a66) + 
  a22*(2*a36*a55*a56 + 2*a35*a56^2 + a35*a55*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==2) & (r[6]==3))
psi <-
3*(2*a26^2*(a36*a55 + 2*a35*a56) + 2*a25*(a25*a36 + 2*a23*a56)*a66 + 
  2*a26*(4*a25*a36*a56 + 2*a23*a56^2 + 2*a25*a35*a66 + a23*a55*a66) + 
  a22*(2*a36*a56^2 + a36*a55*a66 + 2*a35*a56*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==1) & (r[6]==4))
psi <-
3*(8*a26*(a25*a36 + a23*a56)*a66 + 4*a26^2*(2*a36*a56 + a35*a66) + 
  a66*(4*a22*a36*a56 + 2*a23*a25*a66 + a22*a35*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==1) & (r[4]==0) 
& (r[5]==0) & (r[6]==5))
psi <-
15*a66*(4*a26^2*a36 + 2*a23*a26*a66 + a22*a36*a66)
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==6) 
& (r[5]==0) & (r[6]==0))
psi <-
15*a44^2*(6*a24^2 + a22*a44)
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==5) 
& (r[5]==1) & (r[6]==0))
psi <-
15*a44*(2*a24*a25*a44 + 4*a24^2*a45 + a22*a44*a45)
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==5) 
& (r[5]==0) & (r[6]==1))
psi <-
15*a44*(2*a24*a26*a44 + 4*a24^2*a46 + a22*a44*a46)
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==4) 
& (r[5]==2) & (r[6]==0))
psi <-
3*(2*a25^2*a44^2 + 16*a24*a25*a44*a45 + 4*a24^2*(2*a45^2 + a44*a55) + 
  a22*a44*(4*a45^2 + a44*a55))
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==4) 
& (r[5]==1) & (r[6]==1))
psi <-
3*(8*a24*a26*a44*a45 + 2*a25*a44*(a26*a44 + 4*a24*a46) + 
  4*a24^2*(2*a45*a46 + a44*a56) + a22*a44*(4*a45*a46 + a44*a56))
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==4) & (r[5]==0) & (r[6]==2))
psi <-
3*(2*a26^2*a44^2 + 16*a24*a26*a44*a46 + 4*a24^2*(2*a46^2 + a44*a66) + 
  a22*a44*(4*a46^2 + a44*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==3) & (r[5]==3) & (r[6]==0))
psi <-
3*(6*a25^2*a44*a45 + 6*a24*a25*(2*a45^2 + a44*a55) + 
  a45*(2*a22*a45^2 + 6*a24^2*a55 + 3*a22*a44*a55))
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==3) & (r[5]==2) & (r[6]==1))
psi <-
3*(2*a25^2*a44*a46 + 2*a24*a26*(2*a45^2 + a44*a55) + 
  4*a25*(a26*a44*a45 + 2*a24*a45*a46 + a24*a44*a56) + 
  2*a24^2*(a46*a55 + 2*a45*a56) + a22*(2*a45^2*a46 + a44*a46*a55 + 
    2*a44*a45*a56))
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==3) & (r[5]==1) & (r[6]==2))
psi <-
3*(2*a26^2*a44*a45 + 4*a26*(a25*a44*a46 + 2*a24*a45*a46 + a24*a44*a56) + 
  2*a24*a25*(2*a46^2 + a44*a66) + 2*a24^2*(2*a46*a56 + a45*a66) + 
  a22*(2*a45*a46^2 + 2*a44*a46*a56 + a44*a45*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==3) & (r[5]==0) & (r[6]==3))
psi <-
3*(6*a26^2*a44*a46 + 6*a24*a26*(2*a46^2 + a44*a66) + 
  a46*(2*a22*a46^2 + 6*a24^2*a66 + 3*a22*a44*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==2) & (r[5]==4) & (r[6]==0))
psi <-
3*(16*a24*a25*a45*a55 + 4*a25^2*(2*a45^2 + a44*a55) + 
  a55*(4*a22*a45^2 + 2*a24^2*a55 + a22*a44*a55))
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==2) & (r[5]==3) & (r[6]==1))
psi <-
3*(4*a24*a26*a45*a55 + 2*a24^2*a55*a56 + 2*a25^2*(2*a45*a46 + a44*a56) + 
  2*a25*(2*a26*a45^2 + a26*a44*a55 + 2*a24*a46*a55 + 4*a24*a45*a56) + 
  a22*(2*a45*a46*a55 + 2*a45^2*a56 + a44*a55*a56))
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==2) & (r[5]==2) & (r[6]==2))
psi <-
2*a22*a46^2*a55 + 2*a26^2*(2*a45^2 + a44*a55) + 8*a22*a45*a46*a56 + 
 4*a24^2*a56^2 + 2*a22*a44*a56^2 + 
 8*a26*(2*a25*a45*a46 + a24*a46*a55 + a25*a44*a56 + 2*a24*a45*a56) + 
 2*a22*a45^2*a66 + 2*a24^2*a55*a66 + a22*a44*a55*a66 + 
 2*a25^2*(2*a46^2 + a44*a66) + 8*a24*a25*(2*a46*a56 + a45*a66)
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==2) & (r[5]==1) & (r[6]==3))
psi <-
3*(2*a26^2*(2*a45*a46 + a44*a56) + 2*a24*(2*a25*a46 + a24*a56)*a66 + 
  2*a26*(2*a25*a46^2 + 4*a24*a46*a56 + a25*a44*a66 + 2*a24*a45*a66) + 
  a22*(2*a46^2*a56 + 2*a45*a46*a66 + a44*a56*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==2) & (r[5]==0) & (r[6]==4))
psi <-
3*(16*a24*a26*a46*a66 + 4*a26^2*(2*a46^2 + a44*a66) + 
  a66*(4*a22*a46^2 + 2*a24^2*a66 + a22*a44*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==1) & (r[5]==5) & (r[6]==0))
psi <-
15*a55*(4*a25^2*a45 + 2*a24*a25*a55 + a22*a45*a55)
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==1) & (r[5]==4) & (r[6]==1))
psi <-
3*(8*a25*a55*(a26*a45 + a24*a56) + 4*a25^2*(a46*a55 + 2*a45*a56) + 
  a55*(2*a24*a26*a55 + a22*a46*a55 + 4*a22*a45*a56))
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==1) & (r[5]==3) & (r[6]==2))
psi <-
3*(2*a26^2*a45*a55 + 4*a26*(a25*a46*a55 + 2*a25*a45*a56 + a24*a55*a56) + 
  2*a25^2*(2*a46*a56 + a45*a66) + 2*a24*a25*(2*a56^2 + a55*a66) + 
  a22*(2*a46*a55*a56 + 2*a45*a56^2 + a45*a55*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==1) & (r[5]==2) & (r[6]==3))
psi <-
3*(2*a26^2*(a46*a55 + 2*a45*a56) + 2*a25*(a25*a46 + 2*a24*a56)*a66 + 
  2*a26*(4*a25*a46*a56 + 2*a24*a56^2 + 2*a25*a45*a66 + a24*a55*a66) + 
  a22*(2*a46*a56^2 + a46*a55*a66 + 2*a45*a56*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==1) & (r[5]==1) & (r[6]==4))
psi <-
3*(8*a26*(a25*a46 + a24*a56)*a66 + 4*a26^2*(2*a46*a56 + a45*a66) + 
  a66*(4*a22*a46*a56 + 2*a24*a25*a66 + a22*a45*a66))
  if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==1) 
& (r[5]==0) & (r[6]==5))
psi <-
15*a66*(4*a26^2*a46 + 2*a24*a26*a66 + a22*a46*a66)
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==0) & (r[5]==6) & (r[6]==0))
psi <-
15*a55^2*(6*a25^2 + a22*a55)
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==0) & (r[5]==5) & (r[6]==1))
psi <-
15*a55*(2*a25*a26*a55 + 4*a25^2*a56 + a22*a55*a56)
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==0) & (r[5]==4) & (r[6]==2))
psi <-
3*(2*a26^2*a55^2 + 16*a25*a26*a55*a56 + 4*a25^2*(2*a56^2 + a55*a66) + 
  a22*a55*(4*a56^2 + a55*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==0) & (r[5]==3) & (r[6]==3))
psi <-
3*(6*a26^2*a55*a56 + 6*a25*a26*(2*a56^2 + a55*a66) + 
  a56*(2*a22*a56^2 + 6*a25^2*a66 + 3*a22*a55*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==0) & (r[5]==2) & (r[6]==4))
psi <-
3*(16*a25*a26*a56*a66 + 4*a26^2*(2*a56^2 + a55*a66) + 
  a66*(4*a22*a56^2 + 2*a25^2*a66 + a22*a55*a66))
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==0) & (r[5]==1) & (r[6]==5))
psi <-
15*a66*(4*a26^2*a56 + 2*a25*a26*a66 + a22*a56*a66)
 if ((r[1]==0) & (r[2]==2) & (r[3]==0) & (r[4]==0) & (r[5]==0) & (r[6]==6))
psi <-
15*a66^2*(6*a26^2 + a22*a66)
 if ((r[1]==0) & (r[2]==1) & (r[3]==7) & (r[4]==0) & (r[5]==0) & (r[6]==0))
psi <-
105*a23*a33^3
 if ((r[1]==0) & (r[2]==1) & (r[3]==6) & (r[4]==1) & (r[5]==0) & (r[6]==0))
psi <-
15*a33^2*(a24*a33 + 6*a23*a34)
 if ((r[1]==0) & (r[2]==1) & (r[3]==6) & (r[4]==0) & (r[5]==1) & (r[6]==0))
psi <-
15*a33^2*(a25*a33 + 6*a23*a35)
 if ((r[1]==0) & (r[2]==1) & (r[3]==6) & (r[4]==0) & (r[5]==0) & (r[6]==1))
psi <-
15*a33^2*(a26*a33 + 6*a23*a36)
 if ((r[1]==0) & (r[2]==1) & (r[3]==5) & (r[4]==2) & (r[5]==0) & (r[6]==0))
psi <-
15*a33*(2*a24*a33*a34 + 4*a23*a34^2 + a23*a33*a44)
 if ((r[1]==0) & (r[2]==1) & (r[3]==5) & (r[4]==1) & (r[5]==1) & (r[6]==0))
psi <-
15*a33*(a25*a33*a34 + a24*a33*a35 + 4*a23*a34*a35 + a23*a33*a45)
 if ((r[1]==0) & (r[2]==1) & (r[3]==5) & (r[4]==1) & (r[5]==0) & (r[6]==1))
psi <-
15*a33*(a26*a33*a34 + a24*a33*a36 + 4*a23*a34*a36 + a23*a33*a46)
 if ((r[1]==0) & (r[2]==1) & (r[3]==5) & (r[4]==0) & (r[5]==2) & (r[6]==0))
psi <-
15*a33*(2*a25*a33*a35 + 4*a23*a35^2 + a23*a33*a55)
 if ((r[1]==0) & (r[2]==1) & (r[3]==5) & (r[4]==0) & (r[5]==1) & (r[6]==1))
psi <-
15*a33*(a26*a33*a35 + a25*a33*a36 + 4*a23*a35*a36 + a23*a33*a56)
 if ((r[1]==0) & (r[2]==1) & (r[3]==5) & (r[4]==0) & (r[5]==0) & (r[6]==2))
psi <-
15*a33*(2*a26*a33*a36 + 4*a23*a36^2 + a23*a33*a66)
 if ((r[1]==0) & (r[2]==1) & (r[3]==4) & (r[4]==3) & (r[5]==0) & (r[6]==0))
psi <-
9*a24*a33*(4*a34^2 + a33*a44) + 12*a23*a34*(2*a34^2 + 3*a33*a44)
 if ((r[1]==0) & (r[2]==1) & (r[3]==4) & (r[4]==2) & (r[5]==1) & (r[6]==0))
psi <-
3*(a25*a33*(4*a34^2 + a33*a44) + 2*a24*a33*(4*a34*a35 + a33*a45) + 
  4*a23*(2*a34^2*a35 + a33*a35*a44 + 2*a33*a34*a45))
 if ((r[1]==0) & (r[2]==1) & (r[3]==4) & (r[4]==2) & (r[5]==0) & (r[6]==1))
psi <-
3*(a26*a33*(4*a34^2 + a33*a44) + 2*a24*a33*(4*a34*a36 + a33*a46) + 
  4*a23*(2*a34^2*a36 + a33*a36*a44 + 2*a33*a34*a46))
 if ((r[1]==0) & (r[2]==1) & (r[3]==4) & (r[4]==1) & (r[5]==2) & (r[6]==0))
psi <-
3*(2*a25*a33*(4*a34*a35 + a33*a45) + a24*a33*(4*a35^2 + a33*a55) + 
  4*a23*(2*a34*a35^2 + 2*a33*a35*a45 + a33*a34*a55))
 if ((r[1]==0) & (r[2]==1) & (r[3]==4) & (r[4]==1) & (r[5]==1) & (r[6]==1))
psi <-
3*(4*a24*a33*a35*a36 + 8*a23*a34*a35*a36 + 4*a23*a33*a36*a45 + 
  a26*a33*(4*a34*a35 + a33*a45) + 4*a23*a33*a35*a46 + 
  a25*a33*(4*a34*a36 + a33*a46) + a24*a33^2*a56 + 4*a23*a33*a34*a56)
 if ((r[1]==0) & (r[2]==1) & (r[3]==4) & (r[4]==1) & (r[5]==0) & (r[6]==2))
psi <-
3*(2*a26*a33*(4*a34*a36 + a33*a46) + a24*a33*(4*a36^2 + a33*a66) + 
  4*a23*(2*a34*a36^2 + 2*a33*a36*a46 + a33*a34*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==4) & (r[4]==0) & (r[5]==3) & (r[6]==0))
psi <-
9*a25*a33*(4*a35^2 + a33*a55) + 12*a23*a35*(2*a35^2 + 3*a33*a55)
 if ((r[1]==0) & (r[2]==1) & (r[3]==4) & (r[4]==0) & (r[5]==2) & (r[6]==1))
psi <-
3*(a26*a33*(4*a35^2 + a33*a55) + 2*a25*a33*(4*a35*a36 + a33*a56) + 
  4*a23*(2*a35^2*a36 + a33*a36*a55 + 2*a33*a35*a56))
 if ((r[1]==0) & (r[2]==1) & (r[3]==4) & (r[4]==0) & (r[5]==1) & (r[6]==2))
psi <-
3*(2*a26*a33*(4*a35*a36 + a33*a56) + a25*a33*(4*a36^2 + a33*a66) + 
  4*a23*(2*a35*a36^2 + 2*a33*a36*a56 + a33*a35*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==4) & (r[4]==0) & (r[5]==0) & (r[6]==3))
psi <-
9*a26*a33*(4*a36^2 + a33*a66) + 12*a23*a36*(2*a36^2 + 3*a33*a66)
 if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==4) & (r[5]==0) & (r[6]==0))
psi <-
9*a23*a44*(4*a34^2 + a33*a44) + 12*a24*(2*a34^3 + 3*a33*a34*a44)
 if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==3) & (r[5]==1) & (r[6]==0))
psi <-
3*(2*a25*a34^3 + 6*a24*a34^2*a35 + 3*a25*a33*a34*a44 + 3*a24*a33*a35*a44 + 
  6*a23*a34*a35*a44 + 6*a24*a33*a34*a45 + 6*a23*a34^2*a45 + 3*a23*a33*a44*a45)
 if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==3) & (r[5]==0) & (r[6]==1))
psi <-
3*(2*a26*a34^3 + 6*a24*a34^2*a36 + 3*a26*a33*a34*a44 + 3*a24*a33*a36*a44 + 
  6*a23*a34*a36*a44 + 6*a24*a33*a34*a46 + 6*a23*a34^2*a46 + 3*a23*a33*a44*a46)
 if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==2) & (r[5]==2) & (r[6]==0))
psi <-
3*(2*a25*(2*a34^2*a35 + a33*a35*a44 + 2*a33*a34*a45) + 
  2*a24*(2*a34*a35^2 + 2*a33*a35*a45 + a33*a34*a55) + 
  a23*(2*a35^2*a44 + 8*a34*a35*a45 + 2*a33*a45^2 + 2*a34^2*a55 + a33*a44*a55))
 if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==2) & (r[5]==1) & (r[6]==1))
psi <-
3*(4*a24*a34*a35*a36 + 2*a23*a35*a36*a44 + 2*a24*a33*a36*a45 + 
  4*a23*a34*a36*a45 + a26*(2*a34^2*a35 + a33*a35*a44 + 2*a33*a34*a45) + 
  2*a24*a33*a35*a46 + 4*a23*a34*a35*a46 + 2*a23*a33*a45*a46 + 
  a25*(2*a34^2*a36 + a33*a36*a44 + 2*a33*a34*a46) + 2*a24*a33*a34*a56 + 
  2*a23*a34^2*a56 + a23*a33*a44*a56)
 if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==2) & (r[5]==0) & (r[6]==2))
psi <-
3*(2*a26*(2*a34^2*a36 + a33*a36*a44 + 2*a33*a34*a46) + 
  2*a24*(2*a34*a36^2 + 2*a33*a36*a46 + a33*a34*a66) + 
  a23*(2*a36^2*a44 + 8*a34*a36*a46 + 2*a33*a46^2 + 2*a34^2*a66 + a33*a44*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==1) & (r[5]==3) & (r[6]==0))
psi <-
3*(2*a24*a35^3 + 6*a23*a35^2*a45 + 3*a24*a33*a35*a55 + 6*a23*a34*a35*a55 + 
  3*a23*a33*a45*a55 + 3*a25*(2*a34*a35^2 + 2*a33*a35*a45 + a33*a34*a55))
 if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==1) & (r[5]==2) & (r[6]==1))
psi <-
3*(2*a24*a35^2*a36 + 4*a23*a35*a36*a45 + 2*a23*a35^2*a46 + a24*a33*a36*a55 + 
  2*a23*a34*a36*a55 + a23*a33*a46*a55 + 
  a26*(2*a34*a35^2 + 2*a33*a35*a45 + a33*a34*a55) + 2*a24*a33*a35*a56 + 
  4*a23*a34*a35*a56 + 2*a23*a33*a45*a56 + 
  2*a25*(2*a34*a35*a36 + a33*a36*a45 + a33*a35*a46 + a33*a34*a56))
  if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==1) & (r[5]==1) & (r[6]==2))
psi <-
3*(2*a24*a35*a36^2 + 2*a23*a36^2*a45 + 4*a23*a35*a36*a46 + 
  2*a24*a33*a36*a56 + 4*a23*a34*a36*a56 + 2*a23*a33*a46*a56 + 
  2*a26*(2*a34*a35*a36 + a33*a36*a45 + a33*a35*a46 + a33*a34*a56) + 
  a24*a33*a35*a66 + 2*a23*a34*a35*a66 + a23*a33*a45*a66 + 
  a25*(2*a34*a36^2 + 2*a33*a36*a46 + a33*a34*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==1) & (r[5]==0) & (r[6]==3))
psi <-
3*(2*a24*a36^3 + 6*a23*a36^2*a46 + 3*a24*a33*a36*a66 + 6*a23*a34*a36*a66 + 
  3*a23*a33*a46*a66 + 3*a26*(2*a34*a36^2 + 2*a33*a36*a46 + a33*a34*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==0) & (r[5]==4) & (r[6]==0))
psi <-
9*a23*a55*(4*a35^2 + a33*a55) + 12*a25*(2*a35^3 + 3*a33*a35*a55)
 if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==0) & (r[5]==3) & (r[6]==1))
psi <-
3*(2*a26*a35^3 + 6*a25*a35^2*a36 + 3*a26*a33*a35*a55 + 3*a25*a33*a36*a55 + 
  6*a23*a35*a36*a55 + 6*a25*a33*a35*a56 + 6*a23*a35^2*a56 + 3*a23*a33*a55*a56)
 if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==0) & (r[5]==2) & (r[6]==2))
psi <-
3*(2*a26*(2*a35^2*a36 + a33*a36*a55 + 2*a33*a35*a56) + 
  2*a25*(2*a35*a36^2 + 2*a33*a36*a56 + a33*a35*a66) + 
  a23*(2*a36^2*a55 + 8*a35*a36*a56 + 2*a33*a56^2 + 2*a35^2*a66 + a33*a55*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==0) & (r[5]==1) & (r[6]==3))
psi <-
3*(2*a25*a36^3 + 6*a23*a36^2*a56 + 3*a25*a33*a36*a66 + 6*a23*a35*a36*a66 + 
  3*a23*a33*a56*a66 + 3*a26*(2*a35*a36^2 + 2*a33*a36*a56 + a33*a35*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==3) & (r[4]==0) & (r[5]==0) & (r[6]==4))
psi <-
9*a23*a66*(4*a36^2 + a33*a66) + 12*a26*(2*a36^3 + 3*a33*a36*a66)
 if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==5) & (r[5]==0) & (r[6]==0))
psi <-
15*a44*(4*a24*a34^2 + a24*a33*a44 + 2*a23*a34*a44)
 if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==4) & (r[5]==1) & (r[6]==0))
psi <-
3*(a25*a44*(4*a34^2 + a33*a44) + 2*a23*a44*(a35*a44 + 4*a34*a45) + 
  4*a24*(2*a34*a35*a44 + 2*a34^2*a45 + a33*a44*a45))
 if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==4) & (r[5]==0) & (r[6]==1))
psi <-
3*(a26*a44*(4*a34^2 + a33*a44) + 2*a23*a44*(a36*a44 + 4*a34*a46) + 
  4*a24*(2*a34*a36*a44 + 2*a34^2*a46 + a33*a44*a46))
 if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==3) & (r[5]==2) & (r[6]==0))
psi <-
3*(2*a25*(2*a34*a35*a44 + 2*a34^2*a45 + a33*a44*a45) + 
  a24*(2*a35^2*a44 + 8*a34*a35*a45 + 2*a33*a45^2 + 2*a34^2*a55 + 
    a33*a44*a55) + 2*a23*(2*a35*a44*a45 + 2*a34*a45^2 + a34*a44*a55))
 if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==3) & (r[5]==1) & (r[6]==1))
psi <-
3*(2*a24*a35*a36*a44 + 4*a24*a34*a36*a45 + 2*a23*a36*a44*a45 + 
  a26*(2*a34*a35*a44 + 2*a34^2*a45 + a33*a44*a45) + 4*a24*a34*a35*a46 + 
  2*a23*a35*a44*a46 + 2*a24*a33*a45*a46 + 4*a23*a34*a45*a46 + 
  a25*(2*a34*a36*a44 + 2*a34^2*a46 + a33*a44*a46) + 2*a24*a34^2*a56 + 
  a24*a33*a44*a56 + 2*a23*a34*a44*a56)
 if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==3) & (r[5]==0) & (r[6]==2))
psi <-
3*(2*a26*(2*a34*a36*a44 + 2*a34^2*a46 + a33*a44*a46) + 
  a24*(2*a36^2*a44 + 8*a34*a36*a46 + 2*a33*a46^2 + 2*a34^2*a66 + 
    a33*a44*a66) + 2*a23*(2*a36*a44*a46 + 2*a34*a46^2 + a34*a44*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==2) & (r[5]==3) & (r[6]==0))
psi <-
3*(a25*(2*a35^2*a44 + 8*a34*a35*a45 + 2*a33*a45^2 + 2*a34^2*a55 + 
    a33*a44*a55) + 2*(a24*(2*a35^2*a45 + 2*a34*a35*a55 + a33*a45*a55) + 
    a23*(2*a35*a45^2 + a35*a44*a55 + 2*a34*a45*a55)))
 if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==2) & (r[5]==2) & (r[6]==1))
psi <-
a26*(2*a35^2*a44 + 8*a34*a35*a45 + 2*a33*a45^2 + 2*a34^2*a55 + a33*a44*a55) + 
 2*(a25*(2*a35*a36*a44 + 4*a34*a36*a45 + 4*a34*a35*a46 + 2*a33*a45*a46 + 
     2*a34^2*a56 + a33*a44*a56) + a24*(4*a35*a36*a45 + 2*a35^2*a46 + 
     2*a34*a36*a55 + a33*a46*a55 + 4*a34*a35*a56 + 2*a33*a45*a56) + 
   a23*(2*a36*a45^2 + 4*a35*a45*a46 + a36*a44*a55 + 2*a34*a46*a55 + 
     2*a35*a44*a56 + 4*a34*a45*a56))
 if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==2) & (r[5]==1) & (r[6]==2))
psi <-
2*a26*(2*a35*a36*a44 + 4*a34*a36*a45 + 4*a34*a35*a46 + 2*a33*a45*a46 + 
   2*a34^2*a56 + a33*a44*a56) + a25*(2*a36^2*a44 + 8*a34*a36*a46 + 
   2*a33*a46^2 + 2*a34^2*a66 + a33*a44*a66) + 
 2*(a24*(2*a36^2*a45 + 4*a35*a36*a46 + 4*a34*a36*a56 + 2*a33*a46*a56 + 
     2*a34*a35*a66 + a33*a45*a66) + a23*(4*a36*a45*a46 + 2*a35*a46^2 + 
     2*a36*a44*a56 + 4*a34*a46*a56 + a35*a44*a66 + 2*a34*a45*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==2) & (r[5]==0) & (r[6]==3))
psi <-
3*(a26*(2*a36^2*a44 + 8*a34*a36*a46 + 2*a33*a46^2 + 2*a34^2*a66 + 
    a33*a44*a66) + 2*(a24*(2*a36^2*a46 + 2*a34*a36*a66 + a33*a46*a66) + 
    a23*(2*a36*a46^2 + a36*a44*a66 + 2*a34*a46*a66)))
 if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==1) & (r[5]==4) & (r[6]==0))
psi <-
3*(a55*(4*a24*a35^2 + 8*a23*a35*a45 + a24*a33*a55 + 2*a23*a34*a55) + 
  4*a25*(2*a35^2*a45 + 2*a34*a35*a55 + a33*a45*a55))
 if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==1) & (r[5]==3) & (r[6]==1))
psi <-
3*(2*a24*a35*a36*a55 + 2*a23*a36*a45*a55 + 2*a23*a35*a46*a55 + 
  a26*(2*a35^2*a45 + 2*a34*a35*a55 + a33*a45*a55) + 2*a24*a35^2*a56 + 
  4*a23*a35*a45*a56 + a24*a33*a55*a56 + 2*a23*a34*a55*a56 + 
  a25*(4*a35*a36*a45 + 2*a35^2*a46 + 2*a34*a36*a55 + a33*a46*a55 + 
    4*a34*a35*a56 + 2*a33*a45*a56))
 if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==1) & (r[5]==2) & (r[6]==2))
psi <-
2*a24*a36^2*a55 + 4*a23*a36*a46*a55 + 8*a24*a35*a36*a56 + 8*a23*a36*a45*a56 + 
 8*a23*a35*a46*a56 + 2*a24*a33*a56^2 + 4*a23*a34*a56^2 + 
 2*a26*(2*a35^2*a46 + 2*a34*a36*a55 + a33*a46*a55 + 2*a33*a45*a56 + 
   4*a35*(a36*a45 + a34*a56)) + 2*a24*a35^2*a66 + 4*a23*a35*a45*a66 + 
 a24*a33*a55*a66 + 2*a23*a34*a55*a66 + 
 2*a25*(2*a36^2*a45 + 4*a35*a36*a46 + 4*a34*a36*a56 + 2*a33*a46*a56 + 
   2*a34*a35*a66 + a33*a45*a66)
 if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==1) & (r[5]==1) & (r[6]==3))
psi <-
3*(2*a24*a36^2*a56 + 4*a23*a36*a46*a56 + 2*a24*a35*a36*a66 + 
  2*a23*a36*a45*a66 + 2*a23*a35*a46*a66 + a24*a33*a56*a66 + 
  2*a23*a34*a56*a66 + a26*(2*a36^2*a45 + 2*a33*a46*a56 + 
    4*a36*(a35*a46 + a34*a56) + 2*a34*a35*a66 + a33*a45*a66) + 
  a25*(2*a36^2*a46 + 2*a34*a36*a66 + a33*a46*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==1) & (r[5]==0) & (r[6]==4))
psi <-
3*(a66*(4*a24*a36^2 + 8*a23*a36*a46 + a24*a33*a66 + 2*a23*a34*a66) + 
  4*a26*(2*a36^2*a46 + 2*a34*a36*a66 + a33*a46*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==0) & (r[5]==5) & (r[6]==0))
psi <-
15*a55*(4*a25*a35^2 + a25*a33*a55 + 2*a23*a35*a55)
 if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==0) & (r[5]==4) & (r[6]==1))
psi <-
3*(a26*a55*(4*a35^2 + a33*a55) + 2*a23*a55*(a36*a55 + 4*a35*a56) + 
  4*a25*(2*a35*a36*a55 + 2*a35^2*a56 + a33*a55*a56))
 if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==0) & (r[5]==3) & (r[6]==2))
psi <-
3*(2*a26*(2*a35*a36*a55 + 2*a35^2*a56 + a33*a55*a56) + 
  a25*(2*a36^2*a55 + 8*a35*a36*a56 + 2*a33*a56^2 + 2*a35^2*a66 + 
    a33*a55*a66) + 2*a23*(2*a36*a55*a56 + 2*a35*a56^2 + a35*a55*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==0) & (r[5]==2) & (r[6]==3))
psi <-
3*(a26*(2*a36^2*a55 + 8*a35*a36*a56 + 2*a33*a56^2 + 2*a35^2*a66 + 
    a33*a55*a66) + 2*(a25*(2*a36^2*a56 + 2*a35*a36*a66 + a33*a56*a66) + 
    a23*(2*a36*a56^2 + a36*a55*a66 + 2*a35*a56*a66)))
 if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==0) & (r[5]==1) & (r[6]==4))
psi <-
3*(a66*(4*a25*a36^2 + 8*a23*a36*a56 + a25*a33*a66 + 2*a23*a35*a66) + 
  4*a26*(2*a36^2*a56 + 2*a35*a36*a66 + a33*a56*a66))
  if ((r[1]==0) & (r[2]==1) & (r[3]==2) & (r[4]==0) & (r[5]==0) & (r[6]==5))
psi <-
15*a66*(4*a26*a36^2 + a26*a33*a66 + 2*a23*a36*a66)
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==6) & (r[5]==0) & (r[6]==0))
psi <-
15*a44^2*(6*a24*a34 + a23*a44)
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==5) & (r[5]==1) & (r[6]==0))
psi <-
15*a44*(a25*a34*a44 + a24*a35*a44 + 4*a24*a34*a45 + a23*a44*a45)
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==5) & (r[5]==0) & (r[6]==1))
psi <-
15*a44*(a26*a34*a44 + a24*a36*a44 + 4*a24*a34*a46 + a23*a44*a46)
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==4) & (r[5]==2) & (r[6]==0))
psi <-
3*(2*a25*a44*(a35*a44 + 4*a34*a45) + a23*a44*(4*a45^2 + a44*a55) + 
  4*a24*(2*a35*a44*a45 + 2*a34*a45^2 + a34*a44*a55))
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==4) & (r[5]==1) & (r[6]==1))
psi <-
3*(4*a24*a36*a44*a45 + a26*a44*(a35*a44 + 4*a34*a45) + 4*a24*a35*a44*a46 + 
  8*a24*a34*a45*a46 + 4*a23*a44*a45*a46 + a25*a44*(a36*a44 + 4*a34*a46) + 
  4*a24*a34*a44*a56 + a23*a44^2*a56)
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==4) & (r[5]==0) & (r[6]==2))
psi <-
3*(2*a26*a44*(a36*a44 + 4*a34*a46) + a23*a44*(4*a46^2 + a44*a66) + 
  4*a24*(2*a36*a44*a46 + 2*a34*a46^2 + a34*a44*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==3) & (r[5]==3) & (r[6]==0))
psi <-
3*(6*a24*a35*a45^2 + 2*a23*a45^3 + 3*a24*a35*a44*a55 + 6*a24*a34*a45*a55 + 
  3*a23*a44*a45*a55 + 3*a25*(2*a35*a44*a45 + 2*a34*a45^2 + a34*a44*a55))
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==3) & (r[5]==2) & (r[6]==1))
psi <-
3*(2*a24*a36*a45^2 + 4*a24*a35*a45*a46 + 2*a23*a45^2*a46 + a24*a36*a44*a55 + 
  2*a24*a34*a46*a55 + a23*a44*a46*a55 + 
  a26*(2*a35*a44*a45 + 2*a34*a45^2 + a34*a44*a55) + 2*a24*a35*a44*a56 + 
  4*a24*a34*a45*a56 + 2*a23*a44*a45*a56 + 
  2*a25*(a36*a44*a45 + a35*a44*a46 + 2*a34*a45*a46 + a34*a44*a56))
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==3) & (r[5]==1) & (r[6]==2))
psi <-
3*(4*a24*a36*a45*a46 + 2*a24*a35*a46^2 + 2*a23*a45*a46^2 + 
  2*a24*a36*a44*a56 + 4*a24*a34*a46*a56 + 2*a23*a44*a46*a56 + 
  2*a26*(a36*a44*a45 + a35*a44*a46 + 2*a34*a45*a46 + a34*a44*a56) + 
  a24*a35*a44*a66 + 2*a24*a34*a45*a66 + a23*a44*a45*a66 + 
  a25*(2*a36*a44*a46 + 2*a34*a46^2 + a34*a44*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==3) & (r[5]==0) & (r[6]==3))
psi <-
3*(6*a24*a36*a46^2 + 2*a23*a46^3 + 3*a24*a36*a44*a66 + 6*a24*a34*a46*a66 + 
  3*a23*a44*a46*a66 + 3*a26*(2*a36*a44*a46 + 2*a34*a46^2 + a34*a44*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==2) & (r[5]==4) & (r[6]==0))
psi <-
3*(a55*(8*a24*a35*a45 + 4*a23*a45^2 + 2*a24*a34*a55 + a23*a44*a55) + 
  4*a25*(2*a35*a45^2 + a35*a44*a55 + 2*a34*a45*a55))
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==2) & (r[5]==3) & (r[6]==1))
psi <-
3*(2*a24*a36*a45*a55 + 2*a24*a35*a46*a55 + 2*a23*a45*a46*a55 + 
  a26*(2*a35*a45^2 + a35*a44*a55 + 2*a34*a45*a55) + 4*a24*a35*a45*a56 + 
  2*a23*a45^2*a56 + 2*a24*a34*a55*a56 + a23*a44*a55*a56 + 
  a25*(2*a36*a45^2 + 4*a35*a45*a46 + a36*a44*a55 + 2*a34*a46*a55 + 
    2*a35*a44*a56 + 4*a34*a45*a56))
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==2) & (r[5]==2) & (r[6]==2))
psi <-
4*a24*a36*a46*a55 + 2*a23*a46^2*a55 + 8*a24*a36*a45*a56 + 8*a24*a35*a46*a56 + 
 8*a23*a45*a46*a56 + 4*a24*a34*a56^2 + 2*a23*a44*a56^2 + 
 2*a26*(2*a36*a45^2 + 4*a35*a45*a46 + a36*a44*a55 + 2*a34*a46*a55 + 
   2*a35*a44*a56 + 4*a34*a45*a56) + 4*a24*a35*a45*a66 + 2*a23*a45^2*a66 + 
 2*a24*a34*a55*a66 + a23*a44*a55*a66 + 
 2*a25*(4*a36*a45*a46 + 2*a35*a46^2 + 2*a36*a44*a56 + 4*a34*a46*a56 + 
   a35*a44*a66 + 2*a34*a45*a66)
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==2) & (r[5]==1) & (r[6]==3))
psi <-
3*(4*a24*a36*a46*a56 + 2*a23*a46^2*a56 + 2*a24*a36*a45*a66 + 
  2*a24*a35*a46*a66 + 2*a23*a45*a46*a66 + 2*a24*a34*a56*a66 + 
  a23*a44*a56*a66 + a26*(4*a36*a45*a46 + 2*a35*a46^2 + 2*a36*a44*a56 + 
    4*a34*a46*a56 + a35*a44*a66 + 2*a34*a45*a66) + 
  a25*(2*a36*a46^2 + a36*a44*a66 + 2*a34*a46*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==2) & (r[5]==0) & (r[6]==4))
psi <-
3*(a66*(8*a24*a36*a46 + 4*a23*a46^2 + 2*a24*a34*a66 + a23*a44*a66) + 
  4*a26*(2*a36*a46^2 + a36*a44*a66 + 2*a34*a46*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==1) & (r[5]==5) & (r[6]==0))
psi <-
15*a55*(4*a25*a35*a45 + a25*a34*a55 + a24*a35*a55 + a23*a45*a55)
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==1) & (r[5]==4) & (r[6]==1))
psi <-
3*(a26*a55*(4*a35*a45 + a34*a55) + a55*(a24*a36*a55 + a23*a46*a55 + 
    4*a24*a35*a56 + 4*a23*a45*a56) + 
  4*a25*(a36*a45*a55 + a35*a46*a55 + 2*a35*a45*a56 + a34*a55*a56))
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==1) & (r[5]==3) & (r[6]==2))
psi <-
3*(2*a24*a36*a55*a56 + 2*a23*a46*a55*a56 + 2*a24*a35*a56^2 + 
  2*a23*a45*a56^2 + 2*a26*(a36*a45*a55 + a35*a46*a55 + 2*a35*a45*a56 + 
    a34*a55*a56) + a24*a35*a55*a66 + a23*a45*a55*a66 + 
  a25*(2*a36*a46*a55 + 4*a36*a45*a56 + 4*a35*a46*a56 + 2*a34*a56^2 + 
    2*a35*a45*a66 + a34*a55*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==1) & (r[5]==2) & (r[6]==3))
psi <-
3*(2*a24*a36*a56^2 + 2*a23*a46*a56^2 + a24*a36*a55*a66 + a23*a46*a55*a66 + 
  2*a24*a35*a56*a66 + 2*a23*a45*a56*a66 + 
  a26*(2*a36*a46*a55 + 4*a36*a45*a56 + 4*a35*a46*a56 + 2*a34*a56^2 + 
    2*a35*a45*a66 + a34*a55*a66) + 2*a25*(2*a36*a46*a56 + a36*a45*a66 + 
    a35*a46*a66 + a34*a56*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==1) & (r[5]==1) & (r[6]==4))
psi <-
3*(a66*(4*a25*a36*a46 + 4*a24*a36*a56 + 4*a23*a46*a56 + a25*a34*a66 + 
    a24*a35*a66 + a23*a45*a66) + 4*a26*(2*a36*a46*a56 + a36*a45*a66 + 
    a35*a46*a66 + a34*a56*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==1) & (r[5]==0) & (r[6]==5))
psi <-
15*a66*(4*a26*a36*a46 + a26*a34*a66 + a24*a36*a66 + a23*a46*a66)
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==0) & (r[5]==6) & (r[6]==0))
psi <-
15*a55^2*(6*a25*a35 + a23*a55)
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==0) & (r[5]==5) & (r[6]==1))
psi <-
15*a55*(a26*a35*a55 + a25*a36*a55 + 4*a25*a35*a56 + a23*a55*a56)
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==0) & (r[5]==4) & (r[6]==2))
psi <-
3*(2*a26*a55*(a36*a55 + 4*a35*a56) + a23*a55*(4*a56^2 + a55*a66) + 
  4*a25*(2*a36*a55*a56 + 2*a35*a56^2 + a35*a55*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==0) & (r[5]==3) & (r[6]==3))
psi <-
3*(6*a25*a36*a56^2 + 2*a23*a56^3 + 3*a25*a36*a55*a66 + 6*a25*a35*a56*a66 + 
  3*a23*a55*a56*a66 + 3*a26*(2*a36*a55*a56 + 2*a35*a56^2 + a35*a55*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==0) & (r[5]==2) & (r[6]==4))
psi <-
3*(a66*(8*a25*a36*a56 + 4*a23*a56^2 + 2*a25*a35*a66 + a23*a55*a66) + 
  4*a26*(2*a36*a56^2 + a36*a55*a66 + 2*a35*a56*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==0) & (r[5]==1) & (r[6]==5))
psi <-
15*a66*(4*a26*a36*a56 + a26*a35*a66 + a25*a36*a66 + a23*a56*a66)
 if ((r[1]==0) & (r[2]==1) & (r[3]==1) & (r[4]==0) & (r[5]==0) & (r[6]==6))
psi <-
15*a66^2*(6*a26*a36 + a23*a66)
  if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==7) & (r[5]==0) & (r[6]==0))
psi <-
105*a24*a44^3
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==6) & (r[5]==1) & (r[6]==0))
psi <-
15*a44^2*(a25*a44 + 6*a24*a45)
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==6) & (r[5]==0) & (r[6]==1))
psi <-
15*a44^2*(a26*a44 + 6*a24*a46)
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==5) & (r[5]==2) & (r[6]==0))
psi <-
15*a44*(2*a25*a44*a45 + 4*a24*a45^2 + a24*a44*a55)
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==5) & (r[5]==1) & (r[6]==1))
psi <-
15*a44*(a26*a44*a45 + a25*a44*a46 + 4*a24*a45*a46 + a24*a44*a56)
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==5) & (r[5]==0) & (r[6]==2))
psi <-
15*a44*(2*a26*a44*a46 + 4*a24*a46^2 + a24*a44*a66)
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==4) & (r[5]==3) & (r[6]==0))
psi <-
9*a25*a44*(4*a45^2 + a44*a55) + 12*a24*a45*(2*a45^2 + 3*a44*a55)
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==4) & (r[5]==2) & (r[6]==1))
psi <-
3*(a26*a44*(4*a45^2 + a44*a55) + 2*a25*a44*(4*a45*a46 + a44*a56) + 
  4*a24*(2*a45^2*a46 + a44*a46*a55 + 2*a44*a45*a56))
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==4) & (r[5]==1) & (r[6]==2))
psi <-
3*(2*a26*a44*(4*a45*a46 + a44*a56) + a25*a44*(4*a46^2 + a44*a66) + 
  4*a24*(2*a45*a46^2 + 2*a44*a46*a56 + a44*a45*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==4) & (r[5]==0) & (r[6]==3))
psi <-
9*a26*a44*(4*a46^2 + a44*a66) + 12*a24*a46*(2*a46^2 + 3*a44*a66)
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==3) & (r[5]==4) & (r[6]==0))
psi <-
9*a24*a55*(4*a45^2 + a44*a55) + 12*a25*(2*a45^3 + 3*a44*a45*a55)
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==3) & (r[5]==3) & (r[6]==1))
psi <-
3*(2*a26*a45^3 + 6*a25*a45^2*a46 + 3*a26*a44*a45*a55 + 3*a25*a44*a46*a55 + 
  6*a24*a45*a46*a55 + 6*a25*a44*a45*a56 + 6*a24*a45^2*a56 + 3*a24*a44*a55*a56)
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==3) & (r[5]==2) & (r[6]==2))
psi <-
3*(2*a26*(2*a45^2*a46 + a44*a46*a55 + 2*a44*a45*a56) + 
  2*a25*(2*a45*a46^2 + 2*a44*a46*a56 + a44*a45*a66) + 
  a24*(2*a46^2*a55 + 8*a45*a46*a56 + 2*a44*a56^2 + 2*a45^2*a66 + a44*a55*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==3) & (r[5]==1) & (r[6]==3))
psi <-
3*(2*a25*a46^3 + 6*a24*a46^2*a56 + 3*a25*a44*a46*a66 + 6*a24*a45*a46*a66 + 
  3*a24*a44*a56*a66 + 3*a26*(2*a45*a46^2 + 2*a44*a46*a56 + a44*a45*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==3) & (r[5]==0) & (r[6]==4))
psi <-
9*a24*a66*(4*a46^2 + a44*a66) + 12*a26*(2*a46^3 + 3*a44*a46*a66)
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==2) & (r[5]==5) & (r[6]==0))
psi <-
15*a55*(4*a25*a45^2 + a25*a44*a55 + 2*a24*a45*a55)
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==2) & (r[5]==4) & (r[6]==1))
psi <-
3*(a26*a55*(4*a45^2 + a44*a55) + 2*a24*a55*(a46*a55 + 4*a45*a56) + 
  4*a25*(2*a45*a46*a55 + 2*a45^2*a56 + a44*a55*a56))
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==2) & (r[5]==3) & (r[6]==2))
psi <-
3*(2*a26*(2*a45*a46*a55 + 2*a45^2*a56 + a44*a55*a56) + 
  a25*(2*a46^2*a55 + 8*a45*a46*a56 + 2*a44*a56^2 + 2*a45^2*a66 + 
    a44*a55*a66) + 2*a24*(2*a46*a55*a56 + 2*a45*a56^2 + a45*a55*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==2) & (r[5]==2) & (r[6]==3))
psi <-
3*(a26*(2*a46^2*a55 + 8*a45*a46*a56 + 2*a44*a56^2 + 2*a45^2*a66 + 
    a44*a55*a66) + 2*(a25*(2*a46^2*a56 + 2*a45*a46*a66 + a44*a56*a66) + 
    a24*(2*a46*a56^2 + a46*a55*a66 + 2*a45*a56*a66)))
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==2) & (r[5]==1) & (r[6]==4))
psi <-
3*(a66*(4*a25*a46^2 + 8*a24*a46*a56 + a25*a44*a66 + 2*a24*a45*a66) + 
  4*a26*(2*a46^2*a56 + 2*a45*a46*a66 + a44*a56*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==2) & (r[5]==0) & (r[6]==5))
psi <-
15*a66*(4*a26*a46^2 + a26*a44*a66 + 2*a24*a46*a66)
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==1) & (r[5]==6) & (r[6]==0))
psi <-
15*a55^2*(6*a25*a45 + a24*a55)
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==1) & (r[5]==5) & (r[6]==1))
psi <-
15*a55*(a26*a45*a55 + a25*a46*a55 + 4*a25*a45*a56 + a24*a55*a56)
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==1) & (r[5]==4) & (r[6]==2))
psi <-
3*(2*a26*a55*(a46*a55 + 4*a45*a56) + a24*a55*(4*a56^2 + a55*a66) + 
  4*a25*(2*a46*a55*a56 + 2*a45*a56^2 + a45*a55*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==1) & (r[5]==3) & (r[6]==3))
psi <-
3*(6*a25*a46*a56^2 + 2*a24*a56^3 + 3*a25*a46*a55*a66 + 6*a25*a45*a56*a66 + 
  3*a24*a55*a56*a66 + 3*a26*(2*a46*a55*a56 + 2*a45*a56^2 + a45*a55*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==1) & (r[5]==2) & (r[6]==4))
psi <-
3*(a66*(8*a25*a46*a56 + 4*a24*a56^2 + 2*a25*a45*a66 + a24*a55*a66) + 
  4*a26*(2*a46*a56^2 + a46*a55*a66 + 2*a45*a56*a66))
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==1) & (r[5]==1) & (r[6]==5))
psi <-
15*a66*(4*a26*a46*a56 + a26*a45*a66 + a25*a46*a66 + a24*a56*a66)
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==1) & (r[5]==0) & (r[6]==6))
psi <-
15*a66^2*(6*a26*a46 + a24*a66)
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==0) & (r[5]==7) & (r[6]==0))
psi <-
105*a25*a55^3
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==0) & (r[5]==6) & (r[6]==1))
psi <-
15*a55^2*(a26*a55 + 6*a25*a56)
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==0) & (r[5]==5) & (r[6]==2))
psi <-
15*a55*(2*a26*a55*a56 + 4*a25*a56^2 + a25*a55*a66)
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==0) & (r[5]==4) & (r[6]==3))
psi <-
9*a26*a55*(4*a56^2 + a55*a66) + 12*a25*a56*(2*a56^2 + 3*a55*a66)
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==0) & (r[5]==3) & (r[6]==4))
psi <-
9*a25*a66*(4*a56^2 + a55*a66) + 12*a26*(2*a56^3 + 3*a55*a56*a66)
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==0) & (r[5]==2) & (r[6]==5))
psi <-
15*a66*(4*a26*a56^2 + a26*a55*a66 + 2*a25*a56*a66)
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==0) & (r[5]==1) & (r[6]==6))
psi <-
15*a66^2*(6*a26*a56 + a25*a66)
 if ((r[1]==0) & (r[2]==1) & (r[3]==0) & (r[4]==0) & (r[5]==0) & (r[6]==7))
psi <-
105*a26*a66^3
 if ((r[1]==0) & (r[2]==0) & (r[3]==8) & (r[4]==0) & (r[5]==0) & (r[6]==0))
psi <-
105*a33^4
 if ((r[1]==0) & (r[2]==0) & (r[3]==7) & (r[4]==1) & (r[5]==0) & (r[6]==0))
psi <-
105*a33^3*a34
  if ((r[1]==0) & (r[2]==0) & (r[3]==7) & (r[4]==0) & (r[5]==1) & (r[6]==0))
psi <-
105*a33^3*a35
 if ((r[1]==0) & (r[2]==0) & (r[3]==7) & (r[4]==0) & (r[5]==0) & (r[6]==1))
psi <-
105*a33^3*a36
 if ((r[1]==0) & (r[2]==0) & (r[3]==6) & (r[4]==2) & (r[5]==0) & (r[6]==0))
psi <-
15*a33^2*(6*a34^2 + a33*a44)
 if ((r[1]==0) & (r[2]==0) & (r[3]==6) & (r[4]==1) & (r[5]==1) & (r[6]==0))
psi <-
15*a33^2*(6*a34*a35 + a33*a45)
 if ((r[1]==0) & (r[2]==0) & (r[3]==6) & (r[4]==1) & (r[5]==0) & (r[6]==1))
psi <-
15*a33^2*(6*a34*a36 + a33*a46)
 if ((r[1]==0) & (r[2]==0) & (r[3]==6) & (r[4]==0) & (r[5]==2) & (r[6]==0))
psi <-
15*a33^2*(6*a35^2 + a33*a55)
 if ((r[1]==0) & (r[2]==0) & (r[3]==6) & (r[4]==0) & (r[5]==1) & (r[6]==1))
psi <-
15*a33^2*(6*a35*a36 + a33*a56)
 if ((r[1]==0) & (r[2]==0) & (r[3]==6) & (r[4]==0) & (r[5]==0) & (r[6]==2))
psi <-
15*a33^2*(6*a36^2 + a33*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==5) & (r[4]==3) & (r[5]==0) & (r[6]==0))
psi <-
15*a33*a34*(4*a34^2 + 3*a33*a44)
 if ((r[1]==0) & (r[2]==0) & (r[3]==5) & (r[4]==2) & (r[5]==1) & (r[6]==0))
psi <-
15*a33*(4*a34^2*a35 + a33*a35*a44 + 2*a33*a34*a45)
 if ((r[1]==0) & (r[2]==0) & (r[3]==5) & (r[4]==2) & (r[5]==0) & (r[6]==1))
psi <-
15*a33*(4*a34^2*a36 + a33*a36*a44 + 2*a33*a34*a46)
 if ((r[1]==0) & (r[2]==0) & (r[3]==5) & (r[4]==1) & (r[5]==2) & (r[6]==0))
psi <-
15*a33*(4*a34*a35^2 + 2*a33*a35*a45 + a33*a34*a55)
 if ((r[1]==0) & (r[2]==0) & (r[3]==5) & (r[4]==1) & (r[5]==1) & (r[6]==1))
psi <-
15*a33*(4*a34*a35*a36 + a33*a36*a45 + a33*a35*a46 + a33*a34*a56)
 if ((r[1]==0) & (r[2]==0) & (r[3]==5) & (r[4]==1) & (r[5]==0) & (r[6]==2))
psi <-
15*a33*(4*a34*a36^2 + 2*a33*a36*a46 + a33*a34*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==5) & (r[4]==0) & (r[5]==3) & (r[6]==0))
psi <-
15*a33*a35*(4*a35^2 + 3*a33*a55)
 if ((r[1]==0) & (r[2]==0) & (r[3]==5) & (r[4]==0) & (r[5]==2) & (r[6]==1))
psi <-
15*a33*(4*a35^2*a36 + a33*a36*a55 + 2*a33*a35*a56)
 if ((r[1]==0) & (r[2]==0) & (r[3]==5) & (r[4]==0) & (r[5]==1) & (r[6]==2))
psi <-
15*a33*(4*a35*a36^2 + 2*a33*a36*a56 + a33*a35*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==5) & (r[4]==0) & (r[5]==0) & (r[6]==3))
psi <-
15*a33*a36*(4*a36^2 + 3*a33*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==4) & (r[5]==0) & (r[6]==0))
psi <-
24*a34^4 + 72*a33*a34^2*a44 + 9*a33^2*a44^2
 if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==3) & (r[5]==1) & (r[6]==0))
psi <-
24*a34^3*a35 + 36*a33*a34*a35*a44 + 36*a33*a34^2*a45 + 9*a33^2*a44*a45
 if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==3) & (r[5]==0) & (r[6]==1))
psi <-
24*a34^3*a36 + 36*a33*a34*a36*a44 + 36*a33*a34^2*a46 + 9*a33^2*a44*a46
 if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==2) & (r[5]==2) & (r[6]==0))
psi <-
3*(16*a33*a34*a35*a45 + 4*a34^2*(2*a35^2 + a33*a55) + 
  a33*(4*a35^2*a44 + 2*a33*a45^2 + a33*a44*a55))
 if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==2) & (r[5]==1) & (r[6]==1))
psi <-
3*(8*a33*a34*(a36*a45 + a35*a46) + 4*a34^2*(2*a35*a36 + a33*a56) + 
  a33*(4*a35*a36*a44 + 2*a33*a45*a46 + a33*a44*a56))
 if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==2) & (r[5]==0) & (r[6]==2))
psi <-
3*(16*a33*a34*a36*a46 + 4*a34^2*(2*a36^2 + a33*a66) + 
  a33*(4*a36^2*a44 + 2*a33*a46^2 + a33*a44*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==1) & (r[5]==3) & (r[6]==0))
psi <-
9*a33*a45*(4*a35^2 + a33*a55) + 12*a34*(2*a35^3 + 3*a33*a35*a55)
 if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==1) & (r[5]==2) & (r[6]==1))
psi <-
3*(4*a34*(2*a35^2*a36 + a33*a36*a55 + 2*a33*a35*a56) + 
  a33*(8*a35*a36*a45 + 4*a35^2*a46 + a33*a46*a55 + 2*a33*a45*a56))
 if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==1) & (r[5]==1) & (r[6]==2))
psi <-
3*(4*a34*(2*a35*a36^2 + 2*a33*a36*a56 + a33*a35*a66) + 
  a33*(4*a36^2*a45 + 8*a35*a36*a46 + 2*a33*a46*a56 + a33*a45*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==1) & (r[5]==0) & (r[6]==3))
psi <-
9*a33*a46*(4*a36^2 + a33*a66) + 12*a34*(2*a36^3 + 3*a33*a36*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==0) & (r[5]==4) & (r[6]==0))
psi <-
24*a35^4 + 72*a33*a35^2*a55 + 9*a33^2*a55^2
 if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==0) & (r[5]==3) & (r[6]==1))
psi <-
24*a35^3*a36 + 36*a33*a35*a36*a55 + 36*a33*a35^2*a56 + 9*a33^2*a55*a56
 if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==0) & (r[5]==2) & (r[6]==2))
psi <-
3*(16*a33*a35*a36*a56 + 4*a35^2*(2*a36^2 + a33*a66) + 
  a33*(4*a36^2*a55 + 2*a33*a56^2 + a33*a55*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==0) & (r[5]==1) & (r[6]==3))
psi <-
9*a33*a56*(4*a36^2 + a33*a66) + 12*a35*(2*a36^3 + 3*a33*a36*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==4) & (r[4]==0) & (r[5]==0) & (r[6]==4))
psi <-
24*a36^4 + 72*a33*a36^2*a66 + 9*a33^2*a66^2
 if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==5) & (r[5]==0) & (r[6]==0))
psi <-
15*a34*a44*(4*a34^2 + 3*a33*a44)
 if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==4) & (r[5]==1) & (r[6]==0))
psi <-
36*a34^2*a35*a44 + 9*a33*a35*a44^2 + 24*a34^3*a45 + 36*a33*a34*a44*a45
 if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==4) & (r[5]==0) & (r[6]==1))
psi <-
36*a34^2*a36*a44 + 9*a33*a36*a44^2 + 24*a34^3*a46 + 36*a33*a34*a44*a46
 if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==3) & (r[5]==2) & (r[6]==0))
psi <-
3*(12*a34^2*a35*a45 + 6*a33*a35*a44*a45 + 2*a34^3*a55 + 
  3*a34*(2*a35^2*a44 + 2*a33*a45^2 + a33*a44*a55))
 if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==3) & (r[5]==1) & (r[6]==1))
psi <-
3*(6*a34^2*(a36*a45 + a35*a46) + 3*a33*a44*(a36*a45 + a35*a46) + 
  2*a34^3*a56 + 3*a34*(2*a35*a36*a44 + 2*a33*a45*a46 + a33*a44*a56))
 if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==3) & (r[5]==0) & (r[6]==2))
psi <-
3*(12*a34^2*a36*a46 + 6*a33*a36*a44*a46 + 2*a34^3*a66 + 
  3*a34*(2*a36^2*a44 + 2*a33*a46^2 + a33*a44*a66))
  if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==2) & (r[5]==3) & (r[6]==0))
psi <-
3*(2*a35^3*a44 + 12*a34*a35^2*a45 + 6*a33*a34*a45*a55 + 
  3*a35*(2*a33*a45^2 + 2*a34^2*a55 + a33*a44*a55))
 if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==2) & (r[5]==2) & (r[6]==1))
psi <-
3*(2*a35^2*(a36*a44 + 2*a34*a46) + 2*a34^2*a36*a55 + 
  2*a35*(4*a34*a36*a45 + 2*a33*a45*a46 + 2*a34^2*a56 + a33*a44*a56) + 
  a33*(2*a36*a45^2 + a36*a44*a55 + 2*a34*a46*a55 + 4*a34*a45*a56))
 if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==2) & (r[5]==1) & (r[6]==2))
psi <-
3*(a35*(2*a36^2*a44 + 8*a34*a36*a46 + 2*a33*a46^2 + 2*a34^2*a66 + 
    a33*a44*a66) + 2*(2*a34^2*a36*a56 + a33*a36*(2*a45*a46 + a44*a56) + 
    a34*(2*a36^2*a45 + 2*a33*a46*a56 + a33*a45*a66)))
 if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==2) & (r[5]==0) & (r[6]==3))
psi <-
3*(2*a36^3*a44 + 12*a34*a36^2*a46 + 6*a33*a34*a46*a66 + 
  3*a36*(2*a33*a46^2 + 2*a34^2*a66 + a33*a44*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==1) & (r[5]==4) & (r[6]==0))
psi <-
24*a35^3*a45 + 36*a34*a35^2*a55 + 36*a33*a35*a45*a55 + 9*a33*a34*a55^2
 if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==1) & (r[5]==3) & (r[6]==1))
psi <-
3*(2*a35^3*a46 + 6*a35^2*(a36*a45 + a34*a56) + 
  3*a33*a55*(a36*a45 + a34*a56) + 3*a35*(2*a34*a36*a55 + a33*a46*a55 + 
    2*a33*a45*a56))
 if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==1) & (r[5]==2) & (r[6]==2))
psi <-
3*(2*a33*a36*(a46*a55 + 2*a45*a56) + 2*a35^2*(2*a36*a46 + a34*a66) + 
  2*a35*(2*a36^2*a45 + 4*a34*a36*a56 + 2*a33*a46*a56 + a33*a45*a66) + 
  a34*(2*a36^2*a55 + 2*a33*a56^2 + a33*a55*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==1) & (r[5]==1) & (r[6]==3))
psi <-
3*(2*a36^3*a45 + 6*a36^2*(a35*a46 + a34*a56) + 3*a33*(a35*a46 + a34*a56)*
   a66 + 3*a36*(2*a33*a46*a56 + 2*a34*a35*a66 + a33*a45*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==1) & (r[5]==0) & (r[6]==4))
psi <-
24*a36^3*a46 + 36*a34*a36^2*a66 + 36*a33*a36*a46*a66 + 9*a33*a34*a66^2
 if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==0) & (r[5]==5) & (r[6]==0))
psi <-
15*a35*a55*(4*a35^2 + 3*a33*a55)
 if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==0) & (r[5]==4) & (r[6]==1))
psi <-
36*a35^2*a36*a55 + 9*a33*a36*a55^2 + 24*a35^3*a56 + 36*a33*a35*a55*a56
 if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==0) & (r[5]==3) & (r[6]==2))
psi <-
3*(12*a35^2*a36*a56 + 6*a33*a36*a55*a56 + 2*a35^3*a66 + 
  3*a35*(2*a36^2*a55 + 2*a33*a56^2 + a33*a55*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==0) & (r[5]==2) & (r[6]==3))
psi <-
3*(2*a36^3*a55 + 12*a35*a36^2*a56 + 6*a33*a35*a56*a66 + 
  3*a36*(2*a33*a56^2 + 2*a35^2*a66 + a33*a55*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==0) & (r[5]==1) & (r[6]==4))
psi <-
24*a36^3*a56 + 36*a35*a36^2*a66 + 36*a33*a36*a56*a66 + 9*a33*a35*a66^2
 if ((r[1]==0) & (r[2]==0) & (r[3]==3) & (r[4]==0) & (r[5]==0) & (r[6]==5))
psi <-
15*a36*a66*(4*a36^2 + 3*a33*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==6) & (r[5]==0) & (r[6]==0))
psi <-
15*a44^2*(6*a34^2 + a33*a44)
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==5) & (r[5]==1) & (r[6]==0))
psi <-
15*a44*(2*a34*a35*a44 + 4*a34^2*a45 + a33*a44*a45)
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==5) & (r[5]==0) & (r[6]==1))
psi <-
15*a44*(2*a34*a36*a44 + 4*a34^2*a46 + a33*a44*a46)
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==4) & (r[5]==2) & (r[6]==0))
psi <-
3*(2*a35^2*a44^2 + 16*a34*a35*a44*a45 + 4*a34^2*(2*a45^2 + a44*a55) + 
  a33*a44*(4*a45^2 + a44*a55))
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==4) & (r[5]==1) & (r[6]==1))
psi <-
3*(8*a34*a36*a44*a45 + 2*a35*a44*(a36*a44 + 4*a34*a46) + 
  4*a34^2*(2*a45*a46 + a44*a56) + a33*a44*(4*a45*a46 + a44*a56))
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==4) & (r[5]==0) & (r[6]==2))
psi <-
3*(2*a36^2*a44^2 + 16*a34*a36*a44*a46 + 4*a34^2*(2*a46^2 + a44*a66) + 
  a33*a44*(4*a46^2 + a44*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==3) & (r[5]==3) & (r[6]==0))
psi <-
3*(6*a35^2*a44*a45 + 6*a34*a35*(2*a45^2 + a44*a55) + 
  a45*(2*a33*a45^2 + 6*a34^2*a55 + 3*a33*a44*a55))
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==3) & (r[5]==2) & (r[6]==1))
psi <-
3*(2*a35^2*a44*a46 + 2*a34*a36*(2*a45^2 + a44*a55) + 
  4*a35*(a36*a44*a45 + 2*a34*a45*a46 + a34*a44*a56) + 
  2*a34^2*(a46*a55 + 2*a45*a56) + a33*(2*a45^2*a46 + a44*a46*a55 + 
    2*a44*a45*a56))
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==3) & (r[5]==1) & (r[6]==2))
psi <-
3*(2*a36^2*a44*a45 + 4*a36*(a35*a44*a46 + 2*a34*a45*a46 + a34*a44*a56) + 
  2*a34*a35*(2*a46^2 + a44*a66) + 2*a34^2*(2*a46*a56 + a45*a66) + 
  a33*(2*a45*a46^2 + 2*a44*a46*a56 + a44*a45*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==3) & (r[5]==0) & (r[6]==3))
psi <-
3*(6*a36^2*a44*a46 + 6*a34*a36*(2*a46^2 + a44*a66) + 
  a46*(2*a33*a46^2 + 6*a34^2*a66 + 3*a33*a44*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==2) & (r[5]==4) & (r[6]==0))
psi <-
3*(16*a34*a35*a45*a55 + 4*a35^2*(2*a45^2 + a44*a55) + 
  a55*(4*a33*a45^2 + 2*a34^2*a55 + a33*a44*a55))
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==2) & (r[5]==3) & (r[6]==1))
psi <-
3*(4*a34*a36*a45*a55 + 2*a34^2*a55*a56 + 2*a35^2*(2*a45*a46 + a44*a56) + 
  2*a35*(2*a36*a45^2 + a36*a44*a55 + 2*a34*a46*a55 + 4*a34*a45*a56) + 
  a33*(2*a45*a46*a55 + 2*a45^2*a56 + a44*a55*a56))
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==2) & (r[5]==2) & (r[6]==2))
psi <-
2*a33*a46^2*a55 + 2*a36^2*(2*a45^2 + a44*a55) + 8*a33*a45*a46*a56 + 
 4*a34^2*a56^2 + 2*a33*a44*a56^2 + 
 8*a36*(2*a35*a45*a46 + a34*a46*a55 + a35*a44*a56 + 2*a34*a45*a56) + 
 2*a33*a45^2*a66 + 2*a34^2*a55*a66 + a33*a44*a55*a66 + 
 2*a35^2*(2*a46^2 + a44*a66) + 8*a34*a35*(2*a46*a56 + a45*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==2) & (r[5]==1) & (r[6]==3))
psi <-
3*(2*a36^2*(2*a45*a46 + a44*a56) + 2*a34*(2*a35*a46 + a34*a56)*a66 + 
  2*a36*(2*a35*a46^2 + 4*a34*a46*a56 + a35*a44*a66 + 2*a34*a45*a66) + 
  a33*(2*a46^2*a56 + 2*a45*a46*a66 + a44*a56*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==2) & (r[5]==0) & (r[6]==4))
psi <-
3*(16*a34*a36*a46*a66 + 4*a36^2*(2*a46^2 + a44*a66) + 
  a66*(4*a33*a46^2 + 2*a34^2*a66 + a33*a44*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==1) & (r[5]==5) & (r[6]==0))
psi <-
15*a55*(4*a35^2*a45 + 2*a34*a35*a55 + a33*a45*a55)
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==1) & (r[5]==4) & (r[6]==1))
psi <-
3*(8*a35*a55*(a36*a45 + a34*a56) + 4*a35^2*(a46*a55 + 2*a45*a56) + 
  a55*(2*a34*a36*a55 + a33*a46*a55 + 4*a33*a45*a56))
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==1) & (r[5]==3) & (r[6]==2))
psi <-
3*(2*a36^2*a45*a55 + 4*a36*(a35*a46*a55 + 2*a35*a45*a56 + a34*a55*a56) + 
  2*a35^2*(2*a46*a56 + a45*a66) + 2*a34*a35*(2*a56^2 + a55*a66) + 
  a33*(2*a46*a55*a56 + 2*a45*a56^2 + a45*a55*a66))
  if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==1) & (r[5]==2) & (r[6]==3))
psi <-
3*(2*a36^2*(a46*a55 + 2*a45*a56) + 2*a35*(a35*a46 + 2*a34*a56)*a66 + 
  2*a36*(4*a35*a46*a56 + 2*a34*a56^2 + 2*a35*a45*a66 + a34*a55*a66) + 
  a33*(2*a46*a56^2 + a46*a55*a66 + 2*a45*a56*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==1) & (r[5]==1) & (r[6]==4))
psi <-
3*(8*a36*(a35*a46 + a34*a56)*a66 + 4*a36^2*(2*a46*a56 + a45*a66) + 
  a66*(4*a33*a46*a56 + 2*a34*a35*a66 + a33*a45*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==1) & (r[5]==0) & (r[6]==5))
psi <-
15*a66*(4*a36^2*a46 + 2*a34*a36*a66 + a33*a46*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==0) & (r[5]==6) & (r[6]==0))
psi <-
15*a55^2*(6*a35^2 + a33*a55)
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==0) & (r[5]==5) & (r[6]==1))
psi <-
15*a55*(2*a35*a36*a55 + 4*a35^2*a56 + a33*a55*a56)
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==0) & (r[5]==4) & (r[6]==2))
psi <-
3*(2*a36^2*a55^2 + 16*a35*a36*a55*a56 + 4*a35^2*(2*a56^2 + a55*a66) + 
  a33*a55*(4*a56^2 + a55*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==0) & (r[5]==3) & (r[6]==3))
psi <-
3*(6*a36^2*a55*a56 + 6*a35*a36*(2*a56^2 + a55*a66) + 
  a56*(2*a33*a56^2 + 6*a35^2*a66 + 3*a33*a55*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==0) & (r[5]==2) & (r[6]==4))
psi <-
3*(16*a35*a36*a56*a66 + 4*a36^2*(2*a56^2 + a55*a66) + 
  a66*(4*a33*a56^2 + 2*a35^2*a66 + a33*a55*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==0) & (r[5]==1) & (r[6]==5))
psi <-
15*a66*(4*a36^2*a56 + 2*a35*a36*a66 + a33*a56*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==2) & (r[4]==0) & (r[5]==0) & (r[6]==6))
psi <-
15*a66^2*(6*a36^2 + a33*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==7) & (r[5]==0) & (r[6]==0))
psi <-
105*a34*a44^3
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==6) & (r[5]==1) & (r[6]==0))
psi <-
15*a44^2*(a35*a44 + 6*a34*a45)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==6) & (r[5]==0) & (r[6]==1))
psi <-
15*a44^2*(a36*a44 + 6*a34*a46)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==5) & (r[5]==2) & (r[6]==0))
psi <-
15*a44*(2*a35*a44*a45 + 4*a34*a45^2 + a34*a44*a55)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==5) & (r[5]==1) & (r[6]==1))
psi <-
15*a44*(a36*a44*a45 + a35*a44*a46 + 4*a34*a45*a46 + a34*a44*a56)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==5) & (r[5]==0) & (r[6]==2))
psi <-
15*a44*(2*a36*a44*a46 + 4*a34*a46^2 + a34*a44*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==4) & (r[5]==3) & (r[6]==0))
psi <-
9*a35*a44*(4*a45^2 + a44*a55) + 12*a34*a45*(2*a45^2 + 3*a44*a55)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==4) & (r[5]==2) & (r[6]==1))
psi <-
3*(a36*a44*(4*a45^2 + a44*a55) + 2*a35*a44*(4*a45*a46 + a44*a56) + 
  4*a34*(2*a45^2*a46 + a44*a46*a55 + 2*a44*a45*a56))
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==4) & (r[5]==1) & (r[6]==2))
psi <-
3*(2*a36*a44*(4*a45*a46 + a44*a56) + a35*a44*(4*a46^2 + a44*a66) + 
  4*a34*(2*a45*a46^2 + 2*a44*a46*a56 + a44*a45*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==4) & (r[5]==0) & (r[6]==3))
psi <-
9*a36*a44*(4*a46^2 + a44*a66) + 12*a34*a46*(2*a46^2 + 3*a44*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==3) & (r[5]==4) & (r[6]==0))
psi <-
9*a34*a55*(4*a45^2 + a44*a55) + 12*a35*(2*a45^3 + 3*a44*a45*a55)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==3) & (r[5]==3) & (r[6]==1))
psi <-
3*(2*a36*a45^3 + 6*a35*a45^2*a46 + 3*a36*a44*a45*a55 + 3*a35*a44*a46*a55 + 
  6*a34*a45*a46*a55 + 6*a35*a44*a45*a56 + 6*a34*a45^2*a56 + 3*a34*a44*a55*a56)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==3) & (r[5]==2) & (r[6]==2))
psi <-
3*(2*a36*(2*a45^2*a46 + a44*a46*a55 + 2*a44*a45*a56) + 
  2*a35*(2*a45*a46^2 + 2*a44*a46*a56 + a44*a45*a66) + 
  a34*(2*a46^2*a55 + 8*a45*a46*a56 + 2*a44*a56^2 + 2*a45^2*a66 + a44*a55*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==3) & (r[5]==1) & (r[6]==3))
psi <-
3*(2*a35*a46^3 + 6*a34*a46^2*a56 + 3*a35*a44*a46*a66 + 6*a34*a45*a46*a66 + 
  3*a34*a44*a56*a66 + 3*a36*(2*a45*a46^2 + 2*a44*a46*a56 + a44*a45*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==3) & (r[5]==0) & (r[6]==4))
psi <-
9*a34*a66*(4*a46^2 + a44*a66) + 12*a36*(2*a46^3 + 3*a44*a46*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==2) & (r[5]==5) & (r[6]==0))
psi <-
15*a55*(4*a35*a45^2 + a35*a44*a55 + 2*a34*a45*a55)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==2) & (r[5]==4) & (r[6]==1))
psi <-
3*(a36*a55*(4*a45^2 + a44*a55) + 2*a34*a55*(a46*a55 + 4*a45*a56) + 
  4*a35*(2*a45*a46*a55 + 2*a45^2*a56 + a44*a55*a56))
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==2) & (r[5]==3) & (r[6]==2))
psi <-
3*(2*a36*(2*a45*a46*a55 + 2*a45^2*a56 + a44*a55*a56) + 
  a35*(2*a46^2*a55 + 8*a45*a46*a56 + 2*a44*a56^2 + 2*a45^2*a66 + 
    a44*a55*a66) + 2*a34*(2*a46*a55*a56 + 2*a45*a56^2 + a45*a55*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==2) & (r[5]==2) & (r[6]==3))
psi <-
3*(a36*(2*a46^2*a55 + 8*a45*a46*a56 + 2*a44*a56^2 + 2*a45^2*a66 + 
    a44*a55*a66) + 2*(a35*(2*a46^2*a56 + 2*a45*a46*a66 + a44*a56*a66) + 
    a34*(2*a46*a56^2 + a46*a55*a66 + 2*a45*a56*a66)))
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==2) & (r[5]==1) & (r[6]==4))
psi <-
3*(a66*(4*a35*a46^2 + 8*a34*a46*a56 + a35*a44*a66 + 2*a34*a45*a66) + 
  4*a36*(2*a46^2*a56 + 2*a45*a46*a66 + a44*a56*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==2) & (r[5]==0) & (r[6]==5))
psi <-
15*a66*(4*a36*a46^2 + a36*a44*a66 + 2*a34*a46*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==1) & (r[5]==6) & (r[6]==0))
psi <-
15*a55^2*(6*a35*a45 + a34*a55)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==1) & (r[5]==5) & (r[6]==1))
psi <-
15*a55*(a36*a45*a55 + a35*a46*a55 + 4*a35*a45*a56 + a34*a55*a56)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==1) & (r[5]==4) & (r[6]==2))
psi <-
3*(2*a36*a55*(a46*a55 + 4*a45*a56) + a34*a55*(4*a56^2 + a55*a66) + 
  4*a35*(2*a46*a55*a56 + 2*a45*a56^2 + a45*a55*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==1) & (r[5]==3) & (r[6]==3))
psi <-
3*(6*a35*a46*a56^2 + 2*a34*a56^3 + 3*a35*a46*a55*a66 + 6*a35*a45*a56*a66 + 
  3*a34*a55*a56*a66 + 3*a36*(2*a46*a55*a56 + 2*a45*a56^2 + a45*a55*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==1) & (r[5]==2) & (r[6]==4))
psi <-
3*(a66*(8*a35*a46*a56 + 4*a34*a56^2 + 2*a35*a45*a66 + a34*a55*a66) + 
  4*a36*(2*a46*a56^2 + a46*a55*a66 + 2*a45*a56*a66))
  if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==1) & (r[5]==1) & (r[6]==5))
psi <-
15*a66*(4*a36*a46*a56 + a36*a45*a66 + a35*a46*a66 + a34*a56*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==1) & (r[5]==0) & (r[6]==6))
psi <-
15*a66^2*(6*a36*a46 + a34*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==0) & (r[5]==7) & (r[6]==0))
psi <-
105*a35*a55^3
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==0) & (r[5]==6) & (r[6]==1))
psi <-
15*a55^2*(a36*a55 + 6*a35*a56)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==0) & (r[5]==5) & (r[6]==2))
psi <-
15*a55*(2*a36*a55*a56 + 4*a35*a56^2 + a35*a55*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==0) & (r[5]==4) & (r[6]==3))
psi <-
9*a36*a55*(4*a56^2 + a55*a66) + 12*a35*a56*(2*a56^2 + 3*a55*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==0) & (r[5]==3) & (r[6]==4))
psi <-
9*a35*a66*(4*a56^2 + a55*a66) + 12*a36*(2*a56^3 + 3*a55*a56*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==0) & (r[5]==2) & (r[6]==5))
psi <-
15*a66*(4*a36*a56^2 + a36*a55*a66 + 2*a35*a56*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==0) & (r[5]==1) & (r[6]==6))
psi <-
15*a66^2*(6*a36*a56 + a35*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==1) & (r[4]==0) & (r[5]==0) & (r[6]==7))
psi <-
105*a36*a66^3
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==8) & (r[5]==0) & (r[6]==0))
psi <-
105*a44^4
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==7) & (r[5]==1) & (r[6]==0))
psi <-
105*a44^3*a45
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==7) & (r[5]==0) & (r[6]==1))
psi <-
105*a44^3*a46
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==6) & (r[5]==2) & (r[6]==0))
psi <-
15*a44^2*(6*a45^2 + a44*a55)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==6) & (r[5]==1) & (r[6]==1))
psi <-
15*a44^2*(6*a45*a46 + a44*a56)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==6) & (r[5]==0) & (r[6]==2))
psi <-
15*a44^2*(6*a46^2 + a44*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==5) & (r[5]==3) & (r[6]==0))
psi <-
15*a44*a45*(4*a45^2 + 3*a44*a55)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==5) & (r[5]==2) & (r[6]==1))
psi <-
15*a44*(4*a45^2*a46 + a44*a46*a55 + 2*a44*a45*a56)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==5) & (r[5]==1) & (r[6]==2))
psi <-
15*a44*(4*a45*a46^2 + 2*a44*a46*a56 + a44*a45*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==5) & (r[5]==0) & (r[6]==3))
psi <-
15*a44*a46*(4*a46^2 + 3*a44*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==4) & (r[5]==4) & (r[6]==0))
psi <-
24*a45^4 + 72*a44*a45^2*a55 + 9*a44^2*a55^2
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==4) & (r[5]==3) & (r[6]==1))
psi <-
24*a45^3*a46 + 36*a44*a45*a46*a55 + 36*a44*a45^2*a56 + 9*a44^2*a55*a56
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==4) 
& (r[5]==2) & (r[6]==2))
psi <-
3*(16*a44*a45*a46*a56 + 4*a45^2*(2*a46^2 + a44*a66) + 
  a44*(4*a46^2*a55 + 2*a44*a56^2 + a44*a55*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==4) & (r[5]==1) & (r[6]==3))
psi <-
9*a44*a56*(4*a46^2 + a44*a66) + 12*a45*(2*a46^3 + 3*a44*a46*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==4) & (r[5]==0) & (r[6]==4))
psi <-
24*a46^4 + 72*a44*a46^2*a66 + 9*a44^2*a66^2
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==3) & (r[5]==5) & (r[6]==0))
psi <-
15*a45*a55*(4*a45^2 + 3*a44*a55)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==3) & (r[5]==4) & (r[6]==1))
psi <-
36*a45^2*a46*a55 + 9*a44*a46*a55^2 + 24*a45^3*a56 + 36*a44*a45*a55*a56
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==3) & (r[5]==3) & (r[6]==2))
psi <-
3*(12*a45^2*a46*a56 + 6*a44*a46*a55*a56 + 2*a45^3*a66 + 
  3*a45*(2*a46^2*a55 + 2*a44*a56^2 + a44*a55*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==3) & (r[5]==2) & (r[6]==3))
psi <-
3*(2*a46^3*a55 + 12*a45*a46^2*a56 + 6*a44*a45*a56*a66 + 
  3*a46*(2*a44*a56^2 + 2*a45^2*a66 + a44*a55*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==3) & (r[5]==1) & (r[6]==4))
psi <-
24*a46^3*a56 + 36*a45*a46^2*a66 + 36*a44*a46*a56*a66 + 9*a44*a45*a66^2
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==3) & (r[5]==0) & (r[6]==5))
psi <-
15*a46*a66*(4*a46^2 + 3*a44*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==2) & (r[5]==6) & (r[6]==0))
psi <-
15*a55^2*(6*a45^2 + a44*a55)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==2) & (r[5]==5) & (r[6]==1))
psi <-
15*a55*(2*a45*a46*a55 + 4*a45^2*a56 + a44*a55*a56)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==2) & (r[5]==4) & (r[6]==2))
psi <-
3*(2*a46^2*a55^2 + 16*a45*a46*a55*a56 + 4*a45^2*(2*a56^2 + a55*a66) + 
  a44*a55*(4*a56^2 + a55*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==2) & (r[5]==3) & (r[6]==3))
psi <-
3*(6*a46^2*a55*a56 + 6*a45*a46*(2*a56^2 + a55*a66) + 
  a56*(2*a44*a56^2 + 6*a45^2*a66 + 3*a44*a55*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==2) & (r[5]==2) & (r[6]==4))
psi <-
3*(16*a45*a46*a56*a66 + 4*a46^2*(2*a56^2 + a55*a66) + 
  a66*(4*a44*a56^2 + 2*a45^2*a66 + a44*a55*a66))
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==2) & (r[5]==1) & (r[6]==5))
psi <-
15*a66*(4*a46^2*a56 + 2*a45*a46*a66 + a44*a56*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==2) & (r[5]==0) & (r[6]==6))
psi <-
15*a66^2*(6*a46^2 + a44*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==1) & (r[5]==7) & (r[6]==0))
psi <-
105*a45*a55^3
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==1) & (r[5]==6) & (r[6]==1))
psi <-
15*a55^2*(a46*a55 + 6*a45*a56)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==1) & (r[5]==5) & (r[6]==2))
psi <-
15*a55*(2*a46*a55*a56 + 4*a45*a56^2 + a45*a55*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==1) & (r[5]==4) & (r[6]==3))
psi <-
9*a46*a55*(4*a56^2 + a55*a66) + 12*a45*a56*(2*a56^2 + 3*a55*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==1) & (r[5]==3) & (r[6]==4))
psi <-
9*a45*a66*(4*a56^2 + a55*a66) + 12*a46*(2*a56^3 + 3*a55*a56*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==1) & (r[5]==2) & (r[6]==5))
psi <-
15*a66*(4*a46*a56^2 + a46*a55*a66 + 2*a45*a56*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==1) & (r[5]==1) & (r[6]==6))
psi <-
15*a66^2*(6*a46*a56 + a45*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==1) & (r[5]==0) & (r[6]==7))
psi <-105*a46*a66^3
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==8) & (r[6]==0))
psi <-105*a55^4
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==7) & (r[6]==1))
psi <-105*a55^3*a56
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==6) & (r[6]==2))
psi <-15*a55^2*(6*a56^2 + a55*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==5) & (r[6]==3))
psi <-15*a55*a56*(4*a56^2 + 3*a55*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==4) & (r[6]==4))
psi <-24*a56^4 + 72*a55*a56^2*a66 + 9*a55^2*a66^2
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==3) & (r[6]==5))
psi <- 15*a56*a66*(4*a56^2 + 3*a55*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==2) & (r[6]==6))
psi <-15*a66^2*(6*a56^2 + a55*a66)
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==1) & (r[6]==7))
psi <-105*a56*a66^3
 if ((r[1]==0) & (r[2]==0) & (r[3]==0) & (r[4]==0) & (r[5]==0) & (r[6]==8))
  psi <- 105*a66^4

return(psi)

}
