% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lacunarity.R
\name{lacunarity}
\alias{lacunarity}
\title{Calculate gliding-box lacunarity}
\usage{
lacunarity(x, box_sizes = "twos", periodic = FALSE)
}
\arguments{
\item{x}{A 3-dimensional \code{\link[=array]{array}} of integer values}

\item{box_sizes}{Which box sizes to use for calculating lacunarity:
\itemize{
\item \code{"twos"} (the default) returns box sizes for all powers of two less than or
equal to the smallest dimension of \code{x}.
\item \code{"all"} calculates every possible box size up to the smallest dimension of \code{x}.
\item Alternatively, users may supply their own \code{\link[=vector]{vector}} of custom
box sizes. This vector must be of type "\code{numeric}" and can only contain
positive values. Values which exceed the dimensions of \code{x} are ignored.
}}

\item{periodic}{A Boolean. Determines which boundary algorithm to use, the
classic fixed boundary by Allain and Cloitre (default) or the periodic
boundary algorithm introduced by Feagin et al. 2007. The latter is slightly
slower but is more robust to edge effects.}
}
\value{
A \code{\link[=data.frame]{data.frame}} containing box sizes and their
corresponding \eqn{\Lambda(r)}, normalized \eqn{\Lambda(r)}, and \eqn{\mathrm{H}(r)} values. Lacunarity is
always computed for box size 1, even if the user supplies a custom
\code{box_sizes} vector that omits it, as this value is required to calculate
normalized lacunarity.
}
\description{
Generates \eqn{\Lambda(r)} lacunarity curves for a specified set of box sizes, using one
of two versions of the gliding-box algorithm
}
\details{
The raw \eqn{\Lambda(r)} values depend on the proportion of occupied voxels
within the data space. As a result, it is difficult to compare two spatial
patterns with different occupancy proportions because the curves will begin
at different y-intercepts. This is rectified by normalizing the curve,
typically by log-transforming it and dividing by the lacunarity value at
the smallest box size (i.e. \eqn{\log \Lambda(r)/\log \Lambda(1)}). \code{lacunarity()} outputs
both normalized and non-normalized \eqn{\Lambda(r)} curves for convenience.

The function also computes \eqn{\mathrm{H}(r)}, a transformed lacunarity curve introduced
by Feagin 2003. \eqn{\mathrm{H}(r)} rescales normalized \eqn{\Lambda(r)} in terms of the Hurst
exponent, where values greater than 0.5 indicate heterogeneity and values
less than 0.5 indicate homogeneity. Where \eqn{\Lambda(r)} describes a pattern's
deviation from translational invariance, \eqn{\mathrm{H}(r)} describes its deviation from
standard Brownian motion.
}
\examples{
# generate array
a <- array(data = rep(c(1,0), 125), dim = c(5,5,5))
# calculate lacunarity with default options
lacunarity(a)
# supply custom vector of box sizes
lacunarity(a, box_sizes = c(1,3,5))
# calculate lacunarity at all box sizes using the periodic boundary algorithm
lacunarity(a, box_sizes = "all", periodic = TRUE)
}
\references{
Allain, C., & Cloitre, M. (1991). Characterizing the lacunarity
of random and deterministic fractal sets. \emph{Physical Review A}, \strong{44(6)},
3552–3558. \doi{doi:10.1103/PhysRevA.44.3552}.

Feagin, R. A. (2003). Relationship of second-order lacunarity, Hurst
exponent, Brownian motion, and pattern organization. \emph{Physica A:
Statistical Mechanics and its Applications}, \strong{328(3-4)}, 315-321.
\doi{doi:10.1016/S0378-4371(03)00524-7}.

Feagin, R. A., Wu, X. B., & Feagin, T. (2007). Edge effects in lacunarity
analysis. \emph{Ecological Modelling}, \strong{201(3–4)}, 262–268.
\doi{doi:10.1016/j.ecolmodel.2006.09.019}.
}
