% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/matrixcreation.R
\name{fleslie}
\alias{fleslie}
\title{Create Function-based Age-based (Leslie) Matrix Projection Model}
\usage{
fleslie(
  year = "all",
  patch = "all",
  data = NA,
  modelsuite = NA,
  surv_model = NA,
  fec_model = NA,
  paramnames = NA,
  start_age = NA,
  last_age = NA,
  fecage_min = NA,
  fecage_max = NA,
  continue = TRUE,
  inda = NULL,
  indb = NULL,
  indc = NULL,
  surv_dev = 0,
  fec_dev = 0,
  density = NA,
  repmod = 1,
  yearcol = 0,
  patchcol = NA,
  agecol = "obsage",
  year.as.random = FALSE,
  patch.as.random = FALSE,
  random.inda = FALSE,
  random.indb = FALSE,
  random.indc = FALSE,
  randomseed = NA,
  negfec = FALSE,
  exp_tol = 700,
  theta_tol = 1e+08
)
}
\arguments{
\item{year}{A variable corresponding to observation occasion, or a set
of such values, given in values associated with the year term used in linear 
model development. Defaults to \code{"all"}, in which case matrices will be
estimated for all occasions.}

\item{patch}{A variable designating which patches or subpopulations will have
matrices estimated. Defaults to \code{"all"}, but can also be set to specific
patch names.}

\item{data}{The historical vertical demographic data frame used to estimate
vital rates (class \code{hfvdata}). The original data frame is required in
order to initialize occasions and patches properly.}

\item{modelsuite}{An optional \code{lefkoMod} object holding the vital rate
models. If given, then \code{surv_model}, \code{fec_model},
\code{paramnames}, \code{yearcol}, and \code{patchcol} are not required. No
models should include size or reproductive status in occasion \emph{t}-1.}

\item{surv_model}{A linear model predicting survival probability. This can be
a model of class \code{glm} or \code{glmer}, and requires a predicted
binomial variable under a logit link. If given, then will overwrite any
survival probability model given in \code{modelsuite}. This model must have
been developed in a modeling exercise testing only the impacts of occasion
\emph{t}.}

\item{fec_model}{A linear model predicting fecundity. This can be a model of
class \code{glm}, \code{glmer}, \code{glmmTMB}, \code{zeroinfl},
\code{vglm}, \code{lm}, or \code{lmer}. If given, then will overwrite any 
fecundity model given in \code{modelsuite}. This model must have been
developed in a modeling exercise testing only the impacts of occasion
\emph{t}.}

\item{paramnames}{A dataframe with three columns, the first describing all
terms used in linear modeling, the second (must be called \code{mainparams}),
showing the general model terms that will be used in matrix creation (users
should use \code{\link{modelsearch}()} at least once to see the proper
names to be used in this column), and the third showing the equivalent terms
used in modeling (must be named \code{modelparams}). Only required if
\code{modelsuite} is not supplied.}

\item{start_age}{The age from which to start the matrix. Defaults to
\code{NA}, in which case the earliest age in the dataset is used.}

\item{last_age}{The final age to use in the matrix. Defaults to \code{NA}, in
which case the highest age in the dataset is used.}

\item{fecage_min}{The minimum age at which reproduction is possible. Defaults
to \code{NA}, which is interpreted to mean that fecundity should be assessed
starting in the first described age.}

\item{fecage_max}{The maximum age at which reproduction is possible. Defaults
to \code{NA}, which is interpreted to mean that fecundity should be assessed
until the final observed age.}

\item{continue}{A logical value designating whether to allow continued
survival of individuals past the final age noted in the stageframe, using the 
demographic characteristics of the final age. Defaults to \code{TRUE}.}

\item{inda}{Can be a single value to use for individual covariate \code{a}
in all matrices, a pair of values to use for times \emph{t} and \emph{t}-1 in
historical matrices, or a vector of such values corresponding to each
occasion in option \code{year}. Defaults to \code{NULL}.}

\item{indb}{Can be a single value to use for individual covariate \code{b}
in all matrices, a pair of values to use for times \emph{t} and \emph{t}-1 in
historical matrices, or a vector of such values corresponding to each
occasion in option \code{year}. Defaults to \code{NULL}.}

\item{indc}{Can be a single value to use for individual covariate \code{c}
in all matrices, a pair of values to use for times \emph{t} and \emph{t}-1 in
historical matrices, or a vector of such values corresponding to each
occasion in option \code{year}. Defaults to \code{NULL}.}

\item{surv_dev}{A numeric value to be added to the y-intercept in the linear
model for survival probability.}

\item{fec_dev}{A numeric value to be added to the y-intercept in the linear
model for fecundity.}

\item{density}{A numeric value indicating density value to use to propagate
matrices. Only needed if density is an explanatory term used in linear
models. Defaults to \code{NA}.}

\item{repmod}{A scalar multiplier of fecundity. Defaults to \code{1}.}

\item{yearcol}{The variable name or column number corresponding to year in
occasion \emph{t} in the dataset. Not needed if a \code{modelsuite} is
supplied.}

\item{patchcol}{The variable name or column number corresponding to patch in 
the dataset. Not needed if a \code{modelsuite} is supplied.}

\item{agecol}{The name or column number of the variable coding for age in
\code{data}. Defaults to \code{"obsage"}.}

\item{year.as.random}{A logical term indicating whether coefficients for
missing occasions within vital rate models should be estimated as random
intercepts. Defaults to \code{FALSE}, in which case missing monitoring
occasion coefficients are set to \code{0}.}

\item{patch.as.random}{A logical term indicating whether coefficients for
missing patches within vital rate models should be estimated as random
intercepts. Defaults to \code{FALSE}, in which case missing patch
coefficients are set to \code{0}.}

\item{random.inda}{A logical value denoting whether to treat individual
covariate \code{a} as a random, categorical variable. Otherwise is treated as
a fixed, numeric variable. Defaults to \code{FALSE}.}

\item{random.indb}{A logical value denoting whether to treat individual
covariate \code{b} as a random, categorical variable. Otherwise is treated as
a fixed, numeric variable. Defaults to \code{FALSE}.}

\item{random.indc}{A logical value denoting whether to treat individual
covariate \code{c} as a random, categorical variable. Otherwise is treated as
a fixed, numeric variable. Defaults to \code{FALSE}.}

\item{randomseed}{A numeric value used as a seed to generate random estimates
for missing occasion and patch coefficients, if either \code{year.as.random}
or \code{patch.as.random} is set to \code{TRUE}. Defaults to
\code{\link{set.seed}()} default.}

\item{negfec}{A logical value denoting whether fecundity values estimated to
be negative should be reset to \code{0}. Defaults to \code{FALSE}.}

\item{exp_tol}{A numeric value used to indicate a maximum value to set
exponents to in the core kernel to prevent numerical overflow. Defaults to
\code{700}.}

\item{theta_tol}{A numeric value used to indicate a maximum value to theta as
used in the negative binomial probability density kernel. Defaults to
\code{100000000}, but can be reset to other values during error checking.}
}
\value{
If all inputs are properly formatted, then this function will return
an object of class \code{lefkoMat}, which is a list that holds the matrix
projection model and all of its metadata. Its structure is a list with the
following elements:

\item{A}{A list of full projection matrices in order of sorted patches and
occasions. All matrices output in R's \code{matrix} class.}
\item{U}{A list of survival transition matrices sorted as in \code{A}. All 
matrices output in R's \code{matrix} class.}
\item{F}{A list of fecundity matrices sorted as in \code{A}. All matrices 
output in R's \code{matrix} class.}
\item{hstages}{Set to \code{NA} for Leslie MPMs.}
\item{agestages}{Set to \code{NA} for Leslie MPMs.}
\item{ahstages}{A data frame detailing the characteristics of associated
ages, in the form of a modified stageframe including reproduction status.}
\item{labels}{A data frame giving the patch and year of each matrix in order.
In \code{aflefko2()}, only one population may be analyzed at once, and so
\code{pop = NA}}
\item{matrixqc}{A short vector describing the number of non-zero elements
in \code{U} and \code{F} matrices, and the number of annual matrices.}
\item{modelqc}{This is the \code{qc} portion of the modelsuite input.}
}
\description{
Function \code{fleslie()} returns age-based (Leslie) MPMs corresponding to
the patches and occasion times given, including the associated component
transition and fecundity matrices, data frames detailing the characteristics
of the exact ages corresponding to rows and columns in estimated matrices,
and a data frame characterizing the patch and occasion time combinations
corresponding to these matrices.
}
\section{Notes}{

Unlike \code{\link{rlefko2}()}, \code{\link{rlefko3}()}, and
\code{\link{rleslie}()}, this function does not currently distinguish
populations.

This function will yield incorrect estimates if the models utilized
incorporate state in occasion \emph{t}-1, or any size or reproductive status
terms.

Users may at times wish to estimate MPMs using a dataset incorporating
multiple patches or subpopulations, but without discriminating between those
patches or subpopulations. Should the aim of analysis be a general MPM that
does not distinguish these patches or subpopulations, the \code{patchcol}
variable should be set to \code{NA}, which is the default.

Input options including multiple variable names must be entered in the order
of variables in occasion \emph{t}+1 and \emph{t}. Rearranging the order will
lead to erroneous calculations, and may lead to fatal errors.

Care should be taken to match the random status of year and patch to the
states of those variables within the modelsuite. If they do not match, then
they will be treated as zeroes in vital rate estimation.
}

\examples{
\donttest{
data(lathyrus)

lathvert_base <- verticalize3(lathyrus, noyears = 4, firstyear = 1988,
  patchidcol = "SUBPLOT", individcol = "GENET", blocksize = 9,
  sizeacol = "Volume88", repstracol = "FCODE88", fecacol = "Intactseed88",
  deadacol = "Dead1988", censorcol = "Missing1988", censorkeep = NA,
  censor = TRUE, NRasRep = TRUE, NOasObs = TRUE)

lathvert_base$feca3 <- round(lathvert_base$feca3)
lathvert_base$feca2 <- round(lathvert_base$feca2)
lathvert_base$feca1 <- round(lathvert_base$feca1)

lathvert_age <- subset(lathvert_base, firstseen > 1988)

lathmodels2_age <- modelsearch(lathvert_age, historical = FALSE,
  approach = "mixed", suite = "cons", bestfit = "AICc&k", age = "obsage",
  vitalrates = c("surv", "fec"), fecdist = "poisson", indiv = "individ",
  year = "year2", year.as.random = TRUE, show.model.tables = TRUE,
  quiet = TRUE)

lathmat2fleslie <- fleslie(year = "all", data = lathvert_age,
  modelsuite = lathmodels2_age, yearcol = "year2", fecage_min = 1)

summary(lathmat2fleslie)
}
}
\seealso{
\code{\link{flefko3}()}

\code{\link{flefko2}()}

\code{\link{aflefko2}()}

\code{\link{rlefko3}()}

\code{\link{rlefko2}()}

\code{\link{rleslie}()}
}
