% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/grid_metrics.r
\name{grid_metrics}
\alias{grid_metrics}
\title{Rasterize the space and compute metrics for each cell}
\usage{
grid_metrics(.las, func, res = 20, start = c(0, 0), splitlines = FALSE,
  debug = FALSE)
}
\arguments{
\item{.las}{An object of class \code{LAS}}

\item{func}{the function to be applied to each cell}

\item{res}{numeric. The size of the cells. Default 20.}

\item{start}{vector x and y coordinates for the reference raster. Default is (0,0).}

\item{splitlines}{logical. If TRUE the algorithm will compute the metrics for each
flightline individually. It returns the same cells several times in overlap.}

\item{debug}{logical. If you encounter a non trivial error try \code{debug = TRUE}.}
}
\value{
It returns a \code{data.table} containing the metrics for each cell. The table
has the class "lasmetrics" enabling easy plotting.
}
\description{
Computes a series of descriptive statistics defined by the user for a LiDAR dataset within
each pixel of a raster. Output is a data.frame in which each line is a pixel (single grid cell),
and each column is a metric.
}
\details{
\code{grid_metrics} is similar to \link{lasmetrics} or \link{grid_hexametrics} except it computes metrics within each cell
in a predefined grid. The grid cell coordinates are pre-determined for a given resolution.
So the algorithm will always provide the same coordinates independently of the dataset.
When start = (0,0) and res = 20 grid_metrics will produce the following raster centers:
(10,10), (10,30), (30,10) etc.. When start = (-10, -10) and res = 20 grid_metrics will
produce the following raster centers: (0,0), (0,20), (20,0) etc.. In Quebec (Canada) reference
is (-831600,  117980) in the NAD83 coordinate system. The function to be applied to each
cell is a classical function (see examples) that returns a labelled list of metrics.
The following existing function can help the user to compute some metrics:
\itemize{
\item{\link[lidR:stdmetrics]{stdmetrics}}
\item{\link[lidR:entropy]{entropy}}
\item{\link[lidR:VCI]{VCI}}
\item{\link[lidR:LAD]{LAD}}
} Users must write their own functions to create metrics. \code{grid_metrics} will
dispatch the LiDAR data for each cell in the user's function. The user writes their
function without considering grid cells, only a cloud of points (see example).
}
\examples{
LASfile <- system.file("extdata", "Megaplot.laz", package="lidR")
lidar = readLAS(LASfile)

# Canopy surface model with 4 m^2 cells
grid_metrics(lidar, max(Z), 2) \%>\% plot

# Mean height with 400 m^2 cells
grid_metrics(lidar, mean(Z), 20) \%>\% plot

# Define your own new metrics
myMetrics = function(z, i)
{
  metrics = list(
     zwimean = sum(z*i)/sum(i), # Mean elevation weighted by intensities
     zimean  = mean(z*i),       # Mean products of z by intensity
     zsqmean = sqrt(mean(z^2))  # Quadratic mean
   )

   return(metrics)
}

metrics = grid_metrics(lidar, myMetrics(Z, Intensity))

plot(metrics)
plot(metrics, "zwimean")
plot(metrics, "zimean")
plot(metrics, "zsqmean")
#etc.
}
