#' Alert user if NAs are present
#'
#' @param data Data frame from which observations are generated by sample_locally.
#' @param explained_instance A row in an original data frame (as a data.frame).
#'
#' @return Warning message
#'

check_for_na <- function(data, explained_instance) {
  if(any(is.na(explained_instance))) warning("Missing values present in explained instance.")
  if(any(is.na(data))) warning("Missing values present in dataset. NAs will be omitted while sampling.")
}


#' Check if data, explained instance and size make sense.
#' 
#' @param data Data frame from which observations will be sampled.
#' @param explained_instance Instance around which points will be sampled.
#' @param size Number of observation in simulated dataset
#' 
#' @return Produces an error if any of conditions aren't met.
#' 

check_conditions <- function(data, explained_instance, size) {
  if(nrow(data) == 0) stop("Empty data frame")
  if(ncol(data) == 0) stop("Data frame has no columns")
  if(size <= 0) stop("Size has to be a positive integer")
  if(any(colnames(data) != colnames(explained_instance))) 
    stop("Explained instance must have the same variables as data")
}


#' Set date values to one value 
#' 
#' @param data Data frame to change.
#' @param explained_instance 1-row data frame with instance of interest.
#' 

set_constant_dates <- function(data, explained_instance) {
  date_cols <- (1:ncol(data))[unlist(lapply(data, 
                               function(x) lubridate::is.Date(x) | lubridate::is.POSIXt(x)),
                        use.names = FALSE)]
  if(length(date_cols) == 0) {
    return(data)
  } else {
    for(k in date_cols) {
      data.table::set(data, j = as.integer(k), 
                      value = explained_instance[1, as.integer(k)])
    }
    data
  }
}

#' LIME: sampling for local exploration
#'
#' @param data Data frame from which observations will be generated.
#' @param explained_instance A row in an original data frame (as a data.frame).
#' @param size Number of observations to be generated.
#'
#' @return data.frame
#'

generate_neighbourhood <- function(data, explained_instance, size) {
  data <- data.table::as.data.table(data)
  neighbourhood <- data.table::rbindlist(lapply(1:size, function(x) explained_instance))
  for(k in 1:nrow(neighbourhood)) {
    picked_var <- sample(1:ncol(data), 1)
    data.table::set(neighbourhood, i = as.integer(k), j = as.integer(picked_var),
                    data[sample(1:nrow(data), 1), picked_var, with = FALSE])
  }
  as.data.frame(set_constant_dates(neighbourhood, explained_instance))
}


#' Create regression or classification task.
#'
#' @param model Name of a used model in mlr format.
#' @param dataset Data frame on which model will be trained.
#' @param target_var Name of column in dataset containing explained variable.
#'
#' @return mlr task object
#'

create_task <- function(model, dataset, target_var) {
  if(grepl("regr", model)) {
    mlr::makeRegrTask(id = "lime_task",
                      data = as.data.frame(dataset),
                      target = target_var)
  } else {
    mlr::makeClassifTask(id = "lime_task",
                         data = as.data.frame(dataset),
                         target = target_var)
  }
}


#' Generate dataset for local exploration.
#'
#' @param data Data frame from which new dataset will be simulated.
#' @param explained_instance One row data frame with the same variables
#'        as in data argument. Local exploration will be performed around this observation.
#' @param explained_var Name of a column with the variable to be predicted.
#' @param size Number of observations is a simulated dataset.
#' @param standardise If TRUE, numerical variables will be scaled to have mean 0, var 1.
#'
#' @return list
#'
#' @export
#'
#' @examples
#' \dontrun{
#' dataset_for_local_exploration <- sample_locally(data = wine,
#'                                                explained_instance = wine[5, ],
#'                                                explained_var = "quality",
#'                                                size = 50,
#'                                                standardise = TRUE)
#' }
#'

sample_locally <- function(data, explained_instance, explained_var, size, 
                           standardise = FALSE) {
  check_conditions(data, explained_instance, size)
  explained_var_col <- which(colnames(data) == explained_var)
  similar <- generate_neighbourhood(data[, -explained_var_col],
                                    explained_instance[, -explained_var_col], size)
  if(standardise) {
    vscale <- function(x) as.vector(scale(x))
    similar <- dplyr::mutate_if(similar, is.numeric, vscale)
  }

  list(data = similar, target = explained_var)
}


#' Add predictions to generated dataset.
#'
#' @param data Original data frame used to generate new dataset.
#' @param black_box String with mlr signature of a learner or a model with predict interface.
#' @param explained_var Name of a column with the variable to be predicted.
#' @param similar Dataset created for local exploration.
#' @param predict_function Either a "predict" function that returns a vector of the
#'        same type as response or custom function that takes a model as a first argument,
#'        new data used to calculate predictions as a second argument called "newdata"
#'        and returns a vector of the same type as response.
#'        Will be used only if a model object was provided in the black_box argument.
#' @param hyperpars Optional list of (hyper)parameters to be passed to mlr::makeLearner.
#' @param ... Additional parameters to be passed to predict function.
#'
#' @return A list containing black box model object and predictions.
#'

give_predictions <- function(data, black_box, explained_var, similar, predict_function, 
                             hyperpars = list(), ...) {
  if(is.character(black_box)) {
    mlr_task <- create_task(black_box, as.data.frame(data), explained_var)
    lrn <- mlr::makeLearner(black_box, par.vals = hyperpars)
    trained <- mlr::train(lrn, mlr_task)
    pred <- predict(trained, newdata = as.data.frame(similar))
    list(model = mlr::getLearnerModel(trained), 
         predictions = pred[["data"]][["response"]])
  } else {
    list(model = black_box, 
         predictions = predict_function(black_box, similar, ...))
  }
}


#' Add black box predictions to generated dataset
#'
#' @param data Original data frame used to generate new dataset.
#' @param to_explain List return by sample_locally function.
#' @param black_box_model String with mlr signature of a learner or a model with predict interface.
#' @param predict_fun Either a "predict" function that returns a vector of the
#'        same type as response or custom function that takes a model as a first argument,
#'        and data used to calculate predictions as a second argument
#'        and returns a vector of the same type as respone.
#'        Will be used only if a model object was provided in the black_box argument.
#' @param hyperparams Optional list of (hyper)parameters to be passed to mlr::makeLearner.
#' @param ... Additional parameters to be passed to predict function.
#'
#' @return list containing simulated dataset with added predictions,
#'              name of a response variable and black box model object.
#'
#' @export
#'
#' @examples
#' \dontrun{
#' local_exploration1 <- add_predictions(wine, dataset_for_local_exploration,
#'                                       black_box_model = "regr.svm")
#' # Pass trained model to the function.
#' svm_model <- svm(quality ~., data = wine)
#' local_exploration2 <- add_predictions(wkne, dataset_for_local_exploration,
#'                                       black_box_model = svm_model)
#' }
#'

add_predictions <- function(data, to_explain, black_box_model, predict_fun = predict, 
                            hyperparams = list(), ...) {
  trained_black_box <- give_predictions(data = data,
                                        black_box = black_box_model,
                                        explained_var = to_explain$target,
                                        similar = to_explain$data,
                                        predict_function = predict_fun,
                                        hyperpars = hyperparams,
                                        ...)
  to_explain$data[[to_explain$target]] <- trained_black_box$predictions
  
  list(data = to_explain$data, target = to_explain$target, 
       model = trained_black_box$model)
}
