#' Diagnostics for Pareto Smoothed Importance Sampling
#'
#' Print a diagnostic table summarizing the estimated Pareto shape parameters,
#' find the indexes of observations for which the estimated Pareto shape
#' parameter \eqn{k} is larger than some \code{threshold} value, or plot
#' observation indexes vs \eqn{k} estimates.
#'
#' @name pareto-k-diagnostic
#' @param x For \code{pareto_k_ids} and \code{pareto_k_table}, an object created
#'   by \code{\link{loo}} or \code{\link{psislw}}. For \code{plot}, an object
#'   created by \code{\link{loo}}.
#'
#' @inheritSection loo-package Pareto k diagnostic
#'
#' @seealso \code{\link{psislw}} for the implementation of the PSIS algorithm.
#'
#' @template loo-and-psis-references
#'
NULL

#' @rdname pareto-k-diagnostic
#' @export
#' @return \code{pareto_k_table} returns an object of class
#'   \code{"pareto_k_table"}, which is a matrix with columns \code{"Count"} and
#'   \code{"Proportion"} and has its own print method.
pareto_k_table <- function(x) {
  k <- get_pareto_k(x)
  kcut <- .k_cut(k)
  count <- table(kcut)
  out <- cbind(Count = count, Proportion = prop.table(count))
  structure(out, class = c("pareto_k_table", class(out)))
}

#' @export
print.pareto_k_table <- function(x, digits = 1, ...) {
  count <- x[, "Count"]
  prop <- x[, "Proportion"]

  if (sum(count[2:4]) == 0) {
    cat("\nAll Pareto k estimates are good (k < 0.5)\n")
  } else {
    tab <- cbind(
      " " = rep("", 4),
      " " = c("(good)", "(ok)", "(bad)", "(very bad)"),
      "Count" = .fr(count, 0),
      " Pct" = paste0(.fr(100 * prop, digits), "%")
    )
    tab2 <- rbind(tab)
    cat("\nPareto k diagnostic values:\n")
    rownames(tab2) <- format(rownames(tab2), justify = "right")
    print(tab2, quote = FALSE)

    if (sum(count[3:4]) == 0)
      cat("\nAll Pareto k estimates are ok (k < 0.7)\n")

    invisible(x)
  }
}

#' @rdname pareto-k-diagnostic
#' @export
#' @param threshold The threshold value for \eqn{k}.
#' @return \code{pareto_k_ids} returns an integer vector indicating which
#' observations have Pareto \eqn{k} estimates above \code{threshold}.
#'
pareto_k_ids <- function(x, threshold = 0.5) {
  k <- get_pareto_k(x)
  which(k > threshold)
}

#' @rdname pareto-k-diagnostic
#' @export
#' @param label_points,... For the \code{plot} method, if \code{label_points} is
#'   \code{TRUE} the observation numbers corresponding to any values of \eqn{k}
#'   greater than 0.5 will be displayed in the plot. Any arguments specified in
#'   \code{...} will be passed to \code{\link[graphics]{text}} and can be used
#'   to control the appearance of the labels.
#'
#' @return The \code{plot} method is called for its side effect and does not
#'   return anything. If \code{x} is the result of a call to \code{\link{loo}},
#'   \code{plot(x)} produces a plot of the estimates of the Pareto shape
#'   parameter \eqn{k}. There is no \code{plot} method for objects generated by
#'   a call to \code{\link{waic}}.
#'
plot.loo <- function(x, ..., label_points = FALSE) {
  k <- get_pareto_k(x)
  k_inf <- !is.finite(k)
  if (any(k_inf)) {
    warning(signif(100 * mean(k_inf), 2),
            "% of Pareto k estimates are Inf/NA/NaN and not plotted.")
  }
  plot_k(k, ..., label_points = label_points)
}



# internal ----------------------------------------------------------------
get_pareto_k <- function(x) {
  if (is.null(x[["pareto_k"]]))
    stop("No Pareto k estimates found.", call. = FALSE)
  x[["pareto_k"]]
}

#' @importFrom graphics abline axis plot points text
plot_k <- function(k, ..., label_points = FALSE) {
  inrange <- function(a, rr) a >= rr[1L] & a <= rr[2L]
  plot(k, xlab = "Data point", ylab = "Shape parameter k",
       type = "n", bty = "l", yaxt = "n")
  axis(side = 2, las = 1)
  krange <- range(k)
  breaks <- c(0, 0.5, 0.7, 1)
  hex_clrs <- c("#C79999", "#A25050", "#7C0000")
  ltys <- c(3, 4, 2, 1)
  for (j in seq_along(breaks)) {
    val <- breaks[j]
    if (inrange(val, krange))
      abline(h = val, col = ifelse(val == 0, "darkgray", hex_clrs[j-1]),
             lty = ltys[j], lwd = 1)
  }

  breaks <- c(-Inf, 0.5, 1)
  hex_clrs <- c("#6497b1", "#005b96", "#03396c")
  clrs <- ifelse(inrange(k, breaks[1:2]), hex_clrs[1],
                 ifelse(inrange(k, breaks[2:3]), hex_clrs[2], hex_clrs[3]))
  if (all(k < 0.5) || !label_points) {
    points(k, col = clrs, pch = 3, cex = .6)
    return(invisible())
  } else {
    points(k[k < 0.5], col = clrs[k < 0.5], pch = 3, cex = .6)
    sel <- !inrange(k, breaks[1:2])
    dots <- list(...)
    txt_args <- c(list(x = seq_along(k)[sel], y = k[sel],
                       labels = seq_along(k)[sel]),
                  if (length(dots)) dots)
    if (!("adj" %in% names(txt_args)))
      txt_args$adj <- 2/3
    if (!("cex" %in% names(txt_args)))
      txt_args$cex <- 0.75
    if (!("col" %in% names(txt_args)))
      txt_args$col <- clrs[sel]

    do.call("text", txt_args)
  }
}
