% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/psis.R
\name{psis}
\alias{psis}
\alias{psis.array}
\alias{psis.matrix}
\alias{psis.default}
\alias{weights.psis}
\alias{is.psis}
\title{Pareto smoothed importance sampling (PSIS)}
\usage{
psis(log_ratios, ...)

\method{psis}{array}(log_ratios, ..., r_eff = NULL,
  cores = getOption("mc.cores", 1))

\method{psis}{matrix}(log_ratios, ..., r_eff = NULL,
  cores = getOption("mc.cores", 1))

\method{psis}{default}(log_ratios, ..., r_eff = NULL)

\method{weights}{psis}(object, ..., log = TRUE, normalize = TRUE)

is.psis(x)
}
\arguments{
\item{log_ratios}{An array, matrix, or vector of importance ratios on the log
scale (for PSIS-LOO these are \emph{negative} log-likelihood values). See the
\strong{Methods (by class)} section below for a detailed description of how
to specify the inputs for each method.}

\item{...}{Arguments passed on to the various methods.}

\item{r_eff}{Vector of relative effective sample size estimates containing
one element per observation. The values provided should be the relative
effective sample sizes of \code{1/exp(log_ratios)} (i.e., \code{1/ratios}).
This is related to the relative efficiency of estimating the normalizing
term in self-normalizing importance sampling. If \code{r_eff} is not
provided then the reported PSIS effective sample sizes and Monte Carlo
error estimates will be over-optimistic. See the \code{\link{relative_eff}}
helper function for computing \code{r_eff}. If using \code{psis} with
draws of the \code{log_ratios} not obtained from MCMC then the warning
message thrown when not specifying \code{r_eff} can be disabled by
setting \code{r_eff} to \code{NA}.}

\item{cores}{The number of cores to use for parallelization. This defaults to
the option \code{mc.cores} which can be set for an entire R session by
\code{options(mc.cores = NUMBER)}. The old option \code{loo.cores} is now
deprecated but will be given precedence over \code{mc.cores} until
\code{loo.cores} is removed in a future release. \strong{As of version
2.0.0 the default is now 1 core if \code{mc.cores} is not set, but we
recommend using as many (or close to as many) cores as possible.}
Note for Windows 10 users: it is recommended to avoid using the
\code{.Rprofile} file to set \code{mc.cores} (using the \code{cores}
argument or setting \code{mc.cores} interactively or in a script is fine).}

\item{object}{For the \code{weights} method, an object
returned by \code{psis} (a list with class \code{"psis"}).}

\item{log}{For the \code{weights} method, should the weights be returned on
the log scale? Defaults to \code{TRUE}.}

\item{normalize}{For the \code{weights} method, should the weights be
normalized? Defaults to \code{TRUE}.}

\item{x}{For \code{is.psis}, an object to check.}
}
\value{
The \code{psis} methods return an object of class \code{"psis"},
  which is a named list with the following components:

\describe{
  \item{\code{log_weights}}{
    Vector or matrix of smoothed (and truncated) but \emph{unnormalized} log
    weights. To get normalized weights use the \code{weights} method provided
    for objects of class \code{"psis"}.
  }
 \item{\code{diagnostics}}{
   A named list containing two vectors:
   \itemize{
    \item \code{pareto_k}: Estimates of the shape parameter \eqn{k} of the
    generalized Pareto distribution. See the \code{\link{pareto-k-diagnostic}}
    page for details.
    \item \code{n_eff}: PSIS effective sample size estimates.
  }
 }
}

Objects of class \code{"psis"} also have the following
\code{\link{attributes}}:
\describe{
  \item{\code{norm_const_log}}{
    Vector of precomputed values of \code{colLogSumExps(log_weights)} that are
    used internally by the \code{weights} method to normalize the log weights.
  }
  \item{\code{tail_len}}{
    Vector of tail lengths used for fitting the generalized Pareto
    distribution.
  }
  \item{\code{r_eff}}{
    If specified, the user's \code{r_eff} argument.
  }
  \item{\code{dims}}{
    Integer vector of length 2 containing \code{S} (posterior sample size)
    and \code{N} (number of observations).
  }
}

The \code{weights} method returns an object with the same dimensions
  as the \code{log_weights} component of the \code{"psis"} object. The
  \code{normalize} and \code{log} arguments control whether the returned
  weights are normalized and whether or not to return them on the log scale.
}
\description{
Implementation of Pareto smoothed importance sampling (PSIS), a method for
stabilizing importance ratios. The version of PSIS implemented here
corresponds to the algorithm presented in Vehtari, Gelman and Gabry (2017b).
For PSIS diagnostics see the \link{pareto-k-diagnostic} page.
}
\section{Methods (by class)}{
\itemize{
\item \code{array}: An \eqn{I} by \eqn{C} by \eqn{N} array, where \eqn{I}
is the number of MCMC iterations per chain, \eqn{C} is the number of
chains, and \eqn{N} is the number of data points.

\item \code{matrix}: An \eqn{S} by \eqn{N} matrix, where \eqn{S} is the size
of the posterior sample (with all chains merged) and \eqn{N} is the number
of data points.

\item \code{default}: A vector of length \eqn{S} (posterior sample size).
}}

\examples{
log_ratios <- -1 * example_loglik_array()
r_eff <- relative_eff(exp(-log_ratios))
psis_result <- psis(log_ratios, r_eff = r_eff)
str(psis_result)
plot(psis_result)

# extract smoothed weights
lw <- weights(psis_result) # default args are log=TRUE, normalize=TRUE
ulw <- weights(psis_result, normalize=FALSE) # unnormalized log-weights

w <- weights(psis_result, log=FALSE) # normalized weights (not log-weights)
uw <- weights(psis_result, log=FALSE, normalize = FALSE) # unnormalized weights



}
\references{
Vehtari, A., Gelman, A., and Gabry, J. (2017a). Practical
  Bayesian model evaluation using leave-one-out cross-validation and WAIC.
  \emph{Statistics and Computing}. 27(5), 1413--1432.
  doi:10.1007/s11222-016-9696-4.
  (\href{http://link.springer.com/article/10.1007\%2Fs11222-016-9696-4}{
  journal}, \href{http://arxiv.org/abs/1507.04544}{preprint arXiv:1507.04544}).

Vehtari, A., Gelman, A., and Gabry, J. (2017b). Pareto smoothed
importance sampling. arXiv preprint: \url{http://arxiv.org/abs/1507.02646/}
}
\seealso{
\itemize{
\item \code{\link{loo}} for approximate LOO-CV using PSIS.
\item \code{\link{pareto-k-diagnostic}} for PSIS diagnostics.
}
}
