% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rmev.R
\name{rmev}
\alias{rmev}
\title{Exact simulations of multivariate extreme value distributions}
\usage{
rmev(n, d, param, asy, sigma, model = c("log", "alog", "neglog",
  "aneglog", "bilog", "negbilog", "hr", "br", "xstud", "smith",
  "schlather", "ct", "sdir", "dirmix"), alg = c("ef", "sm"), weights,
  vario, coord = NULL, grid = FALSE, ...)
}
\arguments{
\item{n}{number of observations}

\item{d}{dimension of sample}

\item{param}{parameter vector for the logistic, bilogistic, negative bilogistic and extremal Dirichlet (Coles and Tawn) model.
Parameter matrix for the Dirichlet mixture. Degree of freedoms for extremal student model. See \bold{Details}.}

\item{asy}{list of asymmetry parameters, as in \code{\link[evd]{rmvevd}}, of \eqn{2^d-1} vectors of size corresponding to the power set of \code{d}, with sum to one constraints.}

\item{sigma}{covariance matrix for Brown-Resnick and extremal Student-t distributions. Symmetric matrix of squared  coefficients \eqn{\lambda^2} for the Husler-Reiss model, with zero diagonal elements.}

\item{model}{for multivariate extreme value distributions, users can choose between 1-parameter logistic and negative logistic, asymmetric logistic and negative logistic, bilogistic, Husler-Reiss, extremal Dirichlet model (Coles and Tawn) or the Dirichlet mixture. Spatial models include
the Brown-Resnick, Smith, Schlather and extremal Student max-stable processes.}

\item{alg}{algorithm, either simulation via extremal function (\code{'ef'}) or via the spectral measure (\code{'sm'}). Default to \code{ef}.}

\item{weights}{vector of length \code{m} for the \code{m} mixture components. Must sum to one}

\item{vario}{semivariogram function whose first argument must be distance. Used only if provided in conjunction with \code{coord} and if \code{sigma} is missing}

\item{coord}{\code{d} by \code{k} matrix of coordinates, used as input in the variogram \code{vario} or as parameter for the Smith model. If \code{grid} is \code{TRUE}, unique entries should be supplied.}

\item{grid}{Logical. \code{TRUE} if the coordinates are two-dimensional grid points (spatial models).}

\item{...}{additional arguments for the \code{vario} function}
}
\value{
an \code{n} by \code{d} exact sample from the corresponding multivariate extreme value model
}
\description{
Implementation of the random number generators for multivariate extreme-value distributions
and max-stable processes based on the two algorithms described in
Dombry, Engelke and Oesting (2016).
}
\details{
The vector param differs depending on the model
\itemize{
 \item \code{log}: one dimensional parameter greater than 1
 \item \code{alog}: \eqn{2^d-d-1} dimensional parameter for \code{dep}. Values are recycled if needed.
 \item \code{neglog}: one dimensional positive parameter
 \item \code{aneglog}: \eqn{2^d-d-1} dimensional parameter for \code{dep}. Values are recycled if needed.
 \item \code{bilog}: \code{d}-dimensional vector of parameters in \eqn{[0,1]}
 \item \code{negbilog}: \code{d}-dimensional vector of negative parameters
 \item \code{ct, dir, negdir, sdir}: \code{d}-dimensional vector of positive (a)symmetry parameters. For \code{dir} and \code{negdir}, a \eqn{d+1}
 vector consisting of the \code{d} Dirichlet parameters and the last entry is an index of regular variation in \eqn{(-\min(\alpha_1, \ldots, \alpha_d), 1]} treated as shape parameter
 \item \code{xstud}: one dimensional parameter corresponding to degrees of freedom \code{alpha}
 \item \code{dirmix}: \code{d} by \code{m}-dimensional matrix of positive (a)symmetry parameters
}

Stephenson points out that the multivariate asymmetric negative logistic model given in e.g. Coles and Tawn (1991) is not a valid distribution function in dimension \eqn{d>3} unless additional constraints are imposed on the parameter values.
The implementation in \code{mev} uses the same construction as the asymmetric logistic distribution (see the vignette). As such it does not match the bivariate implementation of \link[evd]{rbvevd}.

The dependence parameter of the \code{evd} package for the Husler-Reiss distribution can be recovered taking
for the Brown--Resnick model  \eqn{2/r=\sqrt(2\gamma(h))} where \eqn{h} is the lag vector between sites and \eqn{r=1/\lambda} for the Husler--Reiss.
}
\section{Warning}{

As of version 1.8 (August 16, 2016), there is a distinction between models \code{hr} and \code{br}. The latter is meant to be used in conjunction with variograms. The parametrization differs between the two models.


The family of scaled Dirichlet is now parametrized by a parameter in \eqn{-\min(\alpha)}{-min(\alpha)} appended to the the \code{d} vector \code{param} containing the parameter \code{alpha}
of the Dirichlet model. Arguments \code{model='dir'} and \code{model='negdir'} are still supported internally, but not listed in the options.
}

\examples{
set.seed(1)
rmev(n=100, d=3, param=2.5, model='log', alg='ef')
rmev(n=100, d=4, param=c(0.2,0.1,0.9,0.5), model='bilog', alg='sm')
## Spatial example using power variogram
#NEW: Semi-variogram must take distance as argument
semivario <- function(x, scale, alpha){ scale*x^alpha }
#grid specification
grid.coord <- as.matrix(expand.grid(runif(4), runif(4)))
rmev(n=100, vario=semivario,coord=grid.coord, model='br', scale = 0.5, alpha = 1)
vario2cov <- function(coord, semivario,...){
 sapply(1:nrow(coord), function(i) sapply(1:nrow(coord), function(j)
  semivario(sqrt(sum((coord[i,])^2)), ...) +
  semivario(sqrt(sum((coord[j,])^2)), ...) -
  semivario(sqrt(sum((coord[i,]-coord[j,])^2)), ...)))
}
rmev(n=100, sigma=vario2cov(grid.coord, semivario = semivario, scale = 0.5, alpha = 1), model='br')
#Example with a grid (generating an array)
rmev(n=10, sigma=cbind(c(2,1), c(1,3)), coord=cbind(runif(4), runif(4)), model='smith', grid=TRUE)
## Example with Dirichlet mixture
alpha.mat <- cbind(c(2,1,1),c(1,2,1),c(1,1,2))
rmev(n=100, param=alpha.mat, weights=rep(1/3,3), model='dirmix')
}
\references{
Dombry, Engelke and Oesting (2016). Exact simulation of max-stable processes, \emph{Biometrika}, \bold{103}(2), 303--317.
}
\seealso{
\link{rmevspec}, \link[evd]{rmvevd}, \link[evd]{rbvevd}
}
\author{
Leo Belzile
}
