% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/map.R
\name{mirai_map}
\alias{mirai_map}
\title{mirai Map}
\usage{
mirai_map(.x, .f, ..., .args = list(), .promise = NULL, .compute = "default")
}
\arguments{
\item{.x}{a list or atomic vector. Also accepts a matrix or dataframe, in
which case multiple map is performed over its rows.}

\item{.f}{a function to be applied to each element of \code{.x}, or row of
\code{.x} as the case may be.}

\item{...}{(optional) named arguments (name = value pairs) specifying objects
referenced, but not defined, in \code{.f}.}

\item{.args}{(optional) further constant arguments to \code{.f}, provided as
a list.}

\item{.promise}{(optional) if supplied, registers a promise against each
mirai. Either a function, supplied to the \sQuote{onFulfilled} argument of
\code{promises::then()} or a list of 2 functions, supplied respectively to
\sQuote{onFulfilled} and \sQuote{onRejected} for \code{promises::then()}.
Using this argument requires the \CRANpkg{promises} package.}

\item{.compute}{[default 'default'] character value for the compute profile
to use (each compute profile has its own independent set of daemons).}
}
\value{
A \sQuote{mirai_map} (list of \sQuote{mirai} objects).
}
\description{
Asynchronous parallel map of a function over a list or vector using
\pkg{mirai}, with optional \pkg{promises} integration. Performs multiple map
over the rows of a dataframe or matrix.
}
\details{
Sends each application of function \code{.f} on an element of \code{.x}
(or row of \code{.x}) for computation in a separate \code{\link{mirai}} call.

This simple and transparent behaviour is designed to make full use of
\pkg{mirai} scheduling to minimise overall execution time.

Facilitates recovery from partial failure by returning all
\sQuote{miraiError} / \sQuote{errorValue} as the case may be, thus allowing
only the failures to be re-run.

Note: requires daemons to have previously been set. If not, then one local
daemon is set before the function proceeds.
}
\section{Collection Options}{


\code{x[]} collects the results of a \sQuote{mirai_map} \code{x} and returns
a list. This will wait for all asynchronous operations to complete if still
in progress, blocking but user-interruptible.

\code{x[.flat]} collects and flattens map results to a vector, checking that
they are of the same type to avoid coercion. Note: errors if an
\sQuote{errorValue} has been returned or results are of differing type.

\code{x[.progress]} collects map results whilst showing a progress bar from
the \CRANpkg{cli} package, if installed, with completion percentage and ETA,
or else a simple text progress indicator. Note: if the map operation
completes too quickly then the progress bar may not show at all.

\code{x[.stop]} collects map results applying early stopping, which stops at
the first failure and cancels remaining operations. Note: operations already
in progress continue to completion, although their results are not collected.

The options above may be combined in the manner of: \cr
\code{x[.stop, .progress]} which applies early stopping together with a
progress indicator.
}

\section{Multiple Map}{


If \code{.x} is a matrix or dataframe (or other object with \sQuote{dim}
attributes), \emph{multiple} map is performed over its \strong{rows}.

This allows map over 2 or more arguments, and \code{.f} should accept at
least as many arguments as there are columns. If the dataframe has names, or
the matrix column dimnames, named arguments are provided to \code{.f}.

To map over \strong{columns} instead, first wrap a dataframe in
\code{\link{as.list}}, or transpose a matrix using \code{\link{t}}.
}

\examples{
if (interactive()) {
# Only run examples in interactive R sessions

daemons(4)

# perform and collect mirai map
mm <- mirai_map(c(a = 1, b = 2, c = 3), rnorm)
mm
mm[]

# map with constant args specified via '.args'
mirai_map(1:3, rnorm, .args = list(n = 5, sd = 2))[]

# flatmap with helper function passed via '...'
mirai_map(
  10^(0:9),
  function(x) rnorm(1L, valid(x)),
  valid = function(x) min(max(x, 0L), 100L)
)[.flat]

# unnamed matrix multiple map: arguments passed to function by position
(mat <- matrix(1:4, nrow = 2L))
mirai_map(mat, function(x = 10, y = 0, z = 0) x + y + z)[.flat]

# named matrix multiple map: arguments passed to function by name
dimnames(mat)[[2L]] <- c("y", "z")
mirai_map(mat, function(x = 10, y = 0, z = 0) x + y + z)[.flat]

# dataframe multiple map: using a function taking '...' arguments
df <- data.frame(a = c("Aa", "Bb"), b = c(1L, 4L))
mirai_map(df, function(...) sprintf("\%s: \%d", ...))[.flat]

# indexed map over a vector (using a dataframe)
v <- c("egg", "got", "ten", "nap", "pie")
mirai_map(
  data.frame(1:length(v), v),
  sprintf,
  .args = list(fmt = "\%d_\%s")
)[.flat]

# return a 'mirai_map' object, check for resolution, collect later
mp <- mirai_map(2:4, function(x) runif(1L, x, x + 1))
unresolved(mp)
mp
mp[.flat]
unresolved(mp)

# progress indicator counts up from 0 to 4 seconds
res <- mirai_map(1:4, Sys.sleep)[.progress]

daemons(0)

# generates warning as daemons not set
# stops early when second element returns an error
tryCatch(mirai_map(list(1, "a", 3), sum)[.stop], error = identity)

# promises example that outputs the results, including errors, to the console
if (requireNamespace("promises", quietly = TRUE)) {
daemons(1, dispatcher = FALSE)
ml <- mirai_map(
  1:30,
  function(i) {Sys.sleep(0.1); if (i == 30) stop(i) else i},
  .promise = list(
    function(x) cat(paste(x, "")),
    function(x) { cat(conditionMessage(x), "\n"); daemons(0) }
  )
)
}

}

}
