% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/item.omega.R
\name{item.omega}
\alias{item.omega}
\title{Coefficient Omega, Hierarchical Omega, and Categorical Omega}
\usage{
item.omega(..., data = NULL, rescov = NULL, type = c("omega", "hierarch", "categ"),
           exclude = NULL, std = FALSE, na.omit = FALSE,
           print = c("all", "omega", "item"), digits = 2, conf.level = 0.95,
           as.na = NULL, write = NULL, append = TRUE, check = TRUE,
           output = TRUE)
}
\arguments{
\item{...}{a matrix or data frame. Note that at least three items are
needed for computing omega. Alternatively, an expression
indicating the variable names in \code{data} e.g.,
\code{item.omega(x1, x2, x3, data = dat)}. Note that the operators
\code{.}, \code{+}, \code{-}, \code{~}, \code{:}, \code{::},
and \code{!} can also be used to select variables, see 'Details'
in the \code{\link{df.subset}} function.}

\item{data}{a data frame when specifying one or more variables in the
argument \code{...}. Note that the argument is \code{NULL}
when specifying a matrix or data frame for the argument
\code{...}.}

\item{rescov}{a character vector or a list of character vectors for specifying
residual covariances when computing coefficient omega, e.g.
\code{rescov = c("x1", "x2")} for specifying a residual
covariance between items \code{x1} and \code{x2} or
\code{rescov = list(c("x1", "x2"), c("x3", "x4"))} for specifying
residual covariances between items \code{x1} and \code{x2},
and items \code{x3} and \code{x4}.}

\item{type}{a character string indicating the type of omega to be computed, i.e.,
\code{omega} (default) for coefficient omega, \code{hierarch} for
hierarchical omega, and \code{categ} for categorical omega.}

\item{exclude}{a character vector indicating items to be excluded from the
analysis.}

\item{std}{logical: if \code{TRUE}, the standardized coefficient omega
is computed.}

\item{na.omit}{logical: if \code{TRUE}, incomplete cases are removed before
conducting the analysis (i.e., listwise deletion); if \code{FALSE},
full information maximum likelihood (FIML) is used for computing
coefficient omega or hierarchical omega, while pairwise deletion
is used for computing categorical omega.}

\item{print}{a character vector indicating which results to show, i.e.
\code{"all"} (default), for all results \code{"omega"} for omega,
 and \code{"item"} for item statistics.}

\item{digits}{an integer value indicating the number of decimal places to be
used for displaying omega and standardized factor loadings.}

\item{conf.level}{a numeric value between 0 and 1 indicating the confidence level
of the interval.}

\item{as.na}{a numeric vector indicating user-defined missing values,
i.e. these values are converted to \code{NA} before conducting
the analysis.}

\item{write}{a character string naming a file for writing the output into
either a text file with file extension \code{".txt"} (e.g.,
\code{"Output.txt"}) or Excel file with file extension
\code{".xlsx"}  (e.g., \code{"Output.xlsx"}). If the file
name does not contain any file extension, an Excel file will
be written.}

\item{append}{logical: if \code{TRUE} (default), output will be appended
to an existing text file with extension \code{.txt} specified
in \code{write}, if \code{FALSE} existing text file will be
overwritten.}

\item{check}{logical: if \code{TRUE} (default), argument specification is checked.}

\item{output}{logical: if \code{TRUE} (default), output is shown.}
}
\value{
Returns an object of class \code{misty.object}, which is a list with following
entries:
  \item{\code{call}}{function call}
  \item{\code{type}}{type of analysis}
  \item{\code{data}}{data frame used for the current analysis}
  \item{\code{args}}{specification of function arguments}
  \item{\code{model.fit}}{fitted lavaan object (\code{mod.fit})}
  \item{\code{result}}{list with result tables, i.e., \code{alpha} for a table
                       with coefficient omega and \code{itemstat} for a table with
                       item statistics}
}
\description{
This function computes point estimate and confidence interval for the coefficient
omega (McDonald, 1978), hierarchical omega (Kelley & Pornprasertmanit, 2016),
and categorical omega (Green & Yang, 2009) along with standardized factor loadings
and omega if item deleted.
}
\details{
Omega is computed by estimating a confirmatory factor analysis model using the
\code{cfa()} function in the \pkg{lavaan} package by Yves Rosseel (2019). Maximum
likelihood (\code{"ML"}) estimator is used for computing coefficient omega and
hierarchical omega, while diagonally weighted least squares estimator (\code{"DWLS"})
is used for computing categorical omega.

Approximate confidence intervals are computed using the procedure by Feldt, Woodruff
and Salih (1987). Note that there are at least 10 other procedures for computing
the confidence interval (see Kelley and Pornprasertmanit, 2016), which are implemented
in the \code{ci.reliability()} function in the \pkg{MBESSS} package by Ken Kelley (2019).
}
\note{
Computation of the hierarchical and categorical omega is based on
the \code{ci.reliability()} function in the \pkg{MBESS} package by Ken Kelley
(2019).
}
\examples{
dat <- data.frame(item1 = c(5, 2, 3, 4, 1, 2, 4, 2),
                  item2 = c(5, 3, 3, 5, 2, 2, 5, 1),
                  item3 = c(4, 2, 4, 5, 1, 3, 5, 1),
                  item4 = c(5, 1, 2, 5, 2, 3, 4, 2))

# Example 1a: Compute unstandardized coefficient omega and item statistics
item.omega(dat)

# Example 1b: Alternative specification using the 'data' argument
item.omega(., data = dat)

# Example 2: Compute unstandardized coefficient omega with a residual covariance
# and item statistics
item.omega(dat, rescov = c("item1", "item2"))

# Example 3: Compute unstandardized coefficient omega with residual covariances
# and item statistics
item.omega(dat, rescov = list(c("item1", "item2"), c("item1", "item3")))

# Example 4: Compute unstandardized hierarchical omega and item statistics
item.omega(dat, type = "hierarch")

# Example 5: Compute categorical omega and item statistics
item.omega(dat, type = "categ")

# Example 6: Compute standardized coefficient omega and item statistics
item.omega(dat, std = TRUE)

# Example 7: Compute unstandardized coefficient omega
item.omega(dat, print = "omega")

# Example 8: Compute item statistics
item.omega(dat, print = "item")

# Example 9: Compute unstandardized coefficient omega and item statistics while excluding item3
item.omega(dat, exclude = "item3")

# Example 10: Summary of the CFA model used to compute coefficient omega
lavaan::summary(item.omega(dat, output = FALSE)$model.fit,
                fit.measures = TRUE, standardized = TRUE)

\dontrun{
# Example 11a: Write results into a text file
item.omega(dat, write = "Omega.txt")

# Example 11b: Write results into a Excel file
item.omega(dat, write = "Omega.xlsx")}
}
\references{
Feldt, L. S., Woodruff, D. J., & Salih, F. A. (1987). Statistical inference for
coefficient alpha. \emph{Applied Psychological Measurement}, 11 93-103.

Green, S. B., & Yang, Y. (2009). Reliability of summed item scores using structural
equation modeling: An alternative to coefficient alpha. \emph{Psychometrika, 74},
155-167. https://doi.org/10.1007/s11336-008-9099-3

Kelley, K., & Pornprasertmanit, S. (2016). Confidence intervals for population
reliability coefficients: Evaluation of methods, recommendations, and software
for composite measures. \emph{Psychological Methods, 21}, 69-92.
http://dx.doi.org/10.1037/a0040086

Ken Kelley (2019). \emph{MBESS: The MBESS R Package}. R package version 4.6.0.
https://CRAN.R-project.org/package=MBESS

McDonald, R. P. (1978). Generalizability in factorable domains: Domain validity
and generalizability. \emph{Educational and Psychological Measurement, 38}, 75-79.
}
\seealso{
\code{\link{write.result}}, \code{\link{item.alpha}}, \code{\link{item.cfa}},
\code{\link{item.reverse}}, \code{\link{item.scores}}
}
\author{
Takuya Yanagida \email{takuya.yanagida@univie.ac.at}
}
