% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PipeOpTorch.R
\name{mlr_pipeops_torch}
\alias{mlr_pipeops_torch}
\alias{PipeOpTorch}
\title{Base Class for Torch Module Constructor Wrappers}
\description{
\code{PipeOpTorch} is the base class for all \code{\link[mlr3pipelines:PipeOp]{PipeOp}}s that represent
neural network layers in a \code{\link[mlr3pipelines:Graph]{Graph}}.
During \strong{training}, it generates a \code{\link{PipeOpModule}} that wraps an \code{\link[torch:nn_module]{nn_module}} and attaches it
to the architecture, which is also represented as a \code{\link[mlr3pipelines:Graph]{Graph}} consisting mostly of \code{\link{PipeOpModule}}s
an \code{\link[mlr3pipelines:mlr_pipeops_nop]{PipeOpNOP}}s.

While the former \code{\link[mlr3pipelines:Graph]{Graph}} operates on \code{\link{ModelDescriptor}}s, the latter operates on \link[torch:torch_tensor]{tensors}.

The relationship between a \code{PipeOpTorch} and a \code{\link{PipeOpModule}} is similar to the
relationshop between a \code{nn_module_generator} (like \code{\link[torch:nn_linear]{nn_linear}}) and a
\code{\link[torch:nn_module]{nn_module}} (like the output of \code{nn_linear(...)}).
A crucial difference is that the \code{PipeOpTorch} infers auxiliary parameters (like \code{in_features} for
\code{nn_linear}) automatically from the intermediate tensor shapes that are being communicated through the
\code{\link{ModelDescriptor}}.

During \strong{prediction}, \code{PipeOpTorch} takes in a \code{\link[mlr3:Task]{Task}} in each channel and outputs the same new
\code{\link[mlr3:Task]{Task}} resulting from their \link[mlr3pipelines:mlr_pipeops_featureunion]{feature union} in each channel.
If there is only one input and output channel, the task is simply piped through.
}
\section{Parameters}{

The \code{\link[paradox:ParamSet]{ParamSet}} is specified by the child class inheriting from \code{\link{PipeOpTorch}}.
Usually the parameters are the arguments of the wrapped \code{\link[torch:nn_module]{nn_module}} minus the auxiliary parameter that can
be automatically inferred from the shapes of the input tensors.
}

\section{Inheriting}{

When inheriting from this class, one should overload either the \code{private$.shapes_out()} and the
\code{private$.shape_dependent_params()} methods, or overload \code{private$.make_module()}.
\itemize{
\item \code{.make_module(shapes_in, param_vals, task)}\cr
(\code{list()}, \code{list()}) -> \code{nn_module}\cr
This private method is called to generate the \code{nn_module} that is passed as argument \code{module} to
\code{\link{PipeOpModule}}. It must be overwritten, when no \code{module_generator} is provided.
If left as is, it calls the provided \code{module_generator} with the arguments obtained by
the private method \code{.shape_dependent_params()}.
\item \code{.shapes_out(shapes_in, param_vals, task)}\cr
(\code{list()}, \code{list()}, \code{\link[mlr3:Task]{Task}} or \code{NULL}) -> named \code{list()}\cr
This private method gets a list of \code{integer} vectors (\code{shapes_in}), the parameter values (\code{param_vals}),
as well as an (optional) \code{\link[mlr3:Task]{Task}}.
The \code{shapes_in} can be assumed to be in the same order as the input names of the \code{PipeOp}.
The output shapes must be in the same order as the output names of the \code{PipeOp}.
In case the output shapes depends on the task (as is the case for \code{\link{PipeOpTorchHead}}), the function should return
valid output shapes (possibly containing \code{NA}s) if the \code{task} argument is provided or not.
It is important to properly handle the presence of \code{NA}s in the input shapes.
By default (if construction argument \code{only_batch_unknown} is \code{TRUE}), only the batch dimension can be \code{NA}.
If you set this to \code{FALSE}, you need to take other unknown dimensions into account.
The method can also throw an error if the input shapes violate some assumptions.
\item \code{.shape_dependent_params(shapes_in, param_vals, task)}\cr
(\code{list()}, \code{list()}) -> named \code{list()}\cr
This private method has the same inputs as \code{.shapes_out}.
If \code{.make_module()} is not overwritten, it constructs the arguments passed to \code{module_generator}.
Usually this means that it must infer the auxiliary parameters that can be inferred from the input shapes
and add it to the user-supplied parameter values (\code{param_vals}).
}
}

\section{Input and Output Channels}{

During \emph{training}, all inputs and outputs are of class \code{\link{ModelDescriptor}}.
During \emph{prediction}, all input and output channels are of class \code{\link[mlr3:Task]{Task}}.
}

\section{State}{

The state is the value calculated by the public method \code{shapes_out()}.
}

\section{Internals}{

During training, the \code{PipeOpTorch} creates a \code{\link{PipeOpModule}} for the given parameter specification and the
input shapes from the incoming \code{\link{ModelDescriptor}}s using the private method \code{.make_module()}.
The input shapes are provided by the slot \code{pointer_shape} of the incoming \code{\link{ModelDescriptor}}s.
The channel names of this \code{\link{PipeOpModule}} are identical to the channel names of the generating \code{\link{PipeOpTorch}}.

A \link[=model_descriptor_union]{model descriptor union} of all incoming \code{\link{ModelDescriptor}}s is then created.
Note that this modifies the \code{\link[mlr3pipelines:Graph]{graph}} of the first \code{\link{ModelDescriptor}} \strong{in place} for efficiency.
The \code{\link{PipeOpModule}} is added to the \code{\link[mlr3pipelines:Graph]{graph}} slot of this union and the the edges that connect the
sending \code{PipeOpModule}s to the input channel of this \code{PipeOpModule} are addeded to the graph.
This is possible because every incoming \code{\link{ModelDescriptor}} contains the information about the
\code{id} and the \code{channel} name of the sending \code{PipeOp} in the slot \code{pointer}.

The new graph in the \code{\link{model_descriptor_union}} represents the current state of the neural network
architecture. It is structurally similar to the subgraph that consists of all pipeops of class \code{PipeOpTorch} and
\code{\link{PipeOpTorchIngress}} that are ancestors of this \code{PipeOpTorch}.

For the output, a shallow copy of the \code{\link{ModelDescriptor}} is created and the \code{pointer} and
\code{pointer_shape} are updated accordingly. The shallow copy means that all \code{\link{ModelDescriptor}}s point to the same
\code{\link[mlr3pipelines:Graph]{Graph}} which allows the graph to be modified by-reference in different parts of the code.
}

\examples{
\dontshow{if (torch::torch_is_installed()) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
## Creating a neural network
# In torch

task = tsk("iris")

network_generator = torch::nn_module(
  initialize = function(task, d_hidden) {
    d_in = length(task$feature_names)
    self$linear = torch::nn_linear(d_in, d_hidden)
    self$output = if (task$task_type == "regr") {
      torch::nn_linear(d_hidden, 1)
    } else if (task$task_type == "classif") {
      torch::nn_linear(d_hidden, output_dim_for(task))
    }
  },
  forward = function(x) {
    x = self$linear(x)
    x = torch::nnf_relu(x)
    self$output(x)
  }
)

network = network_generator(task, d_hidden = 50)
x = torch::torch_tensor(as.matrix(task$data(1, task$feature_names)))
y = torch::with_no_grad(network(x))


# In mlr3torch
network_generator = po("torch_ingress_num") \%>>\%
  po("nn_linear", out_features = 50) \%>>\%
  po("nn_head")
md = network_generator$train(task)[[1L]]
network = model_descriptor_to_module(md)
y = torch::with_no_grad(network(torch_ingress_num.input = x))



## Implementing a custom PipeOpTorch

# defining a custom module
nn_custom = nn_module("nn_custom",
  initialize = function(d_in1, d_in2, d_out1, d_out2, bias = TRUE) {
    self$linear1 = nn_linear(d_in1, d_out1, bias)
    self$linear2 = nn_linear(d_in2, d_out2, bias)
  },
  forward = function(input1, input2) {
    output1 = self$linear1(input1)
    output2 = self$linear1(input2)

    list(output1 = output1, output2 = output2)
  }
)

# wrapping the module into a custom PipeOpTorch

library(paradox)

PipeOpTorchCustom = R6::R6Class("PipeOpTorchCustom",
  inherit = PipeOpTorch,
  public = list(
    initialize = function(id = "nn_custom", param_vals = list()) {
      param_set = ps(
        d_out1 = p_int(lower = 1, tags = c("required", "train")),
        d_out2 = p_int(lower = 1, tags = c("required", "train")),
        bias = p_lgl(default = TRUE, tags = "train")
      )
      super$initialize(
        id = id,
        param_vals = param_vals,
        param_set = param_set,
        inname = c("input1", "input2"),
        outname = c("output1", "output2"),
        module_generator = nn_custom
      )
    }
  ),
  private = list(
    .shape_dependent_params = function(shapes_in, param_vals, task) {
      c(param_vals,
        list(d_in1 = tail(shapes_in[["input1"]], 1)), d_in2 = tail(shapes_in[["input2"]], 1)
      )
    },
    .shapes_out = function(shapes_in, param_vals, task) {
      list(
        input1 = c(head(shapes_in[["input1"]], -1), param_vals$d_out1),
        input2 = c(head(shapes_in[["input2"]], -1), param_vals$d_out2)
      )
    }
  )
)

## Training

# generate input
task = tsk("iris")
task1 = task$clone()$select(paste0("Sepal.", c("Length", "Width")))
task2 = task$clone()$select(paste0("Petal.", c("Length", "Width")))
graph = gunion(list(po("torch_ingress_num_1"), po("torch_ingress_num_2")))
mds_in = graph$train(list(task1, task2), single_input = FALSE)

mds_in[[1L]][c("graph", "task", "ingress", "pointer", "pointer_shape")]
mds_in[[2L]][c("graph", "task", "ingress", "pointer", "pointer_shape")]

# creating the PipeOpTorch and training it
po_torch = PipeOpTorchCustom$new()
po_torch$param_set$values = list(d_out1 = 10, d_out2 = 20)
train_input = list(input1 = mds_in[[1L]], input2 = mds_in[[2L]])
mds_out = do.call(po_torch$train, args = list(input = train_input))
po_torch$state

# the new model descriptors

# the resulting graphs are identical
identical(mds_out[[1L]]$graph, mds_out[[2L]]$graph)
# note that as a side-effect, also one of the input graphs is modified in-place for efficiency
mds_in[[1L]]$graph$edges

# The new task has both Sepal and Petal features
identical(mds_out[[1L]]$task, mds_out[[2L]]$task)
mds_out[[2L]]$task

# The new ingress slot contains all ingressors
identical(mds_out[[1L]]$ingress, mds_out[[2L]]$ingress)
mds_out[[1L]]$ingress

# The pointer and pointer_shape slots are different
mds_out[[1L]]$pointer
mds_out[[2L]]$pointer

mds_out[[1L]]$pointer_shape
mds_out[[2L]]$pointer_shape

## Prediction
predict_input = list(input1 = task1, input2 = task2)
tasks_out = do.call(po_torch$predict, args = list(input = predict_input))
identical(tasks_out[[1L]], tasks_out[[2L]])
\dontshow{\}) # examplesIf}
}
\seealso{
Other Graph Network: 
\code{\link{ModelDescriptor}()},
\code{\link{TorchIngressToken}()},
\code{\link{mlr_learners_torch_model}},
\code{\link{mlr_pipeops_module}},
\code{\link{mlr_pipeops_torch_ingress}},
\code{\link{mlr_pipeops_torch_ingress_categ}},
\code{\link{mlr_pipeops_torch_ingress_ltnsr}},
\code{\link{mlr_pipeops_torch_ingress_num}},
\code{\link{model_descriptor_to_learner}()},
\code{\link{model_descriptor_to_module}()},
\code{\link{model_descriptor_union}()},
\code{\link{nn_graph}()}
}
\concept{Graph Network}
\section{Super class}{
\code{\link[mlr3pipelines:PipeOp]{mlr3pipelines::PipeOp}} -> \code{PipeOpTorch}
}
\section{Public fields}{
\if{html}{\out{<div class="r6-fields">}}
\describe{
\item{\code{module_generator}}{(\code{nn_module_generator} or \code{NULL})\cr
The module generator wrapped by this \code{PipeOpTorch}. If \code{NULL}, the private method
\code{private$.make_module(shapes_in, param_vals)} must be overwritte, see section 'Inheriting'.
Do not change this after construction.}
}
\if{html}{\out{</div>}}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-PipeOpTorch-new}{\code{PipeOpTorch$new()}}
\item \href{#method-PipeOpTorch-shapes_out}{\code{PipeOpTorch$shapes_out()}}
\item \href{#method-PipeOpTorch-clone}{\code{PipeOpTorch$clone()}}
}
}
\if{html}{\out{
<details open><summary>Inherited methods</summary>
<ul>
<li><span class="pkg-link" data-pkg="mlr3pipelines" data-topic="PipeOp" data-id="help"><a href='../../mlr3pipelines/html/PipeOp.html#method-PipeOp-help'><code>mlr3pipelines::PipeOp$help()</code></a></span></li>
<li><span class="pkg-link" data-pkg="mlr3pipelines" data-topic="PipeOp" data-id="predict"><a href='../../mlr3pipelines/html/PipeOp.html#method-PipeOp-predict'><code>mlr3pipelines::PipeOp$predict()</code></a></span></li>
<li><span class="pkg-link" data-pkg="mlr3pipelines" data-topic="PipeOp" data-id="print"><a href='../../mlr3pipelines/html/PipeOp.html#method-PipeOp-print'><code>mlr3pipelines::PipeOp$print()</code></a></span></li>
<li><span class="pkg-link" data-pkg="mlr3pipelines" data-topic="PipeOp" data-id="train"><a href='../../mlr3pipelines/html/PipeOp.html#method-PipeOp-train'><code>mlr3pipelines::PipeOp$train()</code></a></span></li>
</ul>
</details>
}}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-PipeOpTorch-new"></a>}}
\if{latex}{\out{\hypertarget{method-PipeOpTorch-new}{}}}
\subsection{Method \code{new()}}{
Creates a new instance of this \link[R6:R6Class]{R6} class.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{PipeOpTorch$new(
  id,
  module_generator,
  param_set = ps(),
  param_vals = list(),
  inname = "input",
  outname = "output",
  packages = "torch",
  tags = NULL,
  only_batch_unknown = TRUE
)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{id}}{(\code{character(1)})\cr
Identifier of the resulting  object.}

\item{\code{module_generator}}{(\code{nn_module_generator})\cr
The torch module generator.}

\item{\code{param_set}}{(\code{\link[paradox:ParamSet]{ParamSet}})\cr
The parameter set.}

\item{\code{param_vals}}{(\code{list()})\cr
List of hyperparameter settings, overwriting the hyperparameter settings that would
otherwise be set during construction.}

\item{\code{inname}}{(\code{character()})\cr
The names of the \code{\link[mlr3pipelines:PipeOp]{PipeOp}}'s input channels. These will be the input channels of the generated \code{\link{PipeOpModule}}.
Unless the wrapped \code{module_generator}'s forward method (if present) has the argument \code{...}, \code{inname} must be
identical to those argument names in order to avoid any ambiguity.\cr
If the forward method has the argument \code{...}, the order of the input channels determines how the tensors
will be passed to the wrapped \code{nn_module}.\cr
If left as \code{NULL} (default), the argument \code{module_generator} must be given and the argument names of the
\code{modue_generator}'s forward function are set as \code{inname}.}

\item{\code{outname}}{(\code{character()}) \cr
The names of the output channels channels. These will be the ouput channels of the generated \code{\link{PipeOpModule}}
and therefore also the names of the list returned by its \verb{$train()}.
In case there is more than one output channel, the \code{nn_module} that is constructed by this
\code{\link[mlr3pipelines:PipeOp]{PipeOp}} during training must return a named \code{list()}, where the names of the list are the
names out the output channels. The default is \code{"output"}.}

\item{\code{packages}}{(\code{character()})\cr
The R packages this object depends on.}

\item{\code{tags}}{(\code{character()})\cr
The tags of the \code{\link[mlr3pipelines:PipeOp]{PipeOp}}. The tags \code{"torch"} is always added.}

\item{\code{only_batch_unknown}}{(\code{logical(1)})\cr
Whether only the batch dimension can be missing in the input shapes or whether other
dimensions can also be unknown.
Default is \code{TRUE}.}
}
\if{html}{\out{</div>}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-PipeOpTorch-shapes_out"></a>}}
\if{latex}{\out{\hypertarget{method-PipeOpTorch-shapes_out}{}}}
\subsection{Method \code{shapes_out()}}{
Calculates the output shapes for the given input shapes, parameters and task.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{PipeOpTorch$shapes_out(shapes_in, task = NULL)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{shapes_in}}{(\code{list()} of \code{integer()})\cr
The input input shapes, which must be in the same order as the input channel names of the \code{PipeOp}.}

\item{\code{task}}{(\code{\link[mlr3:Task]{Task}} or \code{NULL})\cr
The task, which is very rarely used (default is \code{NULL}). An exception is \code{\link{PipeOpTorchHead}}.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A named \code{list()} containing the output shapes. The names are the names of the output channels of
the \code{PipeOp}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-PipeOpTorch-clone"></a>}}
\if{latex}{\out{\hypertarget{method-PipeOpTorch-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{PipeOpTorch$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
