% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/modelsummary.R
\name{msummary}
\alias{msummary}
\title{Beautiful, customizable summaries of statistical models}
\usage{
msummary(
  models,
  statistic = "std.error",
  statistic_override = NULL,
  statistic_vertical = TRUE,
  conf_level = 0.95,
  coef_map = NULL,
  coef_omit = NULL,
  gof_map = modelsummary::gof_map,
  gof_omit = NULL,
  fmt = "\%.3f",
  stars = FALSE,
  stars_note = TRUE,
  title = NULL,
  subtitle = NULL,
  notes = NULL,
  add_rows = NULL,
  add_rows_location = NULL,
  filename = NULL
)
}
\arguments{
\item{models}{a single model object or a (potentially named) list of models
to summarize}

\item{statistic}{string name of the statistic to include in parentheses
below estimates. Must be either "conf.int", or one of the column names
produced by the `broom::tidy` function. Typical values include: "std.error",
"conf.int", "statistic", "p.value". A character vector will stack several
uncertainty estimates on top of one another (in different rows).}

\item{statistic_override}{manually override the uncertainy estimates. This 
argument accepts three types of input:
\itemize{
  \item a function or list of functions of length(models) which produce variance-covariance matrices with row and column names equal to the names of your coefficient estimates. For example, `R` supplies the `vcov` function, and the `sandwich` package supplies `vcovHC`, `vcovHAC`, etc.
  \item a list of length(models) variance-covariance matrices with row and column names equal to the names of your coefficient estimates.
  \item a list of length(models) numeric vectors with names equal to the names of your coefficient estimates.
}}

\item{statistic_vertical}{TRUE if statistics should be printed below
estimates. FALSE if statistics should be printed beside estimates.}

\item{conf_level}{confidence level to use for confidence intervals}

\item{coef_map}{named character vector. Names refer to the original variable
names. Values refer to the variable names that will appear in the table.
Coefficients which are omitted from this vector will be omitted from the
table. The table will be ordered in the same order as this vector.}

\item{coef_omit}{string regular expression. Omits all matching coefficients
from the table (using `stringr::str_detect`).}

\item{gof_map}{data.frame with four columns: `raw`, `clean`, `fmt`, and
`omit`. See `modelsummary::gof_map`}

\item{gof_omit}{string regular expression. Omits all matching gof statistics from
the table (using `stringr::str_detect`).}

\item{fmt}{string which specifies how numeric values will be rounded. This
string is passed to the `sprintf` function. '\%.3f' will keep 3 digits after
the decimal point with trailing zero. '\%.5f' will keep 5 digits. '\%.3e' will
use exponential notation. See `?sprintf` for more options.}

\item{stars}{FALSE for no significance stars. TRUE for default significance
stars (*=.1, **=.05, ***=.01). Named numeric vector for custom significance
stars. For example, `c('*' = .1, '+' = .05)`}

\item{stars_note}{logical include a note at the bottom of the table to describe
the contents of the `stars` argument. The note will be omitted if `stars==NULL`}

\item{title}{string}

\item{subtitle}{string}

\item{notes}{list of notes to append to the bottom of the table.}

\item{add_rows}{list of character vectors, each of length equal to the number
of models + 1.}

\item{add_rows_location}{integer or NULL. custom rows will be added to the
bottom of the table if this parameter is NULL, or after the position set by
this integer.}

\item{filename}{the file name to create on disk. Ensure that an extension
compatible with the output types is provided (`.html`, `.tex`, `.ltx`,
`.rtf`). Read `?gt::gtsave` for further details. When the table produced by
`modelsummary` is post-processed by another `gt` function, users need to use
the `gtsave` function from the `gt` package; using the `filename` argument
will produce an error.}
}
\value{
a 'gt' table object.
}
\description{
Beautiful, customizable summaries of statistical models
}
\examples{
\donttest{
# load data and estimate models
data(trees)
models <- list()
models[['Bivariate']] <- lm(Girth ~ Height, data = trees)
models[['Multivariate']] <- lm(Girth ~ Height + Volume, data = trees)

# simple table
msummary(models)

# confidence intervals, p values, or t-stats instead of standard errors
msummary(models, statistic = 'conf.int', conf_level = 0.99)
msummary(models, statistic = 'p.value', conf_level = 0.99)
msummary(models, statistic = 'statistic', conf_level = 0.99)

# rename and re-order coefficients
msummary(models, coef_map = c('Volume' = 'Large', 'Height' = 'Tall'))

# titles and subtitles
msummary(models, title = 'This is the title', subtitle = 'And a subtitle')

# title with italicized text
msummary(models, title = gt::md('This is *the* title'))

# notes at the bottom of the table (here, the second note includes markdown bold characters)
msummary(models, notes = list('A first note', gt::md('A **bold** note')))
}


}
