% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fitHMM.R
\name{fitHMM}
\alias{fitHMM}
\title{Fit an HMM to the data}
\usage{
fitHMM(data, nbStates, stepPar0, anglePar0, beta0 = NULL, delta0 = NULL,
  formula = ~1, stepDist = c("gamma", "weibull", "lnorm", "exp"),
  angleDist = c("vm", "wrpcauchy", "none"), angleMean = NULL,
  stationary = FALSE, verbose = 0, nlmPar = NULL, fit = TRUE)
}
\arguments{
\item{data}{An object \code{moveData}.}

\item{nbStates}{Number of states of the HMM.}

\item{stepPar0}{Vector of initial state-dependent step length distribution parameters.
The parameters should be in the order expected by the pdf of \code{stepDist}, and the zero-mass
parameter should be the last. Note that zero-mass parameters are mandatory if there are steps of
length zero in the data.
For example, for a 2-state model using the gamma distribution and
including zero-inflation, the vector of initial parameters would be something like:
\code{c(mu1,mu2,sigma1,sigma2,zeromass1,zeromass2)}.}

\item{anglePar0}{Vector of initial state-dependent turning angle distribution parameters.
The parameters should be in the order expected by the pdf of \code{angleDist}. For example, for a 2-state
model using the Von Mises (vm) distribution, the vector of initial parameters would be something like:
\code{c(mu1,mu2,kappa1,kappa2)}.}

\item{beta0}{Initial matrix of regression coefficients for the transition probabilities (more
information in "Details").
Default: \code{NULL}. If not specified, \code{beta0} is initialized such that the diagonal elements
of the transition probability matrix are dominant.}

\item{delta0}{Initial value for the initial distribution of the HMM. Default: \code{rep(1/nbStates,nbStates)}.}

\item{formula}{Regression formula for the covariates. Default: \code{~1} (no covariate effect).}

\item{stepDist}{Name of the distribution of the step lengths (as a character string).
Supported distributions are: gamma, weibull, lnorm, exp. Default: gamma.}

\item{angleDist}{Name of the distribution of the turning angles (as a character string).
Supported distributions are: vm, wrpcauchy. Set to \code{"none"} if the angle distribution should
not be estimated. Default: vm.}

\item{angleMean}{Vector of means of turning angles if not estimated (one for each state).
Default: \code{NULL} (the angle mean is estimated).}

\item{stationary}{\code{FALSE} if there are covariates. If \code{TRUE}, the initial distribution is considered
equal to the stationary distribution. Default: \code{FALSE}.}

\item{verbose}{Determines the print level of the optimizer. The default value of 0 means that no
printing occurs, a value of 1 means that the first and last iterations of the optimization are
detailed, and a value of 2 means that each iteration of the optimization is detailed.}

\item{nlmPar}{List of parameters to pass to the optimization function \code{nlm} (which should be either
'\code{gradtol}', '\code{stepmax}', '\code{steptol}', or '\code{iterlim}' -- see \code{nlm}'s documentation
for more detail)}

\item{fit}{\code{TRUE} if an HMM should be fitted to the data, \code{FALSE} otherwise.
If fit=\code{FALSE}, a model is returned with the MLE replaced by the initial parameters given in
input. This option can be used to assess the initial parameters. Default: \code{TRUE}.}
}
\value{
A \code{moveHMM} object, i.e. a list of:
\item{mle}{The maximum likelihood estimates of the parameters of the model (if the numerical algorithm
has indeed identified the global maximum of the likelihood function), which is a list
of: \code{stepPar} (step distribution parameters), \code{anglePar} (angle distribution
parameters), \code{beta} (transition probabilities regression coefficients - more information
in "Details"), and \code{delta} (initial distribution).}
\item{data}{The movement data}
\item{stepDist}{The step length distribution name}
\item{angleDist}{The turning angle distribution name}
\item{mod}{The object returned by the numerical optimizer \code{nlm}}
\item{conditions}{A few conditions used to fit the model (\code{zeroInflation}, \code{estAngleMean},
\code{stationary}, and \code{formula})}
\item{rawCovs}{Raw covariate values, as found in the data (if any). Used in \code{\link{plot.moveHMM}}.}
}
\description{
Fit an hidden Markov model to the data provided, using numerical optimization of the log-likelihood
function.
}
\details{
\itemize{
\item The matrix \code{beta} of regression coefficients for the transition probabilities has
one row for the intercept, plus one row for each covariate, and one column for
each non-diagonal element of the transition probability matrix. For example, in a 3-state
HMM with 2 covariates, the matrix \code{beta} has three rows (intercept + two covariates)
and six columns (six non-diagonal elements in the 3x3 transition probability matrix -
filled in row-wise).
In a covariate-free model (default), \code{beta} has one row, for the intercept.

\item The choice of initial parameters is crucial to fit a model. The algorithm might not find
the global optimum of the likelihood function if the initial parameters are poorly chosen.
}
}
\examples{
### 1. simulate data
# define all the arguments of simData
nbAnimals <- 2
nbStates <- 2
nbCovs <- 2
mu<-c(15,50)
sigma<-c(10,20)
angleMean <- c(pi,0)
kappa <- c(0.7,1.5)
stepPar <- c(mu,sigma)
anglePar <- c(angleMean,kappa)
stepDist <- "gamma"
angleDist <- "vm"
zeroInflation <- FALSE
obsPerAnimal <- c(50,100)

data <- simData(nbAnimals=nbAnimals,nbStates=nbStates,stepDist=stepDist,angleDist=angleDist,
                 stepPar=stepPar,anglePar=anglePar,nbCovs=nbCovs,zeroInflation=zeroInflation,
                 obsPerAnimal=obsPerAnimal)

### 2. fit the model to the simulated data
# define initial values for the parameters
mu0 <- c(20,70)
sigma0 <- c(10,30)
kappa0 <- c(1,1)
stepPar0 <- c(mu0,sigma0) # no zero-inflation, so no zero-mass included
anglePar0 <- kappa0 # the angle mean is not estimated, so only the concentration parameter is needed
formula <- ~cov1+cos(cov2)

m <- fitHMM(data=data,nbStates=nbStates,stepPar0=stepPar0,anglePar0=anglePar0,formula=formula,
              stepDist=stepDist,angleDist=angleDist,angleMean=angleMean)

print(m)

}
\references{
Patterson T.A., Basson M., Bravington M.V., Gunn J.S. 2009.
Classifying movement behaviour in relation to environmental conditions using hidden Markov models.
Journal of Animal Ecology, 78 (6), 1113-1123.

Langrock R., King R., Matthiopoulos J., Thomas L., Fortin D., Morales J.M. 2012.
Flexible and practical modeling of animal telemetry data: hidden Markov models and extensions.
Ecology, 93 (11), 2336-2342.
}

