% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/frames_spatial.R
\name{frames_spatial}
\alias{frames_spatial}
\title{Create frames of spatial movement maps for animation}
\usage{
frames_spatial(
  m,
  r_list = NULL,
  r_times = NULL,
  r_type = "gradient",
  fade_raster = FALSE,
  crop_raster = TRUE,
  map_service = "osm",
  map_type = "streets",
  map_res = 1,
  map_token = NULL,
  map_dir = NULL,
  margin_factor = 1.1,
  equidistant = NULL,
  ext = NULL,
  path_size = 3,
  path_end = "round",
  path_join = "round",
  path_mitre = 10,
  path_arrow = NULL,
  path_colours = NA,
  path_alpha = 1,
  path_fade = FALSE,
  path_legend = TRUE,
  path_legend_title = "Names",
  tail_length = 19,
  tail_size = 1,
  tail_colour = "white",
  trace_show = FALSE,
  trace_colour = "white",
  cross_dateline = FALSE,
  ...,
  verbose = TRUE
)
}
\arguments{
\item{m}{\code{move} or \code{moveStack} of uniform time scale and time lag, e.g. prepared with \code{\link{align_move}} (recommended). May contain a column named \code{colour} to control path colours (see \code{details}).}

\item{r_list}{list of \code{raster} or \code{rasterStack}. Each list element referrs to the times given in \code{r_times}. Use single-layer \code{raster} objects for gradient or discrete data (see \code{r_type}). Use a  \code{rasterStack} containing three bands for RGB imagery (in the order red, green, blue).}

\item{r_times}{list of \code{POSIXct} times. Each list element represents the time of the corresponding element in \code{r_list}. Must be of same length as \code{r_list}.}

\item{r_type}{character, either \code{"gradient"} or \code{"discrete"}. Ignored, if \code{r_list} contains \code{rasterStacks} of three bands, which are treated as RGB.}

\item{fade_raster}{logical, if \code{TRUE}, \code{r_list} is interpolated over time based on \code{r_times}. If \code{FALSE}, \code{r_list} elements are assigned to those frames closest to the equivalent times in \code{r_times}.}

\item{crop_raster}{logical, whether to crop rasters in \code{r_list} to plot extent before plotting or not.}

\item{map_service}{character, either \code{"osm"}, \code{"carto"} or \code{"mapbox"}. Default is \code{"osm"}.}

\item{map_type}{character, a map type, e.g. \code{"streets"}. For a full list of available map types, see \code{\link{get_maptypes}}.}

\item{map_res}{numeric, resolution of base map in range from 0 to 1.}

\item{map_token}{character, mapbox authentification token for mapbox basemaps. Register at \url{https://www.mapbox.com/} to get a mapbox token. Mapbox is free of charge after registration for up to 50.000 map requests per month. Ignored, if \code{map_service = "osm"}.}

\item{map_dir}{character, directory where downloaded basemap tiles can be stored. By default, a temporary directory is used. 
If you use moveVis often for the same area it is recommended to set this argument to a directory persistent throughout sessions (e.g. in your user folder), 
so that baesmap tiles that had been already downloaded by moveVis do not have to be requested again.}

\item{margin_factor}{numeric, factor relative to the extent of \code{m} by which the frame extent should be increased around the movement area. Ignored, if \code{ext} is set.}

\item{equidistant}{logical, whether to make the map extent equidistant (squared) with y and x axis measuring equal distances or not. Especially in polar regions of the globe it might be necessaray to set \code{equidistant} to \code{FALSE} to avoid strong stretches. By default (\code{equidistant = NULL}), equidistant is set automatically to \code{FALSE}, if \code{ext} is set, otherwise \code{TRUE}. Read more in the details.}

\item{ext}{\code{sf bbox} or \code{sp extent} in same CRS as \code{m}, optional. If set, frames are cropped to this extent. If not set, a squared extent around \code{m}, optional with a margin set by \code{margin_factor}, is used (default).}

\item{path_size}{numeric, size of each path.}

\item{path_end}{character, either \code{"round"}, \code{"butt"} or \code{"square"}, indicating the path end style.}

\item{path_join}{character, either \code{"round"}, \code{"mitre"} or \code{"bevel"}, indicating the path join style.}

\item{path_mitre}{numeric, path mitre limit (number greater than 1).}

\item{path_arrow}{arrow, path arrow specification, as created by grid::arrow().}

\item{path_colours}{character, a vector of colours. Must be of same length as number of individual tracks in \code{m} and refers to the order of tracks in \code{m}. If undefined (\code{NA}) and \code{m} contains a column named \code{colour}, colours provided within \code{m} are used (see details). Othwersie, colours are selected randomly per individual track.}

\item{path_alpha}{numeric, defines alpha (transparency) of the path. Value between 0 and 1. Default is 1.}

\item{path_fade}{logical, whether paths should be faded towards the last frame or not. Useful, if \code{trace_show = TRUE} and you want to hold the last frame using \code{end_pause} in \code{\link{animate_frames}}.}

\item{path_legend}{logical, wether to add a path legend from \code{m} or not. Legend tracks and colours will be ordered by the tracks' temporal apperances, not by their order in \code{m}.}

\item{path_legend_title}{character, path legend title. Default is \code{"Names"}.}

\item{tail_length}{numeric, length of tail per movement path.}

\item{tail_size}{numeric, size of the last tail element. Default is 1.}

\item{tail_colour}{character, colour of the last tail element, to which the path colour is faded. Default is "white".}

\item{trace_show}{logical, whether to show the trace of the complete path or not.}

\item{trace_colour}{character, colour of the trace. Default is "white". It is recommended to define the same colours for both \code{trace_colour} and  \code{tail_colour} to enforce an uninterrupted colour transition form the tail to the trace.}

\item{cross_dateline}{logical, whether tracks are crossing the dateline (longitude 180/-180) or not. If \code{TRUE}, frames are expanded towards the side of the dateline that is smaller in space. Applies only if the CRS of \code{m} is not projected (geographical, lon/lat). If \code{FALSE} (default), frames are clipped at the minimum and maximum longitudes and tracks cannot cross.}

\item{...}{Additional arguments customizing the frame background:
\itemize{
    \item \code{alpha}, numeric, background transparency (0-1).
    \item \code{maxpixels}, maximum number of pixels to be plotted per frame. Defaults to 500000. Reduce to decrease detail and increase rendering speeds.
    \item \code{macColorValue}, numeric, only relevant for RGB backgrounds (i.e. if \code{r_type = "RGB"} or if a default base map is used). Maximum colour value (e.g. 255). Defaults to maximum raster value.
}}

\item{verbose}{logical, if \code{TRUE}, messages and progress information are displayed on the console (default).}
}
\value{
List of ggplot2 objects, each representing a single frame.
}
\description{
\code{frames_spatial} creates a list of \code{ggplot2} maps displaying movement. Each object represents a single frame. Each frame can be viewed or modified individually. The returned list of frames can be animated using \code{\link{animate_frames}}.
}
\details{
If argument \code{path_colours} is not defined (set to \code{NA}), path colours can be defined by adding a character column named \code{colour} to \code{m}, containing a colour code or name per row (e.g. \code{"red"}. This way, for example, column \code{colour} for all rows belonging to individual A can be set to \code{"green"}, while column \code{colour} for all rows belonging to individual B can be set to \code{"red"}.
Colours could also be arranged to change through time or by behavioral segments, geographic locations, age, environmental or health parameters etc. If a column name \code{colour} in \code{m} is missing, colours will be selected automatically. Call \code{colours()} to see all available colours in R.

Basemap colour scales can be changed/added using \code{\link{add_colourscale}} or by using \code{ggplot2} commands (see \code{examples}). For continous scales, use \code{r_type = "gradient"}. For discrete scales, use \code{r_type = "discrete"}.

The projection of \code{m} is treated as target projection. Default base maps accessed through a map service will be reprojected into the projection of \code{m}. Thus, depending on the projection of \code{m}, it may happen that map labels are distorted. To get undistorted map labels, reproject \code{m} to the web mercator projection (the default projection of the base maps): \code{spTransform(m, crs("+init=epsg:3857"))}. The \code{ggplot2} coordinate system will be computed based on the projection of \code{m} using \code{coord_sf}. If argument \code{equidistant} is set, the map extent is calculated (thus enlarged into one axis direction) to represent equal surface distances on the x and y axis.
}
\examples{
library(moveVis)
library(move)
library(ggplot2)

data("move_data")
# align movement
m <- align_move(move_data, res = 4, unit = "mins")

\donttest{
# with osm watercolor base map
frames <- frames_spatial(m, map_service = "osm", map_type = "watercolor")
# take a look at one of the frames, e.g. the 100th
frames[[100]]

# make base map a bit transparent
frames <- frames_spatial(m, map_service = "osm", map_type = "watercolor", alpha = 0.5)
frames[[100]] # take a look

# use a larger margin around extent
frames <- frames_spatial(m, map_service = "osm", map_type = "watercolor", alpha = 0.5,
                         margin_factor = 1.8)

# use a extent object as your AOI
ext <- extent(m)
ext@xmin <- ext@xmin - (ext@xmin*0.003)
ext@xmax <- ext@xmax + (ext@xmax*0.003)
frames <- frames_spatial(m, map_service = "osm", map_type = "watercolor", alpha = 0.5,
                         ext = ext)

# alter path appearance (make it longer and bigger)
frames <- frames_spatial(m, map_service = "osm", map_type = "watercolor", alpha = 0.5,
                         path_size = 4, tail_length = 29)
                         
# adjust path colours manually
frames <- frames_spatial(m, map_service = "osm", map_type = "watercolor", alpha = 0.5,
                         path_colours = c("black", "blue", "purple"))

# or do it directly within your moveStack, e.g. like:
m.list <- split(m) # split m into list by individual
m.list <- mapply(x = m.list, y = c("orange", "purple", "darkgreen"), function(x, y){
  x$colour <- y
  return(x)
}) # add colour per individual
m <- moveStack(m.list) # putting it back together into a moveStack
frames <- frames_spatial(m, map_service = "osm", map_type = "watercolor", alpha = 0.5)
# this way, you do not have to assign colours per individual track
# instead, you could assign colours by segment, age, speed or other variables

# get available map types
get_maptypes()

# use mapbox to get a satellite or other map types (register to on mapbox.com to get a token)
# frames <- frames_spatial(m, map_service = "mapbox",
#                          map_token = "your_token_from_your_mapbox_account",
#                          map_type = "satellite")

# if you make a lot of calls to frames_spatial during mutliple sessions, use a map directory
# to save all base maps offline so that you do not have to query the servers each time
# frames <- frames_spatial(m, map_service = "mapbox",
#                          map_token = "your_token_from_your_mapbox_account",
#                          map_type = "satellite",
#                          map_dir = "your/map_directory/")                         

# use your own custom base maps
data("basemap_data")
r_list <- basemap_data[[1]]
r_times <- basemap_data[[2]]

# using gradient data (e.g. NDVI)
frames <- frames_spatial(m, r_list = r_list, r_times = r_times, r_type = "gradient",
                         fade_raster = TRUE)

# using discrete data (e.g. classifications)
# let's make up some classification data with 10 classes
r_list <- lapply(r_list, function(x){
  y <- raster::setValues(x, round(raster::getValues(x)*10))
  return(y)
})
# turn fade_raster to FALSE, since it makes no sense to temporally interpolate discrete classes
frames <- frames_spatial(m, r_list = r_list, r_times = r_times, r_type = "discrete",
                         fade_raster = FALSE)

# then simply animate the frames using animate_frames
# see ?add_colourscale to learn how to change colours of custom base maps
# see all add_ functions on how to customize your frames created with frames_spatial
# or frames_graph
# see ?animate_frames on how to animate your list of frames
}
}
\seealso{
\code{\link{frames_graph}} \code{\link{join_frames}} \code{\link{animate_frames}}
}
\author{
Jakob Schwalb-Willmann
}
