#' Hessian computation for fitted distance detection function model parameters
#' 
#' Computes hessian to be used for variance-covariance matrix.  The hessian is
#' the outer product of the vector of first partials (see pg 62 of Buckland et
#' al 2002).
#' 
#' 
#' @param ddfobj distance sampling object
#' @param TCI TRUE if point independence assumed (only relevant for double
#'   observer survey analysis)
#' @param misc.options width-transect width (W); int.range-integration range
#'   for observations; showit-if TRUE shows values of parameters and
#'   log-likelihood; doeachint-if TRUE doesn't use cgftab and does each
#'   integral; integral.numeric-if TRUE integral is computed numerically rather
#'   than analytically
#' @return variance-covariance matrix of parameters in the detection function
#' @note This is an internal function used by \code{\link{ddf.ds}} to fit
#'   distance sampling detection functions.  It is not intended for the user to
#'   invoke this function but it is documented here for completeness.
#' @author Jeff Laake
#' @seealso \code{\link{flnl}},\code{\link{flt.lnl}},\code{\link{ddf.ds}}
#' @references Buckland et al. 2002
#' @keywords utility
flt.var <- function(ddfobj, TCI, misc.options)
#   flt.var - computes hessian for v-c matrix (see pg 62 of Buckland et al 2002)
#
#   Arguments: see flt for definitions
#
#   Value:  variance-covariance matrix of parameters
#
#   Functions Used:  flt.lnl
#
{
#
  fpar1<-getpar(ddfobj)    
  fpar=fpar1
  parmat=NULL
#
#   Compute first partial (numerically) of log(f(y)) for each observation 
#   for each parameter and store in parmat (n by length(fpar))
#
  for (i in 1:length(fpar))
  {
	deltap=.0001*fpar1[i]
	if(deltap==0)deltap=0.0001
    fpar[i] <- fpar1[i]- deltap
	if(misc.options$point)
		x1=-fpt.lnl(fpar, ddfobj,TCI,misc.options)
	else
	   x1=-flt.lnl(fpar, ddfobj,TCI,misc.options)
    fpar[i] <- fpar1[i]+deltap
	if(misc.options$point)
		x2=-fpt.lnl(fpar, ddfobj,TCI,misc.options)
	else
		x2=-flt.lnl(fpar, ddfobj,TCI,misc.options)
	parmat=cbind(parmat,(x2-x1)/(2*deltap))
  }
#
#    Compute varmat using first partial approach (pg 62 of Buckland et al 2002)
#
  varmat=matrix(0,ncol=length(fpar1),nrow=length(fpar1))

  for(i in 1:length(fpar1))
    for(j in 1:length(fpar1))
      varmat[i,j]=sum(parmat[,i]*parmat[,j]) 
  
  return(varmat)
}
