# Copyright (C) 2022 Hibiki AI Limited <info@hibiki-ai.com>
#
# This file is part of nanonext.
#
# nanonext is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# nanonext is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# nanonext. If not, see <https://www.gnu.org/licenses/>.

# nanonext - Utilities ---------------------------------------------------------

#' NNG Library Version
#'
#' Returns the version of 'libnng' used and whether TLS is supported.
#'
#' @return A character vector of length 2.
#'
#' @section TLS Support:
#'
#'     Where system installations of 'libnng' and 'libmbedtls' development
#'     headers are detected in the same location, it is assumed that NNG was
#'     built with TLS support (using Mbed TLS) and TLS is configured appropriately.
#'
#'     Otherwise, the environment variable \code{Sys.setenv(NANONEXT_TLS=1)} may
#'     be set prior to installation if:
#'
#'     - your system installations of 'libnng' (built with TLS support) and
#'     'libmbedtls' are in different locations; or
#'
#'     - you have a system installation of 'libmbedtls' but not 'libnng' and want
#'     nanonext to download and build a more recent version of 'libnng' than
#'     available in system repositories against this.
#'
#'     Note: this is not applicable to Windows systems.
#'
#' @examples
#' nng_version()
#'
#' @export
#'
nng_version <- function() .Call(rnng_version)

#' Translate Error Codes
#'
#' Translate integer exit code to human readable form. All package functions
#'     return an integer exit code on error rather than the expected return
#'     value. These are classed 'errorValue' and may be checked by the function
#'     \code{\link{is_error_value}}.
#'
#' @param xc integer exit code to translate.
#'
#' @return A character vector.
#'
#' @section Warnings:
#'
#'     A warning is generated every time an 'errorValue' is returned.
#'
#'     \code{\link{nano_init}} may be used to set the value of option 'warn' and
#'     automatically reverts it upon package unload. The default, applied by
#'     calling \code{nano_init()} with no arguments, is 'immediate', which prints
#'     warnings as they occur.
#'
#'     Further options for warnings may be set manually via \code{\link{options}}:
#'
#'     \itemize{
#'
#'     \item{warning.expression} { - an R code expression to be called if a
#'     warning is
#'     generated, replacing the standard message. If non-null it is called
#'     irrespective of the value of option warn.}
#'
#'     \item{warning.length} { - sets the truncation limit in bytes for error and warning
#'     messages. A non-negative integer, with allowed values 100...8170, default
#'     1000.}
#'
#'     \item{nwarnings} { - the limit for the number of warnings kept when warn = 0,
#'     default 50. This will discard messages if called whilst they are being
#'     collected. If you increase this limit, be aware that the current
#'     implementation pre-allocates the equivalent of a named list for them.}
#'     }
#'
#' @examples
#' nng_error(1L)
#'
#' @export
#'
nng_error <- function(xc) .Call(rnng_strerror, xc)

#' Is Nano
#'
#' Is the object an object created by the nanonext package i.e. a nanoSocket,
#'     nanoContext, nanoStream, nanoListener, nanoDialer or nano Object.
#'
#' @param x an object.
#'
#' @return Logical value TRUE or FALSE.
#'
#' @details Note: does not include Aio objects, for which there is a separate
#'     function \code{\link{is_aio}}.
#'
#' @examples
#' s <- socket()
#' is_nano(s)
#' n <- nano()
#' is_nano(n)
#'
#' close(s)
#' n$close()
#'
#' @export
#'
is_nano <- function(x) inherits(x, c("nano", "nanoObject"))

#' Is Aio
#'
#' Is the object an Aio (sendAio or recvAio).
#'
#' @param x an object.
#'
#' @return Logical value TRUE or FALSE.
#'
#' @examples
#' sock <- socket(listen = "inproc://isaio")
#' r <- recv_aio(sock)
#' s <- send_aio(sock, "test")
#'
#' is_aio(r)
#' is_aio(s)
#'
#' close(sock)
#'
#' @export
#'
is_aio <- function(x) inherits(x, c("recvAio", "sendAio"))

#' Is Nul Byte
#'
#' Is the object a nul byte.
#'
#' @param x an object.
#'
#' @return Logical value TRUE or FALSE.
#'
#' @examples
#' is_nul_byte(as.raw(0L))
#' is_nul_byte(raw(length = 1L))
#' is_nul_byte(writeBin("", con = raw()))
#'
#' is_nul_byte(0L)
#' is_nul_byte(NULL)
#' is_nul_byte(NA)
#'
#' @export
#'
is_nul_byte <- function(x) identical(x, as.raw(0L))

#' Is Error Value
#'
#' Is the object an error value generated by NNG. All returned integer error
#'     codes are classed as 'errorValue' to be distinguishable from integer
#'     message values.
#'
#' @param x an object.
#'
#' @return Logical value TRUE if 'x' is of class 'errorValue', FALSE otherwise.
#'
#' @section Warnings:
#'
#'     A warning is generated every time an 'errorValue' is returned.
#'
#'     \code{\link{nano_init}} may be used to set the value of option 'warn' and
#'     automatically reverts it upon package unload. The default, applied by
#'     calling \code{nano_init()} with no arguments, is 'immediate', which prints
#'     warnings as they occur.
#'
#'     Further options for warnings may be set manually via \code{\link{options}}:
#'
#'     \itemize{
#'
#'     \item{warning.expression} { - an R code expression to be called if a
#'     warning is
#'     generated, replacing the standard message. If non-null it is called
#'     irrespective of the value of option warn.}
#'
#'     \item{warning.length} { - sets the truncation limit in bytes for error and warning
#'     messages. A non-negative integer, with allowed values 100...8170, default
#'     1000.}
#'
#'     \item{nwarnings} { - the limit for the number of warnings kept when warn = 0,
#'     default 50. This will discard messages if called whilst they are being
#'     collected. If you increase this limit, be aware that the current
#'     implementation pre-allocates the equivalent of a named list for them.}
#'     }
#'
#' @examples
#' is_error_value(1L)
#'
#' @export
#'
is_error_value <- function(x) inherits(x, "errorValue")

#' nanonext Initialise
#'
#' Initialise global options - intended to be called immediately after package load.
#'
#' @param warn [default 'immediate'] character string defining how to treat
#'     warnings generated by the package. 'immediate' to print warnings as they
#'     occur, 'deferred' to print warnings when evaluation returns to the top
#'     level, 'error' to upgrade all warnings to errors (stops execution), and
#'     'none' to ignore all warnings.
#'
#' @return Invisibly, the integer \code{code} applied to \code{options(warn = code)}.
#'
#' @section Warnings:
#'
#'     A warning is generated every time an 'errorValue' is returned.
#'
#'     This function sets the global option 'warn' to the appropriate value and
#'     automatically reverts it upon package unload. The default, applied by
#'     calling \code{nano_init()} with no arguments, is 'immediate', which
#'     prints warnings as they occur.
#'
#'     Further options for warnings may be set manually via \code{\link{options}}:
#'
#'     \itemize{
#'
#'     \item{warning.expression} { - an R code expression to be called if a
#'     warning is
#'     generated, replacing the standard message. If non-null it is called
#'     irrespective of the value of option warn.}
#'
#'     \item{warning.length} { - sets the truncation limit in bytes for error
#'     and warning messages. A non-negative integer, with allowed values 100...8170,
#'     default 1000.}
#'
#'     \item{nwarnings} { - the limit for the number of warnings kept when warn = 0,
#'     default 50. This will discard messages if called whilst they are being
#'     collected. If you increase this limit, be aware that the current
#'     implementation pre-allocates the equivalent of a named list for them.}
#'     }
#'
#' @export
#'
nano_init <- function(warn = c("immediate", "deferred", "error", "none")) {

  warn <- switch(match.arg2(warn, c("immediate", "deferred", "error", "none")),
                 1L, 0L, 2L, -1L)
  if (is.null(getOption("nanonext.original.warn")))
    options(nanonext.original.warn = getOption("warn"))
  options(warn = warn)
  invisible(warn)

}

# nanonext - Limited scope exported functions ----------------------------------

#' @export
#'
.mirai_scm <- function() {

  identical(parent.env(parent.env(parent.frame())), getNamespace("mirai")) ||
    stop("this function is for package internal use only")
  .Call(rnng_scm)

}

# nanonext - Non-exported functions --------------------------------------------

match.arg2 <- function(choice, choices) {
  identical(choice, choices) && return(1L)
  index <- pmatch(choice[1L], choices, nomatch = 0L, duplicates.ok = TRUE)
  index || stop(sprintf("'%s' should be one of %s",
                        deparse(substitute(choice)), paste(choices, collapse = ", ")))
  index
}

